@import <Foundation/CPDictionary.j>

@implementation CPDictionaryTest : OJTestCase
{
    CPDictionary    string_dict;
    CPDictionary    json_dict;
    JSObject        json;
}

- (void)setUp
{
    json = {
        "key1": ['1', '2', '3'],
        "key2": "This is a string",
        "key3": {
            "another": "object"
        }
    }

    string_dict = [[CPDictionary alloc] initWithObjects:[@"1", @"2", @"This is a String", @"This is a String"] forKeys:[@"key1", @"key2", @"key3", @"key4"]];
    json_dict = [CPDictionary dictionaryWithJSObject:json recursively:YES];
}

- (void)testInitWithDictionary
{
    var dict = [[CPDictionary alloc] initWithObjects:[@"1", @"2"] forKeys:[@"key1", @"key2"]],
        new_dict = [[CPDictionary alloc] initWithDictionary:dict];

    [self assert:[new_dict objectForKey:@"key1"] equals:[dict objectForKey:@"key1"]];
    [self assert:[new_dict objectForKey:@"key2"] equals:[dict objectForKey:@"key2"]];

    var new_dict_cm = [CPDictionary dictionaryWithDictionary:dict];
    [self assert:[new_dict_cm objectForKey:@"key1"] equals:[dict objectForKey:@"key1"]];
    [self assert:[new_dict_cm objectForKey:@"key2"] equals:[dict objectForKey:@"key2"]];
}

- (void)testInitWithObjects
{
    var dict = [[CPDictionary alloc] initWithObjects:[@"1", @"2"] forKeys:[@"key1", @"key2"]];
    [self assert:[dict objectForKey:@"key1"] equals:@"1"];
    [self assert:[dict objectForKey:@"key2"] equals:@"2"];
    [self assert:[dict count] equals:2];

    var dict_cm = [CPDictionary dictionaryWithObjects:[@"1", @"2"] forKeys:[@"key1", @"key2"]];
    [self assert:[dict_cm objectForKey:@"key1"] equals:@"1"];
    [self assert:[dict_cm objectForKey:@"key2"] equals:@"2"];
    [self assert:[dict_cm count] equals:2];
}

- (void)testDictionaryWithObject
{
    var dict = [CPDictionary dictionaryWithObject:@"1" forKey:@"key1"];
    [self assert:[dict objectForKey:@"key1"] equals:@"1"];
    [self assert:[dict count] equals:1];
}

- (void)testDictionaryWithJSObjectRecursive
{
    var dict = [CPDictionary dictionaryWithJSObject:json recursively:YES];
    [self assert:[json_dict count] equals:3];
    [self assert:[[dict objectForKey:@"key3"] count] equals:1];
}

- (void)testDictionaryWithJSObjectRecursiveWithNull
{
    var json_with_nulls =     {
            "key1": ['1', '2', '3'],
            "key2": "This is a string",
            "key3": null
        },
        dict = [CPDictionary dictionaryWithJSObject:json_with_nulls recursively:YES];

    [self assert:3 equals:[dict count]];
    [self assert:[@"key1", @"key2", @"key3"] equals:[dict allKeys]];
    [self assert:[CPNull null] equals:[dict objectForKey:@"key3"]];
}

- (void)testDictionaryWithJSObjectNonRecursive
{
    var non_recursive_dict = [CPDictionary dictionaryWithJSObject:json recursively:NO];
}

- (void)testCopy
{
    var copy = [string_dict copy];
    [self assert:copy notSame:string_dict];
    [self assert:[copy objectForKey:@"key1"] equals:[string_dict objectForKey:@"key1"]];
}

- (void)testCount
{
    [self assert:[string_dict count] equals:4];
    [self assert:[json_dict count] equals:3];
}

- (void)testAllKeys
{
    [self assert:[string_dict allKeys] equals:[@"key4", @"key3", @"key2", @"key1"]];
    [self assert:[json_dict allKeys] equals:[@"key1", @"key2", @"key3"]];
}

- (void)testAllValues
{
    [self assert:[string_dict allValues] equals:[@"1", @"2", @"This is a String",  @"This is a String"]];
    // Had to get object from key to get test passing
    [self assert:[json_dict allValues] equals:[[json_dict objectForKey:@"key3"], @"This is a string", ['1', '2', '3']]];
}

- (void)testObjectForKey
{
    [self assert:[string_dict objectForKey:@"key1"] equals:"1"];
    [self assert:[json_dict objectForKey:@"key1"] equals:['1', '2', '3']];
}

- (void)testAllKeysForObject
{
    [self assert:[string_dict allKeysForObject:@"This is a String"] equals:[@"key4", @"key3"]];
}

- (void)testKeyEnumerator
{
    var dict = [[CPDictionary alloc] init];
    [self assertNull:[[dict keyEnumerator] nextObject]];
    [self assertNotNull:[[dict keyEnumerator] allObjects]];

    [self assertNotNull:[[string_dict keyEnumerator] nextObject]];
    [self assertNotNull:[[json_dict keyEnumerator] nextObject]];
    [self assertNotNull:[[string_dict keyEnumerator] allObjects]];
    [self assertNotNull:[[json_dict keyEnumerator] allObjects]];
}

- (void)testObjectEnumerator
{
    var dict = [[CPDictionary alloc] init];
    [self assertNull:[[dict objectEnumerator] nextObject]];
    [self assertNotNull:[[dict objectEnumerator] allObjects]];

    [self assertNotNull:[[string_dict objectEnumerator] nextObject]];
    [self assertNotNull:[[json_dict objectEnumerator] nextObject]];
    [self assertNotNull:[[string_dict objectEnumerator] allObjects]];
    [self assertNotNull:[[json_dict objectEnumerator] allObjects]];
}

- (void)testIsEqualToDictionary
{
    [self assertTrue:[string_dict isEqualToDictionary:[string_dict copy]]];
    [self assertTrue:[json_dict isEqualToDictionary:[json_dict copy]]];
    [self assertFalse:[json_dict isEqualToDictionary:[string_dict copy]]];
}

- (void)testRemoveAllObjects
{
    [string_dict removeAllObjects];
    [json_dict removeAllObjects];
    [self assert:[string_dict count] equals:0];
    [self assert:[json_dict count] equals:0];
}

- (void)testRemoveObjectForKey
{
    [string_dict removeObjectForKey:@"key1"];
    [json_dict removeObjectForKey:@"key1"];
    [self assert:[string_dict count] equals:3];
    [self assert:[json_dict count] equals:2];
}

- (void)testRemoveObjectsForKeys
{
    [string_dict removeObjectsForKeys:[@"key1"]];
    [json_dict removeObjectsForKeys:[@"key1", @"key2"]];
    [self assert:[string_dict count] equals:3];
    [self assert:[json_dict count] equals:1];
}

- (void)testSetObjectForKey
{
    var dict = [[CPDictionary alloc] init];
    [dict setObject:@"setObjectForKey test" forKey:@"key1"];
    [self assert:[dict objectForKey:@"key1"] equals:@"setObjectForKey test"];
    [self assert:[dict count] equals:1];
}

- (void)testAddEntriesFromDictionary
{
    var dict = [[CPDictionary alloc] initWithObjects:[@"1", @"2"] forKeys:[@"key5", @"key6"]];
    [string_dict addEntriesFromDictionary:dict]
    [json_dict addEntriesFromDictionary:dict]
    [self assert:[string_dict count] equals:6];
    [self assert:[json_dict count] equals:5];
}

- (void)testDictionaryWithFalsyValues
{
    var dict = [[CPDictionary alloc] initWithObjects:["", 0, [CPNull null]] forKeys:["1", "2", "3"]];
    [self assertTrue:[dict containsKey:"1"]];
    [self assertTrue:[dict containsKey:"2"]];
    [self assertTrue:[dict containsKey:"3"]];
    [self assertFalse:[dict containsKey:"4"]];
}

- (void)testThrowsOnNilValue
{
    [self assertThrows:function(){
        var dict = [[CPDictionary alloc] initWithObjects:[1, nil] forKeys:["1", "2"]];
        [self assertFalse:dict];
    }];
}

- (void)testKeysSortedByValueUsingSelector
{
    var numberDictionary = [CPDictionary dictionaryWithJSObject:{
            key1: 5,
            key2: 1,
            key3: 4,
            key4: 2,
            key5: 3
        }];

    var expected = [@"key2", @"key4", @"key5", @"key3", @"key1"],
        result = [numberDictionary keysSortedByValueUsingSelector:@selector(compare:)];

    [self assert:expected equals:result];

    var stringDictionary = [CPDictionary dictionaryWithJSObject:{
            a: @"Z", b: @"y", c: @"X", d: @"W",
            e: @"V", f: @"u", g: @"T", h: @"s",
            i: @"R", j: @"q", k: @"P", l: @"o"
        }];

    expected = [@"l", @"k", @"j", @"i", @"h", @"g", @"f", @"e", @"d", @"c", @"b", @"a"];
    result = [stringDictionary keysSortedByValueUsingSelector:@selector(caseInsensitiveCompare:)];

    [self assert:expected equals:result];

    expected = [@"k", @"i", @"g", @"e", @"d", @"c", @"a", @"l", @"j", @"h", @"f", @"b"];
    result = [stringDictionary keysSortedByValueUsingSelector:@selector(compare:)];
    [self assert:expected equals:result];
}

@end
