﻿
/*! Copyright (c) Microsoft Corporation.  All Rights Reserved. Licensed under the MIT License. See License.txt in the project root for license information. */
(function () {

    var globalObject = 
        typeof window !== 'undefined' ? window :
        typeof self !== 'undefined' ? self :
        typeof global !== 'undefined' ? global :
        {};
    (function (factory) {
        if (typeof define === 'function' && define.amd) {
            // amd
            define(["./base"], factory);
        } else {
            globalObject.msWriteProfilerMark && msWriteProfilerMark('WinJS.4.0 4.0.1.winjs.2015.6.10 ui.js,StartTM');
            if (typeof module !== 'undefined') {
                // CommonJS
                factory(require("./base"));
            } else {
                // No module system
                factory(globalObject.WinJS);
            }
            globalObject.msWriteProfilerMark && msWriteProfilerMark('WinJS.4.0 4.0.1.winjs.2015.6.10 ui.js,StopTM');
        }
    }(function (WinJS) {


var require = WinJS.Utilities._require;
var define = WinJS.Utilities._define;

// Copyright (c) Microsoft Corporation.  All Rights Reserved. Licensed under the MIT License. See License.txt in the project root for license information.
// Virtualized Data Source
define('WinJS/VirtualizedDataSource/_VirtualizedDataSourceImpl',[
    'exports',
    '../Core/_Global',
    '../Core/_Base',
    '../Core/_BaseUtils',
    '../Core/_ErrorFromName',
    '../Core/_Events',
    '../Core/_Log',
    '../Core/_Resources',
    '../Core/_WriteProfilerMark',
    '../Promise',
    '../Scheduler',
    '../_Signal',
    '../Utilities/_UI'
    ], function listDataSourceInit(exports, _Global, _Base, _BaseUtils, _ErrorFromName, _Events, _Log, _Resources, _WriteProfilerMark, Promise, Scheduler, _Signal, _UI) {
    "use strict";

    _Base.Namespace._moduleDefine(exports, "WinJS.UI", {

        VirtualizedDataSource: _Base.Namespace._lazy(function () {
            var MAX_BEGINREFRESH_COUNT = 100;
            var uniqueID = 1;

            var DataSourceStatus = _UI.DataSourceStatus,
            CountResult = _UI.CountResult,
            FetchError = _UI.FetchError,
            EditError = _UI.EditError;

            // Private statics

            var strings = {
                get listDataAdapterIsInvalid() { return "Invalid argument: listDataAdapter must be an object or an array."; },
                get indexIsInvalid() { return "Invalid argument: index must be a non-negative integer."; },
                get keyIsInvalid() { return "Invalid argument: key must be a string."; },
                get invalidItemReturned() { return "Error: data adapter returned item that is not an object."; },
                get invalidKeyReturned() { return "Error: data adapter returned item with undefined or null key."; },
                get invalidIndexReturned() { return "Error: data adapter should return undefined, null or a non-negative integer for the index."; },
                get invalidCountReturned() { return "Error: data adapter should return undefined, null, CountResult.unknown, or a non-negative integer for the count."; },
                get invalidRequestedCountReturned() { return "Error: data adapter should return CountResult.unknown, CountResult.failure, or a non-negative integer for the count."; },
                get refreshCycleIdentified() { return "refresh cycle found, likely data inconsistency"; },
            };

            var statusChangedEvent = "statuschanged";

            function _baseDataSourceConstructor(listDataAdapter, options) {
                /// <signature helpKeyword="WinJS.UI.VirtualizedDataSource._baseDataSourceConstructor">
                /// <summary locid="WinJS.UI.VirtualizedDataSource._baseDataSourceConstructor">
                /// Initializes the VirtualizedDataSource base class of a custom data source.
                /// </summary>
                /// <param name="listDataAdapter" type="IListDataAdapter" locid="WinJS.UI.VirtualizedDataSource._baseDataSourceConstructor_p:itemIndex">
                /// An object that implements IListDataAdapter and supplies data to the VirtualizedDataSource.
                /// </param>
                /// <param name="options" optional="true" type="Object" locid="WinJS.UI.VirtualizedDataSource._baseDataSourceConstructor_p:options">
                /// An object that contains properties that specify additonal options for the VirtualizedDataSource:
                ///
                /// cacheSize
                /// A Number that specifies minimum number of unrequested items to cache in case they are requested.
                ///
                /// The options parameter is optional.
                /// </param>
                /// </signature>

                // Private members

                /*jshint validthis: true */

                var listDataNotificationHandler,
                    cacheSize,
                    status,
                    statusPending,
                    statusChangePosted,
                    bindingMap,
                    nextListBindingID,
                    nextHandle,
                    nextListenerID,
                    getCountPromise,
                    resultsProcessed,
                    beginEditsCalled,
                    editsInProgress,
                    firstEditInProgress,
                    editQueue,
                    editsQueued,
                    synchronousEdit,
                    waitForRefresh,
                    dataNotificationsInProgress,
                    countDelta,
                    indexUpdateDeferred,
                    nextTempKey,
                    currentRefreshID,
                    fetchesPosted,
                    nextFetchID,
                    fetchesInProgress,
                    fetchCompleteCallbacks,
                    startMarker,
                    endMarker,
                    knownCount,
                    slotsStart,
                    slotListEnd,
                    slotsEnd,
                    handleMap,
                    keyMap,
                    indexMap,
                    releasedSlots,
                    lastSlotReleased,
                    reduceReleasedSlotCountPosted,
                    refreshRequested,
                    refreshInProgress,
                    refreshSignal,
                    refreshFetchesInProgress,
                    refreshItemsFetched,
                    refreshCount,
                    refreshStart,
                    refreshEnd,
                    keyFetchIDs,
                    refreshKeyMap,
                    refreshIndexMap,
                    deletedKeys,
                    synchronousProgress,
                    reentrantContinue,
                    synchronousRefresh,
                    reentrantRefresh;

                var beginRefreshCount = 0,
                    refreshHistory = new Array(100),
                    refreshHistoryPos = -1;

                var itemsFromKey,
                    itemsFromIndex,
                    itemsFromStart,
                    itemsFromEnd,
                    itemsFromDescription;

                if (listDataAdapter.itemsFromKey) {
                    itemsFromKey = function (fetchID, key, countBefore, countAfter, hints) {
                        var perfID = "fetchItemsFromKey id=" + fetchID + " key=" + key + " countBefore=" + countBefore + " countAfter=" + countAfter;
                        profilerMarkStart(perfID);
                        refreshHistory[++refreshHistoryPos % refreshHistory.length] = { kind: "itemsFromKey", key: key, countBefore: countBefore, countAfter: countAfter };
                        var result = listDataAdapter.itemsFromKey(key, countBefore, countAfter, hints);
                        profilerMarkEnd(perfID);
                        return result;
                    };
                }
                if (listDataAdapter.itemsFromIndex) {
                    itemsFromIndex = function (fetchID, index, countBefore, countAfter) {
                        var perfID = "fetchItemsFromIndex id=" + fetchID + " index=" + index + " countBefore=" + countBefore + " countAfter=" + countAfter;
                        profilerMarkStart(perfID);
                        refreshHistory[++refreshHistoryPos % refreshHistory.length] = { kind: "itemsFromIndex", index: index, countBefore: countBefore, countAfter: countAfter };
                        var result = listDataAdapter.itemsFromIndex(index, countBefore, countAfter);
                        profilerMarkEnd(perfID);
                        return result;
                    };
                }
                if (listDataAdapter.itemsFromStart) {
                    itemsFromStart = function (fetchID, count) {
                        var perfID = "fetchItemsFromStart id=" + fetchID + " count=" + count;
                        profilerMarkStart(perfID);
                        refreshHistory[++refreshHistoryPos % refreshHistory.length] = { kind: "itemsFromStart", count: count };
                        var result = listDataAdapter.itemsFromStart(count);
                        profilerMarkEnd(perfID);
                        return result;
                    };
                }
                if (listDataAdapter.itemsFromEnd) {
                    itemsFromEnd = function (fetchID, count) {
                        var perfID = "fetchItemsFromEnd id=" + fetchID + " count=" + count;
                        profilerMarkStart(perfID);
                        refreshHistory[++refreshHistoryPos % refreshHistory.length] = { kind: "itemsFromEnd", count: count };
                        var result = listDataAdapter.itemsFromEnd(count);
                        profilerMarkEnd(perfID);
                        return result;
                    };
                }
                if (listDataAdapter.itemsFromDescription) {
                    itemsFromDescription = function (fetchID, description, countBefore, countAfter) {
                        var perfID = "fetchItemsFromDescription id=" + fetchID + " desc=" + description + " countBefore=" + countBefore + " countAfter=" + countAfter;
                        profilerMarkStart(perfID);
                        refreshHistory[++refreshHistoryPos % refreshHistory.length] = { kind: "itemsFromDescription", description: description, countBefore: countBefore, countAfter: countAfter };
                        var result = listDataAdapter.itemsFromDescription(description, countBefore, countAfter);
                        profilerMarkEnd(perfID);
                        return result;
                    };
                }

                var dataSourceID = ++uniqueID;

                function profilerMarkStart(text) {
                    var message = "WinJS.UI.VirtualizedDataSource:" + dataSourceID + ":" + text + ",StartTM";
                    _WriteProfilerMark(message);
                    _Log.log && _Log.log(message, "winjs vds", "perf");
                }
                function profilerMarkEnd(text) {
                    var message = "WinJS.UI.VirtualizedDataSource:" + dataSourceID + ":" + text + ",StopTM";
                    _WriteProfilerMark(message);
                    _Log.log && _Log.log(message, "winjs vds", "perf");
                }

                function isNonNegativeNumber(n) {
                    return (typeof n === "number") && n >= 0;
                }

                function isNonNegativeInteger(n) {
                    return isNonNegativeNumber(n) && n === Math.floor(n);
                }

                function validateIndexReturned(index) {
                    // Ensure that index is always undefined or a non-negative integer
                    if (index === null) {
                        index = undefined;
                    } else if (index !== undefined && !isNonNegativeInteger(index)) {
                        throw new _ErrorFromName("WinJS.UI.ListDataSource.InvalidIndexReturned", strings.invalidIndexReturned);
                    }

                    return index;
                }

                function validateCountReturned(count) {
                    // Ensure that count is always undefined or a non-negative integer
                    if (count === null) {
                        count = undefined;
                    } else if (count !== undefined && !isNonNegativeInteger(count) && count !== CountResult.unknown) {
                        throw new _ErrorFromName("WinJS.UI.ListDataSource.InvalidCountReturned", strings.invalidCountReturned);
                    }

                    return count;
                }

                // Slot List

                function createSlot() {
                    var handle = (nextHandle++).toString(),
                        slotNew = {
                            handle: handle,
                            item: null,
                            itemNew: null,
                            fetchListeners: null,
                            cursorCount: 0,
                            bindingMap: null
                        };

                    // Deliberately not initialized:
                    //   - directFetchListeners

                    handleMap[handle] = slotNew;

                    return slotNew;
                }

                function createPrimarySlot() {
                    return createSlot();
                }

                function insertSlot(slot, slotNext) {
                    slot.prev = slotNext.prev;
                    slot.next = slotNext;

                    slot.prev.next = slot;
                    slotNext.prev = slot;
                }

                function removeSlot(slot) {
                    if (slot.lastInSequence) {
                        delete slot.lastInSequence;
                        slot.prev.lastInSequence = true;
                    }
                    if (slot.firstInSequence) {
                        delete slot.firstInSequence;
                        slot.next.firstInSequence = true;
                    }
                    slot.prev.next = slot.next;
                    slot.next.prev = slot.prev;
                }

                function sequenceStart(slot) {
                    while (!slot.firstInSequence) {
                        slot = slot.prev;
                    }

                    return slot;
                }

                function sequenceEnd(slot) {
                    while (!slot.lastInSequence) {
                        slot = slot.next;
                    }

                    return slot;
                }

                // Does a little careful surgery to the slot sequence from slotFirst to slotLast before slotNext
                function moveSequenceBefore(slotNext, slotFirst, slotLast) {
                    slotFirst.prev.next = slotLast.next;
                    slotLast.next.prev = slotFirst.prev;

                    slotFirst.prev = slotNext.prev;
                    slotLast.next = slotNext;

                    slotFirst.prev.next = slotFirst;
                    slotNext.prev = slotLast;

                    return true;
                }

                // Does a little careful surgery to the slot sequence from slotFirst to slotLast after slotPrev
                function moveSequenceAfter(slotPrev, slotFirst, slotLast) {
                    slotFirst.prev.next = slotLast.next;
                    slotLast.next.prev = slotFirst.prev;

                    slotFirst.prev = slotPrev;
                    slotLast.next = slotPrev.next;

                    slotPrev.next = slotFirst;
                    slotLast.next.prev = slotLast;

                    return true;
                }

                function mergeSequences(slotPrev) {
                    delete slotPrev.lastInSequence;
                    delete slotPrev.next.firstInSequence;
                }

                function splitSequence(slotPrev) {
                    var slotNext = slotPrev.next;

                    slotPrev.lastInSequence = true;
                    slotNext.firstInSequence = true;

                    if (slotNext === slotListEnd) {
                        // Clear slotListEnd's index, as that's now unknown
                        changeSlotIndex(slotListEnd, undefined);
                    }
                }

                // Inserts a slot in the middle of a sequence or between sequences.  If the latter, mergeWithPrev and mergeWithNext
                // parameters specify whether to merge the slot with the previous sequence, or next, or neither.
                function insertAndMergeSlot(slot, slotNext, mergeWithPrev, mergeWithNext) {
                    insertSlot(slot, slotNext);

                    var slotPrev = slot.prev;

                    if (slotPrev.lastInSequence) {
                        if (mergeWithPrev) {
                            delete slotPrev.lastInSequence;
                        } else {
                            slot.firstInSequence = true;
                        }

                        if (mergeWithNext) {
                            delete slotNext.firstInSequence;
                        } else {
                            slot.lastInSequence = true;
                        }
                    }
                }

                // Keys and Indices

                function setSlotKey(slot, key) {
                    slot.key = key;

                    // Add the slot to the keyMap, so it is possible to quickly find the slot given its key
                    keyMap[slot.key] = slot;
                }

                function setSlotIndex(slot, index, indexMapForSlot) {
                    // Tolerate NaN, so clients can pass (undefined - 1) or (undefined + 1)
                    if (+index === index) {
                        slot.index = index;

                        // Add the slot to the indexMap, so it is possible to quickly find the slot given its index
                        indexMapForSlot[index] = slot;

                        if (!indexUpdateDeferred) {
                            // See if any sequences should be merged
                            if (slot.firstInSequence && slot.prev && slot.prev.index === index - 1) {
                                mergeSequences(slot.prev);
                            }
                            if (slot.lastInSequence && slot.next && slot.next.index === index + 1) {
                                mergeSequences(slot);
                            }
                        }
                    }
                }

                // Creates a new slot and adds it to the slot list before slotNext
                function createAndAddSlot(slotNext, indexMapForSlot) {
                    var slotNew = (indexMapForSlot === indexMap ? createPrimarySlot() : createSlot());

                    insertSlot(slotNew, slotNext);

                    return slotNew;
                }

                function createSlotSequence(slotNext, index, indexMapForSlot) {
                    var slotNew = createAndAddSlot(slotNext, indexMapForSlot);

                    slotNew.firstInSequence = true;
                    slotNew.lastInSequence = true;

                    setSlotIndex(slotNew, index, indexMapForSlot);

                    return slotNew;
                }

                function createPrimarySlotSequence(slotNext, index) {
                    return createSlotSequence(slotNext, index, indexMap);
                }

                function addSlotBefore(slotNext, indexMapForSlot) {
                    var slotNew = createAndAddSlot(slotNext, indexMapForSlot);
                    delete slotNext.firstInSequence;

                    // See if we've bumped into the previous sequence
                    if (slotNew.prev.index === slotNew.index - 1) {
                        delete slotNew.prev.lastInSequence;
                    } else {
                        slotNew.firstInSequence = true;
                    }

                    setSlotIndex(slotNew, slotNext.index - 1, indexMapForSlot);

                    return slotNew;
                }

                function addSlotAfter(slotPrev, indexMapForSlot) {
                    var slotNew = createAndAddSlot(slotPrev.next, indexMapForSlot);
                    delete slotPrev.lastInSequence;

                    // See if we've bumped into the next sequence
                    if (slotNew.next.index === slotNew.index + 1) {
                        delete slotNew.next.firstInSequence;
                    } else {
                        slotNew.lastInSequence = true;
                    }

                    setSlotIndex(slotNew, slotPrev.index + 1, indexMapForSlot);

                    return slotNew;
                }

                function reinsertSlot(slot, slotNext, mergeWithPrev, mergeWithNext) {
                    insertAndMergeSlot(slot, slotNext, mergeWithPrev, mergeWithNext);
                    keyMap[slot.key] = slot;
                    if (slot.index !== undefined) {
                        indexMap[slot.index] = slot;
                    }
                }

                function removeSlotPermanently(slot) {
                    removeSlot(slot);

                    if (slot.key) {
                        delete keyMap[slot.key];
                    }
                    if (slot.index !== undefined && indexMap[slot.index] === slot) {
                        delete indexMap[slot.index];
                    }

                    var bindingMap = slot.bindingMap;
                    for (var listBindingID in bindingMap) {
                        var handle = bindingMap[listBindingID].handle;
                        if (handle && handleMap[handle] === slot) {
                            delete handleMap[handle];
                        }
                    }

                    // Invalidating the slot's handle marks it as deleted
                    if (handleMap[slot.handle] === slot) {
                        delete handleMap[slot.handle];
                    }
                }

                function slotPermanentlyRemoved(slot) {
                    return !handleMap[slot.handle];
                }

                function successorFromIndex(index, indexMapForSlot, listStart, listEnd, skipPreviousIndex) {
                    // Try the previous index
                    var slotNext = (skipPreviousIndex ? null : indexMapForSlot[index - 1]);
                    if (slotNext && (slotNext.next !== listEnd || listEnd.firstInSequence)) {
                        // We want the successor
                        slotNext = slotNext.next;
                    } else {
                        // Try the next index
                        slotNext = indexMapForSlot[index + 1];
                        if (!slotNext) {
                            // Resort to a linear search
                            slotNext = listStart.next;
                            var lastSequenceStart;
                            while (true) {
                                if (slotNext.firstInSequence) {
                                    lastSequenceStart = slotNext;
                                }

                                if (!(index >= slotNext.index) || slotNext === listEnd) {
                                    break;
                                }

                                slotNext = slotNext.next;
                            }

                            if (slotNext === listEnd && !listEnd.firstInSequence) {
                                // Return the last insertion point between sequences, or undefined if none
                                slotNext = (lastSequenceStart && lastSequenceStart.index === undefined ? lastSequenceStart : undefined);
                            }
                        }
                    }

                    return slotNext;
                }

                // Slot Items

                function isPlaceholder(slot) {
                    return !slot.item && !slot.itemNew && slot !== slotListEnd;
                }

                function defineHandleProperty(item, handle) {
                    Object.defineProperty(item, "handle", {
                        value: handle,
                        writable: false,
                        enumerable: false,
                        configurable: true
                    });
                }

                function defineCommonItemProperties(item, slot, handle) {
                    defineHandleProperty(item, handle);

                    Object.defineProperty(item, "index", {
                        get: function () {
                            while (slot.slotMergedWith) {
                                slot = slot.slotMergedWith;
                            }

                            return slot.index;
                        },
                        enumerable: false,
                        configurable: true
                    });
                }

                function validateData(data) {
                    if (data === undefined) {
                        return data;
                    } else {
                        // Convert the data object to JSON to enforce the constraints we want.  For example, we don't want
                        // functions, arrays with extra properties, DOM objects, cyclic or acyclic graphs, or undefined values.
                        var dataValidated = JSON.stringify(data);

                        if (dataValidated === undefined) {
                            throw new _ErrorFromName("WinJS.UI.ListDataSource.ObjectIsNotValidJson", strings.objectIsNotValidJson);
                        }

                        return dataValidated;
                    }
                }

                function itemSignature(item) {
                    return (
                        listDataAdapter.itemSignature ?
                            listDataAdapter.itemSignature(item.data) :
                            validateData(item.data)
                    );
                }

                function prepareSlotItem(slot) {
                    var item = slot.itemNew;
                    slot.itemNew = null;

                    if (item) {
                        item = Object.create(item);
                        defineCommonItemProperties(item, slot, slot.handle);

                        if (!listDataAdapter.compareByIdentity) {
                            // Store the item signature or a stringified copy of the data for comparison later
                            slot.signature = itemSignature(item);
                        }
                    }

                    slot.item = item;

                    delete slot.indexRequested;
                    delete slot.keyRequested;
                }

                // Slot Caching

                function slotRetained(slot) {
                    return slot.bindingMap || slot.cursorCount > 0;
                }

                function slotRequested(slot) {
                    return slotRetained(slot) || slot.fetchListeners || slot.directFetchListeners;
                }

                function slotLive(slot) {
                    return slotRequested(slot) || (!slot.firstInSequence && slotRetained(slot.prev)) || (!slot.lastInSequence && slotRetained(slot.next)) ||
                        (!itemsFromIndex && (
                            (!slot.firstInSequence && slot.prev !== slotsStart && !(slot.prev.item || slot.prev.itemNew)) |
                            (!slot.lastInSequence && slot.next !== slotListEnd && !(slot.next.item || slot.next.itemNew))
                        ));
                }

                function deleteUnnecessarySlot(slot) {
                    splitSequence(slot);
                    removeSlotPermanently(slot);
                }

                function reduceReleasedSlotCount() {
                    // Must not release slots while edits are queued, as undo queue might refer to them
                    if (!editsQueued) {
                        // If lastSlotReleased is no longer valid, use the end of the list instead
                        if (!lastSlotReleased || slotPermanentlyRemoved(lastSlotReleased)) {
                            lastSlotReleased = slotListEnd.prev;
                        }

                        // Now use the simple heuristic of walking outwards in both directions from lastSlotReleased until the
                        // desired cache size is reached, then removing everything else.
                        var slotPrev = lastSlotReleased.prev,
                            slotNext = lastSlotReleased.next,
                            releasedSlotsFound = 0;

                        var considerDeletingSlot = function (slotToDelete) {
                            if (slotToDelete !== slotListEnd && !slotLive(slotToDelete)) {
                                if (releasedSlotsFound <= cacheSize) {
                                    releasedSlotsFound++;
                                } else {
                                    deleteUnnecessarySlot(slotToDelete);
                                }
                            }
                        };

                        while (slotPrev || slotNext) {
                            if (slotPrev) {
                                var slotPrevToDelete = slotPrev;
                                slotPrev = slotPrevToDelete.prev;
                                if (slotPrevToDelete !== slotsStart) {
                                    considerDeletingSlot(slotPrevToDelete);
                                }
                            }
                            if (slotNext) {
                                var slotNextToDelete = slotNext;
                                slotNext = slotNextToDelete.next;
                                if (slotNextToDelete !== slotsEnd) {
                                    considerDeletingSlot(slotNextToDelete);
                                }
                            }
                        }

                        // Reset the count to zero, so this method is only called periodically
                        releasedSlots = 0;
                    }
                }

                function releaseSlotIfUnrequested(slot) {
                    if (!slotRequested(slot)) {

                        releasedSlots++;

                        // Must not release slots while edits are queued, as undo queue might refer to them.  If a refresh is in
                        // progress, retain all slots, just in case the user re-requests some of them before the refresh completes.
                        if (!editsQueued && !refreshInProgress) {
                            // Track which slot was released most recently
                            lastSlotReleased = slot;

                            // See if the number of released slots has exceeded the cache size.  In practice there will be more
                            // live slots than retained slots, so this is just a heuristic to periodically shrink the cache.
                            if (releasedSlots > cacheSize && !reduceReleasedSlotCountPosted) {
                                reduceReleasedSlotCountPosted = true;
                                Scheduler.schedule(function VDS_async_releaseSlotIfUnrequested() {
                                    reduceReleasedSlotCountPosted = false;
                                    reduceReleasedSlotCount();
                                }, Scheduler.Priority.idle, null, "WinJS.UI.VirtualizedDataSource.releaseSlotIfUnrequested");
                            }
                        }
                    }
                }

                // Notifications

                function forEachBindingRecord(callback) {
                    for (var listBindingID in bindingMap) {
                        callback(bindingMap[listBindingID]);
                    }
                }

                function forEachBindingRecordOfSlot(slot, callback) {
                    for (var listBindingID in slot.bindingMap) {
                        callback(slot.bindingMap[listBindingID].bindingRecord, listBindingID);
                    }
                }

                function handlerToNotify(bindingRecord) {
                    if (!bindingRecord.notificationsSent) {
                        bindingRecord.notificationsSent = true;

                        if (bindingRecord.notificationHandler.beginNotifications) {
                            bindingRecord.notificationHandler.beginNotifications();
                        }
                    }
                    return bindingRecord.notificationHandler;
                }

                function finishNotifications() {
                    if (!editsInProgress && !dataNotificationsInProgress) {
                        forEachBindingRecord(function (bindingRecord) {
                            if (bindingRecord.notificationsSent) {
                                bindingRecord.notificationsSent = false;

                                if (bindingRecord.notificationHandler.endNotifications) {
                                    bindingRecord.notificationHandler.endNotifications();
                                }
                            }
                        });
                    }
                }

                function handleForBinding(slot, listBindingID) {
                    var bindingMap = slot.bindingMap;
                    if (bindingMap) {
                        var slotBinding = bindingMap[listBindingID];
                        if (slotBinding) {
                            var handle = slotBinding.handle;
                            if (handle) {
                                return handle;
                            }
                        }
                    }
                    return slot.handle;
                }

                function itemForBinding(item, handle) {
                    if (item && item.handle !== handle) {
                        item = Object.create(item);
                        defineHandleProperty(item, handle);
                    }
                    return item;
                }

                function changeCount(count) {
                    var oldCount = knownCount;
                    knownCount = count;

                    forEachBindingRecord(function (bindingRecord) {
                        if (bindingRecord.notificationHandler && bindingRecord.notificationHandler.countChanged) {
                            handlerToNotify(bindingRecord).countChanged(knownCount, oldCount);
                        }
                    });
                }

                function sendIndexChangedNotifications(slot, indexOld) {
                    forEachBindingRecordOfSlot(slot, function (bindingRecord, listBindingID) {
                        if (bindingRecord.notificationHandler.indexChanged) {
                            handlerToNotify(bindingRecord).indexChanged(handleForBinding(slot, listBindingID), slot.index, indexOld);
                        }
                    });
                }

                function changeSlotIndex(slot, index) {
                    var indexOld = slot.index;

                    if (indexOld !== undefined && indexMap[indexOld] === slot) {
                        // Remove the slot's old index from the indexMap
                        delete indexMap[indexOld];
                    }

                    // Tolerate NaN, so clients can pass (undefined - 1) or (undefined + 1)
                    if (+index === index) {
                        setSlotIndex(slot, index, indexMap);
                    } else if (+indexOld === indexOld) {
                        delete slot.index;
                    } else {
                        // If neither the new index or the old index is defined then there was no index changed.
                        return;
                    }

                    sendIndexChangedNotifications(slot, indexOld);
                }

                function insertionNotificationRecipients(slot, slotPrev, slotNext, mergeWithPrev, mergeWithNext) {
                    var bindingMapRecipients = {};

                    // Start with the intersection of the bindings for the two adjacent slots
                    if ((mergeWithPrev || !slotPrev.lastInSequence) && (mergeWithNext || !slotNext.firstInSequence)) {
                        if (slotPrev === slotsStart) {
                            if (slotNext === slotListEnd) {
                                // Special case: if the list was empty, broadcast the insertion to all ListBindings with
                                // notification handlers.
                                for (var listBindingID in bindingMap) {
                                    bindingMapRecipients[listBindingID] = bindingMap[listBindingID];
                                }
                            } else {
                                // Include every binding on the next slot
                                for (var listBindingID in slotNext.bindingMap) {
                                    bindingMapRecipients[listBindingID] = bindingMap[listBindingID];
                                }
                            }
                        } else if (slotNext === slotListEnd || slotNext.bindingMap) {
                            for (var listBindingID in slotPrev.bindingMap) {
                                if (slotNext === slotListEnd || slotNext.bindingMap[listBindingID]) {
                                    bindingMapRecipients[listBindingID] = bindingMap[listBindingID];
                                }
                            }
                        }
                    }

                    // Use the union of that result with the bindings for the slot being inserted
                    for (var listBindingID in slot.bindingMap) {
                        bindingMapRecipients[listBindingID] = bindingMap[listBindingID];
                    }

                    return bindingMapRecipients;
                }

                function sendInsertedNotification(slot) {
                    var slotPrev = slot.prev,
                        slotNext = slot.next,
                        bindingMapRecipients = insertionNotificationRecipients(slot, slotPrev, slotNext),
                        listBindingID;

                    for (listBindingID in bindingMapRecipients) {
                        var bindingRecord = bindingMapRecipients[listBindingID];
                        if (bindingRecord.notificationHandler) {
                            handlerToNotify(bindingRecord).inserted(bindingRecord.itemPromiseFromKnownSlot(slot),
                                slotPrev.lastInSequence || slotPrev === slotsStart ? null : handleForBinding(slotPrev, listBindingID),
                                slotNext.firstInSequence || slotNext === slotListEnd ? null : handleForBinding(slotNext, listBindingID)
                            );
                        }
                    }
                }

                function changeSlot(slot) {
                    var itemOld = slot.item;
                    prepareSlotItem(slot);

                    forEachBindingRecordOfSlot(slot, function (bindingRecord, listBindingID) {
                        var handle = handleForBinding(slot, listBindingID);
                        handlerToNotify(bindingRecord).changed(itemForBinding(slot.item, handle), itemForBinding(itemOld, handle));
                    });
                }

                function moveSlot(slot, slotMoveBefore, mergeWithPrev, mergeWithNext, skipNotifications) {
                    var slotMoveAfter = slotMoveBefore.prev,
                        listBindingID;

                    // If the slot is being moved before or after itself, adjust slotMoveAfter or slotMoveBefore accordingly. If
                    // nothing is going to change in the slot list, don't send a notification.
                    if (slotMoveBefore === slot) {
                        if (!slot.firstInSequence || !mergeWithPrev) {
                            return;
                        }
                        slotMoveBefore = slot.next;
                    } else if (slotMoveAfter === slot) {
                        if (!slot.lastInSequence || !mergeWithNext) {
                            return;
                        }
                        slotMoveAfter = slot.prev;
                    }

                    if (!skipNotifications) {
                        // Determine which bindings to notify

                        var bindingMapRecipients = insertionNotificationRecipients(slot, slotMoveAfter, slotMoveBefore, mergeWithPrev, mergeWithNext);

                        // Send the notification before the move
                        for (listBindingID in bindingMapRecipients) {
                            var bindingRecord = bindingMapRecipients[listBindingID];
                            handlerToNotify(bindingRecord).moved(bindingRecord.itemPromiseFromKnownSlot(slot),
                                ((slotMoveAfter.lastInSequence || slotMoveAfter === slot.prev) && !mergeWithPrev) || slotMoveAfter === slotsStart ? null : handleForBinding(slotMoveAfter, listBindingID),
                                ((slotMoveBefore.firstInSequence || slotMoveBefore === slot.next) && !mergeWithNext) || slotMoveBefore === slotListEnd ? null : handleForBinding(slotMoveBefore, listBindingID)
                            );
                        }

                        // If a ListBinding cursor is at the slot that's moving, adjust the cursor
                        forEachBindingRecord(function (bindingRecord) {
                            bindingRecord.adjustCurrentSlot(slot);
                        });
                    }

                    removeSlot(slot);
                    insertAndMergeSlot(slot, slotMoveBefore, mergeWithPrev, mergeWithNext);
                }

                function deleteSlot(slot, mirage) {
                    completeFetchPromises(slot, true);

                    forEachBindingRecordOfSlot(slot, function (bindingRecord, listBindingID) {
                        handlerToNotify(bindingRecord).removed(handleForBinding(slot, listBindingID), mirage);
                    });

                    // If a ListBinding cursor is at the slot that's being removed, adjust the cursor
                    forEachBindingRecord(function (bindingRecord) {
                        bindingRecord.adjustCurrentSlot(slot);
                    });

                    removeSlotPermanently(slot);
                }

                function deleteMirageSequence(slot) {
                    // Remove the slots in order

                    while (!slot.firstInSequence) {
                        slot = slot.prev;
                    }

                    var last;
                    do {
                        last = slot.lastInSequence;

                        var slotNext = slot.next;
                        deleteSlot(slot, true);
                        slot = slotNext;
                    } while (!last);
                }

                // Deferred Index Updates

                // Returns the index of the slot taking into account any outstanding index updates
                function adjustedIndex(slot) {
                    var undefinedIndex;

                    if (!slot) {
                        return undefinedIndex;
                    }

                    var delta = 0;
                    while (!slot.firstInSequence) {
                        delta++;
                        slot = slot.prev;
                    }

                    return (
                        typeof slot.indexNew === "number" ?
                            slot.indexNew + delta :
                        typeof slot.index === "number" ?
                            slot.index + delta :
                            undefinedIndex
                    );
                }

                // Updates the new index of the first slot in each sequence after the given slot
                function updateNewIndicesAfterSlot(slot, indexDelta) {
                    // Adjust all the indexNews after this slot
                    for (slot = slot.next; slot; slot = slot.next) {
                        if (slot.firstInSequence) {
                            var indexNew = (slot.indexNew !== undefined ? slot.indexNew : slot.index);
                            if (indexNew !== undefined) {
                                slot.indexNew = indexNew + indexDelta;
                            }
                        }
                    }

                    // Adjust the overall count
                    countDelta += indexDelta;

                    indexUpdateDeferred = true;

                    // Increment currentRefreshID so any outstanding fetches don't cause trouble.  If a refresh is in progress,
                    // restart it (which will also increment currentRefreshID).
                    if (refreshInProgress) {
                        beginRefresh();
                    } else {
                        currentRefreshID++;
                    }
                }

                // Updates the new index of the given slot if necessary, and all subsequent new indices
                function updateNewIndices(slot, indexDelta) {
                    // If this slot is at the start of a sequence, transfer the indexNew
                    if (slot.firstInSequence) {
                        var indexNew;

                        if (indexDelta < 0) {
                            // The given slot is about to be removed
                            indexNew = slot.indexNew;
                            if (indexNew !== undefined) {
                                delete slot.indexNew;
                            } else {
                                indexNew = slot.index;
                            }

                            if (!slot.lastInSequence) {
                                // Update the next slot now
                                slot = slot.next;
                                if (indexNew !== undefined) {
                                    slot.indexNew = indexNew;
                                }
                            }
                        } else {
                            // The given slot was just inserted
                            if (!slot.lastInSequence) {
                                var slotNext = slot.next;

                                indexNew = slotNext.indexNew;
                                if (indexNew !== undefined) {
                                    delete slotNext.indexNew;
                                } else {
                                    indexNew = slotNext.index;
                                }

                                if (indexNew !== undefined) {
                                    slot.indexNew = indexNew;
                                }
                            }
                        }
                    }

                    updateNewIndicesAfterSlot(slot, indexDelta);
                }

                // Updates the new index of the first slot in each sequence after the given new index
                function updateNewIndicesFromIndex(index, indexDelta) {
                    for (var slot = slotsStart; slot !== slotListEnd; slot = slot.next) {
                        var indexNew = slot.indexNew;

                        if (indexNew !== undefined && index <= indexNew) {
                            updateNewIndicesAfterSlot(slot, indexDelta);
                            break;
                        }
                    }
                }

                // Adjust the indices of all slots to be consistent with any indexNew properties, and strip off the indexNews
                function updateIndices() {
                    var slot,
                        slotFirstInSequence,
                        indexNew;

                    for (slot = slotsStart; ; slot = slot.next) {
                        if (slot.firstInSequence) {
                            slotFirstInSequence = slot;
                            if (slot.indexNew !== undefined) {
                                indexNew = slot.indexNew;
                                delete slot.indexNew;
                                if (isNaN(indexNew)) {
                                    break;
                                }
                            } else {
                                indexNew = slot.index;
                            }

                            // See if this sequence should be merged with the previous one
                            if (slot !== slotsStart && slot.prev.index === indexNew - 1) {
                                mergeSequences(slot.prev);
                            }
                        }

                        if (slot.lastInSequence) {
                            var index = indexNew;
                            for (var slotUpdate = slotFirstInSequence; slotUpdate !== slot.next; slotUpdate = slotUpdate.next) {
                                if (index !== slotUpdate.index) {
                                    changeSlotIndex(slotUpdate, index);
                                }
                                if (+index === index) {
                                    index++;
                                }
                            }
                        }

                        if (slot === slotListEnd) {
                            break;
                        }
                    }

                    // Clear any indices on slots that were moved adjacent to slots without indices
                    for (; slot !== slotsEnd; slot = slot.next) {
                        if (slot.index !== undefined && slot !== slotListEnd) {
                            changeSlotIndex(slot, undefined);
                        }
                    }

                    indexUpdateDeferred = false;

                    if (countDelta && +knownCount === knownCount) {
                        if (getCountPromise) {
                            getCountPromise.reset();
                        } else {
                            changeCount(knownCount + countDelta);
                        }

                        countDelta = 0;
                    }
                }

                // Fetch Promises

                function createFetchPromise(slot, listenersProperty, listenerID, listBindingID, onComplete) {
                    if (slot.item) {
                        return new Promise(function (complete) {
                            if (onComplete) {
                                onComplete(complete, slot.item);
                            } else {
                                complete(slot.item);
                            }
                        });
                    } else {
                        var listener = {
                            listBindingID: listBindingID,
                            retained: false
                        };

                        if (!slot[listenersProperty]) {
                            slot[listenersProperty] = {};
                        }
                        slot[listenersProperty][listenerID] = listener;

                        listener.promise = new Promise(function (complete, error) {
                            listener.complete = (onComplete ? function (item) {
                                onComplete(complete, item);
                            } : complete);
                            listener.error = error;
                        }, function () {
                            // By now the slot might have been merged with another

                            while (slot.slotMergedWith) {
                                slot = slot.slotMergedWith;
                            }

                            var fetchListeners = slot[listenersProperty];
                            if (fetchListeners) {
                                delete fetchListeners[listenerID];

                                // See if there are any other listeners
                                if (Object.keys(fetchListeners).length > 0) {
                                    return;
                                }
                                delete slot[listenersProperty];
                            }
                            releaseSlotIfUnrequested(slot);
                        });

                        return listener.promise;
                    }
                }

                function completePromises(item, listeners) {
                    for (var listenerID in listeners) {
                        listeners[listenerID].complete(item);
                    }
                }

                function completeFetchPromises(slot, completeSynchronously) {
                    var fetchListeners = slot.fetchListeners,
                        directFetchListeners = slot.directFetchListeners;

                    if (fetchListeners || directFetchListeners) {
                        prepareSlotItem(slot);

                        // By default, complete asynchronously to minimize reentrancy

                        var item = slot.item;

                        var completeOrQueuePromises = function (listeners) {
                            if (completeSynchronously) {
                                completePromises(item, listeners);
                            } else {
                                fetchCompleteCallbacks.push(function () {
                                    completePromises(item, listeners);
                                });
                            }
                        };

                        if (directFetchListeners) {
                            slot.directFetchListeners = null;
                            completeOrQueuePromises(directFetchListeners);
                        }

                        if (fetchListeners) {
                            slot.fetchListeners = null;
                            completeOrQueuePromises(fetchListeners);
                        }

                        releaseSlotIfUnrequested(slot);
                    }
                }

                function callFetchCompleteCallbacks() {
                    var callbacks = fetchCompleteCallbacks;

                    // Clear fetchCompleteCallbacks first to avoid reentrancy problems
                    fetchCompleteCallbacks = [];

                    for (var i = 0, len = callbacks.length; i < len; i++) {
                        callbacks[i]();
                    }
                }

                function returnDirectFetchError(slot, error) {
                    var directFetchListeners = slot.directFetchListeners;
                    if (directFetchListeners) {
                        slot.directFetchListeners = null;

                        for (var listenerID in directFetchListeners) {
                            directFetchListeners[listenerID].error(error);
                        }

                        releaseSlotIfUnrequested(slot);
                    }
                }

                // Item Requests

                function requestSlot(slot) {
                    // Ensure that there's a slot on either side of each requested item
                    if (slot.firstInSequence) {
                        addSlotBefore(slot, indexMap);
                    }
                    if (slot.lastInSequence) {
                        addSlotAfter(slot, indexMap);
                    }

                    // If the item has already been fetched, prepare it now to be returned to the client
                    if (slot.itemNew) {
                        prepareSlotItem(slot);
                    }

                    // Start a new fetch if necessary
                    postFetch();

                    return slot;
                }

                function requestSlotBefore(slotNext) {
                    // First, see if the previous slot already exists
                    if (!slotNext.firstInSequence) {
                        var slotPrev = slotNext.prev;

                        // Next, see if the item is known to not exist
                        return (slotPrev === slotsStart ? null : requestSlot(slotPrev));
                    }

                    return requestSlot(addSlotBefore(slotNext, indexMap));
                }

                function requestSlotAfter(slotPrev) {
                    // First, see if the next slot already exists
                    if (!slotPrev.lastInSequence) {
                        var slotNext = slotPrev.next;

                        // Next, see if the item is known to not exist
                        return (slotNext === slotListEnd ? null : requestSlot(slotNext));
                    }

                    return requestSlot(addSlotAfter(slotPrev, indexMap));
                }

                function itemDirectlyFromSlot(slot) {
                    // Return a complete promise for a non-existent slot
                    return (
                        slot ?
                            createFetchPromise(slot, "directFetchListeners", (nextListenerID++).toString()) :
                            Promise.wrap(null)
                    );
                }

                function validateKey(key) {
                    if (typeof key !== "string" || !key) {
                        throw new _ErrorFromName("WinJS.UI.ListDataSource.KeyIsInvalid", strings.keyIsInvalid);
                    }
                }

                function createSlotForKey(key) {
                    var slot = createPrimarySlotSequence(slotsEnd);

                    setSlotKey(slot, key);
                    slot.keyRequested = true;

                    return slot;
                }

                function slotFromKey(key, hints) {
                    validateKey(key);

                    var slot = keyMap[key];

                    if (!slot) {
                        slot = createSlotForKey(key);
                        slot.hints = hints;
                    }

                    return requestSlot(slot);
                }

                function slotFromIndex(index) {
                    if (typeof index !== "number" || index < 0) {
                        throw new _ErrorFromName("WinJS.UI.ListDataSource.IndexIsInvalid", strings.indexIsInvalid);
                    }

                    if (slotListEnd.index <= index) {
                        return null;
                    }

                    var slot = indexMap[index];

                    if (!slot) {
                        var slotNext = successorFromIndex(index, indexMap, slotsStart, slotListEnd);

                        if (!slotNext) {
                            // The complete list has been observed, and this index isn't a part of it; a refresh may be necessary
                            return null;
                        }

                        if (slotNext === slotListEnd && index >= slotListEnd) {
                            // Clear slotListEnd's index, as that's now unknown
                            changeSlotIndex(slotListEnd, undefined);
                        }

                        // Create a new slot and start a request for it
                        if (slotNext.prev.index === index - 1) {
                            slot = addSlotAfter(slotNext.prev, indexMap);
                        } else if (slotNext.index === index + 1) {
                            slot = addSlotBefore(slotNext, indexMap);
                        } else {
                            slot = createPrimarySlotSequence(slotNext, index);
                        }
                    }

                    if (!slot.item) {
                        slot.indexRequested = true;
                    }

                    return requestSlot(slot);
                }

                function slotFromDescription(description) {
                    // Create a new slot and start a request for it
                    var slot = createPrimarySlotSequence(slotsEnd);

                    slot.description = description;

                    return requestSlot(slot);
                }

                // Status
                var that = this;
                function setStatus(statusNew) {
                    statusPending = statusNew;
                    if (status !== statusPending) {
                        var dispatch = function () {
                            statusChangePosted = false;

                            if (status !== statusPending) {
                                status = statusPending;
                                that.dispatchEvent(statusChangedEvent, status);
                            }
                        };
                        if (statusPending === DataSourceStatus.failure) {
                            dispatch();
                        } else if (!statusChangePosted) {
                            statusChangePosted = true;

                            // Delay the event to filter out rapid changes
                            _Global.setTimeout(dispatch, 40);
                        }
                    }
                }

                // Slot Fetching

                function slotFetchInProgress(slot) {
                    var fetchID = slot.fetchID;
                    return fetchID && fetchesInProgress[fetchID];
                }

                function setFetchID(slot, fetchID) {
                    slot.fetchID = fetchID;
                }

                function newFetchID() {
                    var fetchID = nextFetchID;
                    nextFetchID++;

                    fetchesInProgress[fetchID] = true;

                    return fetchID;
                }

                function setFetchIDs(slot, countBefore, countAfter) {
                    var fetchID = newFetchID();
                    setFetchID(slot, fetchID);

                    var slotBefore = slot;
                    while (!slotBefore.firstInSequence && countBefore > 0) {
                        slotBefore = slotBefore.prev;
                        countBefore--;
                        setFetchID(slotBefore, fetchID);
                    }

                    var slotAfter = slot;
                    while (!slotAfter.lastInSequence && countAfter > 0) {
                        slotAfter = slotAfter.next;
                        countAfter--;
                        setFetchID(slotAfter, fetchID);
                    }

                    return fetchID;
                }

                // Adds markers on behalf of the data adapter if their presence can be deduced
                function addMarkers(fetchResult) {
                    var items = fetchResult.items,
                        offset = fetchResult.offset,
                        totalCount = fetchResult.totalCount,
                        absoluteIndex = fetchResult.absoluteIndex,
                        atStart = fetchResult.atStart,
                        atEnd = fetchResult.atEnd;

                    if (isNonNegativeNumber(absoluteIndex)) {
                        if (isNonNegativeNumber(totalCount)) {
                            var itemsLength = items.length;
                            if (absoluteIndex - offset + itemsLength === totalCount) {
                                atEnd = true;
                            }
                        }

                        if (offset === absoluteIndex) {
                            atStart = true;
                        }
                    }

                    if (atStart) {
                        items.unshift(startMarker);
                        fetchResult.offset++;
                    }

                    if (atEnd) {
                        items.push(endMarker);
                    }
                }

                function resultsValid(slot, refreshID, fetchID) {
                    // This fetch has completed, whatever it has returned
                    delete fetchesInProgress[fetchID];

                    if (refreshID !== currentRefreshID || slotPermanentlyRemoved(slot)) {
                        // This information is out of date, or the slot has since been discarded

                        postFetch();
                        return false;
                    }

                    return true;
                }

                function fetchItems(slot, fetchID, promiseItems, index) {
                    var refreshID = currentRefreshID;
                    promiseItems.then(function (fetchResult) {
                        if (fetchResult.items && fetchResult.items.length) {
                            var perfID = "itemsFetched id=" + fetchID + " count=" + fetchResult.items.length;
                            profilerMarkStart(perfID);
                            if (resultsValid(slot, refreshID, fetchID)) {
                                if (+index === index) {
                                    fetchResult.absoluteIndex = index;
                                }
                                addMarkers(fetchResult);
                                processResults(slot, fetchResult.items, fetchResult.offset, fetchResult.totalCount, fetchResult.absoluteIndex);
                            }
                            profilerMarkEnd(perfID);
                        } else {
                            return Promise.wrapError(new _ErrorFromName(FetchError.doesNotExist));
                        }
                    }).then(null, function (error) {
                        if (resultsValid(slot, refreshID, fetchID)) {
                            processErrorResult(slot, error);
                        }
                    });
                }

                function fetchItemsForIndex(indexRequested, slot, fetchID, promiseItems) {
                    var refreshID = currentRefreshID;
                    promiseItems.then(function (fetchResult) {
                        if (fetchResult.items && fetchResult.items.length) {
                            var perfID = "itemsFetched id=" + fetchID + " count=" + fetchResult.items.length;
                            profilerMarkStart(perfID);
                            if (resultsValid(slot, refreshID, fetchID)) {
                                fetchResult.absoluteIndex = indexRequested;
                                addMarkers(fetchResult);
                                processResultsForIndex(indexRequested, slot, fetchResult.items, fetchResult.offset, fetchResult.totalCount, fetchResult.absoluteIndex);
                            }
                            profilerMarkEnd(perfID);
                        } else {
                            return Promise.wrapError(new _ErrorFromName(FetchError.doesNotExist));
                        }
                    }).then(null, function () {
                        if (resultsValid(slot, refreshID, fetchID)) {
                            processErrorResultForIndex(indexRequested, slot, refreshID);
                        }
                    });
                }

                function fetchItemsFromStart(slot, count) {
                    var fetchID = setFetchIDs(slot, 0, count - 1);
                    if (itemsFromStart) {
                        fetchItems(slot, fetchID, itemsFromStart(fetchID, count), 0);
                    } else {
                        fetchItems(slot, fetchID, itemsFromIndex(fetchID, 0, 0, count - 1), 0);
                    }
                }

                function fetchItemsFromEnd(slot, count) {
                    var fetchID = setFetchIDs(slot, count - 1, 0);
                    fetchItems(slot, fetchID, itemsFromEnd(fetchID, count));
                }

                function fetchItemsFromKey(slot, countBefore, countAfter) {
                    var fetchID = setFetchIDs(slot, countBefore, countAfter);
                    fetchItems(slot, fetchID, itemsFromKey(fetchID, slot.key, countBefore, countAfter, slot.hints));
                }

                function fetchItemsFromIndex(slot, countBefore, countAfter) {
                    var index = slot.index;

                    // Don't ask for items with negative indices
                    if (countBefore > index) {
                        countBefore = index;
                    }

                    if (itemsFromIndex) {
                        var fetchID = setFetchIDs(slot, countBefore, countAfter);
                        fetchItems(slot, fetchID, itemsFromIndex(fetchID, index, countBefore, countAfter), index);
                    } else {
                        // If the slot key is known, we just need to request the surrounding items
                        if (slot.key) {
                            fetchItemsFromKey(slot, countBefore, countAfter);
                        } else {
                            // Search for the slot with the closest index that has a known key (using the start of the list as a
                            // last resort).
                            var slotClosest = slotsStart,
                                closestDelta = index + 1,
                                slotSearch,
                                delta;

                            // First search backwards
                            for (slotSearch = slot.prev; slotSearch !== slotsStart; slotSearch = slotSearch.prev) {
                                if (slotSearch.index !== undefined && slotSearch.key) {
                                    delta = index - slotSearch.index;
                                    if (closestDelta > delta) {
                                        closestDelta = delta;
                                        slotClosest = slotSearch;
                                    }
                                    break;
                                }
                            }

                            // Then search forwards
                            for (slotSearch = slot.next; slotSearch !== slotListEnd; slotSearch = slotSearch.next) {
                                if (slotSearch.index !== undefined && slotSearch.key) {
                                    delta = slotSearch.index - index;
                                    if (closestDelta > delta) {
                                        closestDelta = delta;
                                        slotClosest = slotSearch;
                                    }
                                    break;
                                }
                            }

                            if (slotClosest === slotsStart) {
                                var fetchID = setFetchIDs(slot, 0, index + 1);
                                fetchItemsForIndex(0, slot, fetchID, itemsFromStart(fetchID, index + 1));
                            } else {
                                var fetchBefore = Math.max(slotClosest.index - index, 0);
                                var fetchAfter = Math.max(index - slotClosest.index, 0);
                                var fetchID = setFetchIDs(slotClosest, fetchBefore, fetchAfter);
                                fetchItemsForIndex(slotClosest.index, slot, fetchID, itemsFromKey(fetchID,
                                    slotClosest.key,
                                    fetchBefore,
                                    fetchAfter,
                                    slot.hints
                                ));
                            }
                        }
                    }
                }

                function fetchItemsFromDescription(slot, countBefore, countAfter) {
                    var fetchID = setFetchIDs(slot, countBefore, countAfter);
                    fetchItems(slot, fetchID, itemsFromDescription(fetchID, slot.description, countBefore, countAfter));
                }

                function fetchItemsForAllSlots() {
                    if (!refreshInProgress) {
                        var slotFirstPlaceholder,
                            placeholderCount,
                            fetchInProgress = false,
                            fetchesInProgress = false,
                            slotRequestedByKey,
                            requestedKeyOffset,
                            slotRequestedByDescription,
                            requestedDescriptionOffset,
                            slotRequestedByIndex,
                            requestedIndexOffset;

                        for (var slot = slotsStart.next; slot !== slotsEnd;) {
                            var slotNext = slot.next;

                            if (slot !== slotListEnd && isPlaceholder(slot)) {
                                fetchesInProgress = true;

                                if (!slotFirstPlaceholder) {
                                    slotFirstPlaceholder = slot;
                                    placeholderCount = 1;
                                } else {
                                    placeholderCount++;
                                }

                                if (slotFetchInProgress(slot)) {
                                    fetchInProgress = true;
                                }

                                if (slot.keyRequested && !slotRequestedByKey) {
                                    slotRequestedByKey = slot;
                                    requestedKeyOffset = placeholderCount - 1;
                                }

                                if (slot.description !== undefined && !slotRequestedByDescription) {
                                    slotRequestedByDescription = slot;
                                    requestedDescriptionOffset = placeholderCount - 1;
                                }

                                if (slot.indexRequested && !slotRequestedByIndex) {
                                    slotRequestedByIndex = slot;
                                    requestedIndexOffset = placeholderCount - 1;
                                }

                                if (slot.lastInSequence || slotNext === slotsEnd || !isPlaceholder(slotNext)) {
                                    if (fetchInProgress) {
                                        fetchInProgress = false;
                                    } else {
                                        resultsProcessed = false;

                                        // Start a new fetch for this placeholder sequence

                                        // Prefer fetches in terms of a known item
                                        if (!slotFirstPlaceholder.firstInSequence && slotFirstPlaceholder.prev.key && itemsFromKey) {
                                            fetchItemsFromKey(slotFirstPlaceholder.prev, 0, placeholderCount);
                                        } else if (!slot.lastInSequence && slotNext.key && itemsFromKey) {
                                            fetchItemsFromKey(slotNext, placeholderCount, 0);
                                        } else if (slotFirstPlaceholder.prev === slotsStart && !slotFirstPlaceholder.firstInSequence && (itemsFromStart || itemsFromIndex)) {
                                            fetchItemsFromStart(slotFirstPlaceholder, placeholderCount);
                                        } else if (slotNext === slotListEnd && !slot.lastInSequence && itemsFromEnd) {
                                            fetchItemsFromEnd(slot, placeholderCount);
                                        } else if (slotRequestedByKey) {
                                            fetchItemsFromKey(slotRequestedByKey, requestedKeyOffset, placeholderCount - 1 - requestedKeyOffset);
                                        } else if (slotRequestedByDescription) {
                                            fetchItemsFromDescription(slotRequestedByDescription, requestedDescriptionOffset, placeholderCount - 1 - requestedDescriptionOffset);
                                        } else if (slotRequestedByIndex) {
                                            fetchItemsFromIndex(slotRequestedByIndex, requestedIndexOffset, placeholderCount - 1 - requestedIndexOffset);
                                        } else if (typeof slotFirstPlaceholder.index === "number") {
                                            fetchItemsFromIndex(slotFirstPlaceholder, placeholderCount - 1, 0);
                                        } else {
                                            // There is no way to fetch anything in this sequence
                                            deleteMirageSequence(slotFirstPlaceholder);
                                        }

                                        if (resultsProcessed) {
                                            // A re-entrant fetch might have altered the slots list - start again
                                            postFetch();
                                            return;
                                        }

                                        if (refreshInProgress) {
                                            // A re-entrant fetch might also have caused a refresh
                                            return;
                                        }
                                    }

                                    slotFirstPlaceholder = slotRequestedByIndex = slotRequestedByKey = null;
                                }
                            }

                            slot = slotNext;
                        }

                        setStatus(fetchesInProgress ? DataSourceStatus.waiting : DataSourceStatus.ready);
                    }
                }

                function postFetch() {
                    if (!fetchesPosted) {
                        fetchesPosted = true;
                        Scheduler.schedule(function VDS_async_postFetch() {
                            fetchesPosted = false;
                            fetchItemsForAllSlots();

                            // A mirage sequence might have been removed
                            finishNotifications();
                        }, Scheduler.Priority.max, null, "WinJS.UI.ListDataSource._fetch");
                    }
                }

                // Fetch Result Processing

                function itemChanged(slot) {
                    var itemNew = slot.itemNew;

                    if (!itemNew) {
                        return false;
                    }

                    var item = slot.item;

                    for (var property in item) {
                        switch (property) {
                            case "data":
                                // This is handled below
                                break;

                            default:
                                if (item[property] !== itemNew[property]) {
                                    return true;
                                }
                                break;
                        }
                    }

                    return (
                        listDataAdapter.compareByIdentity ?
                            item.data !== itemNew.data :
                            slot.signature !== itemSignature(itemNew)
                    );
                }

                function changeSlotIfNecessary(slot) {
                    if (!slotRequested(slot)) {
                        // There's no need for any notifications, just delete the old item
                        slot.item = null;
                    } else if (itemChanged(slot)) {
                        changeSlot(slot);
                    } else {
                        slot.itemNew = null;
                    }
                }

                function updateSlotItem(slot) {
                    if (slot.item) {
                        changeSlotIfNecessary(slot);
                    } else {
                        completeFetchPromises(slot);
                    }
                }

                function updateSlot(slot, item) {
                    if (!slot.key) {
                        setSlotKey(slot, item.key);
                    }
                    slot.itemNew = item;

                    updateSlotItem(slot);
                }

                function sendMirageNotifications(slot, slotToDiscard, listBindingIDsToDelete) {
                    var bindingMap = slotToDiscard.bindingMap;
                    if (bindingMap) {
                        for (var listBindingID in listBindingIDsToDelete) {
                            if (bindingMap[listBindingID]) {
                                var fetchListeners = slotToDiscard.fetchListeners;
                                for (var listenerID in fetchListeners) {
                                    var listener = fetchListeners[listenerID];

                                    if (listener.listBindingID === listBindingID && listener.retained) {
                                        delete fetchListeners[listenerID];
                                        listener.complete(null);
                                    }
                                }

                                var bindingRecord = bindingMap[listBindingID].bindingRecord;
                                handlerToNotify(bindingRecord).removed(handleForBinding(slotToDiscard, listBindingID), true, handleForBinding(slot, listBindingID));

                                // A re-entrant call to release from the removed handler might have cleared slotToDiscard.bindingMap
                                if (slotToDiscard.bindingMap) {
                                    delete slotToDiscard.bindingMap[listBindingID];
                                }
                            }
                        }
                    }
                }

                function mergeSlots(slot, slotToDiscard) {
                    // This shouldn't be called on a slot that has a pending change notification
                    // Only one of the two slots should have a key
                    // If slotToDiscard is about to acquire an index, send the notifications now; in rare cases, multiple
                    // indexChanged notifications will be sent for a given item during a refresh, but that's fine.
                    if (slot.index !== slotToDiscard.index) {
                        // If slotToDiscard has a defined index, that should have been transferred already
                        var indexOld = slotToDiscard.index;
                        slotToDiscard.index = slot.index;
                        sendIndexChangedNotifications(slotToDiscard, indexOld);
                    }

                    slotToDiscard.slotMergedWith = slot;

                    // Transfer the slotBindings from slotToDiscard to slot
                    var bindingMap = slotToDiscard.bindingMap;
                    for (var listBindingID in bindingMap) {
                        if (!slot.bindingMap) {
                            slot.bindingMap = {};
                        }

                        var slotBinding = bindingMap[listBindingID];

                        if (!slotBinding.handle) {
                            slotBinding.handle = slotToDiscard.handle;
                        }
                        handleMap[slotBinding.handle] = slot;

                        slot.bindingMap[listBindingID] = slotBinding;
                    }

                    // Update any ListBinding cursors pointing to slotToDiscard
                    forEachBindingRecord(function (bindingRecord) {
                        bindingRecord.adjustCurrentSlot(slotToDiscard, slot);
                    });

                    // See if the item needs to be transferred from slotToDiscard to slot
                    var item = slotToDiscard.itemNew || slotToDiscard.item;
                    if (item) {
                        item = Object.create(item);
                        defineCommonItemProperties(item, slot, slot.handle);
                        updateSlot(slot, item);
                    }

                    // Transfer the fetch listeners from slotToDiscard to slot, or complete them if item is known
                    if (slot.item) {
                        if (slotToDiscard.directFetchListeners) {
                            fetchCompleteCallbacks.push(function () {
                                completePromises(slot.item, slotToDiscard.directFetchListeners);
                            });
                        }
                        if (slotToDiscard.fetchListeners) {
                            fetchCompleteCallbacks.push(function () {
                                completePromises(slot.item, slotToDiscard.fetchListeners);
                            });
                        }
                    } else {
                        var listenerID;

                        for (listenerID in slotToDiscard.directFetchListeners) {
                            if (!slot.directFetchListeners) {
                                slot.directFetchListeners = {};
                            }
                            slot.directFetchListeners[listenerID] = slotToDiscard.directFetchListeners[listenerID];
                        }

                        for (listenerID in slotToDiscard.fetchListeners) {
                            if (!slot.fetchListeners) {
                                slot.fetchListeners = {};
                            }
                            slot.fetchListeners[listenerID] = slotToDiscard.fetchListeners[listenerID];
                        }
                    }

                    // This might be the first time this slot's item can be prepared
                    if (slot.itemNew) {
                        completeFetchPromises(slot);
                    }

                    // Give slotToDiscard an unused handle so it appears to be permanently removed
                    slotToDiscard.handle = (nextHandle++).toString();

                    splitSequence(slotToDiscard);
                    removeSlotPermanently(slotToDiscard);
                }

                function mergeSlotsAndItem(slot, slotToDiscard, item) {
                    if (slotToDiscard && slotToDiscard.key) {
                        if (!item) {
                            item = slotToDiscard.itemNew || slotToDiscard.item;
                        }

                        // Free up the key for the promoted slot
                        delete slotToDiscard.key;
                        delete keyMap[item.key];

                        slotToDiscard.itemNew = null;
                        slotToDiscard.item = null;
                    }

                    if (item) {
                        updateSlot(slot, item);
                    }

                    if (slotToDiscard) {
                        mergeSlots(slot, slotToDiscard);
                    }
                }

                function slotFromResult(result) {
                    if (typeof result !== "object") {
                        throw new _ErrorFromName("WinJS.UI.ListDataSource.InvalidItemReturned", strings.invalidItemReturned);
                    } else if (result === startMarker) {
                        return slotsStart;
                    } else if (result === endMarker) {
                        return slotListEnd;
                    } else if (!result.key) {
                        throw new _ErrorFromName("WinJS.UI.ListDataSource.InvalidKeyReturned", strings.invalidKeyReturned);
                    } else {
                        if (_BaseUtils.validation) {
                            validateKey(result.key);
                        }
                        return keyMap[result.key];
                    }
                }

                function matchSlot(slot, result) {
                    // First see if there is an existing slot that needs to be merged
                    var slotExisting = slotFromResult(result);
                    if (slotExisting === slot) {
                        slotExisting = null;
                    }

                    if (slotExisting) {
                        sendMirageNotifications(slot, slotExisting, slot.bindingMap);
                    }

                    mergeSlotsAndItem(slot, slotExisting, result);
                }

                function promoteSlot(slot, item, index, insertionPoint) {
                    if (item && slot.key && slot.key !== item.key) {
                        // A contradiction has been found
                        beginRefresh();
                        return false;
                    }

                    // The slot with the key "wins"; slots without bindings can be merged without any change in observable behavior

                    var slotWithIndex = indexMap[index];
                    if (slotWithIndex) {
                        if (slotWithIndex === slot) {
                            slotWithIndex = null;
                        } else if (slotWithIndex.key && (slot.key || (item && slotWithIndex.key !== item.key))) {
                            // A contradiction has been found
                            beginRefresh();
                            return false;
                        } else if (!slot.key && slotWithIndex.bindingMap) {
                            return false;
                        }
                    }

                    var slotWithKey;
                    if (item) {
                        slotWithKey = keyMap[item.key];

                        if (slotWithKey === slot) {
                            slotWithKey = null;
                        } else if (slotWithKey && slotWithKey.bindingMap) {
                            return false;
                        }
                    }

                    if (slotWithIndex) {
                        sendMirageNotifications(slot, slotWithIndex, slot.bindingMap);

                        // Transfer the index to the promoted slot
                        delete indexMap[index];
                        changeSlotIndex(slot, index);

                        // See if this sequence should be merged with its neighbors
                        if (slot.prev.index === index - 1) {
                            mergeSequences(slot.prev);
                        }
                        if (slot.next.index === index + 1) {
                            mergeSequences(slot);
                        }

                        insertionPoint.slotNext = slotWithIndex.slotNext;

                        if (!item) {
                            item = slotWithIndex.itemNew || slotWithIndex.item;
                            if (item) {
                                slotWithKey = keyMap[item.key];
                            }
                        }
                    } else {
                        changeSlotIndex(slot, index);
                    }

                    if (slotWithKey && slotWithIndex !== slotWithKey) {
                        sendMirageNotifications(slot, slotWithKey, slot.bindingMap);
                    }

                    mergeSlotsAndItem(slot, slotWithKey, item);

                    // Do this after mergeSlotsAndItem, since its call to updateSlot might send changed notifications, and those
                    // wouldn't make sense to clients that never saw the old item.
                    if (slotWithIndex && slotWithIndex !== slotWithKey) {
                        mergeSlots(slot, slotWithIndex);
                    }

                    return true;
                }

                function mergeAdjacentSlot(slotExisting, slot, listBindingIDsToDelete) {
                    if (slot.key && slotExisting.key && slot.key !== slotExisting.key) {
                        // A contradiction has been found
                        beginRefresh();
                        return false;
                    }

                    for (var listBindingID in slotExisting.bindingMap) {
                        listBindingIDsToDelete[listBindingID] = true;
                    }

                    sendMirageNotifications(slotExisting, slot, listBindingIDsToDelete);
                    mergeSlotsAndItem(slotExisting, slot);

                    return true;
                }

                function mergeSlotsBefore(slot, slotExisting) {
                    var listBindingIDsToDelete = {};

                    while (slot) {
                        var slotPrev = (slot.firstInSequence ? null : slot.prev);

                        if (!slotExisting.firstInSequence && slotExisting.prev === slotsStart) {
                            deleteSlot(slot, true);
                        } else {
                            if (slotExisting.firstInSequence) {
                                slotExisting = addSlotBefore(slotExisting, indexMap);
                            } else {
                                slotExisting = slotExisting.prev;
                            }

                            if (!mergeAdjacentSlot(slotExisting, slot, listBindingIDsToDelete)) {
                                return;
                            }
                        }

                        slot = slotPrev;
                    }
                }

                function mergeSlotsAfter(slot, slotExisting) {
                    var listBindingIDsToDelete = {};

                    while (slot) {
                        var slotNext = (slot.lastInSequence ? null : slot.next);

                        if (!slotExisting.lastInSequence && slotExisting.next === slotListEnd) {
                            deleteSlot(slot, true);
                        } else {
                            if (slotExisting.lastInSequence) {
                                slotExisting = addSlotAfter(slotExisting, indexMap);
                            } else {
                                slotExisting = slotExisting.next;
                            }

                            if (!mergeAdjacentSlot(slotExisting, slot, listBindingIDsToDelete)) {
                                return;
                            }
                        }

                        slot = slotNext;
                    }
                }

                function mergeSequencePairs(sequencePairsToMerge) {
                    for (var i = 0; i < sequencePairsToMerge.length; i++) {
                        var sequencePairToMerge = sequencePairsToMerge[i];
                        mergeSlotsBefore(sequencePairToMerge.slotBeforeSequence, sequencePairToMerge.slotFirstInSequence);
                        mergeSlotsAfter(sequencePairToMerge.slotAfterSequence, sequencePairToMerge.slotLastInSequence);
                    }
                }

                // Removes any placeholders with indices that exceed the given upper bound on the count
                function removeMirageIndices(countMax, indexFirstKnown) {
                    var placeholdersAtEnd = 0;

                    function removePlaceholdersAfterSlot(slotRemoveAfter) {
                        for (var slot2 = slotListEnd.prev; !(slot2.index < countMax) && slot2 !== slotRemoveAfter;) {
                            var slotPrev2 = slot2.prev;
                            if (slot2.index !== undefined) {
                                deleteSlot(slot2, true);
                            }
                            slot2 = slotPrev2;
                        }

                        placeholdersAtEnd = 0;
                    }

                    for (var slot = slotListEnd.prev; !(slot.index < countMax) || placeholdersAtEnd > 0;) {
                        var slotPrev = slot.prev;

                        if (slot === slotsStart) {
                            removePlaceholdersAfterSlot(slotsStart);
                            break;
                        } else if (slot.key) {
                            if (slot.index >= countMax) {
                                beginRefresh();
                                return false;
                            } else if (slot.index >= indexFirstKnown) {
                                removePlaceholdersAfterSlot(slot);
                            } else {
                                if (itemsFromKey) {
                                    fetchItemsFromKey(slot, 0, placeholdersAtEnd);
                                } else {
                                    fetchItemsFromIndex(slot, 0, placeholdersAtEnd);
                                }

                                // Wait until the fetch has completed before doing anything
                                return false;
                            }
                        } else if (slot.indexRequested || slot.firstInSequence) {
                            removePlaceholdersAfterSlot(slotPrev);
                        } else {
                            placeholdersAtEnd++;
                        }

                        slot = slotPrev;
                    }

                    return true;
                }

                // Merges the results of a fetch into the slot list data structure, and determines if any notifications need to be
                // synthesized.
                function processResults(slot, results, offset, count, index) {
                    var perfId = "WinJS.UI.ListDataSource.processResults";
                    profilerMarkStart(perfId);

                    index = validateIndexReturned(index);
                    count = validateCountReturned(count);

                    // If there are edits queued, we need to wait until the slots get back in sync with the data
                    if (editsQueued) {
                        profilerMarkEnd(perfId);
                        return;
                    }

                    if (indexUpdateDeferred) {
                        updateIndices();
                    }

                    // If the count has changed, and the end of the list has been reached, that's a contradiction
                    if ((isNonNegativeNumber(count) || count === CountResult.unknown) && count !== knownCount && !slotListEnd.firstInSequence) {
                        beginRefresh();
                        profilerMarkEnd(perfId);
                        return;
                    }

                    resultsProcessed = true;

                    (function () {
                        var i,
                            j,
                            resultsCount = results.length,
                            slotExisting,
                            slotBefore;

                        // If an index wasn't passed in, see if the indices of these items can be determined
                        if (typeof index !== "number") {
                            for (i = 0; i < resultsCount; i++) {
                                slotExisting = slotFromResult(results[i]);
                                if (slotExisting && slotExisting.index !== undefined) {
                                    index = slotExisting.index + offset - i;
                                    break;
                                }
                            }
                        }

                        // See if these results include the end of the list
                        if (typeof index === "number" && results[resultsCount - 1] === endMarker) {
                            // If the count wasn't known, it is now
                            count = index - offset + resultsCount - 1;
                        } else if (isNonNegativeNumber(count) && (index === undefined || index === null)) {
                            // If the index wasn't known, it is now
                            index = count - (resultsCount - 1) + offset;
                        }

                        // If the count is known, remove any mirage placeholders at the end
                        if (isNonNegativeNumber(count) && !removeMirageIndices(count, index - offset)) {
                            // "Forget" the count - a subsequent fetch or refresh will update the count and list end
                            count = undefined;
                        }

                        // Find any existing slots that correspond to the results, and check for contradictions
                        var offsetMap = new Array(resultsCount);
                        for (i = 0; i < resultsCount; i++) {
                            var slotBestMatch = null;

                            slotExisting = slotFromResult(results[i]);

                            if (slotExisting) {
                                // See if this item is currently adjacent to a different item, or has a different index
                                if ((i > 0 && !slotExisting.firstInSequence && slotExisting.prev.key && slotExisting.prev.key !== results[i - 1].key) ||
                                        (typeof index === "number" && slotExisting.index !== undefined && slotExisting.index !== index - offset + i)) {
                                    // A contradiction has been found, so we can't proceed further
                                    beginRefresh();
                                    return;
                                }

                                if (slotExisting === slotsStart || slotExisting === slotListEnd || slotExisting.bindingMap) {
                                    // First choice is a slot with the given key and at least one binding (or an end of the list)
                                    slotBestMatch = slotExisting;
                                }
                            }

                            if (typeof index === "number") {
                                slotExisting = indexMap[index - offset + i];

                                if (slotExisting) {
                                    if (slotExisting.key && slotExisting.key !== results[i].key) {
                                        // A contradiction has been found, so we can't proceed further
                                        beginRefresh();
                                        return;
                                    }

                                    if (!slotBestMatch && slotExisting.bindingMap) {
                                        // Second choice is a slot with the given index and at least one binding
                                        slotBestMatch = slotExisting;
                                    }
                                }
                            }

                            if (i === offset) {
                                if ((slot.key && slot.key !== results[i].key) || (typeof slot.index === "number" && typeof index === "number" && slot.index !== index)) {
                                    // A contradiction has been found, so we can't proceed further
                                    beginRefresh();
                                    return;
                                }

                                if (!slotBestMatch) {
                                    // Third choice is the slot that was passed in
                                    slotBestMatch = slot;
                                }
                            }

                            offsetMap[i] = slotBestMatch;
                        }

                        // Update items with known indices (and at least one binding) first, as they will not be merged with
                        // anything.
                        for (i = 0; i < resultsCount; i++) {
                            slotExisting = offsetMap[i];
                            if (slotExisting && slotExisting.index !== undefined && slotExisting !== slotsStart && slotExisting !== slotListEnd) {
                                matchSlot(slotExisting, results[i]);
                            }
                        }

                        var sequencePairsToMerge = [];

                        // Now process the sequences without indices
                        var firstSequence = true;
                        var slotBeforeSequence;
                        var slotAfterSequence;
                        for (i = 0; i < resultsCount; i++) {
                            slotExisting = offsetMap[i];
                            if (slotExisting && slotExisting !== slotListEnd) {
                                var iLast = i;

                                if (slotExisting.index === undefined) {
                                    var insertionPoint = {};

                                    promoteSlot(slotExisting, results[i], index - offset + i, insertionPoint);

                                    // Find the extents of the sequence of slots that we can use
                                    var slotFirstInSequence = slotExisting,
                                        slotLastInSequence = slotExisting,
                                        result;

                                    for (j = i - 1; !slotFirstInSequence.firstInSequence; j--) {
                                        // Keep going until we hit the start marker or a slot that we can't use or promote (it's ok
                                        // if j leaves the results range).

                                        result = results[j];
                                        if (result === startMarker) {
                                            break;
                                        }

                                        // Avoid assigning negative indices to slots
                                        var index2 = index - offset + j;
                                        if (index2 < 0) {
                                            break;
                                        }

                                        if (promoteSlot(slotFirstInSequence.prev, result, index2, insertionPoint)) {
                                            slotFirstInSequence = slotFirstInSequence.prev;
                                            if (j >= 0) {
                                                offsetMap[j] = slotFirstInSequence;
                                            }
                                        } else {
                                            break;
                                        }
                                    }

                                    for (j = i + 1; !slotLastInSequence.lastInSequence; j++) {
                                        // Keep going until we hit the end marker or a slot that we can't use or promote (it's ok
                                        // if j leaves the results range).

                                        // If slotListEnd is in this sequence, it should not be separated from any predecessor
                                        // slots, but they may need to be promoted.
                                        result = results[j];
                                        if ((result === endMarker || j === count) && slotLastInSequence.next !== slotListEnd) {
                                            break;
                                        }

                                        if (slotLastInSequence.next === slotListEnd || promoteSlot(slotLastInSequence.next, result, index - offset + j, insertionPoint)) {
                                            slotLastInSequence = slotLastInSequence.next;
                                            if (j < resultsCount) {
                                                offsetMap[j] = slotLastInSequence;
                                            }

                                            iLast = j;

                                            if (slotLastInSequence === slotListEnd) {
                                                break;
                                            }
                                        } else {
                                            break;
                                        }
                                    }

                                    slotBeforeSequence = (slotFirstInSequence.firstInSequence ? null : slotFirstInSequence.prev);
                                    slotAfterSequence = (slotLastInSequence.lastInSequence ? null : slotLastInSequence.next);

                                    if (slotBeforeSequence) {
                                        splitSequence(slotBeforeSequence);
                                    }
                                    if (slotAfterSequence) {
                                        splitSequence(slotLastInSequence);
                                    }

                                    // Move the sequence if necessary
                                    if (typeof index === "number") {
                                        if (slotLastInSequence === slotListEnd) {
                                            // Instead of moving the list end, move the sequence before out of the way
                                            if (slotBeforeSequence) {
                                                moveSequenceAfter(slotListEnd, sequenceStart(slotBeforeSequence), slotBeforeSequence);
                                            }
                                        } else {
                                            var slotInsertBefore = insertionPoint.slotNext;
                                            if (!slotInsertBefore) {
                                                slotInsertBefore = successorFromIndex(slotLastInSequence.index, indexMap, slotsStart, slotListEnd, true);
                                            }
                                            moveSequenceBefore(slotInsertBefore, slotFirstInSequence, slotLastInSequence);
                                        }
                                        if (slotFirstInSequence.prev.index === slotFirstInSequence.index - 1) {
                                            mergeSequences(slotFirstInSequence.prev);
                                        }
                                        if (slotLastInSequence.next.index === slotLastInSequence.index + 1) {
                                            mergeSequences(slotLastInSequence);
                                        }
                                    } else if (!firstSequence) {
                                        slotBefore = offsetMap[i - 1];

                                        if (slotBefore) {
                                            if (slotFirstInSequence.prev !== slotBefore) {
                                                if (slotLastInSequence === slotListEnd) {
                                                    // Instead of moving the list end, move the sequence before out of the way and
                                                    // the predecessor sequence into place.
                                                    if (slotBeforeSequence) {
                                                        moveSequenceAfter(slotListEnd, sequenceStart(slotBeforeSequence), slotBeforeSequence);
                                                    }
                                                    moveSequenceBefore(slotFirstInSequence, sequenceStart(slotBefore), slotBefore);
                                                } else {
                                                    moveSequenceAfter(slotBefore, slotFirstInSequence, slotLastInSequence);
                                                }
                                            }
                                            mergeSequences(slotBefore);
                                        }
                                    }
                                    firstSequence = false;

                                    if (refreshRequested) {
                                        return;
                                    }

                                    sequencePairsToMerge.push({
                                        slotBeforeSequence: slotBeforeSequence,
                                        slotFirstInSequence: slotFirstInSequence,
                                        slotLastInSequence: slotLastInSequence,
                                        slotAfterSequence: slotAfterSequence
                                    });
                                }

                                // See if the fetched slot needs to be merged
                                if (i === offset && slotExisting !== slot && !slotPermanentlyRemoved(slot)) {
                                    slotBeforeSequence = (slot.firstInSequence ? null : slot.prev);
                                    slotAfterSequence = (slot.lastInSequence ? null : slot.next);

                                    sendMirageNotifications(slotExisting, slot, slotExisting.bindingMap);
                                    mergeSlots(slotExisting, slot);

                                    sequencePairsToMerge.push({
                                        slotBeforeSequence: slotBeforeSequence,
                                        slotFirstInSequence: slotExisting,
                                        slotLastInSequence: slotExisting,
                                        slotAfterSequence: slotAfterSequence
                                    });
                                }

                                // Skip past all the other items in the sequence we just processed
                                i = iLast;
                            }
                        }

                        // If the count is known, set the index of the list end (wait until now because promoteSlot can sometimes
                        // delete it; do this before mergeSequencePairs so the list end can have slots inserted immediately before
                        // it).
                        if (isNonNegativeNumber(count) && slotListEnd.index !== count) {
                            changeSlotIndex(slotListEnd, count);
                        }

                        // Now that all the sequences have been moved, merge any colliding slots
                        mergeSequencePairs(sequencePairsToMerge);

                        // Match or cache any leftover items
                        for (i = 0; i < resultsCount; i++) {
                            // Find the first matched item
                            slotExisting = offsetMap[i];
                            if (slotExisting) {
                                for (j = i - 1; j >= 0; j--) {
                                    var slotAfter = offsetMap[j + 1];
                                    matchSlot(offsetMap[j] = (slotAfter.firstInSequence ? addSlotBefore(offsetMap[j + 1], indexMap) : slotAfter.prev), results[j]);
                                }
                                for (j = i + 1; j < resultsCount; j++) {
                                    slotBefore = offsetMap[j - 1];
                                    slotExisting = offsetMap[j];
                                    if (!slotExisting) {
                                        matchSlot(offsetMap[j] = (slotBefore.lastInSequence ? addSlotAfter(slotBefore, indexMap) : slotBefore.next), results[j]);
                                    } else if (slotExisting.firstInSequence) {
                                        // Adding the cached items may result in some sequences merging
                                        if (slotExisting.prev !== slotBefore) {
                                            moveSequenceAfter(slotBefore, slotExisting, sequenceEnd(slotExisting));
                                        }
                                        mergeSequences(slotBefore);
                                    }
                                }
                                break;
                            }
                        }

                        // The description is no longer required
                        delete slot.description;
                    })();

                    if (!refreshRequested) {
                        // If the count changed, but that's the only thing, just send the notification
                        if (count !== undefined && count !== knownCount) {
                            changeCount(count);
                        }

                        // See if there are more requests we can now fulfill
                        postFetch();
                    }

                    finishNotifications();

                    // Finally complete any promises for newly obtained items
                    callFetchCompleteCallbacks();
                    profilerMarkEnd(perfId);
                }

                function processErrorResult(slot, error) {
                    switch (error.name) {
                        case FetchError.noResponse:
                            setStatus(DataSourceStatus.failure);
                            returnDirectFetchError(slot, error);
                            break;

                        case FetchError.doesNotExist:
                            // Don't return an error, just complete with null (when the slot is deleted)

                            if (slot.indexRequested) {
                                // We now have an upper bound on the count
                                removeMirageIndices(slot.index);
                            } else if (slot.keyRequested || slot.description) {
                                // This item, and any items in the same sequence, count as mirages, since they might never have
                                // existed.
                                deleteMirageSequence(slot);
                            }

                            finishNotifications();

                            // It's likely that the client requested this item because something has changed since the client's
                            // latest observations of the data.  Begin a refresh just in case.
                            beginRefresh();
                            break;
                    }
                }

                function processResultsForIndex(indexRequested, slot, results, offset, count, index) {
                    index = validateIndexReturned(index);
                    count = validateCountReturned(count);

                    var indexFirst = indexRequested - offset;

                    var resultsCount = results.length;
                    if (slot.index >= indexFirst && slot.index < indexFirst + resultsCount) {
                        // The item is in this batch of results - process them all
                        processResults(slot, results, slot.index - indexFirst, count, slot.index);
                    } else if ((offset === resultsCount - 1 && indexRequested < slot.index) || (isNonNegativeNumber(count) && count <= slot.index)) {
                        // The requested index does not exist
                        processErrorResult(slot, new _ErrorFromName(FetchError.doesNotExist));
                    } else {
                        // We didn't get all the results we requested - pick up where they left off
                        if (slot.index < indexFirst) {
                            var fetchID = setFetchIDs(slot, 0, indexFirst - slot.index);
                            fetchItemsForIndex(indexFirst, slot, fetchID, itemsFromKey(
                                fetchID,
                                results[0].key,
                                indexFirst - slot.index,
                                0
                            ));
                        } else {
                            var indexLast = indexFirst + resultsCount - 1;
                            var fetchID = setFetchIDs(slot, slot.index - indexLast, 0);
                            fetchItemsForIndex(indexLast, slot, fetchID, itemsFromKey(
                                fetchID,
                                results[resultsCount - 1].key,
                                0,
                                slot.index - indexLast
                            ));
                        }
                    }
                }

                function processErrorResultForIndex(indexRequested, slot, error) {
                    // If the request was for an index other than the initial one, and the result was doesNotExist, this doesn't
                    switch (error.name) {
                        case FetchError.doesNotExist:
                            if (indexRequested === slotsStart.index) {
                                // The request was for the start of the list, so the item must not exist, and we now have an upper
                                // bound of zero for the count.
                                removeMirageIndices(0);

                                processErrorResult(slot, error);

                                // No need to check return value of removeMirageIndices, since processErrorResult is going to start
                                // a refresh anyway.
                            } else {
                                // Something has changed, but this index might still exist, so request a refresh
                                beginRefresh();
                            }
                            break;

                        default:
                            processErrorResult(slot, error);
                            break;
                    }
                }

                // Refresh

                function identifyRefreshCycle() {
                    // find refresh cycles, find the first beginRefresh in the refreshHistory and see whether it
                    // matches the next beginRefresh, if so then move the data source into an error state and stop
                    // refreshing.
                    var start = 0;
                    for (; start < refreshHistory.length; start++) {
                        if (refreshHistory[start].kind === "beginRefresh") {
                            break;
                        }
                    }
                    var end = start;
                    for (; end < refreshHistory.length; end++) {
                        if (refreshHistory[end].kind === "beginRefresh") {
                            break;
                        }
                    }
                    if (end > start && (end + (end - start) < refreshHistory.length)) {
                        var match = true;
                        var length = end - start;
                        for (var i = 0; i < length; i++) {
                            if (refreshHistory[start + i].kind !== refreshHistory[end + i].kind) {
                                match = false;
                                break;
                            }
                        }
                        if (match) {
                            if (_Log.log) {
                                _Log.log(strings.refreshCycleIdentified, "winjs vds", "error");
                                for (var i = start; i < end; i++) {
                                    _Log.log("" + (i - start) + ": " + JSON.stringify(refreshHistory[i]), "winjs vds", "error");
                                }
                            }
                        }
                        return match;
                    }
                }

                function resetRefreshState() {
                    if (++beginRefreshCount > MAX_BEGINREFRESH_COUNT) {
                        if (identifyRefreshCycle()) {
                            setStatus(DataSourceStatus.failure);
                            return;
                        }
                    }
                    refreshHistory[++refreshHistoryPos % refreshHistory.length] = { kind: "beginRefresh" };

                    // Give the start sentinel an index so we can always use predecessor + 1
                    refreshStart = {
                        firstInSequence: true,
                        lastInSequence: true,
                        index: -1
                    };
                    refreshEnd = {
                        firstInSequence: true,
                        lastInSequence: true
                    };
                    refreshStart.next = refreshEnd;
                    refreshEnd.prev = refreshStart;

                    refreshItemsFetched = false;
                    refreshCount = undefined;
                    keyFetchIDs = {};
                    refreshKeyMap = {};
                    refreshIndexMap = {};
                    refreshIndexMap[-1] = refreshStart;
                    deletedKeys = {};
                }

                function beginRefresh() {
                    if (refreshRequested) {
                        // There's already a refresh that has yet to start
                        return;
                    }

                    refreshRequested = true;

                    setStatus(DataSourceStatus.waiting);

                    if (waitForRefresh) {
                        waitForRefresh = false;

                        // The edit queue has been paused until the next refresh - resume it now
                        applyNextEdit();
                        return;
                    }

                    if (editsQueued) {
                        // The refresh will be started once the edit queue empties out
                        return;
                    }

                    var refreshID = ++currentRefreshID;
                    refreshInProgress = true;
                    refreshFetchesInProgress = 0;

                    // Batch calls to beginRefresh
                    Scheduler.schedule(function VDS_async_beginRefresh() {
                        if (currentRefreshID !== refreshID) {
                            return;
                        }

                        refreshRequested = false;

                        resetRefreshState();

                        // Remove all slots that aren't live, so we don't waste time fetching them
                        for (var slot = slotsStart.next; slot !== slotsEnd;) {
                            var slotNext = slot.next;

                            if (!slotLive(slot) && slot !== slotListEnd) {
                                deleteUnnecessarySlot(slot);
                            }

                            slot = slotNext;
                        }

                        startRefreshFetches();
                    }, Scheduler.Priority.high, null, "WinJS.VirtualizedDataSource.beginRefresh");
                }

                function requestRefresh() {
                    refreshSignal = refreshSignal || new _Signal();

                    beginRefresh();

                    return refreshSignal.promise;
                }

                function resultsValidForRefresh(refreshID, fetchID) {
                    // This fetch has completed, whatever it has returned
                    delete fetchesInProgress[fetchID];

                    if (refreshID !== currentRefreshID) {
                        // This information is out of date.  Ignore it.
                        return false;
                    }

                    refreshFetchesInProgress--;

                    return true;
                }

                function fetchItemsForRefresh(key, fromStart, fetchID, promiseItems, index) {
                    var refreshID = currentRefreshID;

                    refreshFetchesInProgress++;

                    promiseItems.then(function (fetchResult) {
                        if (fetchResult.items && fetchResult.items.length) {
                            var perfID = "itemsFetched id=" + fetchID + " count=" + fetchResult.items.length;
                            profilerMarkStart(perfID);
                            if (resultsValidForRefresh(refreshID, fetchID)) {
                                addMarkers(fetchResult);
                                processRefreshResults(key, fetchResult.items, fetchResult.offset, fetchResult.totalCount, (typeof index === "number" ? index : fetchResult.absoluteIndex));
                            }
                            profilerMarkEnd(perfID);
                        } else {
                            return Promise.wrapError(new _ErrorFromName(FetchError.doesNotExist));
                        }
                    }).then(null, function (error) {
                        if (resultsValidForRefresh(refreshID, fetchID)) {
                            processRefreshErrorResult(key, fromStart, error);
                        }
                    });
                }

                function refreshRange(slot, fetchID, countBefore, countAfter) {
                    if (itemsFromKey) {
                        // Keys are the preferred identifiers when the item might have moved
                        fetchItemsForRefresh(slot.key, false, fetchID, itemsFromKey(fetchID, slot.key, countBefore, countAfter, slot.hints));
                    } else {
                        // Request additional items to try to locate items that have moved
                        var searchDelta = 10,
                            index = slot.index;

                        if (refreshIndexMap[index] && refreshIndexMap[index].firstInSequence) {
                            // Ensure at least one element is observed before this one
                            fetchItemsForRefresh(slot.key, false, fetchID, itemsFromIndex(fetchID, index - 1, Math.min(countBefore + searchDelta, index) - 1, countAfter + 1 + searchDelta), index - 1);
                        } else if (refreshIndexMap[index] && refreshIndexMap[index].lastInSequence) {
                            // Ask for the next index we need directly
                            fetchItemsForRefresh(slot.key, false, fetchID, itemsFromIndex(fetchID, index + 1, Math.min(countBefore + searchDelta, index) + 1, countAfter - 1 + searchDelta), index + 1);
                        } else {
                            fetchItemsForRefresh(slot.key, false, fetchID, itemsFromIndex(fetchID, index, Math.min(countBefore + searchDelta, index), countAfter + searchDelta), index);
                        }
                    }
                }

                function refreshFirstItem(fetchID) {
                    if (itemsFromStart) {
                        fetchItemsForRefresh(null, true, fetchID, itemsFromStart(fetchID, 1), 0);
                    } else if (itemsFromIndex) {
                        fetchItemsForRefresh(null, true, fetchID, itemsFromIndex(fetchID, 0, 0, 0), 0);
                    }
                }

                function keyFetchInProgress(key) {
                    return fetchesInProgress[keyFetchIDs[key]];
                }

                function refreshRanges(slotFirst, allRanges) {
                    // Fetch a few extra items each time, to catch insertions without requiring an extra fetch
                    var refreshFetchExtra = 3;

                    var refreshID = currentRefreshID;

                    var slotFetchFirst,
                        slotRefreshFirst,
                        fetchCount = 0,
                        fetchID;

                    // Walk through the slot list looking for keys we haven't fetched or attempted to fetch yet.  Rely on the
                    // heuristic that items that were close together before the refresh are likely to remain so after, so batched
                    // fetches will locate most of the previously fetched items.
                    for (var slot = slotFirst; slot !== slotsEnd; slot = slot.next) {
                        if (!slotFetchFirst && slot.key && !deletedKeys[slot.key] && !keyFetchInProgress(slot.key)) {
                            var slotRefresh = refreshKeyMap[slot.key];

                            // Keep attempting to fetch an item until at least one item on either side of it has been observed, so
                            // we can determine its position relative to others.
                            if (!slotRefresh || slotRefresh.firstInSequence || slotRefresh.lastInSequence) {
                                slotFetchFirst = slot;
                                slotRefreshFirst = slotRefresh;
                                fetchID = newFetchID();
                            }
                        }

                        if (!slotFetchFirst) {
                            // Also attempt to fetch placeholders for requests for specific keys, just in case those items no
                            // longer exist.
                            if (slot.key && isPlaceholder(slot) && !deletedKeys[slot.key]) {
                                // Fulfill each "itemFromKey" request
                                if (!refreshKeyMap[slot.key]) {
                                    // Fetch at least one item before and after, just to verify item's position in list
                                    fetchID = newFetchID();
                                    fetchItemsForRefresh(slot.key, false, fetchID, itemsFromKey(fetchID, slot.key, 1, 1, slot.hints));
                                }
                            }
                        } else {
                            var keyAlreadyFetched = keyFetchInProgress(slot.key);

                            if (!deletedKeys[slot.key] && !refreshKeyMap[slot.key] && !keyAlreadyFetched) {
                                if (slot.key) {
                                    keyFetchIDs[slot.key] = fetchID;
                                }
                                fetchCount++;
                            }

                            if (slot.lastInSequence || slot.next === slotListEnd || keyAlreadyFetched) {
                                refreshRange(slotFetchFirst, fetchID, (!slotRefreshFirst || slotRefreshFirst.firstInSequence ? refreshFetchExtra : 0), fetchCount - 1 + refreshFetchExtra);

                                if (!allRanges) {
                                    break;
                                }

                                slotFetchFirst = null;
                                fetchCount = 0;
                            }
                        }
                    }

                    if (refreshFetchesInProgress === 0 && !refreshItemsFetched && currentRefreshID === refreshID) {
                        // If nothing was successfully fetched, try fetching the first item, to detect an empty list
                        refreshFirstItem(newFetchID());
                    }

                }

                function startRefreshFetches() {
                    var refreshID = currentRefreshID;

                    do {
                        synchronousProgress = false;
                        reentrantContinue = true;
                        refreshRanges(slotsStart.next, true);
                        reentrantContinue = false;
                    } while (refreshFetchesInProgress === 0 && synchronousProgress && currentRefreshID === refreshID && refreshInProgress);

                    if (refreshFetchesInProgress === 0 && currentRefreshID === refreshID) {
                        concludeRefresh();
                    }
                }

                function continueRefresh(key) {
                    var refreshID = currentRefreshID;

                    // If the key is absent, then the attempt to fetch the first item just completed, and there is nothing else to
                    // fetch.
                    if (key) {
                        var slotContinue = keyMap[key];
                        if (!slotContinue) {
                            // In a rare case, the slot might have been deleted; just start scanning from the beginning again
                            slotContinue = slotsStart.next;
                        }

                        do {
                            synchronousRefresh = false;
                            reentrantRefresh = true;
                            refreshRanges(slotContinue, false);
                            reentrantRefresh = false;
                        } while (synchronousRefresh && currentRefreshID === refreshID && refreshInProgress);
                    }

                    if (reentrantContinue) {
                        synchronousProgress = true;
                    } else {
                        if (refreshFetchesInProgress === 0 && currentRefreshID === refreshID) {
                            // Walk through the entire list one more time, in case any edits were made during the refresh
                            startRefreshFetches();
                        }
                    }
                }

                function slotRefreshFromResult(result) {
                    if (typeof result !== "object" || !result) {
                        throw new _ErrorFromName("WinJS.UI.ListDataSource.InvalidItemReturned", strings.invalidItemReturned);
                    } else if (result === startMarker) {
                        return refreshStart;
                    } else if (result === endMarker) {
                        return refreshEnd;
                    } else if (!result.key) {
                        throw new _ErrorFromName("WinJS.UI.ListDataSource.InvalidKeyReturned", strings.invalidKeyReturned);
                    } else {
                        return refreshKeyMap[result.key];
                    }
                }

                function processRefreshSlotIndex(slot, expectedIndex) {
                    while (slot.index === undefined) {
                        setSlotIndex(slot, expectedIndex, refreshIndexMap);

                        if (slot.firstInSequence) {
                            return true;
                        }

                        slot = slot.prev;
                        expectedIndex--;
                    }

                    if (slot.index !== expectedIndex) {
                        // Something has changed since the refresh began; start again
                        beginRefresh();
                        return false;
                    }

                    return true;
                }

                function setRefreshSlotResult(slotRefresh, result) {
                    slotRefresh.key = result.key;
                    refreshKeyMap[slotRefresh.key] = slotRefresh;

                    slotRefresh.item = result;
                }

                // Returns the slot after the last insertion point between sequences
                function lastRefreshInsertionPoint() {
                    var slotNext = refreshEnd;
                    while (!slotNext.firstInSequence) {
                        slotNext = slotNext.prev;

                        if (slotNext === refreshStart) {
                            return null;
                        }
                    }

                    return slotNext;
                }

                function processRefreshResults(key, results, offset, count, index) {
                    index = validateIndexReturned(index);
                    count = validateCountReturned(count);

                    var keyPresent = false;

                    refreshItemsFetched = true;

                    var indexFirst = index - offset,
                        result = results[0];

                    if (result.key === key) {
                        keyPresent = true;
                    }

                    var slot = slotRefreshFromResult(result);
                    if (!slot) {
                        if (refreshIndexMap[indexFirst]) {
                            // Something has changed since the refresh began; start again
                            beginRefresh();
                            return;
                        }

                        // See if these results should be appended to an existing sequence
                        var slotPrev;
                        if (index !== undefined && (slotPrev = refreshIndexMap[indexFirst - 1])) {
                            if (!slotPrev.lastInSequence) {
                                // Something has changed since the refresh began; start again
                                beginRefresh();
                                return;
                            }
                            slot = addSlotAfter(slotPrev, refreshIndexMap);
                        } else {
                            // Create a new sequence
                            var slotSuccessor = (
                                +indexFirst === indexFirst ?
                                    successorFromIndex(indexFirst, refreshIndexMap, refreshStart, refreshEnd) :
                                    lastRefreshInsertionPoint(refreshStart, refreshEnd)
                            );

                            if (!slotSuccessor) {
                                // Something has changed since the refresh began; start again
                                beginRefresh();
                                return;
                            }

                            slot = createSlotSequence(slotSuccessor, indexFirst, refreshIndexMap);
                        }

                        setRefreshSlotResult(slot, results[0]);
                    } else {
                        if (+indexFirst === indexFirst) {
                            if (!processRefreshSlotIndex(slot, indexFirst)) {
                                return;
                            }
                        }
                    }

                    var resultsCount = results.length;
                    for (var i = 1; i < resultsCount; i++) {
                        result = results[i];

                        if (result.key === key) {
                            keyPresent = true;
                        }

                        var slotNext = slotRefreshFromResult(result);

                        if (!slotNext) {
                            if (!slot.lastInSequence) {
                                // Something has changed since the refresh began; start again
                                beginRefresh();
                                return;
                            }
                            slotNext = addSlotAfter(slot, refreshIndexMap);
                            setRefreshSlotResult(slotNext, result);
                        } else {
                            if (slot.index !== undefined && !processRefreshSlotIndex(slotNext, slot.index + 1)) {
                                return;
                            }

                            // If the slots aren't adjacent, see if it's possible to reorder sequences to make them so
                            if (slotNext !== slot.next) {
                                if (!slot.lastInSequence || !slotNext.firstInSequence) {
                                    // Something has changed since the refresh began; start again
                                    beginRefresh();
                                    return;
                                }

                                var slotLast = sequenceEnd(slotNext);
                                if (slotLast !== refreshEnd) {
                                    moveSequenceAfter(slot, slotNext, slotLast);
                                } else {
                                    var slotFirst = sequenceStart(slot);
                                    if (slotFirst !== refreshStart) {
                                        moveSequenceBefore(slotNext, slotFirst, slot);
                                    } else {
                                        // Something has changed since the refresh began; start again
                                        beginRefresh();
                                        return;
                                    }
                                }

                                mergeSequences(slot);
                            } else if (slot.lastInSequence) {
                                mergeSequences(slot);
                            }
                        }

                        slot = slotNext;
                    }

                    if (!keyPresent) {
                        deletedKeys[key] = true;
                    }

                    // If the count wasn't provided, see if it can be determined from the end of the list.
                    if (!isNonNegativeNumber(count) && !refreshEnd.firstInSequence) {
                        var indexLast = refreshEnd.prev.index;
                        if (indexLast !== undefined) {
                            count = indexLast + 1;
                        }
                    }

                    if (isNonNegativeNumber(count) || count === CountResult.unknown) {
                        if (isNonNegativeNumber(refreshCount)) {
                            if (count !== refreshCount) {
                                // Something has changed since the refresh began; start again
                                beginRefresh();
                                return;
                            }
                        } else {
                            refreshCount = count;
                        }

                        if (isNonNegativeNumber(refreshCount) && !refreshIndexMap[refreshCount]) {
                            setSlotIndex(refreshEnd, refreshCount, refreshIndexMap);
                        }
                    }

                    if (reentrantRefresh) {
                        synchronousRefresh = true;
                    } else {
                        continueRefresh(key);
                    }
                }

                function processRefreshErrorResult(key, fromStart, error) {
                    switch (error.name) {
                        case FetchError.noResponse:
                            setStatus(DataSourceStatus.failure);
                            break;

                        case FetchError.doesNotExist:
                            if (fromStart) {
                                // The attempt to fetch the first item failed, so the list must be empty
                                setSlotIndex(refreshEnd, 0, refreshIndexMap);
                                refreshCount = 0;

                                concludeRefresh();
                            } else {
                                deletedKeys[key] = true;

                                if (reentrantRefresh) {
                                    synchronousRefresh = true;
                                } else {
                                    continueRefresh(key);
                                }
                            }
                            break;
                    }
                }

                function slotFromSlotRefresh(slotRefresh) {
                    if (slotRefresh === refreshStart) {
                        return slotsStart;
                    } else if (slotRefresh === refreshEnd) {
                        return slotListEnd;
                    } else {
                        return keyMap[slotRefresh.key];
                    }
                }

                function slotRefreshFromSlot(slot) {
                    if (slot === slotsStart) {
                        return refreshStart;
                    } else if (slot === slotListEnd) {
                        return refreshEnd;
                    } else {
                        return refreshKeyMap[slot.key];
                    }
                }

                function mergeSequencesForRefresh(slotPrev) {
                    mergeSequences(slotPrev);

                    // Mark the merge point, so we can distinguish insertions from unrequested items
                    slotPrev.next.mergedForRefresh = true;
                }

                function copyRefreshSlotData(slotRefresh, slot) {
                    setSlotKey(slot, slotRefresh.key);
                    slot.itemNew = slotRefresh.item;
                }

                function addNewSlotFromRefresh(slotRefresh, slotNext, insertAfter) {
                    var slotNew = createPrimarySlot();

                    copyRefreshSlotData(slotRefresh, slotNew);
                    insertAndMergeSlot(slotNew, slotNext, insertAfter, !insertAfter);

                    var index = slotRefresh.index;
                    if (+index !== index) {
                        index = (insertAfter ? slotNew.prev.index + 1 : slotNext.next.index - 1);
                    }

                    setSlotIndex(slotNew, index, indexMap);

                    return slotNew;
                }

                function matchSlotForRefresh(slotExisting, slot, slotRefresh) {
                    if (slotExisting) {
                        sendMirageNotifications(slotExisting, slot, slotExisting.bindingMap);
                        mergeSlotsAndItem(slotExisting, slot, slotRefresh.item);
                    } else {
                        copyRefreshSlotData(slotRefresh, slot);

                        // If the index was requested, complete the promises now, as the index might be about to change
                        if (slot.indexRequested) {
                            updateSlotItem(slot);
                        }
                    }
                }

                function updateSlotForRefresh(slotExisting, slot, slotRefresh) {
                    if (!slot.key) {
                        if (slotExisting) {
                            // Record the relationship between the slot to discard and its neighbors
                            slotRefresh.mergeWithPrev = !slot.firstInSequence;
                            slotRefresh.mergeWithNext = !slot.lastInSequence;
                        } else {
                            slotRefresh.stationary = true;
                        }
                        matchSlotForRefresh(slotExisting, slot, slotRefresh);
                        return true;
                    } else {
                        return false;
                    }
                }

                function indexForRefresh(slot) {
                    var indexNew;

                    if (slot.indexRequested) {
                        indexNew = slot.index;
                    } else {
                        var slotRefresh = slotRefreshFromSlot(slot);
                        if (slotRefresh) {
                            indexNew = slotRefresh.index;
                        }
                    }

                    return indexNew;
                }

                function concludeRefresh() {
                    beginRefreshCount = 0;
                    refreshHistory = new Array(100);
                    refreshHistoryPos = -1;

                    indexUpdateDeferred = true;

                    keyFetchIDs = {};

                    var i,
                        j,
                        slot,
                        slotPrev,
                        slotNext,
                        slotBefore,
                        slotAfter,
                        slotRefresh,
                        slotExisting,
                        slotFirstInSequence,
                        sequenceCountOld,
                        sequencesOld = [],
                        sequenceOld,
                        sequenceOldPrev,
                        sequenceOldBestMatch,
                        sequenceCountNew,
                        sequencesNew = [],
                        sequenceNew,
                        index,
                        offset;

                    // Assign a sequence number to each refresh slot
                    sequenceCountNew = 0;
                    for (slotRefresh = refreshStart; slotRefresh; slotRefresh = slotRefresh.next) {
                        slotRefresh.sequenceNumber = sequenceCountNew;

                        if (slotRefresh.firstInSequence) {
                            slotFirstInSequence = slotRefresh;
                        }

                        if (slotRefresh.lastInSequence) {
                            sequencesNew[sequenceCountNew] = {
                                first: slotFirstInSequence,
                                last: slotRefresh,
                                matchingItems: 0
                            };
                            sequenceCountNew++;
                        }
                    }

                    // Remove unnecessary information from main slot list, and update the items
                    lastSlotReleased = null;
                    releasedSlots = 0;
                    for (slot = slotsStart.next; slot !== slotsEnd;) {
                        slotRefresh = refreshKeyMap[slot.key];
                        slotNext = slot.next;

                        if (slot !== slotListEnd) {
                            if (!slotLive(slot)) {
                                // Some more items might have been released since the refresh started.  Strip them away from the
                                // main slot list, as they'll just get in the way from now on.  Since we're discarding these, but
                                // don't know if they're actually going away, split the sequence as our starting assumption must be
                                // that the items on either side are in separate sequences.
                                deleteUnnecessarySlot(slot);
                            } else if (slot.key && !slotRefresh) {
                                // Remove items that have been deleted (or moved far away) and send removed notifications
                                deleteSlot(slot, false);
                            } else if (refreshCount === 0 || (slot.indexRequested && slot.index >= refreshCount)) {
                                // Remove items that can't exist in the list and send mirage removed notifications
                                deleteSlot(slot, true);
                            } else if (slot.item || slot.keyRequested) {
                                // Store the new item; this value will be compared with that stored in slot.item later
                                slot.itemNew = slotRefresh.item;
                            } else {
                                // Clear keys and items that have never been observed by client
                                if (slot.key) {
                                    if (!slot.keyRequested) {
                                        delete keyMap[slot.key];
                                        delete slot.key;
                                    }
                                    slot.itemNew = null;
                                }
                            }
                        }

                        slot = slotNext;
                    }

                    // Placeholders generated by itemsAtIndex should not move.  Match these to items now if possible, or merge them
                    // with existing items if necessary.
                    for (slot = slotsStart.next; slot !== slotListEnd;) {
                        slotNext = slot.next;

                        if (slot.indexRequested) {
                            slotRefresh = refreshIndexMap[slot.index];
                            if (slotRefresh) {
                                matchSlotForRefresh(slotFromSlotRefresh(slotRefresh), slot, slotRefresh);
                            }
                        }

                        slot = slotNext;
                    }

                    // Match old sequences to new sequences
                    var bestMatch,
                        bestMatchCount,
                        previousBestMatch = 0,
                        newSequenceCounts = [],
                        slotIndexRequested,
                        sequenceIndexEnd,
                        sequenceOldEnd;

                    sequenceCountOld = 0;
                    for (slot = slotsStart; slot !== slotsEnd; slot = slot.next) {
                        if (slot.firstInSequence) {
                            slotFirstInSequence = slot;
                            slotIndexRequested = null;
                            for (i = 0; i < sequenceCountNew; i++) {
                                newSequenceCounts[i] = 0;
                            }
                        }

                        if (slot.indexRequested) {
                            slotIndexRequested = slot;
                        }

                        slotRefresh = slotRefreshFromSlot(slot);
                        if (slotRefresh) {
                            newSequenceCounts[slotRefresh.sequenceNumber]++;
                        }

                        if (slot.lastInSequence) {
                            // Determine which new sequence is the best match for this old one.  Use a simple greedy algorithm to
                            // ensure the relative ordering of matched sequences is the same; out-of-order sequences will require
                            // move notifications.
                            bestMatchCount = 0;
                            for (i = previousBestMatch; i < sequenceCountNew; i++) {
                                if (bestMatchCount < newSequenceCounts[i]) {
                                    bestMatchCount = newSequenceCounts[i];
                                    bestMatch = i;
                                }
                            }

                            sequenceOld = {
                                first: slotFirstInSequence,
                                last: slot,
                                sequenceNew: (bestMatchCount > 0 ? sequencesNew[bestMatch] : undefined),
                                matchingItems: bestMatchCount
                            };

                            if (slotIndexRequested) {
                                sequenceOld.indexRequested = true;
                                sequenceOld.stationarySlot = slotIndexRequested;
                            }

                            sequencesOld[sequenceCountOld] = sequenceOld;

                            if (slot === slotListEnd) {
                                sequenceIndexEnd = sequenceCountOld;
                                sequenceOldEnd = sequenceOld;
                            }

                            sequenceCountOld++;

                            if (sequencesNew[bestMatch].first.index !== undefined) {
                                previousBestMatch = bestMatch;
                            }
                        }
                    }

                    // Special case: split the old start into a separate sequence if the new start isn't its best match
                    if (sequencesOld[0].sequenceNew !== sequencesNew[0]) {
                        splitSequence(slotsStart);
                        sequencesOld[0].first = slotsStart.next;
                        sequencesOld.unshift({
                            first: slotsStart,
                            last: slotsStart,
                            sequenceNew: sequencesNew[0],
                            matchingItems: 1
                        });
                        sequenceIndexEnd++;
                        sequenceCountOld++;
                    }

                    var listEndObserved = !slotListEnd.firstInSequence;

                    // Special case: split the old end into a separate sequence if the new end isn't its best match
                    if (sequenceOldEnd.sequenceNew !== sequencesNew[sequenceCountNew - 1]) {
                        splitSequence(slotListEnd.prev);
                        sequenceOldEnd.last = slotListEnd.prev;
                        sequenceIndexEnd++;
                        sequencesOld.splice(sequenceIndexEnd, 0, {
                            first: slotListEnd,
                            last: slotListEnd,
                            sequenceNew: sequencesNew[sequenceCountNew - 1],
                            matchingItems: 1
                        });
                        sequenceCountOld++;
                        sequenceOldEnd = sequencesOld[sequenceIndexEnd];
                    }

                    // Map new sequences to old sequences
                    for (i = 0; i < sequenceCountOld; i++) {
                        sequenceNew = sequencesOld[i].sequenceNew;
                        if (sequenceNew && sequenceNew.matchingItems < sequencesOld[i].matchingItems) {
                            sequenceNew.matchingItems = sequencesOld[i].matchingItems;
                            sequenceNew.sequenceOld = sequencesOld[i];
                        }
                    }

                    // The old end must always be the best match for the new end (if the new end is also the new start, they will
                    // be merged below).
                    sequencesNew[sequenceCountNew - 1].sequenceOld = sequenceOldEnd;
                    sequenceOldEnd.stationarySlot = slotListEnd;

                    // The old start must always be the best match for the new start
                    sequencesNew[0].sequenceOld = sequencesOld[0];
                    sequencesOld[0].stationarySlot = slotsStart;

                    // Merge additional indexed old sequences when possible

                    // First do a forward pass
                    for (i = 0; i <= sequenceIndexEnd; i++) {
                        sequenceOld = sequencesOld[i];

                        if (sequenceOld.sequenceNew && (sequenceOldBestMatch = sequenceOld.sequenceNew.sequenceOld) === sequenceOldPrev && sequenceOldPrev.last !== slotListEnd) {
                            mergeSequencesForRefresh(sequenceOldBestMatch.last);
                            sequenceOldBestMatch.last = sequenceOld.last;
                            delete sequencesOld[i];
                        } else {
                            sequenceOldPrev = sequenceOld;
                        }
                    }

                    // Now do a reverse pass
                    sequenceOldPrev = null;
                    for (i = sequenceIndexEnd; i >= 0; i--) {
                        sequenceOld = sequencesOld[i];
                        // From this point onwards, some members of sequencesOld may be undefined
                        if (sequenceOld) {
                            if (sequenceOld.sequenceNew && (sequenceOldBestMatch = sequenceOld.sequenceNew.sequenceOld) === sequenceOldPrev && sequenceOld.last !== slotListEnd) {
                                mergeSequencesForRefresh(sequenceOld.last);
                                sequenceOldBestMatch.first = sequenceOld.first;
                                delete sequencesOld[i];
                            } else {
                                sequenceOldPrev = sequenceOld;
                            }
                        }
                    }

                    // Since we may have forced the list end into a separate sequence, the mergedForRefresh flag may be incorrect
                    if (listEndObserved) {
                        delete slotListEnd.mergedForRefresh;
                    }

                    var sequencePairsToMerge = [];

                    // Find unchanged sequences without indices that can be merged with existing sequences without move
                    // notifications.
                    for (i = sequenceIndexEnd + 1; i < sequenceCountOld; i++) {
                        sequenceOld = sequencesOld[i];
                        if (sequenceOld && (!sequenceOld.sequenceNew || sequenceOld.sequenceNew.sequenceOld !== sequenceOld)) {
                            // If the order of the known items in the sequence is unchanged, then the sequence probably has not
                            // moved, but we now know where it belongs relative to at least one other sequence.
                            var orderPreserved = true,
                                slotRefreshFirst = null,
                                slotRefreshLast = null,
                                sequenceLength = 0;
                            slotRefresh = slotRefreshFromSlot(sequenceOld.first);
                            if (slotRefresh) {
                                slotRefreshFirst = slotRefreshLast = slotRefresh;
                                sequenceLength = 1;
                            }
                            for (slot = sequenceOld.first; slot !== sequenceOld.last; slot = slot.next) {
                                var slotRefreshNext = slotRefreshFromSlot(slot.next);

                                if (slotRefresh && slotRefreshNext && (slotRefresh.lastInSequence || slotRefresh.next !== slotRefreshNext)) {
                                    orderPreserved = false;
                                    break;
                                }

                                if (slotRefresh && !slotRefreshFirst) {
                                    slotRefreshFirst = slotRefreshLast = slotRefresh;
                                }

                                if (slotRefreshNext && slotRefreshFirst) {
                                    slotRefreshLast = slotRefreshNext;
                                    sequenceLength++;
                                }

                                slotRefresh = slotRefreshNext;
                            }

                            // If the stationary sequence has indices, verify that there is enough space for this sequence - if
                            // not, then something somewhere has moved after all.
                            if (orderPreserved && slotRefreshFirst && slotRefreshFirst.index !== undefined) {
                                var indexBefore;
                                if (!slotRefreshFirst.firstInSequence) {
                                    slotBefore = slotFromSlotRefresh(slotRefreshFirst.prev);
                                    if (slotBefore) {
                                        indexBefore = slotBefore.index;
                                    }
                                }

                                var indexAfter;
                                if (!slotRefreshLast.lastInSequence) {
                                    slotAfter = slotFromSlotRefresh(slotRefreshLast.next);
                                    if (slotAfter) {
                                        indexAfter = slotAfter.index;
                                    }
                                }

                                if ((!slotAfter || slotAfter.lastInSequence || slotAfter.mergedForRefresh) &&
                                        (indexBefore === undefined || indexAfter === undefined || indexAfter - indexBefore - 1 >= sequenceLength)) {
                                    sequenceOld.locationJustDetermined = true;

                                    // Mark the individual refresh slots as not requiring move notifications
                                    for (slotRefresh = slotRefreshFirst; ; slotRefresh = slotRefresh.next) {
                                        slotRefresh.locationJustDetermined = true;

                                        if (slotRefresh === slotRefreshLast) {
                                            break;
                                        }
                                    }

                                    // Store any adjacent placeholders so they can be merged once the moves and insertions have
                                    // been processed.
                                    var slotFirstInSequence = slotFromSlotRefresh(slotRefreshFirst),
                                        slotLastInSequence = slotFromSlotRefresh(slotRefreshLast);
                                    sequencePairsToMerge.push({
                                        slotBeforeSequence: (slotFirstInSequence.firstInSequence ? null : slotFirstInSequence.prev),
                                        slotFirstInSequence: slotFirstInSequence,
                                        slotLastInSequence: slotLastInSequence,
                                        slotAfterSequence: (slotLastInSequence.lastInSequence ? null : slotLastInSequence.next)
                                    });
                                }
                            }
                        }
                    }

                    // Remove placeholders in old sequences that don't map to new sequences (and don't contain requests for a
                    // specific index or key), as they no longer have meaning.
                    for (i = 0; i < sequenceCountOld; i++) {
                        sequenceOld = sequencesOld[i];
                        if (sequenceOld && !sequenceOld.indexRequested && !sequenceOld.locationJustDetermined && (!sequenceOld.sequenceNew || sequenceOld.sequenceNew.sequenceOld !== sequenceOld)) {
                            sequenceOld.sequenceNew = null;

                            slot = sequenceOld.first;

                            var sequenceEndReached;
                            do {
                                sequenceEndReached = (slot === sequenceOld.last);

                                slotNext = slot.next;

                                if (slot !== slotsStart && slot !== slotListEnd && slot !== slotsEnd && !slot.item && !slot.keyRequested) {
                                    deleteSlot(slot, true);
                                    if (sequenceOld.first === slot) {
                                        if (sequenceOld.last === slot) {
                                            delete sequencesOld[i];
                                            break;
                                        } else {
                                            sequenceOld.first = slot.next;
                                        }
                                    } else if (sequenceOld.last === slot) {
                                        sequenceOld.last = slot.prev;
                                    }
                                }

                                slot = slotNext;
                            } while (!sequenceEndReached);
                        }
                    }

                    // Locate boundaries of new items in new sequences
                    for (i = 0; i < sequenceCountNew; i++) {
                        sequenceNew = sequencesNew[i];
                        for (slotRefresh = sequenceNew.first; !slotFromSlotRefresh(slotRefresh) && !slotRefresh.lastInSequence; slotRefresh = slotRefresh.next) {
                            /*@empty*/
                        }
                        if (slotRefresh.lastInSequence && !slotFromSlotRefresh(slotRefresh)) {
                            sequenceNew.firstInner = sequenceNew.lastInner = null;
                        } else {
                            sequenceNew.firstInner = slotRefresh;
                            for (slotRefresh = sequenceNew.last; !slotFromSlotRefresh(slotRefresh) ; slotRefresh = slotRefresh.prev) {
                                /*@empty*/
                            }
                            sequenceNew.lastInner = slotRefresh;
                        }
                    }

                    // Determine which items to move
                    for (i = 0; i < sequenceCountNew; i++) {
                        sequenceNew = sequencesNew[i];
                        if (sequenceNew && sequenceNew.firstInner) {
                            sequenceOld = sequenceNew.sequenceOld;
                            if (sequenceOld) {
                                // Number the slots in each new sequence with their offset in the corresponding old sequence (or
                                // undefined if in a different old sequence).
                                var ordinal = 0;
                                for (slot = sequenceOld.first; true; slot = slot.next, ordinal++) {
                                    slotRefresh = slotRefreshFromSlot(slot);
                                    if (slotRefresh && slotRefresh.sequenceNumber === sequenceNew.firstInner.sequenceNumber) {
                                        slotRefresh.ordinal = ordinal;
                                    }

                                    if (slot.lastInSequence) {
                                        break;
                                    }
                                }

                                // Determine longest subsequence of items that are in the same order before and after
                                var piles = [];
                                for (slotRefresh = sequenceNew.firstInner; true; slotRefresh = slotRefresh.next) {
                                    ordinal = slotRefresh.ordinal;
                                    if (ordinal !== undefined) {
                                        var searchFirst = 0,
                                            searchLast = piles.length - 1;
                                        while (searchFirst <= searchLast) {
                                            var searchMidpoint = Math.floor(0.5 * (searchFirst + searchLast));
                                            if (piles[searchMidpoint].ordinal < ordinal) {
                                                searchFirst = searchMidpoint + 1;
                                            } else {
                                                searchLast = searchMidpoint - 1;
                                            }
                                        }
                                        piles[searchFirst] = slotRefresh;
                                        if (searchFirst > 0) {
                                            slotRefresh.predecessor = piles[searchFirst - 1];
                                        }
                                    }

                                    if (slotRefresh === sequenceNew.lastInner) {
                                        break;
                                    }
                                }

                                // The items in the longest ordered subsequence don't move; everything else does
                                var stationaryItems = [],
                                    stationaryItemCount = piles.length;
                                slotRefresh = piles[stationaryItemCount - 1];
                                for (j = stationaryItemCount; j--;) {
                                    slotRefresh.stationary = true;
                                    stationaryItems[j] = slotRefresh;
                                    slotRefresh = slotRefresh.predecessor;
                                }
                                sequenceOld.stationarySlot = slotFromSlotRefresh(stationaryItems[0]);

                                // Try to match new items before the first stationary item to placeholders
                                slotRefresh = stationaryItems[0];
                                slot = slotFromSlotRefresh(slotRefresh);
                                slotPrev = slot.prev;
                                var sequenceBoundaryReached = slot.firstInSequence;
                                while (!slotRefresh.firstInSequence) {
                                    slotRefresh = slotRefresh.prev;
                                    slotExisting = slotFromSlotRefresh(slotRefresh);
                                    if (!slotExisting || slotRefresh.locationJustDetermined) {
                                        // Find the next placeholder walking backwards
                                        while (!sequenceBoundaryReached && slotPrev !== slotsStart) {
                                            slot = slotPrev;
                                            slotPrev = slot.prev;
                                            sequenceBoundaryReached = slot.firstInSequence;

                                            if (updateSlotForRefresh(slotExisting, slot, slotRefresh)) {
                                                break;
                                            }
                                        }
                                    }
                                }

                                // Try to match new items between stationary items to placeholders
                                for (j = 0; j < stationaryItemCount - 1; j++) {
                                    slotRefresh = stationaryItems[j];
                                    slot = slotFromSlotRefresh(slotRefresh);
                                    var slotRefreshStop = stationaryItems[j + 1],
                                        slotRefreshMergePoint = null,
                                        slotStop = slotFromSlotRefresh(slotRefreshStop),
                                        slotExisting;
                                    // Find all the new items
                                    slotNext = slot.next;
                                    for (slotRefresh = slotRefresh.next; slotRefresh !== slotRefreshStop && !slotRefreshMergePoint && slot !== slotStop; slotRefresh = slotRefresh.next) {
                                        slotExisting = slotFromSlotRefresh(slotRefresh);
                                        if (!slotExisting || slotRefresh.locationJustDetermined) {
                                            // Find the next placeholder
                                            while (slotNext !== slotStop) {
                                                // If a merge point is reached, match the remainder of the placeholders by walking backwards
                                                if (slotNext.mergedForRefresh) {
                                                    slotRefreshMergePoint = slotRefresh.prev;
                                                    break;
                                                }

                                                slot = slotNext;
                                                slotNext = slot.next;

                                                if (updateSlotForRefresh(slotExisting, slot, slotRefresh)) {
                                                    break;
                                                }
                                            }
                                        }
                                    }

                                    // Walk backwards to the first merge point if necessary
                                    if (slotRefreshMergePoint) {
                                        slotPrev = slotStop.prev;
                                        for (slotRefresh = slotRefreshStop.prev; slotRefresh !== slotRefreshMergePoint && slotStop !== slot; slotRefresh = slotRefresh.prev) {
                                            slotExisting = slotFromSlotRefresh(slotRefresh);
                                            if (!slotExisting || slotRefresh.locationJustDetermined) {
                                                // Find the next placeholder walking backwards
                                                while (slotPrev !== slot) {
                                                    slotStop = slotPrev;
                                                    slotPrev = slotStop.prev;

                                                    if (updateSlotForRefresh(slotExisting, slotStop, slotRefresh)) {
                                                        break;
                                                    }
                                                }
                                            }
                                        }
                                    }

                                    // Delete remaining placeholders, sending notifications
                                    while (slotNext !== slotStop) {
                                        slot = slotNext;
                                        slotNext = slot.next;

                                        if (slot !== slotsStart && isPlaceholder(slot) && !slot.keyRequested) {
                                            // This might occur due to two sequences - requested by different clients - being
                                            // merged.  However, since only sequences with indices are merged, if this placehholder
                                            // is no longer necessary, it means an item actually was removed, so this doesn't count
                                            // as a mirage.
                                            deleteSlot(slot);
                                        }
                                    }
                                }

                                // Try to match new items after the last stationary item to placeholders
                                slotRefresh = stationaryItems[stationaryItemCount - 1];
                                slot = slotFromSlotRefresh(slotRefresh);
                                slotNext = slot.next;
                                sequenceBoundaryReached = slot.lastInSequence;
                                while (!slotRefresh.lastInSequence) {
                                    slotRefresh = slotRefresh.next;
                                    slotExisting = slotFromSlotRefresh(slotRefresh);
                                    if (!slotExisting || slotRefresh.locationJustDetermined) {
                                        // Find the next placeholder
                                        while (!sequenceBoundaryReached && slotNext !== slotListEnd) {
                                            slot = slotNext;
                                            slotNext = slot.next;
                                            sequenceBoundaryReached = slot.lastInSequence;

                                            if (updateSlotForRefresh(slotExisting, slot, slotRefresh)) {
                                                break;
                                            }
                                        }
                                    }
                                }
                            }
                        }
                    }

                    // Move items and send notifications
                    for (i = 0; i < sequenceCountNew; i++) {
                        sequenceNew = sequencesNew[i];

                        if (sequenceNew.firstInner) {
                            slotPrev = null;
                            for (slotRefresh = sequenceNew.firstInner; true; slotRefresh = slotRefresh.next) {
                                slot = slotFromSlotRefresh(slotRefresh);
                                if (slot) {
                                    if (!slotRefresh.stationary) {
                                        var slotMoveBefore,
                                            mergeWithPrev = false,
                                            mergeWithNext = false;
                                        if (slotPrev) {
                                            slotMoveBefore = slotPrev.next;
                                            mergeWithPrev = true;
                                        } else {
                                            // The first item will be inserted before the first stationary item, so find that now
                                            var slotRefreshStationary;
                                            for (slotRefreshStationary = sequenceNew.firstInner; !slotRefreshStationary.stationary && slotRefreshStationary !== sequenceNew.lastInner; slotRefreshStationary = slotRefreshStationary.next) {
                                                /*@empty*/
                                            }

                                            if (!slotRefreshStationary.stationary) {
                                                // There are no stationary items, as all the items are moving from another old
                                                // sequence.

                                                index = slotRefresh.index;

                                                // Find the best place to insert the new sequence
                                                if (index === 0) {
                                                    // Index 0 is a special case
                                                    slotMoveBefore = slotsStart.next;
                                                    mergeWithPrev = true;
                                                } else if (index === undefined) {
                                                    slotMoveBefore = slotsEnd;
                                                } else {
                                                    // Use a linear search; unlike successorFromIndex, prefer the last insertion
                                                    // point between sequences over the precise index
                                                    slotMoveBefore = slotsStart.next;
                                                    var lastSequenceStart = null;
                                                    while (true) {
                                                        if (slotMoveBefore.firstInSequence) {
                                                            lastSequenceStart = slotMoveBefore;
                                                        }

                                                        if ((index < slotMoveBefore.index && lastSequenceStart) || slotMoveBefore === slotListEnd) {
                                                            break;
                                                        }

                                                        slotMoveBefore = slotMoveBefore.next;
                                                    }

                                                    if (!slotMoveBefore.firstInSequence && lastSequenceStart) {
                                                        slotMoveBefore = lastSequenceStart;
                                                    }
                                                }
                                            } else {
                                                slotMoveBefore = slotFromSlotRefresh(slotRefreshStationary);
                                                mergeWithNext = true;
                                            }
                                        }

                                        // Preserve merge boundaries
                                        if (slot.mergedForRefresh) {
                                            delete slot.mergedForRefresh;
                                            if (!slot.lastInSequence) {
                                                slot.next.mergedForRefresh = true;
                                            }
                                        }

                                        mergeWithPrev = mergeWithPrev || slotRefresh.mergeWithPrev;
                                        mergeWithNext = mergeWithNext || slotRefresh.mergeWithNext;

                                        var skipNotifications = slotRefresh.locationJustDetermined;

                                        moveSlot(slot, slotMoveBefore, mergeWithPrev, mergeWithNext, skipNotifications);

                                        if (skipNotifications && mergeWithNext) {
                                            // Since this item was moved without a notification, this is an implicit merge of
                                            // sequences.  Mark the item's successor as mergedForRefresh.
                                            slotMoveBefore.mergedForRefresh = true;
                                        }
                                    }

                                    slotPrev = slot;
                                }

                                if (slotRefresh === sequenceNew.lastInner) {
                                    break;
                                }
                            }
                        }
                    }

                    // Insert new items (with new indices) and send notifications
                    for (i = 0; i < sequenceCountNew; i++) {
                        sequenceNew = sequencesNew[i];

                        if (sequenceNew.firstInner) {
                            slotPrev = null;
                            for (slotRefresh = sequenceNew.firstInner; true; slotRefresh = slotRefresh.next) {
                                slot = slotFromSlotRefresh(slotRefresh);
                                if (!slot) {
                                    var slotInsertBefore;
                                    if (slotPrev) {
                                        slotInsertBefore = slotPrev.next;
                                    } else {
                                        // The first item will be inserted *before* the first old item, so find that now
                                        var slotRefreshOld;
                                        for (slotRefreshOld = sequenceNew.firstInner; !slotFromSlotRefresh(slotRefreshOld) ; slotRefreshOld = slotRefreshOld.next) {
                                            /*@empty*/
                                        }
                                        slotInsertBefore = slotFromSlotRefresh(slotRefreshOld);
                                    }

                                    // Create a new slot for the item
                                    slot = addNewSlotFromRefresh(slotRefresh, slotInsertBefore, !!slotPrev);

                                    var slotRefreshNext = slotRefreshFromSlot(slotInsertBefore);

                                    if (!slotInsertBefore.mergedForRefresh && (!slotRefreshNext || !slotRefreshNext.locationJustDetermined)) {
                                        prepareSlotItem(slot);

                                        // Send the notification after the insertion
                                        sendInsertedNotification(slot);
                                    }
                                }
                                slotPrev = slot;

                                if (slotRefresh === sequenceNew.lastInner) {
                                    break;
                                }
                            }
                        }
                    }

                    // Rebuild the indexMap from scratch, so it is possible to detect colliding indices
                    indexMap = [];

                    // Send indexChanged and changed notifications
                    var indexFirst = -1;
                    for (slot = slotsStart, offset = 0; slot !== slotsEnd; offset++) {
                        var slotNext = slot.next;

                        if (slot.firstInSequence) {
                            slotFirstInSequence = slot;
                            offset = 0;
                        }

                        if (indexFirst === undefined) {
                            var indexNew = indexForRefresh(slot);
                            if (indexNew !== undefined) {
                                indexFirst = indexNew - offset;
                            }
                        }

                        // See if the next slot would cause a contradiction, in which case split the sequences
                        if (indexFirst !== undefined && !slot.lastInSequence) {
                            var indexNewNext = indexForRefresh(slot.next);
                            if (indexNewNext !== undefined && indexNewNext !== indexFirst + offset + 1) {
                                splitSequence(slot);

                                // 'Move' the items in-place individually, so move notifications are sent.  In rare cases, this
                                // will result in multiple move notifications being sent for a given item, but that's fine.
                                var first = true;
                                for (var slotMove = slot.next, lastInSequence = false; !lastInSequence && slotMove !== slotListEnd;) {
                                    var slotMoveNext = slotMove.next;

                                    lastInSequence = slotMove.lastInSequence;

                                    moveSlot(slotMove, slotMoveNext, !first, false);

                                    first = false;
                                    slotMove = slotMoveNext;
                                }
                            }
                        }

                        if (slot.lastInSequence) {
                            index = indexFirst;
                            for (var slotUpdate = slotFirstInSequence; slotUpdate !== slotNext;) {
                                var slotUpdateNext = slotUpdate.next;

                                if (index >= refreshCount && slotUpdate !== slotListEnd) {
                                    deleteSlot(slotUpdate, true);
                                } else {
                                    var slotWithIndex = indexMap[index];

                                    if (index !== slotUpdate.index) {
                                        delete indexMap[index];
                                        changeSlotIndex(slotUpdate, index);
                                    } else if (+index === index && indexMap[index] !== slotUpdate) {
                                        indexMap[index] = slotUpdate;
                                    }

                                    if (slotUpdate.itemNew) {
                                        updateSlotItem(slotUpdate);
                                    }

                                    if (slotWithIndex) {
                                        // Two slots' indices have collided - merge them
                                        if (slotUpdate.key) {
                                            sendMirageNotifications(slotUpdate, slotWithIndex, slotUpdate.bindingMap);
                                            mergeSlots(slotUpdate, slotWithIndex);
                                            if (+index === index) {
                                                indexMap[index] = slotUpdate;
                                            }
                                        } else {
                                            sendMirageNotifications(slotWithIndex, slotUpdate, slotWithIndex.bindingMap);
                                            mergeSlots(slotWithIndex, slotUpdate);
                                            if (+index === index) {
                                                indexMap[index] = slotWithIndex;
                                            }
                                        }
                                    }

                                    if (+index === index) {
                                        index++;
                                    }
                                }

                                slotUpdate = slotUpdateNext;
                            }

                            indexFirst = undefined;
                        }

                        slot = slotNext;
                    }

                    // See if any sequences need to be moved and/or merged
                    var indexMax = -2,
                        listEndReached;

                    for (slot = slotsStart, offset = 0; slot !== slotsEnd; offset++) {
                        var slotNext = slot.next;

                        if (slot.firstInSequence) {
                            slotFirstInSequence = slot;
                            offset = 0;
                        }

                        // Clean up during this pass
                        delete slot.mergedForRefresh;

                        if (slot.lastInSequence) {
                            // Move sequence if necessary
                            if (slotFirstInSequence.index === undefined) {
                                slotBefore = slotFirstInSequence.prev;
                                var slotRefreshBefore;
                                if (slotBefore && (slotRefreshBefore = slotRefreshFromSlot(slotBefore)) && !slotRefreshBefore.lastInSequence &&
                                        (slotRefresh = slotRefreshFromSlot(slot)) && slotRefresh.prev === slotRefreshBefore) {
                                    moveSequenceAfter(slotBefore, slotFirstInSequence, slot);
                                    mergeSequences(slotBefore);
                                } else if (slot !== slotListEnd && !listEndReached) {
                                    moveSequenceBefore(slotsEnd, slotFirstInSequence, slot);
                                }
                            } else {
                                if (indexMax < slot.index && !listEndReached) {
                                    indexMax = slot.index;
                                } else {
                                    // Find the correct insertion point
                                    for (slotAfter = slotsStart.next; slotAfter.index < slot.index; slotAfter = slotAfter.next) {
                                        /*@empty*/
                                    }

                                    // Move the items individually, so move notifications are sent
                                    for (var slotMove = slotFirstInSequence; slotMove !== slotNext;) {
                                        var slotMoveNext = slotMove.next;
                                        slotRefresh = slotRefreshFromSlot(slotMove);
                                        moveSlot(slotMove, slotAfter, slotAfter.prev.index === slotMove.index - 1, slotAfter.index === slotMove.index + 1, slotRefresh && slotRefresh.locationJustDetermined);
                                        slotMove = slotMoveNext;
                                    }
                                }

                                // Store slotBefore here since the sequence might have just been moved
                                slotBefore = slotFirstInSequence.prev;

                                // See if this sequence should be merged with the previous one
                                if (slotBefore && slotBefore.index === slotFirstInSequence.index - 1) {
                                    mergeSequences(slotBefore);
                                }
                            }
                        }

                        if (slot === slotListEnd) {
                            listEndReached = true;
                        }

                        slot = slotNext;
                    }

                    indexUpdateDeferred = false;

                    // Now that all the sequences have been moved, merge any colliding slots
                    mergeSequencePairs(sequencePairsToMerge);

                    // Send countChanged notification
                    if (refreshCount !== undefined && refreshCount !== knownCount) {
                        changeCount(refreshCount);
                    }

                    finishNotifications();

                    // Before discarding the refresh slot list, see if any fetch requests can be completed by pretending each range
                    // of refresh slots is an incoming array of results.
                    var fetchResults = [];
                    for (i = 0; i < sequenceCountNew; i++) {
                        sequenceNew = sequencesNew[i];

                        var results = [];

                        slot = null;
                        offset = 0;

                        var slotOffset;

                        for (slotRefresh = sequenceNew.first; true; slotRefresh = slotRefresh.next, offset++) {
                            if (slotRefresh === refreshStart) {
                                results.push(startMarker);
                            } else if (slotRefresh === refreshEnd) {
                                results.push(endMarker);
                            } else {
                                results.push(slotRefresh.item);

                                if (!slot) {
                                    slot = slotFromSlotRefresh(slotRefresh);
                                    slotOffset = offset;
                                }
                            }

                            if (slotRefresh.lastInSequence) {
                                break;
                            }
                        }

                        if (slot) {
                            fetchResults.push({
                                slot: slot,
                                results: results,
                                offset: slotOffset
                            });
                        }
                    }

                    resetRefreshState();
                    refreshInProgress = false;

                    // Complete any promises for newly obtained items
                    callFetchCompleteCallbacks();

                    // Now process the 'extra' results from the refresh list
                    for (i = 0; i < fetchResults.length; i++) {
                        var fetchResult = fetchResults[i];
                        processResults(fetchResult.slot, fetchResult.results, fetchResult.offset, knownCount, fetchResult.slot.index);
                    }

                    if (refreshSignal) {
                        var signal = refreshSignal;

                        refreshSignal = null;

                        signal.complete();
                    }

                    // Finally, kick-start fetches for any remaining placeholders
                    postFetch();
                }

                // Edit Queue

                // Queues an edit and immediately "optimistically" apply it to the slots list, sending re-entrant notifications
                function queueEdit(applyEdit, editType, complete, error, keyUpdate, updateSlots, undo) {
                    var editQueueTail = editQueue.prev,
                        edit = {
                            prev: editQueueTail,
                            next: editQueue,
                            applyEdit: applyEdit,
                            editType: editType,
                            complete: complete,
                            error: error,
                            keyUpdate: keyUpdate
                        };
                    editQueueTail.next = edit;
                    editQueue.prev = edit;
                    editsQueued = true;

                    // If there's a refresh in progress, abandon it, but request that a new one be started once the edits complete
                    if (refreshRequested || refreshInProgress) {
                        currentRefreshID++;
                        refreshInProgress = false;
                        refreshRequested = true;
                    }

                    if (editQueue.next === edit) {
                        // Attempt the edit immediately, in case it completes synchronously
                        applyNextEdit();
                    }

                    // If the edit succeeded or is still pending, apply it to the slots (in the latter case, "optimistically")
                    if (!edit.failed) {
                        updateSlots();

                        // Supply the undo function now
                        edit.undo = undo;
                    }

                    if (!editsInProgress) {
                        completeEdits();
                    }
                }

                function dequeueEdit() {
                    firstEditInProgress = false;

                    var editNext = editQueue.next.next;

                    editQueue.next = editNext;
                    editNext.prev = editQueue;
                }

                // Undo all queued edits, starting with the most recent
                function discardEditQueue() {
                    while (editQueue.prev !== editQueue) {
                        var editLast = editQueue.prev;

                        if (editLast.error) {
                            editLast.error(new _ErrorFromName(EditError.canceled));
                        }

                        // Edits that haven't been applied to the slots yet don't need to be undone
                        if (editLast.undo && !refreshRequested) {
                            editLast.undo();
                        }

                        editQueue.prev = editLast.prev;
                    }
                    editQueue.next = editQueue;

                    editsInProgress = false;

                    completeEdits();
                }

                var EditType = {
                    insert: "insert",
                    change: "change",
                    move: "move",
                    remove: "remove"
                };

                function attemptEdit(edit) {
                    if (firstEditInProgress) {
                        return;
                    }

                    var reentrant = true;

                    function continueEdits() {
                        if (!waitForRefresh) {
                            if (reentrant) {
                                synchronousEdit = true;
                            } else {
                                applyNextEdit();
                            }
                        }
                    }

                    var keyUpdate = edit.keyUpdate;

                    function onEditComplete(item) {
                        if (item) {
                            var slot;
                            if (keyUpdate && keyUpdate.key !== item.key) {
                                var keyNew = item.key;
                                if (!edit.undo) {
                                    // If the edit is in the process of being queued, we can use the correct key when we update the
                                    // slots, so there's no need for a later update.
                                    keyUpdate.key = keyNew;
                                } else {
                                    slot = keyUpdate.slot;
                                    if (slot) {
                                        var keyOld = slot.key;
                                        if (keyOld) {
                                            delete keyMap[keyOld];
                                        }
                                        setSlotKey(slot, keyNew);
                                        slot.itemNew = item;
                                        if (slot.item) {
                                            changeSlot(slot);
                                            finishNotifications();
                                        } else {
                                            completeFetchPromises(slot);
                                        }
                                    }
                                }
                            } else if (edit.editType === EditType.change) {
                                slot.itemNew = item;

                                if (!reentrant) {
                                    changeSlotIfNecessary(slot);
                                }
                            }
                        }

                        dequeueEdit();

                        if (edit.complete) {
                            edit.complete(item);
                        }

                        continueEdits();
                    }

                    function onEditError(error) {
                        switch (error.Name) {
                            case EditError.noResponse:
                                // Report the failure to the client, but do not dequeue the edit
                                setStatus(DataSourceStatus.failure);
                                waitForRefresh = true;

                                firstEditInProgress = false;

                                // Don't report the error, as the edit will be attempted again on the next refresh
                                return;

                            case EditError.notPermitted:
                                break;

                            case EditError.noLongerMeaningful:
                                // Something has changed, so request a refresh
                                beginRefresh();
                                break;

                            default:
                                break;
                        }

                        // Discard all remaining edits, rather than try to determine which subsequent ones depend on this one
                        edit.failed = true;
                        dequeueEdit();

                        discardEditQueue();

                        if (edit.error) {
                            edit.error(error);
                        }

                        continueEdits();
                    }

                    if (listDataAdapter.beginEdits && !beginEditsCalled) {
                        beginEditsCalled = true;
                        listDataAdapter.beginEdits();
                    }

                    // Call the applyEdit function for the given edit, passing in our own wrapper of the error handler that the
                    // client passed in.
                    firstEditInProgress = true;
                    edit.applyEdit().then(onEditComplete, onEditError);
                    reentrant = false;
                }

                function applyNextEdit() {
                    // See if there are any outstanding edits, and try to process as many as possible synchronously
                    while (editQueue.next !== editQueue) {
                        synchronousEdit = false;
                        attemptEdit(editQueue.next);
                        if (!synchronousEdit) {
                            return;
                        }
                    }

                    // The queue emptied out synchronously (or was empty to begin with)
                    concludeEdits();
                }

                function completeEdits() {
                    updateIndices();

                    finishNotifications();

                    callFetchCompleteCallbacks();

                    if (editQueue.next === editQueue) {
                        concludeEdits();
                    }
                }

                // Once the edit queue has emptied, update state appropriately and resume normal operation
                function concludeEdits() {
                    editsQueued = false;

                    if (listDataAdapter.endEdits && beginEditsCalled && !editsInProgress) {
                        beginEditsCalled = false;
                        listDataAdapter.endEdits();
                    }

                    // See if there's a refresh that needs to begin
                    if (refreshRequested) {
                        refreshRequested = false;
                        beginRefresh();
                    } else {
                        // Otherwise, see if anything needs to be fetched
                        postFetch();
                    }
                }

                // Editing Operations

                function getSlotForEdit(key) {
                    validateKey(key);

                    return keyMap[key] || createSlotForKey(key);
                }

                function insertNewSlot(key, itemNew, slotInsertBefore, mergeWithPrev, mergeWithNext) {
                    // Create a new slot, but don't worry about its index, as indices will be updated during endEdits
                    var slot = createPrimarySlot();

                    insertAndMergeSlot(slot, slotInsertBefore, mergeWithPrev, mergeWithNext);
                    if (key) {
                        setSlotKey(slot, key);
                    }
                    slot.itemNew = itemNew;

                    updateNewIndices(slot, 1);

                    // If this isn't part of a batch of changes, set the slot index now so renderers can see it
                    if (!editsInProgress && !dataNotificationsInProgress) {
                        if (!slot.firstInSequence && typeof slot.prev.index === "number") {
                            setSlotIndex(slot, slot.prev.index + 1, indexMap);
                        } else if (!slot.lastInSequence && typeof slot.next.index === "number") {
                            setSlotIndex(slot, slot.next.index - 1, indexMap);
                        }
                    }

                    prepareSlotItem(slot);

                    // Send the notification after the insertion
                    sendInsertedNotification(slot);

                    return slot;
                }

                function insertItem(key, data, slotInsertBefore, append, applyEdit) {
                    var keyUpdate = { key: key };

                    return new Promise(function (complete, error) {
                        queueEdit(
                            applyEdit, EditType.insert, complete, error, keyUpdate,

                            // updateSlots
                            function () {
                                if (slotInsertBefore) {
                                    var itemNew = {
                                        key: keyUpdate.key,
                                        data: data
                                    };

                                    keyUpdate.slot = insertNewSlot(keyUpdate.key, itemNew, slotInsertBefore, append, !append);
                                }
                            },

                            // undo
                            function () {
                                var slot = keyUpdate.slot;

                                if (slot) {
                                    updateNewIndices(slot, -1);
                                    deleteSlot(slot, false);
                                }
                            }
                        );
                    });
                }

                function moveItem(slot, slotMoveBefore, append, applyEdit) {
                    return new Promise(function (complete, error) {
                        var mergeAdjacent,
                            slotNext,
                            firstInSequence,
                            lastInSequence;

                        queueEdit(
                            applyEdit, EditType.move, complete, error,

                            // keyUpdate
                            null,

                            // updateSlots
                            function () {
                                slotNext = slot.next;
                                firstInSequence = slot.firstInSequence;
                                lastInSequence = slot.lastInSequence;

                                var slotPrev = slot.prev;

                                mergeAdjacent = (typeof slot.index !== "number" && (firstInSequence || !slotPrev.item) && (lastInSequence || !slotNext.item));

                                updateNewIndices(slot, -1);
                                moveSlot(slot, slotMoveBefore, append, !append);
                                updateNewIndices(slot, 1);

                                if (mergeAdjacent) {
                                    splitSequence(slotPrev);

                                    if (!firstInSequence) {
                                        mergeSlotsBefore(slotPrev, slot);
                                    }
                                    if (!lastInSequence) {
                                        mergeSlotsAfter(slotNext, slot);
                                    }
                                }
                            },

                            // undo
                            function () {
                                if (!mergeAdjacent) {
                                    updateNewIndices(slot, -1);
                                    moveSlot(slot, slotNext, !firstInSequence, !lastInSequence);
                                    updateNewIndices(slot, 1);
                                } else {
                                    beginRefresh();
                                }
                            }
                        );
                    });
                }

                function ListDataNotificationHandler() {
                    /// <signature helpKeyword="WinJS.UI.VirtualizedDataSource.ListDataNotificationHandler">
                    /// <summary locid="WinJS.UI.VirtualizedDataSource.ListDataNotificationHandler">
                    /// An implementation of IListDataNotificationHandler that is passed to the
                    /// IListDataAdapter.setNotificationHandler method.
                    /// </summary>
                    /// </signature>

                    this.invalidateAll = function () {
                        /// <signature helpKeyword="WinJS.UI.VirtualizedDataSource.ListDataNotificationHandler.invalidateAll">
                        /// <summary locid="WinJS.UI.VirtualizedDataSource.ListDataNotificationHandler.invalidateAll">
                        /// Notifies the VirtualizedDataSource that some data has changed, without specifying which data. It might
                        /// be impractical for some data sources to call this method for any or all changes, so this call is optional.
                        /// But if a given data adapter never calls it, the application should periodically call
                        /// invalidateAll on the VirtualizedDataSource to refresh the data.
                        /// </summary>
                        /// <returns type="Promise" locid="WinJS.UI.VirtualizedDataSource.ListDataNotificationHandler.invalidateAll_returnValue">
                        /// A Promise that completes when the data has been completely refreshed and all change notifications have
                        /// been sent.
                        /// </returns>
                        /// </signature>

                        if (knownCount === 0) {
                            this.reload();
                            return Promise.wrap();
                        }

                        return requestRefresh();
                    };

                    this.reload = function () {
                        /// <signature helpKeyword="WinJS.UI.VirtualizedDataSource.ListDataNotificationHandler.reload">
                        /// <summary locid="WinJS.UI.VirtualizedDataSource.ListDataNotificationHandler.reload">
                        /// Notifies the list data source that the list data has changed so much that it is better
                        /// to reload the data from scratch.
                        /// </summary>
                        /// </signature>

                        // Cancel all promises

                        if (getCountPromise) {
                            getCountPromise.cancel();
                        }

                        if (refreshSignal) {
                            refreshSignal.cancel();
                        }

                        for (var slot = slotsStart.next; slot !== slotsEnd; slot = slot.next) {
                            var fetchListeners = slot.fetchListeners;
                            for (var listenerID in fetchListeners) {
                                fetchListeners[listenerID].promise.cancel();
                            }
                            var directFetchListeners = slot.directFetchListeners;
                            for (var listenerID in directFetchListeners) {
                                directFetchListeners[listenerID].promise.cancel();
                            }
                        }

                        resetState();

                        forEachBindingRecord(function (bindingRecord) {
                            if (bindingRecord.notificationHandler) {
                                bindingRecord.notificationHandler.reload();
                            }
                        });
                    };

                    this.beginNotifications = function () {
                        /// <signature helpKeyword="WinJS.UI.VirtualizedDataSource.ListDataNotificationHandler.beginNotifications">
                        /// <summary locid="WinJS.UI.VirtualizedDataSource.ListDataNotificationHandler.beginNotifications">
                        /// Indicates the start of a notification batch.
                        /// Call it before a sequence of other notification calls to minimize the number of countChanged and
                        /// indexChanged notifications sent to the client of the VirtualizedDataSource. You must pair it with a call
                        /// to endNotifications, and pairs can't be nested.
                        /// </summary>
                        /// </signature>

                        dataNotificationsInProgress = true;
                    };

                    function completeNotification() {
                        if (!dataNotificationsInProgress) {
                            updateIndices();
                            finishNotifications();

                            callFetchCompleteCallbacks();
                        }
                    }

                    this.inserted = function (newItem, previousKey, nextKey, index) {
                        /// <signature helpKeyword="WinJS.UI.VirtualizedDataSource.ListDataNotificationHandler.inserted">
                        /// <summary locid="WinJS.UI.VirtualizedDataSource.ListDataNotificationHandler.inserted">
                        /// Raises a notification that an item was inserted.
                        /// </summary>
                        /// <param name="newItem" type="Object" locid="WinJS.UI.VirtualizedDataSource.ListDataNotificationHandler.inserted_p:newItem">
                        /// The inserted item. It must have a key and a data property (it must implement the IItem interface).
                        /// </param>
                        /// <param name="previousKey" mayBeNull="true" type="String" locid="WinJS.UI.VirtualizedDataSource.ListDataNotificationHandler.inserted_p:previousKey">
                        /// The key of the item before the insertion point, or null if the item was inserted at the start of the
                        /// list.  It can be null if you specified nextKey.
                        /// </param>
                        /// <param name="nextKey" mayBeNull="true" type="String" locid="WinJS.UI.VirtualizedDataSource.ListDataNotificationHandler.inserted_p:nextKey">
                        /// The key of the item after the insertion point, or null if the item was inserted at the end of the list.
                        /// It can be null if you specified previousKey.
                        /// </param>
                        /// <param name="index" optional="true" type="Number" integer="true" locid="WinJS.UI.VirtualizedDataSource.ListDataNotificationHandler.inserted_p:index">
                        /// The index of the inserted item.
                        /// </param>
                        /// </signature>

                        if (editsQueued) {
                            // We can't change the slots out from under any queued edits
                            beginRefresh();
                        } else {
                            var key = newItem.key,
                                slotPrev = keyMap[previousKey],
                                slotNext = keyMap[nextKey];

                            var havePreviousKey = typeof previousKey === "string",
                                haveNextKey = typeof nextKey === "string";

                            // Only one of previousKey, nextKey needs to be passed in
                            //
                            if (havePreviousKey) {
                                if (slotNext && !slotNext.firstInSequence) {
                                    slotPrev = slotNext.prev;
                                }
                            } else if (haveNextKey) {
                                if (slotPrev && !slotPrev.lastInSequence) {
                                    slotNext = slotPrev.next;
                                }
                            }

                            // If the VDS believes the list is empty but the data adapter believes the item has
                            // a adjacent item start a refresh.
                            //
                            if ((havePreviousKey || haveNextKey) && !(slotPrev || slotNext) && (slotsStart.next === slotListEnd)) {
                                beginRefresh();
                                return;
                            }

                            // If this key is known, something has changed, start a refresh.
                            //
                            if (keyMap[key]) {
                                beginRefresh();
                                return;
                            }

                            // If the slots aren't adjacent or are thought to be distinct sequences by the
                            //  VDS something has changed so start a refresh.
                            //
                            if (slotPrev && slotNext) {
                                if (slotPrev.next !== slotNext || slotPrev.lastInSequence || slotNext.firstInSequence) {
                                    beginRefresh();
                                    return;
                                }
                            }

                            // If one of the adjacent keys or indicies has only just been requested - rare,
                            //  and easier to deal with in a refresh.
                            //
                            if ((slotPrev && (slotPrev.keyRequested || slotPrev.indexRequested)) ||
                                (slotNext && (slotNext.keyRequested || slotNext.indexRequested))) {
                                beginRefresh();
                                return;
                            }

                            if (slotPrev || slotNext) {
                                insertNewSlot(key, newItem, (slotNext ? slotNext : slotPrev.next), !!slotPrev, !!slotNext);
                            } else if (slotsStart.next === slotListEnd) {
                                insertNewSlot(key, newItem, slotsStart.next, true, true);
                            } else if (index !== undefined) {
                                updateNewIndicesFromIndex(index, 1);
                            } else {
                                // We could not find a previous or next slot and an index was not provided, start a refresh
                                //
                                beginRefresh();
                                return;
                            }

                            completeNotification();
                        }
                    };

                    this.changed = function (item) {
                        /// <signature helpKeyword="WinJS.UI.VirtualizedDataSource.ListDataNotificationHandler.changed">
                        /// <summary locid="WinJS.UI.VirtualizedDataSource.ListDataNotificationHandler.changed">
                        /// Raises a notification that an item changed.
                        /// </summary>
                        /// <param name="item" type="Object" locid="WinJS.UI.VirtualizedDataSource.ListDataNotificationHandler.changed_p:item">
                        /// An IItem that represents the item that changed.
                        /// </param>
                        /// </signature>

                        if (editsQueued) {
                            // We can't change the slots out from under any queued edits
                            beginRefresh();
                        } else {
                            var key = item.key,
                                slot = keyMap[key];

                            if (slot) {
                                if (slot.keyRequested) {
                                    // The key has only just been requested - rare, and easier to deal with in a refresh
                                    beginRefresh();
                                } else {
                                    slot.itemNew = item;

                                    if (slot.item) {
                                        changeSlot(slot);

                                        completeNotification();
                                    }
                                }
                            }
                        }
                    };

                    this.moved = function (item, previousKey, nextKey, oldIndex, newIndex) {
                        /// <signature helpKeyword="WinJS.UI.VirtualizedDataSource.ListDataNotificationHandler.moved">
                        /// <summary locid="WinJS.UI.VirtualizedDataSource.ListDataNotificationHandler.moved">
                        /// Raises a notfication that an item was moved.
                        /// </summary>
                        /// <param name="item" type="Object" locid="WinJS.UI.VirtualizedDataSource.ListDataNotificationHandler.moved_p:item">
                        /// The item that was moved.
                        /// </param>
                        /// <param name="previousKey" mayBeNull="true" type="String" locid="WinJS.UI.VirtualizedDataSource.ListDataNotificationHandler.moved_p:previousKey">
                        /// The key of the item before the insertion point, or null if the item was moved to the beginning of the list.
                        /// It can be null if you specified nextKey.
                        /// </param>
                        /// <param name="nextKey" mayBeNull="true" type="String" locid="WinJS.UI.VirtualizedDataSource.ListDataNotificationHandler.moved_p:nextKey">
                        /// The key of the item after the insertion point, or null if the item was moved to the end of the list.
                        /// It can be null if you specified previousKey.
                        /// </param>
                        /// <param name="oldIndex" optional="true" type="Number" integer="true" locid="WinJS.UI.VirtualizedDataSource.ListDataNotificationHandler.moved_p:oldIndex">
                        /// The index of the item before it was moved.
                        /// </param>
                        /// <param name="newIndex" optional="true" type="Number" integer="true" locid="WinJS.UI.VirtualizedDataSource.ListDataNotificationHandler.moved_p:newIndex">
                        /// The index of the item after it was moved.
                        /// </param>
                        /// </signature>

                        if (editsQueued) {
                            // We can't change the slots out from under any queued edits
                            beginRefresh();
                        } else {
                            var key = item.key,
                                slot = keyMap[key],
                                slotPrev = keyMap[previousKey],
                                slotNext = keyMap[nextKey];

                            if ((slot && slot.keyRequested) || (slotPrev && slotPrev.keyRequested) || (slotNext && slotNext.keyRequested)) {
                                // One of the keys has only just been requested - rare, and easier to deal with in a refresh
                                beginRefresh();
                            } else if (slot) {
                                if (slotPrev && slotNext && (slotPrev.next !== slotNext || slotPrev.lastInSequence || slotNext.firstInSequence)) {
                                    // Something has changed, start a refresh
                                    beginRefresh();
                                } else if (!slotPrev && !slotNext) {
                                    // If we can't tell where the item moved to, treat this like a removal
                                    updateNewIndices(slot, -1);
                                    deleteSlot(slot, false);

                                    if (oldIndex !== undefined) {
                                        if (oldIndex < newIndex) {
                                            newIndex--;
                                        }

                                        updateNewIndicesFromIndex(newIndex, 1);
                                    }

                                    completeNotification();
                                } else {
                                    updateNewIndices(slot, -1);
                                    moveSlot(slot, (slotNext ? slotNext : slotPrev.next), !!slotPrev, !!slotNext);
                                    updateNewIndices(slot, 1);

                                    completeNotification();
                                }
                            } else if (slotPrev || slotNext) {
                                // If previousKey or nextKey is known, but key isn't, treat this like an insertion.

                                if (oldIndex !== undefined) {
                                    updateNewIndicesFromIndex(oldIndex, -1);

                                    if (oldIndex < newIndex) {
                                        newIndex--;
                                    }
                                }

                                this.inserted(item, previousKey, nextKey, newIndex);
                            } else if (oldIndex !== undefined) {
                                updateNewIndicesFromIndex(oldIndex, -1);

                                if (oldIndex < newIndex) {
                                    newIndex--;
                                }

                                updateNewIndicesFromIndex(newIndex, 1);

                                completeNotification();
                            }
                        }
                    };

                    this.removed = function (key, index) {
                        /// <signature helpKeyword="WinJS.UI.VirtualizedDataSource.ListDataNotificationHandler.removed">
                        /// <summary locid="WinJS.UI.VirtualizedDataSource.ListDataNotificationHandler.removed">
                        /// Raises a notification that an item was removed.
                        /// </summary>
                        /// <param name="key" mayBeNull="true" type="String" locid="WinJS.UI.VirtualizedDataSource.ListDataNotificationHandler.removed_p:key">
                        /// The key of the item that was removed.
                        /// </param>
                        /// <param name="index" optional="true" type="Number" integer="true" locid="WinJS.UI.VirtualizedDataSource.ListDataNotificationHandler.removed_p:index">
                        /// The index of the item that was removed.
                        /// </param>
                        /// </signature>

                        if (editsQueued) {
                            // We can't change the slots out from under any queued edits
                            beginRefresh();
                        } else {
                            var slot;

                            if (typeof key === "string") {
                                slot = keyMap[key];
                            } else {
                                slot = indexMap[index];
                            }

                            if (slot) {
                                if (slot.keyRequested) {
                                    // The key has only just been requested - rare, and easier to deal with in a refresh
                                    beginRefresh();
                                } else {
                                    updateNewIndices(slot, -1);
                                    deleteSlot(slot, false);

                                    completeNotification();
                                }
                            } else if (index !== undefined) {
                                updateNewIndicesFromIndex(index, -1);
                                completeNotification();
                            }
                        }
                    };

                    this.endNotifications = function () {
                        /// <signature helpKeyword="WinJS.UI.VirtualizedDataSource.ListDataNotificationHandler.endNotifications">
                        /// <summary locid="WinJS.UI.VirtualizedDataSource.ListDataNotificationHandler.endNotifications">
                        /// Concludes a sequence of notifications that began with a call to beginNotifications.
                        /// </summary>
                        /// </signature>

                        dataNotificationsInProgress = false;
                        completeNotification();
                    };

                } // ListDataNotificationHandler

                function resetState() {
                    setStatus(DataSourceStatus.ready);

                    // Track count promises
                    getCountPromise = null;

                    // Track whether listDataAdapter.endEdits needs to be called
                    beginEditsCalled = false;

                    // Track whether finishNotifications should be called after each edit
                    editsInProgress = false;

                    // Track whether the first queued edit should be attempted
                    firstEditInProgress = false;

                    // Queue of edis that have yet to be completed
                    editQueue = {};
                    editQueue.next = editQueue;
                    editQueue.prev = editQueue;

                    // Track whether there are currently edits queued
                    editsQueued = false;

                    // If an edit has returned noResponse, the edit queue will be reapplied when the next refresh is requested
                    waitForRefresh = false;

                    // Change to count while multiple edits are taking place
                    countDelta = 0;

                    // True while the indices are temporarily in a bad state due to multiple edits
                    indexUpdateDeferred = false;

                    // Next temporary key to use
                    nextTempKey = 0;

                    // Set of fetches for which results have not yet arrived
                    fetchesInProgress = {};

                    // Queue of complete callbacks for fetches
                    fetchCompleteCallbacks = [];

                    // Tracks the count returned explicitly or implicitly by the data adapter
                    knownCount = CountResult.unknown;

                    // Sentinel objects for list of slots
                    // Give the start sentinel an index so we can always use predecessor + 1.
                    slotsStart = {
                        firstInSequence: true,
                        lastInSequence: true,
                        index: -1
                    };
                    slotListEnd = {
                        firstInSequence: true,
                        lastInSequence: true
                    };
                    slotsEnd = {
                        firstInSequence: true,
                        lastInSequence: true
                    };
                    slotsStart.next = slotListEnd;
                    slotListEnd.prev = slotsStart;
                    slotListEnd.next = slotsEnd;
                    slotsEnd.prev = slotListEnd;

                    // Map of request IDs to slots
                    handleMap = {};

                    // Map of keys to slots
                    keyMap = {};

                    // Map of indices to slots
                    indexMap = {};
                    indexMap[-1] = slotsStart;

                    // Count of slots that have been released but not deleted
                    releasedSlots = 0;

                    lastSlotReleased = null;

                    // At most one call to reduce the number of refresh slots should be posted at any given time
                    reduceReleasedSlotCountPosted = false;

                    // Multiple refresh requests are coalesced
                    refreshRequested = false;

                    // Requests do not cause fetches while a refresh is in progress
                    refreshInProgress = false;

                    // Refresh requests yield the same promise until a refresh completes
                    refreshSignal = null;
                }

                // Construction

                // Process creation parameters
                if (!listDataAdapter) {
                    throw new _ErrorFromName("WinJS.UI.ListDataSource.ListDataAdapterIsInvalid", strings.listDataAdapterIsInvalid);
                }

                // Minimum number of released slots to retain
                cacheSize = (listDataAdapter.compareByIdentity ? 0 : 200);

                if (options) {
                    if (typeof options.cacheSize === "number") {
                        cacheSize = options.cacheSize;
                    }
                }

                // Cached listDataNotificationHandler initially undefined
                if (listDataAdapter.setNotificationHandler) {
                    listDataNotificationHandler = new ListDataNotificationHandler();

                    listDataAdapter.setNotificationHandler(listDataNotificationHandler);
                }

                // Current status
                status = DataSourceStatus.ready;

                // Track whether a change to the status has been posted already
                statusChangePosted = false;

                // Map of bindingIDs to binding records
                bindingMap = {};

                // ID to assign to the next ListBinding, incremented each time one is created
                nextListBindingID = 0;

                // ID assigned to a slot, incremented each time one is created - start with 1 so "if (handle)" tests are valid
                nextHandle = 1;

                // ID assigned to a fetch listener, incremented each time one is created
                nextListenerID = 0;

                // ID of the refresh in progress, incremented each time a new refresh is started
                currentRefreshID = 0;

                // Track whether fetchItemsForAllSlots has been posted already
                fetchesPosted = false;

                // ID of a fetch, incremented each time a new fetch is initiated - start with 1 so "if (fetchID)" tests are valid
                nextFetchID = 1;

                // Sentinel objects for results arrays
                startMarker = {};
                endMarker = {};

                resetState();

                // Public methods

                this.createListBinding = function (notificationHandler) {
                    /// <signature helpKeyword="WinJS.UI.IListDataSource.createListBinding">
                    /// <summary locid="WinJS.UI.IListDataSource.createListBinding">
                    /// Creates an IListBinding object that allows a client to read from the list and receive notifications for
                    /// changes that affect those portions of the list that the client already read.
                    /// </summary>
                    /// <param name="notificationHandler" optional="true" locid="WinJS.UI.IListDataSource.createListBinding_p:notificationHandler">
                    /// An object that implements the IListNotificationHandler interface.  If you omit this parameter,
                    /// change notifications won't be available.
                    /// </param>
                    /// <returns type="IListBinding" locid="WinJS.UI.IListDataSource.createListBinding_returnValue">
                    /// An object that implements the IListBinding interface.
                    /// </returns>
                    /// </signature>

                    var listBindingID = (nextListBindingID++).toString(),
                        slotCurrent = null,
                        released = false;

                    function retainSlotForCursor(slot) {
                        if (slot) {
                            slot.cursorCount++;
                        }
                    }

                    function releaseSlotForCursor(slot) {
                        if (slot) {
                            if (--slot.cursorCount === 0) {
                                releaseSlotIfUnrequested(slot);
                            }
                        }
                    }

                    function moveCursor(slot) {
                        // Retain the new slot first just in case it's the same slot
                        retainSlotForCursor(slot);
                        releaseSlotForCursor(slotCurrent);
                        slotCurrent = slot;
                    }

                    function adjustCurrentSlot(slot, slotNew) {
                        if (slot === slotCurrent) {
                            if (!slotNew) {
                                slotNew = (
                                    !slotCurrent || slotCurrent.lastInSequence || slotCurrent.next === slotListEnd ?
                                        null :
                                        slotCurrent.next
                                );
                            }
                            moveCursor(slotNew);
                        }
                    }

                    function releaseSlotFromListBinding(slot) {
                        var bindingMap = slot.bindingMap,
                            bindingHandle = bindingMap[listBindingID].handle;

                        delete slot.bindingMap[listBindingID];

                        // See if there are any listBindings left in the map
                        var releaseBindingMap = true,
                            releaseHandle = true;
                        for (var listBindingID2 in bindingMap) {
                            releaseBindingMap = false;
                            if (bindingHandle && bindingMap[listBindingID2].handle === bindingHandle) {
                                releaseHandle = false;
                                break;
                            }
                        }

                        if (bindingHandle && releaseHandle) {
                            delete handleMap[bindingHandle];
                        }
                        if (releaseBindingMap) {
                            slot.bindingMap = null;
                            releaseSlotIfUnrequested(slot);
                        }
                    }

                    function retainItem(slot, listenerID) {
                        if (!slot.bindingMap) {
                            slot.bindingMap = {};
                        }

                        var slotBinding = slot.bindingMap[listBindingID];
                        if (slotBinding) {
                            slotBinding.count++;
                        } else {
                            slot.bindingMap[listBindingID] = {
                                bindingRecord: bindingMap[listBindingID],
                                count: 1
                            };
                        }

                        if (slot.fetchListeners) {
                            var listener = slot.fetchListeners[listenerID];
                            if (listener) {
                                listener.retained = true;
                            }
                        }
                    }

                    function releaseItem(handle) {
                        var slot = handleMap[handle];

                        if (slot) {
                            var slotBinding = slot.bindingMap[listBindingID];
                            if (--slotBinding.count === 0) {
                                var fetchListeners = slot.fetchListeners;
                                for (var listenerID in fetchListeners) {
                                    var listener = fetchListeners[listenerID];
                                    if (listener.listBindingID === listBindingID) {
                                        listener.retained = false;
                                    }
                                }

                                releaseSlotFromListBinding(slot);
                            }
                        }
                    }

                    function itemPromiseFromKnownSlot(slot) {
                        var handle = handleForBinding(slot, listBindingID),
                            listenerID = (nextListenerID++).toString();

                        var itemPromise = createFetchPromise(slot, "fetchListeners", listenerID, listBindingID,
                            function (complete, item) {
                                complete(itemForBinding(item, handle));
                            }
                        );

                        defineCommonItemProperties(itemPromise, slot, handle);

                        // Only implement retain and release methods if a notification handler has been supplied
                        if (notificationHandler) {
                            itemPromise.retain = function () {
                                listBinding._retainItem(slot, listenerID);
                                return itemPromise;
                            };

                            itemPromise.release = function () {
                                listBinding._releaseItem(handle);
                            };
                        }

                        return itemPromise;
                    }

                    bindingMap[listBindingID] = {
                        notificationHandler: notificationHandler,
                        notificationsSent: false,
                        adjustCurrentSlot: adjustCurrentSlot,
                        itemPromiseFromKnownSlot: itemPromiseFromKnownSlot,
                    };

                    function itemPromiseFromSlot(slot) {
                        var itemPromise;

                        if (!released && slot) {
                            itemPromise = itemPromiseFromKnownSlot(slot);
                        } else {
                            // Return a complete promise for a non-existent slot
                            if (released) {
                                itemPromise = new Promise(function () { });
                                itemPromise.cancel();
                            } else {
                                itemPromise = Promise.wrap(null);
                            }
                            defineHandleProperty(itemPromise, null);
                            // Only implement retain and release methods if a notification handler has been supplied
                            if (notificationHandler) {
                                itemPromise.retain = function () { return itemPromise; };
                                itemPromise.release = function () { };
                            }
                        }

                        moveCursor(slot);

                        return itemPromise;
                    }

                    /// <signature helpKeyword="WinJS.UI.IListBinding">
                    /// <summary locid="WinJS.UI.IListBinding">
                    /// An interface that enables a client to read from the list and receive notifications for changes that affect
                    /// those portions of the list that the client already read.  IListBinding can also enumerate through lists
                    /// that can change at any time.
                    /// </summary>
                    /// </signature>
                    var listBinding = {
                        _retainItem: function (slot, listenerID) {
                            retainItem(slot, listenerID);
                        },

                        _releaseItem: function (handle) {
                            releaseItem(handle);
                        },

                        jumpToItem: function (item) {
                            /// <signature helpKeyword="WinJS.UI.IListBinding.jumpToItem">
                            /// <summary locid="WinJS.UI.IListBinding.jumpToItem">
                            /// Makes the specified item the current item.
                            /// </summary>
                            /// <param name="item" type="Object" locid="WinJS.UI.IListBinding.jumpToItem_p:item">
                            /// The IItem or IItemPromise to make the current item.
                            /// </param>
                            /// <returns type="IItemPromise" locid="WinJS.UI.IListBinding.jumpToItem_returnValue">
                            /// An object that implements the IItemPromise interface and serves as a promise for the specified item.  If
                            /// the specified item is not in the list, the promise completes with a value of null.
                            /// </returns>
                            /// </signature>

                            return itemPromiseFromSlot(item ? handleMap[item.handle] : null);
                        },

                        current: function () {
                            /// <signature helpKeyword="WinJS.UI.IListBinding.current">
                            /// <summary locid="WinJS.UI.IListBinding.current">
                            /// Retrieves the current item.
                            /// </summary>
                            /// <returns type="IItemPromise" locid="WinJS.UI.IListBinding.current_returnValue">
                            /// An object that implements the IItemPromise interface and serves as a promise for the current item.
                            /// If the cursor has moved past the start or end of the list, the promise completes with a value
                            /// of null.  If the current item has been deleted or moved, the promise returns an error.
                            /// </returns>
                            /// </signature>

                            return itemPromiseFromSlot(slotCurrent);
                        },

                        previous: function () {
                            /// <signature helpKeyword="WinJS.UI.IListBinding.previous">
                            /// <summary locid="WinJS.UI.IListBinding.previous">
                            /// Retrieves the item before the current item and makes it the current item.
                            /// </summary>
                            /// <returns type="IItemPromise" locid="WinJS.UI.IListBinding.previous_returnValue">
                            /// An object that implements the IItemPromise interface and serves as a promise for the previous item.
                            /// If the cursor moves past the start of the list, the promise completes with a value of null.
                            /// </returns>
                            /// </signature>

                            return itemPromiseFromSlot(slotCurrent ? requestSlotBefore(slotCurrent) : null);
                        },

                        next: function () {
                            /// <signature helpKeyword="WinJS.UI.IListBinding.next">
                            /// <summary locid="WinJS.UI.IListBinding.next">
                            /// Retrieves the item after the current item and makes it the current item.
                            /// </summary>
                            /// <returns type="IItemPromise" locid="WinJS.UI.IListBinding.next_returnValue">
                            /// An object that implements the IItemPromise interface and serves as a promise for the next item.  If
                            /// the cursor moves past the end of the list, the promise completes with a value of null.
                            /// </returns>
                            /// </signature>

                            return itemPromiseFromSlot(slotCurrent ? requestSlotAfter(slotCurrent) : null);
                        },

                        releaseItem: function (item) {
                            /// <signature helpKeyword="WinJS.UI.IListBinding.releaseItem">
                            /// <summary locid="WinJS.UI.IListBinding.releaseItem">
                            /// Creates a request to stop change notfications for the specified item. The item is released only when the
                            /// number of release calls matches the number of IItemPromise.retain calls. The number of release calls cannot
                            /// exceed the number of retain calls. This method is present only if you passed an IListNotificationHandler
                            /// to IListDataSource.createListBinding when it created this IListBinding.
                            /// </summary>
                            /// <param name="item" type="Object" locid="WinJS.UI.IListBinding.releaseItem_p:item">
                            /// The IItem or IItemPromise to release.
                            /// </param>
                            /// </signature>

                            this._releaseItem(item.handle);
                        },

                        release: function () {
                            /// <signature helpKeyword="WinJS.UI.IListBinding.release">
                            /// <summary locid="WinJS.UI.IListBinding.release">
                            /// Releases resources, stops notifications, and cancels outstanding promises
                            /// for all tracked items that this IListBinding returned.
                            /// </summary>
                            /// </signature>

                            released = true;

                            releaseSlotForCursor(slotCurrent);
                            slotCurrent = null;

                            for (var slot = slotsStart.next; slot !== slotsEnd;) {
                                var slotNext = slot.next;

                                var fetchListeners = slot.fetchListeners;
                                for (var listenerID in fetchListeners) {
                                    var listener = fetchListeners[listenerID];
                                    if (listener.listBindingID === listBindingID) {
                                        listener.promise.cancel();
                                        delete fetchListeners[listenerID];
                                    }
                                }

                                if (slot.bindingMap && slot.bindingMap[listBindingID]) {
                                    releaseSlotFromListBinding(slot);
                                }

                                slot = slotNext;
                            }

                            delete bindingMap[listBindingID];
                        }
                    };

                    // Only implement each navigation method if the data adapter implements certain methods

                    if (itemsFromStart || itemsFromIndex) {
                        listBinding.first = function () {
                            /// <signature helpKeyword="WinJS.UI.IListBinding.first">
                            /// <summary locid="WinJS.UI.IListBinding.first">
                            /// Retrieves the first item in the list and makes it the current item.
                            /// </summary>
                            /// <returns type="IItemPromise" locid="WinJS.UI.IListBinding.first_returnValue">
                            /// An IItemPromise that serves as a promise for the requested item.
                            /// If the list is empty, the Promise completes with a value of null.
                            /// </returns>
                            /// </signature>

                            return itemPromiseFromSlot(requestSlotAfter(slotsStart));
                        };
                    }

                    if (itemsFromEnd) {
                        listBinding.last = function () {
                            /// <signature helpKeyword="WinJS.UI.IListBinding.last">
                            /// <summary locid="WinJS.UI.IListBinding.last">
                            /// Retrieves the last item in the list and makes it the current item.
                            /// </summary>
                            /// <returns type="IItemPromise" locid="WinJS.UI.IListBinding.last_returnValue">
                            /// An IItemPromise that serves as a promise for the requested item.
                            /// If the list is empty, the Promise completes with a value of null.
                            /// </returns>
                            /// </signature>

                            return itemPromiseFromSlot(requestSlotBefore(slotListEnd));
                        };
                    }

                    if (itemsFromKey) {
                        listBinding.fromKey = function (key, hints) {
                            /// <signature helpKeyword="WinJS.UI.IListBinding.fromKey">
                            /// <summary locid="WinJS.UI.IListBinding.fromKey">
                            /// Retrieves the item with the specified key and makes it the current item.
                            /// </summary>
                            /// <param name="key" type="String" locid="WinJS.UI.IListBinding.fromKey_p:key">
                            /// The key of the requested item. It must be a non-empty string.
                            /// </param>
                            /// <param name="hints" locid="WinJS.UI.IListBinding.fromKey_p:hints">
                            /// Domain-specific hints to the IListDataAdapter
                            /// about the location of the item to improve retrieval time.
                            /// </param>
                            /// <returns type="IItemPromise" locid="WinJS.UI.IListBinding.fromKey_returnValue">
                            /// An IItemPromise that serves as a promise for the requested item.
                            /// If the list doesn't contain an item with the specified key, the Promise completes with a value of null.
                            /// </returns>
                            /// </signature>

                            return itemPromiseFromSlot(slotFromKey(key, hints));
                        };
                    }

                    if (itemsFromIndex || (itemsFromStart && itemsFromKey)) {
                        listBinding.fromIndex = function (index) {
                            /// <signature helpKeyword="WinJS.UI.IListBinding.fromIndex">
                            /// <summary locid="WinJS.UI.IListBinding.fromIndex">
                            /// Retrieves the item with the specified index and makes it the current item.
                            /// </summary>
                            /// <param name="index" type="Nunmber" integer="true" locid="WinJS.UI.IListBinding.fromIndex_p:index">
                            /// A value greater than or equal to 0 that is the index of the item to retrieve.
                            /// </param>
                            /// <returns type="IItemPromise" locid="WinJS.UI.IListBinding.fromIndex_returnValue">
                            /// An IItemPromise that serves as a promise for the requested item.
                            /// If the list doesn't contain an item with the specified index, the IItemPromise completes with a value of null.
                            /// </returns>
                            /// </signature>

                            return itemPromiseFromSlot(slotFromIndex(index));
                        };
                    }

                    if (itemsFromDescription) {
                        listBinding.fromDescription = function (description) {
                            /// <signature helpKeyword="WinJS.UI.IListBinding.fromDescription">
                            /// <summary locid="WinJS.UI.IListBinding.fromDescription">
                            /// Retrieves the item with the specified description and makes it the current item.
                            /// </summary>
                            /// <param name="description" locid="WinJS.UI.IListDataSource.fromDescription_p:description">
                            /// The domain-specific description of the requested item, to be interpreted by the list data adapter.
                            /// </param>
                            /// <returns type="Promise" locid="WinJS.UI.IListDataSource.fromDescription_returnValue">
                            /// A Promise for the requested item. If the list doesn't contain an item with the specified description,
                            /// the IItemPromise completes with a value of null.
                            /// </returns>
                            /// </signature>

                            return itemPromiseFromSlot(slotFromDescription(description));
                        };
                    }

                    return listBinding;
                };

                this.invalidateAll = function () {
                    /// <signature helpKeyword="WinJS.UI.IListDataSource.invalidateAll">
                    /// <summary locid="WinJS.UI.IListDataSource.invalidateAll">
                    /// Makes the data source refresh its cached items by re-requesting them from the data adapter.
                    /// The data source generates notifications if the data has changed.
                    /// </summary>
                    /// <returns type="Promise" locid="WinJS.UI.IListDataSource.invalidateAll_returnValue">
                    /// A Promise that completes when the data has been completely refreshed and all change notifications have been
                    /// sent.
                    /// </returns>
                    /// </signature>

                    return requestRefresh();
                };

                // Create a helper which issues new promises for the result of the input promise
                //  but have their cancelations ref-counted so that any given consumer canceling
                //  their promise doesn't result in the incoming promise being canceled unless
                //  all consumers are no longer interested in the result.
                //
                var countedCancelation = function (incomingPromise, dataSource) {
                    var signal = new _Signal();
                    incomingPromise.then(
                        function (v) { signal.complete(v); },
                        function (e) { signal.error(e); }
                    );
                    var resultPromise = signal.promise.then(null, function (e) {
                        if (e.name === "WinJS.UI.VirtualizedDataSource.resetCount") {
                            getCountPromise = null;
                            return incomingPromise = dataSource.getCount();
                        }
                        return Promise.wrapError(e);
                    });
                    var count = 0;
                    var currentGetCountPromise = {
                        get: function () {
                            count++;
                            return new Promise(
                                function (c, e) { resultPromise.then(c, e); },
                                function () {
                                    if (--count === 0) {
                                        // when the count reaches zero cancel the incoming promise
                                        signal.promise.cancel();
                                        incomingPromise.cancel();
                                        if (currentGetCountPromise === getCountPromise) {
                                            getCountPromise = null;
                                        }
                                    }
                                }
                            );
                        },
                        reset: function () {
                            signal.error(new _ErrorFromName("WinJS.UI.VirtualizedDataSource.resetCount"));
                        },
                        cancel: function () {
                            // if explicitly asked to cancel the incoming promise
                            signal.promise.cancel();
                            incomingPromise.cancel();
                            if (currentGetCountPromise === getCountPromise) {
                                getCountPromise = null;
                            }
                        }
                    };
                    return currentGetCountPromise;
                };

                this.getCount = function () {
                    /// <signature helpKeyword="WinJS.UI.IListDataSource.getCount">
                    /// <summary locid="WinJS.UI.IListDataSource.getCount">
                    /// Retrieves the number of items in the data source.
                    /// </summary>
                    /// </signature>

                    if (listDataAdapter.getCount) {
                        // Always do a fetch, even if there is a cached result
                        //
                        var that = this;
                        return Promise.wrap().then(function () {
                            if (editsInProgress || editsQueued) {
                                return knownCount;
                            }

                            var requestPromise;

                            if (!getCountPromise) {

                                var relatedGetCountPromise;

                                // Make a request for the count
                                //
                                requestPromise = listDataAdapter.getCount();
                                var synchronous;
                                requestPromise.then(
                                    function () {
                                        if (getCountPromise === relatedGetCountPromise) {
                                            getCountPromise = null;
                                        }
                                        synchronous = true;
                                    },
                                    function () {
                                        if (getCountPromise === relatedGetCountPromise) {
                                            getCountPromise = null;
                                        }
                                        synchronous = true;
                                    }
                                );

                                // Every time we make a new request for the count we can consider the
                                //  countDelta to be invalidated
                                //
                                countDelta = 0;

                                // Wrap the result in a cancelation counter which will block cancelation
                                //  of the outstanding promise unless all consumers cancel.
                                //
                                if (!synchronous) {
                                    relatedGetCountPromise = getCountPromise = countedCancelation(requestPromise, that);
                                }
                            }

                            return getCountPromise ? getCountPromise.get() : requestPromise;

                        }).then(function (count) {
                            if (!isNonNegativeInteger(count) && count !== undefined) {
                                throw new _ErrorFromName("WinJS.UI.ListDataSource.InvalidRequestedCountReturned", strings.invalidRequestedCountReturned);
                            }

                            if (count !== knownCount) {
                                if (knownCount === CountResult.unknown) {
                                    knownCount = count;
                                } else {
                                    changeCount(count);
                                    finishNotifications();
                                }
                            }

                            if (count === 0) {
                                if (slotsStart.next !== slotListEnd || slotListEnd.next !== slotsEnd) {
                                    // A contradiction has been found
                                    beginRefresh();
                                } else if (slotsStart.lastInSequence) {
                                    // Now we know the list is empty
                                    mergeSequences(slotsStart);
                                    slotListEnd.index = 0;
                                }
                            }

                            return count;
                        }).then(null, function (error) {
                            if (error.name === _UI.CountError.noResponse) {
                                // Report the failure, but still report last known count
                                setStatus(DataSourceStatus.failure);
                                return knownCount;
                            }
                            return Promise.wrapError(error);
                        });
                    } else {
                        // If the data adapter doesn't support the count method, return the VirtualizedDataSource's
                        //  reckoning of the count.
                        return Promise.wrap(knownCount);
                    }
                };

                if (itemsFromKey) {
                    this.itemFromKey = function (key, hints) {
                        /// <signature helpKeyword="WinJS.UI.IListDataSource.itemFromKey">
                        /// <summary locid="WinJS.UI.IListDataSource.itemFromKey">
                        /// Retrieves the item with the specified key.
                        /// </summary>
                        /// <param name="key" type="String" locid="WinJS.UI.IListDataSource.itemFromKey_p:key">
                        /// The key of the requested item. It must be a non-empty string.
                        /// </param>
                        /// <param name="hints" locid="WinJS.UI.IListDataSource.itemFromKey_p:hints">
                        /// Domain-specific hints to IListDataAdapter about the location of the item
                        /// to improve the retrieval time.
                        /// </param>
                        /// <returns type="Promise" locid="WinJS.UI.IListDataSource.itemFromKey_returnValue">
                        /// A Promise for the requested item. If the list doesn't contain an item with the specified key,
                        /// the Promise completes with a value of null.
                        /// </returns>
                        /// </signature>

                        return itemDirectlyFromSlot(slotFromKey(key, hints));
                    };
                }

                if (itemsFromIndex || (itemsFromStart && itemsFromKey)) {
                    this.itemFromIndex = function (index) {
                        /// <signature helpKeyword="WinJS.UI.IListDataSource.itemFromIndex">
                        /// <summary locid="WinJS.UI.IListDataSource.itemFromIndex">
                        /// Retrieves the item at the specified index.
                        /// </summary>
                        /// <param name="index" type="Number" integer="true" locid="WinJS.UI.IListDataSource.itemFromIndex_p:index">
                        /// A value greater than or equal to zero that is the index of the requested item.
                        /// </param>
                        /// <returns type="Promise" locid="WinJS.UI.IListDataSource.itemFromIndex_returnValue">
                        /// A Promise for the requested item. If the list doesn't contain an item with the specified index,
                        /// the Promise completes with a value of null.
                        /// </returns>
                        /// </signature>

                        return itemDirectlyFromSlot(slotFromIndex(index));
                    };
                }

                if (itemsFromDescription) {
                    this.itemFromDescription = function (description) {
                        /// <signature helpKeyword="WinJS.UI.IListDataSource.itemFromDescription">
                        /// <summary locid="WinJS.UI.IListDataSource.itemFromDescription">
                        /// Retrieves the item with the specified description.
                        /// </summary>
                        /// <param name="description" locid="WinJS.UI.IListDataSource.itemFromDescription_p:description">
                        /// Domain-specific info that describes the item to retrieve, to be interpreted by the IListDataAdapter,
                        /// </param>
                        /// <returns type="Promise" locid="WinJS.UI.IListDataSource.itemFromDescription_returnValue">
                        /// A Promise for the requested item. If the list doesn't contain an item with the specified description,
                        /// the Promise completes with a value of null.
                        /// </returns>
                        /// </signature>

                        return itemDirectlyFromSlot(slotFromDescription(description));
                    };
                }

                this.beginEdits = function () {
                    /// <signature helpKeyword="WinJS.UI.IListDataSource.beginEdits">
                    /// <summary locid="WinJS.UI.IListDataSource.beginEdits">
                    /// Notifies the data source that a sequence of edits is about to begin.  The data source calls
                    /// IListNotificationHandler.beginNotifications and endNotifications each one time for a sequence of edits.
                    /// </summary>
                    /// </signature>

                    editsInProgress = true;
                };

                // Only implement each editing method if the data adapter implements the corresponding ListDataAdapter method

                if (listDataAdapter.insertAtStart) {
                    this.insertAtStart = function (key, data) {
                        /// <signature helpKeyword="WinJS.UI.IListDataSource.insertAtStart">
                        /// <summary locid="WinJS.UI.IListDataSource.insertAtStart">
                        /// Adds an item to the beginning of the data source.
                        /// </summary>
                        /// <param name="key" mayBeNull="true" type="String" locid="WinJS.UI.IListDataSource.insertAtStart_p:key">
                        /// The key of the item to insert, if known; otherwise, null.
                        /// </param>
                        /// <param name="data" locid="WinJS.UI.IListDataSource.insertAtStart_p:data">
                        /// The data for the item to add.
                        /// </param>
                        /// <returns type="Promise" locid="WinJS.UI.IListDataSource.insertAtStart_returnValue">
                        /// A Promise that contains the IItem that was added or an EditError if an error occurred.
                        /// </returns>
                        /// </signature>

                        // Add item to start of list, only notify if the first item was requested
                        return insertItem(
                            key, data,

                            // slotInsertBefore, append
                            (slotsStart.lastInSequence ? null : slotsStart.next), true,

                            // applyEdit
                            function () {
                                return listDataAdapter.insertAtStart(key, data);
                            }
                        );
                    };
                }

                if (listDataAdapter.insertBefore) {
                    this.insertBefore = function (key, data, nextKey) {
                        /// <signature helpKeyword="WinJS.UI.IListDataSource.insertBefore">
                        /// <summary locid="WinJS.UI.IListDataSource.insertBefore">
                        /// Inserts an item before another item.
                        /// </summary>
                        /// <param name="key" mayBeNull="true" type="String" locid="WinJS.UI.IListDataSource.insertBefore_p:key">
                        /// The key of the item to insert, if known; otherwise, null.
                        /// </param>
                        /// <param name="data" locid="WinJS.UI.IListDataSource.insertBefore_p:data">
                        /// The data for the item to insert.
                        /// </param>
                        /// <param name="nextKey" type="String" locid="WinJS.UI.IListDataSource.insertBefore_p:nextKey">
                        /// The key of an item in the data source. The new data is inserted before this item.
                        /// </param>
                        /// <returns type="Promise" locid="WinJS.UI.IListDataSource.insertBefore_returnValue">
                        /// A Promise that contains the IItem that was added or an EditError if an error occurred.
                        /// </returns>
                        /// </signature>

                        var slotNext = getSlotForEdit(nextKey);

                        // Add item before given item and send notification
                        return insertItem(
                            key, data,

                            // slotInsertBefore, append
                            slotNext, false,

                            // applyEdit
                            function () {
                                return listDataAdapter.insertBefore(key, data, nextKey, adjustedIndex(slotNext));
                            }
                        );
                    };
                }

                if (listDataAdapter.insertAfter) {
                    this.insertAfter = function (key, data, previousKey) {
                        /// <signature helpKeyword="WinJS.UI.IListDataSource.insertAfter">
                        /// <summary locid="WinJS.UI.IListDataSource.insertAfter">
                        /// Inserts an item after another item.
                        /// </summary>
                        /// <param name="key" mayBeNull="true" type="String" locid="WinJS.UI.IListDataSource.insertAfter_p:key">
                        /// The key of the item to insert, if known; otherwise, null.
                        /// </param>
                        /// <param name="data" locid="WinJS.UI.IListDataSource.insertAfter_p:data">
                        /// The data for the item to insert.
                        /// </param>
                        /// <param name="previousKey" type="String" locid="WinJS.UI.IListDataSource.insertAfter_p:previousKey">
                        /// The key for an item in the data source. The new item is added after this item.
                        /// </param>
                        /// <returns type="Promise" locid="WinJS.UI.IListDataSource.insertAfter_returnValue">
                        /// A Promise that contains the IItem that was added or an EditError if an error occurred.
                        /// </returns>
                        /// </signature>

                        var slotPrev = getSlotForEdit(previousKey);

                        // Add item after given item and send notification
                        return insertItem(
                            key, data,

                            // slotInsertBefore, append
                            (slotPrev ? slotPrev.next : null), true,

                            // applyEdit
                            function () {
                                return listDataAdapter.insertAfter(key, data, previousKey, adjustedIndex(slotPrev));
                            }
                        );
                    };
                }

                if (listDataAdapter.insertAtEnd) {
                    this.insertAtEnd = function (key, data) {
                        /// <signature helpKeyword="WinJS.UI.IListDataSource.insertAtEnd">
                        /// <summary locid="WinJS.UI.IListDataSource.insertAtEnd">
                        /// Adds an item to the end of the data source.
                        /// </summary>
                        /// <param name="key" mayBeNull="true" type="String" locid="WinJS.UI.IListDataSource.insertAtEnd_p:key">
                        /// The key of the item to insert, if known; otherwise, null.
                        /// </param>
                        /// <param name="data" locid="WinJS.UI.IListDataSource.insertAtEnd_data">
                        /// The data for the item to insert.
                        /// </param>
                        /// <returns type="Promise" locid="WinJS.UI.IListDataSource.insertAtEnd_returnValue">
                        /// A Promise that contains the IItem that was added or an EditError if an error occurred.
                        /// </returns>
                        /// </signature>

                        // Add item to end of list, only notify if the last item was requested
                        return insertItem(
                            key, data,

                            // slotInsertBefore, append
                            (slotListEnd.firstInSequence ? null : slotListEnd), false,

                            // applyEdit
                            function () {
                                return listDataAdapter.insertAtEnd(key, data);
                            }
                        );
                    };
                }

                if (listDataAdapter.change) {
                    this.change = function (key, newData) {
                        /// <signature helpKeyword="WinJS.UI.IListDataSource.change">
                        /// <summary locid="WinJS.UI.IListDataSource.change">
                        /// Overwrites the data of the specified item.
                        /// </summary>
                        /// <param name="key" type="String" locid="WinJS.UI.IListDataSource.change_p:key">
                        /// The key for the item to replace.
                        /// </param>
                        /// <param name="newData" type="Object" locid="WinJS.UI.IListDataSource.change_p:newData">
                        /// The new data for the item.
                        /// </param>
                        /// <returns type="Promise" locid="WinJS.UI.IListDataSource.change_returnValue">
                        /// A Promise that contains the IItem that was updated or an EditError if an error occurred.
                        /// </returns>
                        /// </signature>

                        var slot = getSlotForEdit(key);

                        return new Promise(function (complete, error) {
                            var itemOld;

                            queueEdit(
                                // applyEdit
                                function () {
                                    return listDataAdapter.change(key, newData, adjustedIndex(slot));
                                },

                                EditType.change, complete, error,

                                // keyUpdate
                                null,

                                // updateSlots
                                function () {
                                    itemOld = slot.item;

                                    slot.itemNew = {
                                        key: key,
                                        data: newData
                                    };

                                    if (itemOld) {
                                        changeSlot(slot);
                                    } else {
                                        completeFetchPromises(slot);
                                    }
                                },

                                // undo
                                function () {
                                    if (itemOld) {
                                        slot.itemNew = itemOld;
                                        changeSlot(slot);
                                    } else {
                                        beginRefresh();
                                    }
                                }
                            );
                        });
                    };
                }

                if (listDataAdapter.moveToStart) {
                    this.moveToStart = function (key) {
                        /// <signature helpKeyword="WinJS.UI.IListDataSource.moveToStart">
                        /// <summary locid="WinJS.UI.IListDataSource.moveToStart">
                        /// Moves the specified item to the beginning of the data source.
                        /// </summary>
                        /// <param name="key" type="String" locid="WinJS.UI.IListDataSource.moveToStart_p:key">
                        /// The key of the item to move.
                        /// </param>
                        /// <returns type="Promise" locid="WinJS.UI.IListDataSource.moveToStart_returnValue">
                        /// A Promise that contains the IItem that was moved or an EditError if an error occurred.
                        /// </returns>
                        /// </signature>

                        var slot = getSlotForEdit(key);

                        return moveItem(
                            slot,

                            // slotMoveBefore, append
                            slotsStart.next, true,

                            // applyEdit
                            function () {
                                return listDataAdapter.moveToStart(key, adjustedIndex(slot));
                            }
                        );
                    };
                }

                if (listDataAdapter.moveBefore) {
                    this.moveBefore = function (key, nextKey) {
                        /// <signature helpKeyword="WinJS.UI.IListDataSource.moveBefore">
                        /// <summary locid="WinJS.UI.IListDataSource.moveBefore">
                        /// Moves the specified item before another item.
                        /// </summary>
                        /// <param name="key" type="String" locid="WinJS.UI.IListDataSource.moveBefore_p:key">
                        /// The key of the item to move.
                        /// </param>
                        /// <param name="nextKey" type="String" locid="WinJS.UI.IListDataSource.moveBefore_p:nextKey">
                        /// The key of another item in the data source. The item specified by the key parameter
                        /// is moved to a position immediately before this item.
                        /// </param>
                        /// <returns type="Promise" locid="WinJS.UI.IListDataSource.moveBefore_returnValue">
                        /// A Promise that contains the IItem that was moved or an EditError if an error occurred.
                        /// </returns>
                        /// </signature>

                        var slot = getSlotForEdit(key),
                            slotNext = getSlotForEdit(nextKey);

                        return moveItem(
                            slot,

                            // slotMoveBefore, append
                            slotNext, false,

                            // applyEdit
                            function () {
                                return listDataAdapter.moveBefore(key, nextKey, adjustedIndex(slot), adjustedIndex(slotNext));
                            }
                        );
                    };
                }

                if (listDataAdapter.moveAfter) {
                    this.moveAfter = function (key, previousKey) {
                        /// <signature helpKeyword="WinJS.UI.IListDataSource.moveAfter">
                        /// <summary locid="WinJS.UI.IListDataSource.moveAfter">
                        /// Moves an item after another item.
                        /// </summary>
                        /// <param name="key" type="String" locid="WinJS.UI.IListDataSource.moveAfter_p:key">
                        /// The key of the item to move.
                        /// </param>
                        /// <param name="previousKey" type="String" locid="WinJS.UI.IListDataSource.moveAfter_p:previousKey">
                        /// The key of another item in the data source. The item specified by the key parameter will
                        /// is moved to a position immediately after this item.
                        /// </param>
                        /// <returns type="Promise" locid="WinJS.UI.IListDataSource.moveAfter_returnValue">
                        /// A Promise that contains the IItem that was moved or an EditError if an error occurred.
                        /// </returns>
                        /// </signature>

                        var slot = getSlotForEdit(key),
                            slotPrev = getSlotForEdit(previousKey);

                        return moveItem(
                            slot,

                            // slotMoveBefore, append
                            slotPrev.next, true,

                            // applyEdit
                            function () {
                                return listDataAdapter.moveAfter(key, previousKey, adjustedIndex(slot), adjustedIndex(slotPrev));
                            }
                        );
                    };
                }

                if (listDataAdapter.moveToEnd) {
                    this.moveToEnd = function (key) {
                        /// <signature helpKeyword="WinJS.UI.IListDataSource.moveToEnd">
                        /// <summary locid="WinJS.UI.IListDataSource.moveToEnd">
                        /// Moves an item to the end of the data source.
                        /// </summary>
                        /// <param name="key" type="String" locid="WinJS.UI.IListDataSource.moveToEnd_p:key">
                        /// The key of the item to move.
                        /// </param>
                        /// <returns type="Promise" locid="WinJS.UI.IListDataSource.moveToEnd_returnValue">
                        /// A Promise that contains the IItem that was moved or an EditError if an error occurred.
                        /// </returns>
                        /// </signature>

                        var slot = getSlotForEdit(key);

                        return moveItem(
                            slot,

                            // slotMoveBefore, append
                            slotListEnd, false,

                            // applyEdit
                            function () {
                                return listDataAdapter.moveToEnd(key, adjustedIndex(slot));
                            }
                        );
                    };
                }

                if (listDataAdapter.remove) {
                    this.remove = function (key) {
                        /// <signature helpKeyword="WinJS.UI.IListDataSource.remove">
                        /// <summary locid="WinJS.UI.IListDataSource.remove">
                        /// Removes an item from the data source.
                        /// </summary>
                        /// <param name="key" type="String" locid="WinJS.UI.IListDataSource.remove_p:key">
                        /// The key of the item to remove.
                        /// </param>
                        /// <returns type="Promise" locid="WinJS.UI.IListDataSource.remove_returnValue">
                        /// A Promise that contains nothing if the operation was successful or an EditError if an error occurred.
                        /// </returns>
                        /// </signature>

                        validateKey(key);

                        var slot = keyMap[key];

                        return new Promise(function (complete, error) {
                            var slotNext,
                                firstInSequence,
                                lastInSequence;

                            queueEdit(
                                // applyEdit
                                function () {
                                    return listDataAdapter.remove(key, adjustedIndex(slot));
                                },

                                EditType.remove, complete, error,

                                // keyUpdate
                                null,

                                // updateSlots
                                function () {
                                    if (slot) {
                                        slotNext = slot.next;
                                        firstInSequence = slot.firstInSequence;
                                        lastInSequence = slot.lastInSequence;

                                        updateNewIndices(slot, -1);
                                        deleteSlot(slot, false);
                                    }
                                },

                                // undo
                                function () {
                                    if (slot) {
                                        reinsertSlot(slot, slotNext, !firstInSequence, !lastInSequence);
                                        updateNewIndices(slot, 1);
                                        sendInsertedNotification(slot);
                                    }
                                }
                            );
                        });
                    };
                }

                this.endEdits = function () {
                    /// <signature helpKeyword="WinJS.UI.IListDataSource.endEdits">
                    /// <summary locid="WinJS.UI.IListDataSource.endEdits">
                    /// Notifies the data source that a sequence of edits has ended.  The data source will call
                    /// IListNotificationHandler.beginNotifications and endNotifications once each for a sequence of edits.
                    /// </summary>
                    /// </signature>

                    editsInProgress = false;
                    completeEdits();
                };

            } // _baseDataSourceConstructor

            var VDS = _Base.Class.define(function () {
                /// <signature helpKeyword="WinJS.UI.VirtualizedDataSource">
                /// <summary locid="WinJS.UI.VirtualizedDataSource">
                /// Use as a base class when defining a custom data source. Do not instantiate directly.
                /// </summary>
                /// <event name="statuschanged" locid="WinJS.UI.VirtualizedDataSource_e:statuschanged">
                /// Raised when the status of the VirtualizedDataSource changes between ready, waiting, and failure states.
                /// </event>
                /// </signature>
            }, {
                _baseDataSourceConstructor: _baseDataSourceConstructor,
                _isVirtualizedDataSource: true
            }, { // Static Members
                supportedForProcessing: false
            });
            _Base.Class.mix(VDS, _Events.eventMixin);
            return VDS;
        })

    });

});
// Copyright (c) Microsoft Corporation.  All Rights Reserved. Licensed under the MIT License. See License.txt in the project root for license information.
// Group Data Source

define('WinJS/VirtualizedDataSource/_GroupDataSource',[
    'exports',
    '../Core/_Base',
    '../Core/_ErrorFromName',
    '../Promise',
    '../Scheduler',
    '../Utilities/_UI',
    './_VirtualizedDataSourceImpl'
    ], function groupDataSourceInit(exports, _Base, _ErrorFromName, Promise, Scheduler, _UI, VirtualizedDataSource) {
    "use strict";

    _Base.Namespace._moduleDefine(exports, "WinJS.UI", {

        _GroupDataSource: _Base.Namespace._lazy(function () {

            // Private statics

            function errorDoesNotExist() {
                return new _ErrorFromName(_UI.FetchError.doesNotExist);
            }

            var batchSizeDefault = 101;

            function groupReady(group) {
                return group && group.firstReached && group.lastReached;
            }

            var ListNotificationHandler = _Base.Class.define(function ListNotificationHandler_ctor(groupDataAdapter) {
                // Constructor

                this._groupDataAdapter = groupDataAdapter;
            }, {
                // Public methods

                beginNotifications: function () {
                },

                // itemAvailable: not implemented

                inserted: function (itemPromise, previousHandle, nextHandle) {
                    this._groupDataAdapter._inserted(itemPromise, previousHandle, nextHandle);
                },

                changed: function (newItem, oldItem) {
                    this._groupDataAdapter._changed(newItem, oldItem);
                },

                moved: function (itemPromise, previousHandle, nextHandle) {
                    this._groupDataAdapter._moved(itemPromise, previousHandle, nextHandle);
                },

                removed: function (handle, mirage) {
                    this._groupDataAdapter._removed(handle, mirage);
                },

                countChanged: function (newCount, oldCount) {
                    if (newCount === 0 && oldCount !== 0) {
                        this._groupDataAdapter.invalidateGroups();
                    }
                },

                indexChanged: function (handle, newIndex, oldIndex) {
                    this._groupDataAdapter._indexChanged(handle, newIndex, oldIndex);
                },

                endNotifications: function () {
                    this._groupDataAdapter._endNotifications();
                },

                reload: function () {
                    this._groupDataAdapter._reload();
                }
            }, {
                supportedForProcessing: false,
            });

            var GroupDataAdapter = _Base.Class.define(function GroupDataAdapater_ctor(listDataSource, groupKey, groupData, options) {
                // Constructor

                this._listBinding = listDataSource.createListBinding(new ListNotificationHandler(this));

                this._groupKey = groupKey;
                this._groupData = groupData;

                // _initializeState clears the count, so call this before processing the groupCountEstimate option
                this._initializeState();

                this._batchSize = batchSizeDefault;
                this._count = null;

                if (options) {
                    if (typeof options.groupCountEstimate === "number") {
                        this._count = (options.groupCountEstimate < 0 ? null : Math.max(options.groupCountEstimate, 1));
                    }
                    if (typeof options.batchSize === "number") {
                        this._batchSize = options.batchSize + 1;
                    }
                }

                if (this._listBinding.last) {
                    this.itemsFromEnd = function (count) {
                        var that = this;
                        return this._fetchItems(
                            // getGroup
                            function () {
                                return that._lastGroup;
                            },

                            // mayExist
                            function (failed) {
                                if (failed) {
                                    return false;
                                }
                                var count = that._count;
                                if (+count !== count) {
                                    return true;
                                }
                                if (count > 0) {
                                    return true;
                                }
                            },

                            // fetchInitialBatch
                            function () {
                                that._fetchBatch(that._listBinding.last(), that._batchSize - 1, 0);
                            },

                            count - 1, 0
                        );
                    };
                }
            }, {
                // Public members

                setNotificationHandler: function (notificationHandler) {
                    this._listDataNotificationHandler = notificationHandler;
                },

                // The ListDataSource should always compare these items by identity; in rare cases, it will do some unnecessary
                // rerendering, but at least fetching will not stringify items we already know to be valid and that we know
                // have not changed.
                compareByIdentity: true,

                // itemsFromStart: not implemented

                // itemsFromEnd: implemented in constructor

                itemsFromKey: function (key, countBefore, countAfter, hints) {
                    var that = this;
                    return this._fetchItems(
                        // getGroup
                        function () {
                            return that._keyMap[key];
                        },

                        // mayExist
                        function () {
                            var lastGroup = that._lastGroup;
                            if (!lastGroup) {
                                return true;
                            }
                            if (+lastGroup.index !== lastGroup.index) {
                                return true;
                            }
                        },

                        // fetchInitialBatch
                        function () {
                            hints = hints || {};
                            var itemPromise = (
                                typeof hints.groupMemberKey === "string" && that._listBinding.fromKey ?
                                    that._listBinding.fromKey(hints.groupMemberKey) :
                                typeof hints.groupMemberIndex === "number" && that._listBinding.fromIndex ?
                                    that._listBinding.fromIndex(hints.groupMemberIndex) :
                                hints.groupMemberDescription !== undefined && that._listBinding.fromDescription ?
                                    that._listBinding.fromDescription(hints.groupMemberDescription) :
                                    that._listBinding.first()
                            );

                            var fetchBefore = Math.floor(0.5 * (that._batchSize - 1));
                            that._fetchBatch(itemPromise, fetchBefore, that._batchSize - 1 - fetchBefore);
                        },

                        countBefore, countAfter
                    );
                },

                itemsFromIndex: function (index, countBefore, countAfter) {
                    var that = this;
                    return this._fetchItems(
                        // getGroup
                        function () {
                            return that._indexMap[index];
                        },

                        // mayExist
                        function () {
                            var lastGroup = that._lastGroup;
                            if (!lastGroup) {
                                return true;
                            }
                            if (+lastGroup.index !== lastGroup.index) {
                                return true;
                            }
                            if (index <= lastGroup.index) {
                                return true;
                            }
                        },

                        // fetchInitialBatch
                        function () {
                            that._fetchNextIndex();
                        },

                        countBefore, countAfter
                    );
                },

                // itemsFromDescription: not implemented

                getCount: function () {
                    if (this._lastGroup && typeof this._lastGroup.index === "number") {
                        return Promise.wrap(this._count);
                    } else {
                        // Even if there's a current estimate for _count, consider this call to be a request to determine the true
                        // count.

                        var that = this;
                        var countPromise = new Promise(function (complete) {
                            var fetch = {
                                initialBatch: function () {
                                    that._fetchNextIndex();
                                },
                                getGroup: function () { return null; },
                                countBefore: 0,
                                countAfter: 0,
                                complete: function (failed) {
                                    if (failed) {
                                        that._count = 0;
                                    }

                                    var count = that._count;
                                    if (typeof count === "number") {
                                        complete(count);
                                        return true;
                                    } else {
                                        return false;
                                    }
                                }
                            };

                            that._fetchQueue.push(fetch);

                            if (!that._itemBatch) {
                                that._continueFetch(fetch);
                            }
                        });

                        return (typeof this._count === "number" ? Promise.wrap(this._count) : countPromise);
                    }
                },

                invalidateGroups: function () {
                    this._beginRefresh();
                    this._initializeState();
                },

                // Editing methods not implemented

                // Private members

                _initializeState: function () {
                    this._count = null;
                    this._indexMax = null;

                    this._keyMap = {};
                    this._indexMap = {};
                    this._lastGroup = null;
                    this._handleMap = {};

                    this._fetchQueue = [];

                    this._itemBatch = null;
                    this._itemsToFetch = 0;

                    this._indicesChanged = false;
                },

                _releaseItem: function (item) {
                    delete this._handleMap[item.handle];
                    this._listBinding.releaseItem(item);
                },

                _processBatch: function () {
                    var previousItem = null,
                        previousGroup = null,
                        firstItemInGroup = null,
                        itemsSinceStart = 0,
                        failed = true;
                    for (var i = 0; i < this._batchSize; i++) {
                        var item = this._itemBatch[i],
                            groupKey = (item ? this._groupKey(item) : null);

                        if (item) {
                            failed = false;
                        }

                        if (previousGroup && groupKey !== null && groupKey === previousGroup.key) {
                            // This item is in the same group as the last item.  The only thing to do is advance the group's
                            // lastItem if this is definitely the last item that has been processed for the group.
                            itemsSinceStart++;
                            if (previousGroup.lastItem === previousItem) {
                                if (previousGroup.lastItem.handle !== previousGroup.firstItem.handle) {
                                    this._releaseItem(previousGroup.lastItem);
                                }
                                previousGroup.lastItem = item;
                                this._handleMap[item.handle] = previousGroup;

                                previousGroup.size++;
                            } else if (previousGroup.firstItem === item) {
                                if (previousGroup.firstItem.handle !== previousGroup.lastItem.handle) {
                                    this._releaseItem(previousGroup.firstItem);
                                }
                                previousGroup.firstItem = firstItemInGroup;
                                this._handleMap[firstItemInGroup.handle] = previousGroup;

                                previousGroup.size += itemsSinceStart;
                            }
                        } else {
                            var index = null;

                            if (previousGroup) {
                                previousGroup.lastReached = true;

                                if (typeof previousGroup.index === "number") {
                                    index = previousGroup.index + 1;
                                }
                            }

                            if (item) {
                                // See if the current group has already been processed
                                var group = this._keyMap[groupKey];

                                if (!group) {
                                    group = {
                                        key: groupKey,
                                        data: this._groupData(item),
                                        firstItem: item,
                                        lastItem: item,
                                        size: 1
                                    };
                                    this._keyMap[group.key] = group;
                                    this._handleMap[item.handle] = group;
                                }

                                if (i > 0) {
                                    group.firstReached = true;

                                    if (!previousGroup) {
                                        index = 0;
                                    }
                                }

                                if (typeof group.index !== "number" && typeof index === "number") {
                                    // Set the indices of as many groups as possible
                                    for (var group2 = group; group2; group2 = this._nextGroup(group2)) {
                                        group2.index = index;
                                        this._indexMap[index] = group2;

                                        index++;
                                    }

                                    this._indexMax = index;
                                    if (typeof this._count === "number" && !this._lastGroup && this._count <= this._indexMax) {
                                        this._count = this._indexMax + 1;
                                    }
                                }

                                firstItemInGroup = item;
                                itemsSinceStart = 0;

                                previousGroup = group;
                            } else {
                                if (previousGroup) {
                                    this._lastGroup = previousGroup;

                                    if (typeof previousGroup.index === "number") {
                                        this._count = (previousGroup.index + 1);
                                    }

                                    // Force a client refresh (which should be fast) to ensure that a countChanged notification is
                                    // sent.
                                    this._listDataNotificationHandler.invalidateAll();

                                    previousGroup = null;
                                }
                            }
                        }

                        previousItem = item;
                    }

                    // See how many fetches have now completed
                    var fetch;
                    for (fetch = this._fetchQueue[0]; fetch && fetch.complete(failed) ; fetch = this._fetchQueue[0]) {
                        this._fetchQueue.splice(0, 1);
                    }

                    // Continue work on the next fetch, if any
                    if (fetch) {
                        var that = this;
                        // Avoid reentering _processBatch
                        Scheduler.schedule(function GroupDataSource_async_processBatch() {
                            that._continueFetch(fetch);
                        }, Scheduler.Priority.normal, null, "WinJS.UI._GroupDataSource._continueFetch");
                    } else {
                        this._itemBatch = null;
                    }
                },

                _processPromise: function (itemPromise, batchIndex) {
                    itemPromise.retain();

                    this._itemBatch[batchIndex] = itemPromise;

                    var that = this;
                    itemPromise.then(function (item) {
                        that._itemBatch[batchIndex] = item;
                        if (--that._itemsToFetch === 0) {
                            that._processBatch();
                        }
                    });
                },

                _fetchBatch: function (itemPromise, countBefore) {
                    this._itemBatch = new Array(this._batchSize);
                    this._itemsToFetch = this._batchSize;

                    this._processPromise(itemPromise, countBefore);

                    var batchIndex;

                    this._listBinding.jumpToItem(itemPromise);
                    for (batchIndex = countBefore - 1; batchIndex >= 0; batchIndex--) {
                        this._processPromise(this._listBinding.previous(), batchIndex);
                    }

                    this._listBinding.jumpToItem(itemPromise);
                    for (batchIndex = countBefore + 1; batchIndex < this._batchSize; batchIndex++) {
                        this._processPromise(this._listBinding.next(), batchIndex);
                    }
                },

                _fetchAdjacent: function (item, after) {
                    // Batches overlap by one so group boundaries always fall within at least one batch
                    this._fetchBatch(
                        (this._listBinding.fromKey ? this._listBinding.fromKey(item.key) : this._listBinding.fromIndex(item.index)),
                        (after ? 0 : this._batchSize - 1),
                        (after ? this._batchSize - 1 : 0)
                    );
                },

                _fetchNextIndex: function () {
                    var groupHighestIndex = this._indexMap[this._indexMax - 1];
                    if (groupHighestIndex) {
                        // We've already fetched some of the first items, so continue where we left off
                        this._fetchAdjacent(groupHighestIndex.lastItem, true);
                    } else {
                        // Fetch one non-existent item before the list so _processBatch knows the start was reached
                        this._fetchBatch(this._listBinding.first(), 1, this._batchSize - 2);
                    }
                },

                _continueFetch: function (fetch) {
                    if (fetch.initialBatch) {
                        fetch.initialBatch();
                        fetch.initialBatch = null;
                    } else {
                        var group = fetch.getGroup();
                        if (group) {
                            var groupPrev,
                                groupNext;

                            if (!group.firstReached) {
                                this._fetchAdjacent(group.firstItem, false);
                            } else if (!group.lastReached) {
                                this._fetchAdjacent(group.lastItem, true);
                            } else if (fetch.countBefore > 0 && group.index !== 0 && !groupReady(groupPrev = this._previousGroup(group))) {
                                this._fetchAdjacent((groupPrev && groupPrev.lastReached ? groupPrev.firstItem : group.firstItem), false);
                            } else {
                                groupNext = this._nextGroup(group);
                                this._fetchAdjacent((groupNext && groupNext.firstReached ? groupNext.lastItem : group.lastItem), true);
                            }
                        } else {
                            // Assume we're searching for a key, index or the count by brute force
                            this._fetchNextIndex();
                        }
                    }
                },

                _fetchComplete: function (group, countBefore, countAfter, firstRequest, complete) {
                    if (groupReady(group)) {
                        // Check if the minimal requirements for the request are met
                        var groupPrev = this._previousGroup(group);
                        if (firstRequest || groupReady(groupPrev) || group.index === 0 || countBefore === 0) {
                            var groupNext = this._nextGroup(group);
                            if (firstRequest || groupReady(groupNext) || this._lastGroup === group || countAfter === 0) {
                                // Time to return the fetch results

                                // Find the first available group to return (don't return more than asked for)
                                var countAvailableBefore = 0,
                                    groupFirst = group;
                                while (countAvailableBefore < countBefore) {
                                    groupPrev = this._previousGroup(groupFirst);

                                    if (!groupReady(groupPrev)) {
                                        break;
                                    }

                                    groupFirst = groupPrev;
                                    countAvailableBefore++;
                                }

                                // Find the last available group to return
                                var countAvailableAfter = 0,
                                    groupLast = group;
                                while (countAvailableAfter < countAfter) {
                                    groupNext = this._nextGroup(groupLast);

                                    if (!groupReady(groupNext)) {
                                        break;
                                    }

                                    groupLast = groupNext;
                                    countAvailableAfter++;
                                }

                                // Now create the items to return
                                var len = countAvailableBefore + 1 + countAvailableAfter,
                                    items = new Array(len);

                                for (var i = 0; i < len; i++) {
                                    var item = {
                                        key: groupFirst.key,
                                        data: groupFirst.data,
                                        firstItemKey: groupFirst.firstItem.key,
                                        groupSize: groupFirst.size
                                    };

                                    var firstItemIndex = groupFirst.firstItem.index;
                                    if (typeof firstItemIndex === "number") {
                                        item.firstItemIndexHint = firstItemIndex;
                                    }

                                    items[i] = item;

                                    groupFirst = this._nextGroup(groupFirst);
                                }

                                var result = {
                                    items: items,
                                    offset: countAvailableBefore
                                };

                                result.totalCount = (
                                    typeof this._count === "number" ?
                                        this._count :
                                        _UI.CountResult.unknown
                                );

                                if (typeof group.index === "number") {
                                    result.absoluteIndex = group.index;
                                }

                                if (groupLast === this._lastGroup) {
                                    result.atEnd = true;
                                }

                                complete(result);
                                return true;
                            }
                        }
                    }

                    return false;
                },

                _fetchItems: function (getGroup, mayExist, fetchInitialBatch, countBefore, countAfter) {
                    var that = this;
                    return new Promise(function (complete, error) {
                        var group = getGroup(),
                            firstRequest = !group,
                            failureCount = 0;

                        function fetchComplete(failed) {
                            var group2 = getGroup();

                            if (group2) {
                                return that._fetchComplete(group2, countBefore, countAfter, firstRequest, complete, error);
                            } else if (firstRequest && !mayExist(failed)) {
                                error(errorDoesNotExist());
                                return true;
                            } else if (failureCount > 2) {
                                error(errorDoesNotExist());
                                return true;
                            } else {
                                // only consider consecutive failures
                                if (failed) {
                                    failureCount++;
                                } else {
                                    failureCount = 0;
                                }
                                // _continueFetch will switch to a brute force search
                                return false;
                            }
                        }

                        if (!fetchComplete()) {
                            var fetch = {
                                initialBatch: firstRequest ? fetchInitialBatch : null,
                                getGroup: getGroup,
                                countBefore: countBefore,
                                countAfter: countAfter,
                                complete: fetchComplete
                            };

                            that._fetchQueue.push(fetch);

                            if (!that._itemBatch) {
                                that._continueFetch(fetch);
                            }
                        }
                    });
                },

                _previousGroup: function (group) {
                    if (group && group.firstReached) {
                        this._listBinding.jumpToItem(group.firstItem);

                        return this._handleMap[this._listBinding.previous().handle];
                    } else {
                        return null;
                    }
                },

                _nextGroup: function (group) {
                    if (group && group.lastReached) {
                        this._listBinding.jumpToItem(group.lastItem);

                        return this._handleMap[this._listBinding.next().handle];
                    } else {
                        return null;
                    }
                },

                _invalidateIndices: function (group) {
                    this._count = null;
                    this._lastGroup = null;

                    if (typeof group.index === "number") {
                        this._indexMax = (group.index > 0 ? group.index : null);
                    }

                    // Delete the indices of this and all subsequent groups
                    for (var group2 = group; group2 && typeof group2.index === "number"; group2 = this._nextGroup(group2)) {
                        delete this._indexMap[group2.index];
                        group2.index = null;
                    }
                },

                _releaseGroup: function (group) {
                    this._invalidateIndices(group);

                    delete this._keyMap[group.key];

                    if (this._lastGroup === group) {
                        this._lastGroup = null;
                    }

                    if (group.firstItem !== group.lastItem) {
                        this._releaseItem(group.firstItem);
                    }
                    this._releaseItem(group.lastItem);
                },

                _beginRefresh: function () {
                    // Abandon all current fetches

                    this._fetchQueue = [];

                    if (this._itemBatch) {
                        for (var i = 0; i < this._batchSize; i++) {
                            var item = this._itemBatch[i];
                            if (item) {
                                if (item.cancel) {
                                    item.cancel();
                                }
                                this._listBinding.releaseItem(item);
                            }
                        }

                        this._itemBatch = null;
                    }

                    this._itemsToFetch = 0;

                    this._listDataNotificationHandler.invalidateAll();
                },

                _processInsertion: function (item, previousHandle, nextHandle) {
                    var groupPrev = this._handleMap[previousHandle],
                        groupNext = this._handleMap[nextHandle],
                        groupKey = null;

                    if (groupPrev) {
                        // If an item in a different group from groupPrev is being inserted after it, no need to discard groupPrev
                        if (!groupPrev.lastReached || previousHandle !== groupPrev.lastItem.handle || (groupKey = this._groupKey(item)) === groupPrev.key) {
                            this._releaseGroup(groupPrev);
                        } else if (this._lastGroup === groupPrev) {
                            this._lastGroup = null;
                            this._count = null;
                        }
                        this._beginRefresh();
                    }

                    if (groupNext && groupNext !== groupPrev) {
                        this._invalidateIndices(groupNext);

                        // If an item in a different group from groupNext is being inserted before it, no need to discard groupNext
                        if (!groupNext.firstReached || nextHandle !== groupNext.firstItem.handle || (groupKey !== null ? groupKey : this._groupKey(item)) === groupNext.key) {
                            this._releaseGroup(groupNext);
                        }
                        this._beginRefresh();
                    }
                },

                _processRemoval: function (handle) {
                    var group = this._handleMap[handle];

                    if (group && (handle === group.firstItem.handle || handle === group.lastItem.handle)) {
                        this._releaseGroup(group);
                        this._beginRefresh();
                    } else if (this._itemBatch) {
                        for (var i = 0; i < this._batchSize; i++) {
                            var item = this._itemBatch[i];
                            if (item && item.handle === handle) {
                                this._beginRefresh();
                                break;
                            }
                        }
                    }
                },

                _inserted: function (itemPromise, previousHandle, nextHandle) {
                    var that = this;
                    itemPromise.then(function (item) {
                        that._processInsertion(item, previousHandle, nextHandle);
                    });
                },

                _changed: function (newItem, oldItem) {
                    // A change to the first item could affect the group item
                    var group = this._handleMap[newItem.handle];
                    if (group && newItem.handle === group.firstItem.handle) {
                        this._releaseGroup(group);
                        this._beginRefresh();
                    }

                    // If the item is now in a different group, treat this as a move
                    if (this._groupKey(newItem) !== this._groupKey(oldItem)) {
                        this._listBinding.jumpToItem(newItem);
                        var previousHandle = this._listBinding.previous().handle;
                        this._listBinding.jumpToItem(newItem);
                        var nextHandle = this._listBinding.next().handle;

                        this._processRemoval(newItem.handle);
                        this._processInsertion(newItem, previousHandle, nextHandle);
                    }
                },

                _moved: function (itemPromise, previousHandle, nextHandle) {
                    this._processRemoval(itemPromise.handle);

                    var that = this;
                    itemPromise.then(function (item) {
                        that._processInsertion(item, previousHandle, nextHandle);
                    });
                },

                _removed: function (handle, mirage) {
                    // Mirage removals will just result in null items, which can be ignored
                    if (!mirage) {
                        this._processRemoval(handle);
                    }
                },

                _indexChanged: function (handle, newIndex, oldIndex) {
                    if (typeof oldIndex === "number") {
                        this._indicesChanged = true;
                    }
                },

                _endNotifications: function () {
                    if (this._indicesChanged) {
                        this._indicesChanged = false;

                        // Update the group sizes
                        for (var key in this._keyMap) {
                            var group = this._keyMap[key];

                            if (group.firstReached && group.lastReached) {
                                var newSize = group.lastItem.index + 1 - group.firstItem.index;
                                if (!isNaN(newSize)) {
                                    group.size = newSize;
                                }
                            }
                        }

                        // Invalidate the client, since some firstItemIndexHint properties have probably changed
                        this._beginRefresh();
                    }
                },

                _reload: function () {
                    this._initializeState();
                    this._listDataNotificationHandler.reload();
                }
            }, {
                supportedForProcessing: false,
            });

            return _Base.Class.derive(VirtualizedDataSource.VirtualizedDataSource, function (listDataSource, groupKey, groupData, options) {
                var groupDataAdapter = new GroupDataAdapter(listDataSource, groupKey, groupData, options);

                this._baseDataSourceConstructor(groupDataAdapter);

                this.extensions = {
                    invalidateGroups: function () {
                        groupDataAdapter.invalidateGroups();
                    }
                };
            }, {
                /* empty */
            }, {
                supportedForProcessing: false,
            });
        })

    });

});


// Copyright (c) Microsoft Corporation.  All Rights Reserved. Licensed under the MIT License. See License.txt in the project root for license information.
// Grouped Item Data Source

define('WinJS/VirtualizedDataSource/_GroupedItemDataSource',[
    '../Core/_Base',
    './_GroupDataSource'
    ], function groupedItemDataSourceInit(_Base, _GroupDataSource) {
    "use strict";

    _Base.Namespace.define("WinJS.UI", {

        computeDataSourceGroups: function (listDataSource, groupKey, groupData, options) {
            /// <signature helpKeyword="WinJS.UI.computeDataSourceGroups">
            /// <summary locid="WinJS.UI.computeDataSourceGroups">
            /// Returns a data source that adds group information to the items of another data source.  The "groups" property
            /// of this data source evaluates to yet another data source that enumerates the groups themselves.
            /// </summary>
            /// <param name="listDataSource" type="VirtualizedDataSource" locid="WinJS.UI.computeDataSourceGroups_p:listDataSource">
            /// The data source for the individual items to group.
            /// </param>
            /// <param name="groupKey" type="Function" locid="WinJS.UI.computeDataSourceGroups_p:groupKey">
            /// A callback function that takes an item in the list as an argument. The function is called
            /// for each item in the list and returns the group key for the item as a string.
            /// </param>
            /// <param name="groupData" type="Function" locid="WinJS.UI.computeDataSourceGroups_p:groupData">
            /// A callback function that takes an item in the IListDataSource as an argument.
            /// The function is called on one item in each group and returns
            /// an object that represents the header of that group.
            /// </param>
            /// <param name="options" type="Object" locid="WinJS.UI.computeDataSourceGroups_p:options">
            /// An object that can contain properties that specify additional options:
            ///
            /// groupCountEstimate:
            /// A Number value that is the initial estimate for the number of groups. If you specify -1,
            /// this function returns no result is until the actual number of groups
            /// has been determined.
            ///
            /// batchSize:
            /// A Number greater than 0 that specifies the number of items to fetch during each processing pass when
            /// searching for groups. (In addition to the number specified, one item from the previous batch
            /// is always included.)
            /// </param>
            /// <returns type="IListDataSource" locid="WinJS.UI.computeDataSourceGroups_returnValue">
            /// An IListDataSource that contains the items in the original data source and provides additional
            /// group info in a "groups" property. The "groups" property returns another
            /// IListDataSource that enumerates the different groups in the list.
            /// </returns>
            /// </signature>

            var groupedItemDataSource = Object.create(listDataSource);

            function createGroupedItem(item) {
                if (item) {
                    var groupedItem = Object.create(item);

                    groupedItem.groupKey = groupKey(item);

                    if (groupData) {
                        groupedItem.groupData = groupData(item);
                    }

                    return groupedItem;
                } else {
                    return null;
                }
            }

            function createGroupedItemPromise(itemPromise) {
                var groupedItemPromise = Object.create(itemPromise);

                groupedItemPromise.then = function (onComplete, onError, onCancel) {
                    return itemPromise.then(function (item) {
                        return onComplete(createGroupedItem(item));
                    }, onError, onCancel);
                };

                return groupedItemPromise;
            }

            groupedItemDataSource.createListBinding = function (notificationHandler) {
                var groupedNotificationHandler;

                if (notificationHandler) {
                    groupedNotificationHandler = Object.create(notificationHandler);

                    groupedNotificationHandler.inserted = function (itemPromise, previousHandle, nextHandle) {
                        return notificationHandler.inserted(createGroupedItemPromise(itemPromise), previousHandle, nextHandle);
                    };

                    groupedNotificationHandler.changed = function (newItem, oldItem) {
                        return notificationHandler.changed(createGroupedItem(newItem), createGroupedItem(oldItem));
                    };

                    groupedNotificationHandler.moved = function (itemPromise, previousHandle, nextHandle) {
                        return notificationHandler.moved(createGroupedItemPromise(itemPromise), previousHandle, nextHandle);
                    };
                } else {
                    groupedNotificationHandler = null;
                }

                var listBinding = listDataSource.createListBinding(groupedNotificationHandler),
                    groupedItemListBinding = Object.create(listBinding);

                var listBindingMethods = [
                    "first",
                    "last",
                    "fromDescription",
                    "jumpToItem",
                    "current"
                ];

                for (var i = 0, len = listBindingMethods.length; i < len; i++) {
                    (function (listBindingMethod) {
                        if (listBinding[listBindingMethod]) {
                            groupedItemListBinding[listBindingMethod] = function () {
                                return createGroupedItemPromise(listBinding[listBindingMethod].apply(listBinding, arguments));
                            };
                        }
                    })(listBindingMethods[i]);
                }

                // The following methods should be fast

                if (listBinding.fromKey) {
                    groupedItemListBinding.fromKey = function (key) {
                        return createGroupedItemPromise(listBinding.fromKey(key));
                    };
                }

                if (listBinding.fromIndex) {
                    groupedItemListBinding.fromIndex = function (index) {
                        return createGroupedItemPromise(listBinding.fromIndex(index));
                    };
                }

                groupedItemListBinding.prev = function () {
                    return createGroupedItemPromise(listBinding.prev());
                };

                groupedItemListBinding.next = function () {
                    return createGroupedItemPromise(listBinding.next());
                };

                return groupedItemListBinding;
            };

            var listDataSourceMethods = [
                "itemFromKey",
                "itemFromIndex",
                "itemFromDescription",
                "insertAtStart",
                "insertBefore",
                "insertAfter",
                "insertAtEnd",
                "change",
                "moveToStart",
                "moveBefore",
                "moveAfter",
                "moveToEnd"
                // remove does not return an itemPromise
            ];

            for (var i = 0, len = listDataSourceMethods.length; i < len; i++) {
                (function (listDataSourceMethod) {
                    if (listDataSource[listDataSourceMethod]) {
                        groupedItemDataSource[listDataSourceMethod] = function () {
                            return createGroupedItemPromise(listDataSource[listDataSourceMethod].apply(listDataSource, arguments));
                        };
                    }
                })(listDataSourceMethods[i]);
            }

            ["addEventListener", "removeEventListener", "dispatchEvent"].forEach(function (methodName) {
                if (listDataSource[methodName]) {
                    groupedItemDataSource[methodName] = function () {
                        return listDataSource[methodName].apply(listDataSource, arguments);
                    };
                }
            });

            var groupDataSource = null;

            Object.defineProperty(groupedItemDataSource, "groups", {
                get: function () {
                    if (!groupDataSource) {
                        groupDataSource = new _GroupDataSource._GroupDataSource(listDataSource, groupKey, groupData, options);
                    }
                    return groupDataSource;
                },
                enumerable: true,
                configurable: true
            });

            return groupedItemDataSource;
        }

    });

});


// Copyright (c) Microsoft Corporation.  All Rights Reserved. Licensed under the MIT License. See License.txt in the project root for license information.
// Storage Item Data Source

define('WinJS/VirtualizedDataSource/_StorageDataSource',[
    'exports',
    '../Core/_WinRT',
    '../Core/_Global',
    '../Core/_Base',
    '../Core/_ErrorFromName',
    '../Core/_WriteProfilerMark',
    '../Animations',
    '../Promise',
    '../Utilities/_UI',
    './_VirtualizedDataSourceImpl'
    ], function storageDataSourceInit(exports, _WinRT, _Global, _Base, _ErrorFromName, _WriteProfilerMark, Animations, Promise, _UI, VirtualizedDataSource) {
    "use strict";

    _Base.Namespace._moduleDefine(exports, "WinJS.UI", {
        StorageDataSource: _Base.Namespace._lazy(function () {
            var StorageDataAdapter = _Base.Class.define(function StorageDataAdapter_ctor(query, options) {
                // Constructor
                _WriteProfilerMark("WinJS.UI.StorageDataSource:constructor,StartTM");

                var mode = _WinRT.Windows.Storage.FileProperties.ThumbnailMode.singleItem,
                    size = 256,
                    flags = _WinRT.Windows.Storage.FileProperties.ThumbnailOptions.useCurrentScale,
                    delayLoad = true,
                    library;

                if (query === "Pictures") {
                    mode = _WinRT.Windows.Storage.FileProperties.ThumbnailMode.picturesView;
                    library = _WinRT.Windows.Storage.KnownFolders.picturesLibrary;
                    size = 190;
                } else if (query === "Music") {
                    mode = _WinRT.Windows.Storage.FileProperties.ThumbnailMode.musicView;
                    library = _WinRT.Windows.Storage.KnownFolders.musicLibrary;
                    size = 256;
                } else if (query === "Documents") {
                    mode = _WinRT.Windows.Storage.FileProperties.ThumbnailMode.documentsView;
                    library = _WinRT.Windows.Storage.KnownFolders.documentsLibrary;
                    size = 40;
                } else if (query === "Videos") {
                    mode = _WinRT.Windows.Storage.FileProperties.ThumbnailMode.videosView;
                    library = _WinRT.Windows.Storage.KnownFolders.videosLibrary;
                    size = 190;
                }

                if (!library) {
                    this._query = query;
                } else {
                    var queryOptions = new _WinRT.Windows.Storage.Search.QueryOptions();
                    queryOptions.folderDepth = _WinRT.Windows.Storage.Search.FolderDepth.deep;
                    queryOptions.indexerOption = _WinRT.Windows.Storage.Search.IndexerOption.useIndexerWhenAvailable;
                    this._query = library.createFileQueryWithOptions(queryOptions);
                }

                if (options) {
                    if (typeof options.mode === "number") {
                        mode = options.mode;
                    }
                    if (typeof options.requestedThumbnailSize === "number") {
                        size = Math.max(1, Math.min(options.requestedThumbnailSize, 1024));
                    } else {
                        switch (mode) {
                            case _WinRT.Windows.Storage.FileProperties.ThumbnailMode.picturesView:
                            case _WinRT.Windows.Storage.FileProperties.ThumbnailMode.videosView:
                                size = 190;
                                break;
                            case _WinRT.Windows.Storage.FileProperties.ThumbnailMode.documentsView:
                            case _WinRT.Windows.Storage.FileProperties.ThumbnailMode.listView:
                                size = 40;
                                break;
                            case _WinRT.Windows.Storage.FileProperties.ThumbnailMode.musicView:
                            case _WinRT.Windows.Storage.FileProperties.ThumbnailMode.singleItem:
                                size = 256;
                                break;
                        }
                    }
                    if (typeof options.thumbnailOptions === "number") {
                        flags = options.thumbnailOptions;
                    }
                    if (typeof options.waitForFileLoad === "boolean") {
                        delayLoad = !options.waitForFileLoad;
                    }
                }

                this._loader = new _WinRT.Windows.Storage.BulkAccess.FileInformationFactory(this._query, mode, size, flags, delayLoad);
                this.compareByIdentity = false;
                this.firstDataRequest = true;
                _WriteProfilerMark("WinJS.UI.StorageDataSource:constructor,StopTM");
            }, {
                // Public members

                setNotificationHandler: function (notificationHandler) {
                    this._notificationHandler = notificationHandler;
                    this._query.addEventListener("contentschanged", function () {
                        notificationHandler.invalidateAll();
                    });
                    this._query.addEventListener("optionschanged", function () {
                        notificationHandler.invalidateAll();
                    });
                },

                itemsFromEnd: function (count) {
                    var that = this;
                    _WriteProfilerMark("WinJS.UI.StorageDataSource:itemsFromEnd,info");
                    return this.getCount().then(function (totalCount) {
                        if (totalCount === 0) {
                            return Promise.wrapError(new _ErrorFromName(_UI.FetchError.doesNotExist));
                        }
                        // Intentionally passing countAfter = 1 to go one over the end so that itemsFromIndex will
                        // report the vector size since its known.
                        return that.itemsFromIndex(totalCount - 1, Math.min(totalCount - 1, count - 1), 1);
                    });
                },

                itemsFromIndex: function (index, countBefore, countAfter) {
                    // don't allow more than 64 items to be retrieved at once
                    if (countBefore + countAfter > 64) {
                        countBefore = Math.min(countBefore, 32);
                        countAfter = 64 - (countBefore + 1);
                    }

                    var first = (index - countBefore),
                        count = (countBefore + 1 + countAfter);
                    var that = this;
                    // Fetch a minimum of 32 items on the first request for smoothness. Otherwise
                    // listview displays 2 items first and then the rest of the page.
                    if (that.firstDataRequest) {
                        that.firstDataRequest = false;
                        count = Math.max(count, 32);
                    }
                    function listener(ev) {
                        that._notificationHandler.changed(that._item(ev.target));
                    }

                    var perfId = "WinJS.UI.StorageDataSource:itemsFromIndex(" + first + "-" + (first + count - 1) + ")";
                    _WriteProfilerMark(perfId + ",StartTM");
                    return this._loader.getItemsAsync(first, count).then(function (itemsVector) {
                        var vectorSize = itemsVector.size;
                        if (vectorSize <= countBefore) {
                            return Promise.wrapError(new _ErrorFromName(_UI.FetchError.doesNotExist));
                        }
                        var items = new Array(vectorSize);
                        var localItemsVector = new Array(vectorSize);
                        itemsVector.getMany(0, localItemsVector);
                        for (var i = 0; i < vectorSize; i++) {
                            items[i] = that._item(localItemsVector[i]);
                            localItemsVector[i].addEventListener("propertiesupdated", listener);
                        }
                        var result = {
                            items: items,
                            offset: countBefore,
                            absoluteIndex: index
                        };
                        // set the totalCount only when we know it (when we retrieived fewer items than were asked for)
                        if (vectorSize < count) {
                            result.totalCount = first + vectorSize;
                        }
                        _WriteProfilerMark(perfId + ",StopTM");
                        return result;
                    });
                },

                itemsFromDescription: function (description, countBefore, countAfter) {
                    var that = this;
                    _WriteProfilerMark("WinJS.UI.StorageDataSource:itemsFromDescription,info");
                    return this._query.findStartIndexAsync(description).then(function (index) {
                        return that.itemsFromIndex(index, countBefore, countAfter);
                    });
                },

                getCount: function () {
                    _WriteProfilerMark("WinJS.UI.StorageDataSource:getCount,info");
                    return this._query.getItemCountAsync();
                },

                itemSignature: function (item) {
                    return item.folderRelativeId;
                },

                // compareByIdentity: set in constructor
                // itemsFromStart: not implemented
                // itemsFromKey: not implemented
                // insertAtStart: not implemented
                // insertBefore: not implemented
                // insertAfter: not implemented
                // insertAtEnd: not implemented
                // change: not implemented
                // moveToStart: not implemented
                // moveBefore: not implemented
                // moveAfter: not implemented
                // moveToEnd: not implemented
                // remove: not implemented

                // Private members

                _item: function (item) {
                    return {
                        key: item.path || item.folderRelativeId,
                        data: item
                    };
                }
            }, {
                supportedForProcessing: false,
            });

            return _Base.Class.derive(VirtualizedDataSource.VirtualizedDataSource, function (query, options) {
                /// <signature helpKeyword="WinJS.UI.StorageDataSource">
                /// <summary locid="WinJS.UI.StorageDataSource">
                /// Creates a data source that enumerates an IStorageQueryResultBase.
                /// </summary>
                /// <param name="query" type="Windows.Storage.Search.IStorageQueryResultBase" locid="WinJS.UI.StorageDataSource_p:query">
                /// The object to enumerate. It must support IStorageQueryResultBase.
                /// </param>
                /// <param name="options" mayBeNull="true" optional="true" type="Object" locid="WinJS.UI.StorageDataSource_p:options">
                /// An object that specifies options for the data source. This parameter is optional. It can contain these properties:
                ///
                /// mode:
                /// A Windows.Storage.FileProperties.ThumbnailMode - a value that specifies whether to request
                /// thumbnails and the type of thumbnails to request.
                ///
                /// requestedThumbnailSize:
                /// A Number that specifies the size of the thumbnails.
                ///
                /// thumbnailOptions:
                /// A Windows.Storage.FileProperties.ThumbnailOptions value that specifies additional options for the thumbnails.
                ///
                /// waitForFileLoad:
                /// If you set this to true, the data source returns items only after it loads their properties and thumbnails.
                ///
                /// </param>
                /// </signature>
                this._baseDataSourceConstructor(new StorageDataAdapter(query, options));
            }, {
                /* empty */
            }, {
                loadThumbnail: function (item, image) {
                    /// <signature>
                    /// <summary locid="WinJS.UI.StorageDataSource.loadThumbnail">
                    /// Returns a promise for an image element that completes when the full quality thumbnail of the provided item is drawn to the
                    /// image element.
                    /// </summary>
                    /// <param name="item" type="ITemplateItem" locid="WinJS.UI.StorageDataSource.loadThumbnail_p:item">
                    /// The item to retrieve a thumbnail for.
                    /// </param>
                    /// <param name="image" type="Object" domElement="true" optional="true" locid="WinJS.UI.StorageDataSource.loadThumbnail_p:image">
                    /// The image element to use. If not provided, a new image element is created.
                    /// </param>
                    /// </signature>
                    var thumbnailUpdateHandler,
                        thumbnailPromise,
                        shouldRespondToThumbnailUpdate = false;

                    return new Promise(function (complete) {
                        // Load a thumbnail if it exists. The promise completes when a full quality thumbnail is visible.
                        var tagSupplied = (image ? true : false);
                        var processThumbnail = function (thumbnail) {
                            if (thumbnail) {
                                var url = _Global.URL.createObjectURL(thumbnail, {oneTimeOnly: true});

                                // If this is the first version of the thumbnail we're loading, fade it in.
                                if (!thumbnailPromise) {
                                    thumbnailPromise = item.loadImage(url, image).then(function (image) {
                                        // Wrapping the fadeIn call in a promise for the image returned by loadImage allows us to
                                        // pipe the result of loadImage to further chained promises.  This is necessary because the
                                        // image element provided to loadThumbnail is optional, and loadImage will create an image
                                        // element if none is provided.
                                        return item.isOnScreen().then(function (visible) {
                                            var imagePromise;
                                            if (visible && tagSupplied) {
                                                imagePromise = Animations.fadeIn(image).then(function () {
                                                    return image;
                                                });
                                            } else {
                                                image.style.opacity = 1;
                                                imagePromise = Promise.wrap(image);
                                            }
                                            return imagePromise;
                                        });
                                    });
                                } else { // Otherwise, replace the existing version without animation.
                                    thumbnailPromise = thumbnailPromise.then(function (image) {
                                        return item.loadImage(url, image);
                                    });
                                }

                                // If we have the full resolution thumbnail, we can cancel further updates and complete the promise
                                // when current work is complete.
                                if ((thumbnail.type !== _WinRT.Windows.Storage.FileProperties.ThumbnailType.icon) && !thumbnail.returnedSmallerCachedSize) {
                                    _WriteProfilerMark("WinJS.UI.StorageDataSource:loadThumbnail complete,info");
                                    item.data.removeEventListener("thumbnailupdated", thumbnailUpdateHandler);
                                    shouldRespondToThumbnailUpdate = false;
                                    thumbnailPromise = thumbnailPromise.then(function (image) {
                                        thumbnailUpdateHandler = null;
                                        thumbnailPromise = null;
                                        complete(image);
                                    });
                                }
                            }
                        };

                        thumbnailUpdateHandler = function (e) {
                            // Ensure that a zombie update handler does not get invoked.
                            if (shouldRespondToThumbnailUpdate) {
                                processThumbnail(e.target.thumbnail);
                            }
                        };
                        item.data.addEventListener("thumbnailupdated", thumbnailUpdateHandler);
                        shouldRespondToThumbnailUpdate = true;

                        // If we already have a thumbnail we should render it now.
                        processThumbnail(item.data.thumbnail);
                    }, function () {
                        item.data.removeEventListener("thumbnailupdated", thumbnailUpdateHandler);
                        shouldRespondToThumbnailUpdate = false;
                        thumbnailUpdateHandler = null;
                        if (thumbnailPromise) {
                            thumbnailPromise.cancel();
                            thumbnailPromise = null;
                        }
                    });
                },

                supportedForProcessing: false,
            });
        })
    });

});
// Copyright (c) Microsoft Corporation.  All Rights Reserved. Licensed under the MIT License. See License.txt in the project root for license information.
define('WinJS/VirtualizedDataSource',[
    './VirtualizedDataSource/_VirtualizedDataSourceImpl',
    './VirtualizedDataSource/_GroupDataSource',
    './VirtualizedDataSource/_GroupedItemDataSource',
    './VirtualizedDataSource/_StorageDataSource'
     ], function () {

    //wrapper module
});
define('WinJS/_Accents',["require", "exports", "./Core/_Global", "./Core/_WinRT", "./Core/_Base", "./Core/_BaseUtils"], function (require, exports, _Global, _WinRT, _Base, _BaseUtils) {
    var Constants = {
        accentStyleId: "WinJSAccentsStyle",
        themeDetectionTag: "winjs-themedetection-tag",
        hoverSelector: "html.win-hoverable",
        lightThemeSelector: ".win-ui-light",
        darkThemeSelector: ".win-ui-dark"
    };
    var CSSSelectorTokens = [".", "#", ":"];
    var UISettings = null;
    var colors = [];
    var isDarkTheme = false;
    var rules = [];
    var writeRulesTOHandle = -1;
    // Public APIs
    //
    // Enum values align with the colors array indices
    (function (ColorTypes) {
        ColorTypes[ColorTypes["accent"] = 0] = "accent";
        ColorTypes[ColorTypes["listSelectRest"] = 1] = "listSelectRest";
        ColorTypes[ColorTypes["listSelectHover"] = 2] = "listSelectHover";
        ColorTypes[ColorTypes["listSelectPress"] = 3] = "listSelectPress";
        ColorTypes[ColorTypes["_listSelectRestInverse"] = 4] = "_listSelectRestInverse";
        ColorTypes[ColorTypes["_listSelectHoverInverse"] = 5] = "_listSelectHoverInverse";
        ColorTypes[ColorTypes["_listSelectPressInverse"] = 6] = "_listSelectPressInverse";
    })(exports.ColorTypes || (exports.ColorTypes = {}));
    var ColorTypes = exports.ColorTypes;
    function createAccentRule(selector, props) {
        rules.push({ selector: selector, props: props });
        scheduleWriteRules();
    }
    exports.createAccentRule = createAccentRule;
    // Private helpers
    //
    function scheduleWriteRules() {
        if (rules.length === 0 || writeRulesTOHandle !== -1) {
            return;
        }
        writeRulesTOHandle = _BaseUtils._setImmediate(function () {
            writeRulesTOHandle = -1;
            cleanup();
            var inverseThemeSelector = isDarkTheme ? Constants.lightThemeSelector : Constants.darkThemeSelector;
            var inverseThemeHoverSelector = Constants.hoverSelector + " " + inverseThemeSelector;
            var style = _Global.document.createElement("style");
            style.id = Constants.accentStyleId;
            style.textContent = rules.map(function (rule) {
                // example rule: { selector: "  .foo,   html.win-hoverable   .bar:hover ,  div:hover  ", props: [{ name: "color", value: 0 }, { name: "background-color", value: 1 } }
                var body = "  " + rule.props.map(function (prop) { return prop.name + ": " + colors[prop.value] + ";"; }).join("\n  ");
                // body = color: *accent*; background-color: *listSelectHover*
                var selectorSplit = rule.selector.split(",").map(function (str) { return sanitizeSpaces(str); }); // [".foo", ".bar:hover", "div"]
                var selector = selectorSplit.join(",\n"); // ".foo, html.win-hoverable .bar:hover, div:hover"
                var css = selector + " {\n" + body + "\n}";
                // css = .foo, html.win-hoverable .bar:hover, div:hover { *body* }
                // Inverse Theme Selectors
                var isThemedColor = rule.props.some(function (prop) { return prop.value !== 0 /* accent */; });
                if (isThemedColor) {
                    var inverseBody = "  " + rule.props.map(function (prop) { return prop.name + ": " + colors[(prop.value ? (prop.value + 3) : prop.value)] + ";"; }).join("\n  ");
                    // inverseBody = "color: *accent*; background-color: *listSelectHoverInverse"
                    var themedSelectors = [];
                    selectorSplit.forEach(function (sel) {
                        if (sel.indexOf(Constants.hoverSelector) !== -1 && sel.indexOf(inverseThemeHoverSelector) === -1) {
                            themedSelectors.push(sel.replace(Constants.hoverSelector, inverseThemeHoverSelector));
                            var selWithoutHover = sel.replace(Constants.hoverSelector, "").trim();
                            if (CSSSelectorTokens.indexOf(selWithoutHover[0]) !== -1) {
                                themedSelectors.push(sel.replace(Constants.hoverSelector + " ", inverseThemeHoverSelector));
                            }
                        }
                        else {
                            themedSelectors.push(inverseThemeSelector + " " + sel);
                            if (CSSSelectorTokens.indexOf(sel[0]) !== -1) {
                                themedSelectors.push(inverseThemeSelector + sel);
                            }
                        }
                        css += "\n" + themedSelectors.join(",\n") + " {\n" + inverseBody + "\n}";
                    });
                }
                return css;
            }).join("\n");
            _Global.document.head.appendChild(style);
        });
    }
    function handleColorsChanged() {
        var UIColorType = _WinRT.Windows.UI.ViewManagement.UIColorType;
        var uiColor = UISettings.getColorValue(_WinRT.Windows.UI.ViewManagement.UIColorType.accent);
        var accent = colorToString(uiColor, 1);
        if (colors[0] === accent) {
            return;
        }
        // Establish colors
        // The order of the colors align with the ColorTypes enum values
        colors.length = 0;
        colors.push(accent, colorToString(uiColor, (isDarkTheme ? 0.6 : 0.4)), colorToString(uiColor, (isDarkTheme ? 0.8 : 0.6)), colorToString(uiColor, (isDarkTheme ? 0.9 : 0.7)), colorToString(uiColor, (isDarkTheme ? 0.4 : 0.6)), colorToString(uiColor, (isDarkTheme ? 0.6 : 0.8)), colorToString(uiColor, (isDarkTheme ? 0.7 : 0.9)));
        scheduleWriteRules();
    }
    function colorToString(color, alpha) {
        return "rgba(" + color.r + "," + color.g + "," + color.b + "," + alpha + ")";
    }
    function sanitizeSpaces(str) {
        return str.replace(/  /g, " ").replace(/  /g, " ").trim();
    }
    function cleanup() {
        var style = _Global.document.head.querySelector("#" + Constants.accentStyleId);
        style && style.parentNode.removeChild(style);
    }
    function _reset() {
        rules.length = 0;
        cleanup();
    }
    // Module initialization
    //
    // Figure out color theme
    var tag = _Global.document.createElement(Constants.themeDetectionTag);
    _Global.document.head.appendChild(tag);
    var theme = _Global.getComputedStyle(tag).opacity;
    isDarkTheme = theme === "0";
    tag.parentElement.removeChild(tag);
    try {
        UISettings = new _WinRT.Windows.UI.ViewManagement.UISettings();
        UISettings.addEventListener("colorvalueschanged", handleColorsChanged);
        handleColorsChanged();
    }
    catch (e) {
        // No WinRT - use hardcoded blue accent color
        // The order of the colors align with the ColorTypes enum values
        colors.push("rgb(0, 120, 215)", "rgba(0, 120, 215, " + (isDarkTheme ? "0.6" : "0.4") + ")", "rgba(0, 120, 215, " + (isDarkTheme ? "0.8" : "0.6") + ")", "rgba(0, 120, 215, " + (isDarkTheme ? "0.9" : "0.7") + ")", "rgba(0, 120, 215, " + (isDarkTheme ? "0.4" : "0.6") + ")", "rgba(0, 120, 215, " + (isDarkTheme ? "0.6" : "0.8") + ")", "rgba(0, 120, 215, " + (isDarkTheme ? "0.7" : "0.9") + ")");
    }
    // Publish to WinJS namespace
    var toPublish = {
        ColorTypes: ColorTypes,
        createAccentRule: createAccentRule,
        // Exposed for tests    
        _colors: colors,
        _reset: _reset,
        _isDarkTheme: isDarkTheme
    };
    _Base.Namespace.define("WinJS.UI._Accents", toPublish);
});

define('require-style',{load: function(id){throw new Error("Dynamic load not allowed: " + id);}});

define('require-style!less/styles-intrinsic',[],function(){});

define('require-style!less/colors-intrinsic',[],function(){});
// Copyright (c) Microsoft Corporation.  All Rights Reserved. Licensed under the MIT License. See License.txt in the project root for license information.
define('WinJS/Controls/IntrinsicControls',[
    '../Utilities/_Hoverable',
    '../_Accents',
    'require-style!less/styles-intrinsic',
    'require-style!less/colors-intrinsic'
], function (_Hoverable, _Accents) {
    "use strict";

    // Shared color rule 
    _Accents.createAccentRule(
        ".win-link,\
         .win-progress-bar,\
         .win-progress-ring,\
         .win-ring",
        [{ name: "color", value: _Accents.ColorTypes.accent }]);

    // Shared background-color rule
    _Accents.createAccentRule(
        "::selection,\
         .win-button.win-button-primary,\
         .win-dropdown option:checked,\
         select[multiple].win-dropdown option:checked",
        [{ name: "background-color", value: _Accents.ColorTypes.accent }]);

    // Shared border-color rule
    _Accents.createAccentRule(
        ".win-textbox:focus,\
         .win-textarea:focus,\
         .win-textbox:focus:hover,\
         .win-textarea:focus:hover",
        [{ name: "border-color", value: _Accents.ColorTypes.accent }]);

    // Edge-specific color rule
    _Accents.createAccentRule(
        ".win-textbox::-ms-clear:hover:not(:active),\
         .win-textbox::-ms-reveal:hover:not(:active)",
        [{ name: "color", value: _Accents.ColorTypes.accent }]);

    // Edge-specific background-color rule
    _Accents.createAccentRule(
        ".win-checkbox:checked::-ms-check,\
         .win-textbox::-ms-clear:active,\
         .win-textbox::-ms-reveal:active",
        [{ name: "background-color", value: _Accents.ColorTypes.accent }]);

    // Webkit-specific background-color rule
    _Accents.createAccentRule(
        ".win-progress-bar::-webkit-progress-value,\
         .win-progress-ring::-webkit-progress-value,\
         .win-ring::-webkit-progress-value",
        [{ name: "background-color", value: _Accents.ColorTypes.accent }]);

    // Mozilla-specific background-color rule
    _Accents.createAccentRule(
        ".win-progress-bar:not(:indeterminate)::-moz-progress-bar,\
         .win-progress-ring:not(:indeterminate)::-moz-progress-bar,\
         .win-ring:not(:indeterminate)::-moz-progress-bar",
        [{ name: "background-color", value: _Accents.ColorTypes.accent }]);

    // Edge-specific border-color rule
    _Accents.createAccentRule(
        ".win-checkbox:indeterminate::-ms-check,\
         .win-checkbox:hover:indeterminate::-ms-check,\
         .win-radio:checked::-ms-check",
        [{ name: "border-color", value: _Accents.ColorTypes.accent }]);


    // Note the use of background instead of background-color
    // FF slider styling doesn't work with background-color
    // so using background for everything here for consistency

    // Edge-specific background rule
    _Accents.createAccentRule(
        ".win-slider::-ms-thumb,\
         .win-slider::-ms-fill-lower", /* Fill-Lower only supported in IE */
        [{ name: "background", value: _Accents.ColorTypes.accent }]);

    // Webkit-specific background rule
    _Accents.createAccentRule(
        ".win-slider::-webkit-slider-thumb",
        [{ name: "background", value: _Accents.ColorTypes.accent }]);

    // Mozilla-specific background rule
    _Accents.createAccentRule(
        ".win-slider::-moz-range-thumb",
        [{ name: "background", value: _Accents.ColorTypes.accent }]);
});
// Copyright (c) Microsoft Corporation.  All Rights Reserved. Licensed under the MIT License. See License.txt in the project root for license information.
define('WinJS/Controls/ItemContainer/_Constants',[
    'exports',
    '../../Core/_Base'
    ], function constantsInit(exports, _Base) {
    "use strict";

    var members = {};
    members._listViewClass = "win-listview";
    members._viewportClass = "win-viewport";
    members._rtlListViewClass = "win-rtl";
    members._horizontalClass = "win-horizontal";
    members._verticalClass = "win-vertical";
    members._scrollableClass = "win-surface";
    members._itemsContainerClass = "win-itemscontainer";
    members._listHeaderContainerClass = "win-headercontainer";
    members._listFooterContainerClass = "win-footercontainer";
    members._padderClass = "win-itemscontainer-padder";
    members._proxyClass = "_win-proxy";
    members._itemClass = "win-item";
    members._itemBoxClass = "win-itembox";
    members._itemsBlockClass = "win-itemsblock";
    members._containerClass = "win-container";
    members._containerEvenClass = "win-container-even";
    members._containerOddClass = "win-container-odd";
    members._backdropClass = "win-backdrop";
    members._footprintClass = "win-footprint";
    members._groupsClass = "win-groups";
    members._selectedClass = "win-selected";
    members._selectionBorderClass = "win-selectionborder";
    members._selectionBackgroundClass = "win-selectionbackground";
    members._selectionCheckmarkClass = "win-selectioncheckmark";
    members._selectionCheckmarkBackgroundClass = "win-selectioncheckmarkbackground";
    members._pressedClass = "win-pressed";
    members._headerClass = "win-groupheader";
    members._headerContainerClass = "win-groupheadercontainer";
    members._groupLeaderClass = "win-groupleader";
    members._progressClass = "win-progress";
    members._revealedClass = "win-revealed";
    members._itemFocusClass = "win-focused";
    members._itemFocusOutlineClass = "win-focusedoutline";
    members._zoomingXClass = "win-zooming-x";
    members._zoomingYClass = "win-zooming-y";
    members._listLayoutClass = "win-listlayout";
    members._gridLayoutClass = "win-gridlayout";
    members._headerPositionTopClass = "win-headerpositiontop";
    members._headerPositionLeftClass = "win-headerpositionleft";
    members._structuralNodesClass = "win-structuralnodes";
    members._singleItemsBlockClass = "win-single-itemsblock";
    members._uniformGridLayoutClass = "win-uniformgridlayout";
    members._uniformListLayoutClass = "win-uniformlistlayout";
    members._cellSpanningGridLayoutClass = "win-cellspanninggridlayout";
    members._laidOutClass = "win-laidout";
    members._nonDraggableClass = "win-nondraggable";
    members._nonSelectableClass = "win-nonselectable";
    members._dragOverClass = "win-dragover";
    members._dragSourceClass = "win-dragsource";
    members._clipClass = "win-clip";
    members._selectionModeClass = "win-selectionmode";
    members._noCSSGrid = "win-nocssgrid";
    members._hidingSelectionMode = "win-hidingselectionmode";
    members._hidingSelectionModeAnimationTimeout = 250;

    members._INVALID_INDEX = -1;
    members._UNINITIALIZED = -1;

    members._LEFT_MSPOINTER_BUTTON = 0;
    members._RIGHT_MSPOINTER_BUTTON = 2;

    members._TAP_END_THRESHOLD = 10;

    members._DEFAULT_PAGES_TO_LOAD = 5;
    members._DEFAULT_PAGE_LOAD_THRESHOLD = 2;

    members._MIN_AUTOSCROLL_RATE = 150;
    members._MAX_AUTOSCROLL_RATE = 1500;
    members._AUTOSCROLL_THRESHOLD = 100;
    members._AUTOSCROLL_DELAY = 50;

    members._DEFERRED_ACTION = 250;
    members._DEFERRED_SCROLL_END = 250;

    members._SELECTION_CHECKMARK = "\uE081";

    members._LISTVIEW_PROGRESS_DELAY = 2000;

    var ScrollToPriority = {
        uninitialized: 0,
        low: 1,             // used by layoutSite.invalidateLayout, forceLayout, _processReload, _update and _onMSElementResize - operations that preserve the scroll position
        medium: 2,          // used by dataSource change, layout change and etc - operations that reset the scroll position to 0
        high: 3             // used by indexOfFirstVisible, ensureVisible, scrollPosition - operations in which the developer explicitly sets the scroll position
    };

    var ViewChange = {
        rebuild: 0,
        remeasure: 1,
        relayout: 2,
        realize: 3
    };

    members._ScrollToPriority = ScrollToPriority;
    members._ViewChange = ViewChange;

    _Base.Namespace._moduleDefine(exports, "WinJS.UI", members);
});

// Copyright (c) Microsoft Corporation.  All Rights Reserved. Licensed under the MIT License. See License.txt in the project root for license information.
define('WinJS/Controls/ItemContainer/_ItemEventsHandler',[
    'exports',
    '../../Core/_Global',
    '../../Core/_WinRT',
    '../../Core/_Base',
    '../../Core/_BaseUtils',
    '../../Core/_WriteProfilerMark',
    '../../Animations',
    '../../Animations/_TransitionAnimation',
    '../../Promise',
    '../../Utilities/_ElementUtilities',
    '../../Utilities/_UI',
    './_Constants'
], function itemEventsHandlerInit(exports, _Global, _WinRT, _Base, _BaseUtils, _WriteProfilerMark, Animations, _TransitionAnimation, Promise, _ElementUtilities, _UI, _Constants) {
    "use strict";

    var transformNames = _BaseUtils._browserStyleEquivalents["transform"];

    // Returns a CSS transformation to rotate and shrink an element when it is
    // pressed. The closer the click is to the center of the item, the more it
    // shrinks and the less it rotates.
    // *elementRect* should be of the form returned by getBoundingClientRect. All
    // of the parameters must be relative to the same coordinate system.
    // This function was translated from the Splash implementation.
    function tiltTransform(clickX, clickY, elementRect) {
        var minSize = 44,
            maxSize = 750,
            minRotationX = 2,
            maxRotationX = 9,
            minRotationY = 2.11,
            maxRotationY = 13,
            sizeRange = maxSize - minSize,
            halfWidth = elementRect.width / 2,
            halfHeight = elementRect.height / 2;

        var clampedWidth = _ElementUtilities._clamp(elementRect.width, minSize, maxSize);
        var clampedHeight = _ElementUtilities._clamp(elementRect.height, minSize, maxSize);

        // The maximum rotation that any element is capable of is calculated by using its width and height and clamping it into the range calculated above.
        // minRotationX|Y and maxRotationX|Y are the absolute minimums and maximums that any generic element can be rotated by, but in order to determine
        // what the min/max rotations for our current element is (which will be inside of the absolute min/max described above), we need
        // to calculate the max rotations for this element by clamping the sizes and doing a linear interpolation:
        var maxElementRotationX = maxRotationX - (((clampedHeight - minSize) / sizeRange) * (maxRotationX - minRotationX));
        var maxElementRotationY = maxRotationY - (((clampedWidth - minSize) / sizeRange) * (maxRotationY - minRotationY));

        // Now we calculate the distance of our current point from the center of our element and normalize it to be in the range of 0 - 1
        var normalizedOffsetX = ((clickX - elementRect.left) - halfWidth) / halfWidth;
        var normalizedOffsetY = ((clickY - elementRect.top) - halfHeight) / halfHeight;

        // Finally, we calculate the appropriate rotations and scale for the element by using the normalized click offsets and the
        // maximum element rotation.
        var rotationX = maxElementRotationX * normalizedOffsetY;
        var rotationY = maxElementRotationY * normalizedOffsetX;
        var scale = 0.97 + 0.03 * (Math.abs(normalizedOffsetX) + Math.abs(normalizedOffsetY)) / 2.0;
        var transform = "perspective(800px) scale(" + scale + ") rotateX(" + -rotationX + "deg) rotateY(" + rotationY + "deg)";
        return transform;
    }

    _Base.Namespace._moduleDefine(exports, "WinJS.UI", {
        // Expose these to the unit tests
        _tiltTransform: tiltTransform,

        _ItemEventsHandler: _Base.Namespace._lazy(function () {
            var PT_TOUCH = _ElementUtilities._MSPointerEvent.MSPOINTER_TYPE_TOUCH || "touch";

            function createNodeWithClass(className, skipAriaHidden) {
                var element = _Global.document.createElement("div");
                element.className = className;
                if (!skipAriaHidden) {
                    element.setAttribute("aria-hidden", true);
                }
                return element;
            }

            var ItemEventsHandler = _Base.Class.define(function ItemEventsHandler_ctor(site) {
                this._site = site;

                this._work = [];
                this._animations = {};
            }, {
                dispose: function () {
                    if (this._disposed) {
                        return;
                    }
                    this._disposed = true;
                    _ElementUtilities._removeEventListener(_Global, "pointerup", this._resetPointerDownStateBound);
                    _ElementUtilities._removeEventListener(_Global, "pointercancel", this._resetPointerDownStateBound);
                },

                onPointerDown: function ItemEventsHandler_onPointerDown(eventObject) {
                    _WriteProfilerMark("WinJS.UI._ItemEventsHandler:MSPointerDown,StartTM");
                    var site = this._site,
                        touchInput = (eventObject.pointerType === PT_TOUCH),
                        leftButton,
                        rightButton;
                    site.pressedElement = eventObject.target;
                    if (_WinRT.Windows.UI.Input.PointerPoint) {
                        // xButton is true when you've x-clicked with a mouse or pen. Otherwise it is false.
                        var currentPoint = this._getCurrentPoint(eventObject);
                        var pointProps = currentPoint.properties;
                        if (!(touchInput || pointProps.isInverted || pointProps.isEraser || pointProps.isMiddleButtonPressed)) {
                            rightButton = pointProps.isRightButtonPressed;
                            leftButton = !rightButton && pointProps.isLeftButtonPressed;
                        } else {
                            leftButton = rightButton = false;
                        }
                    } else {
                        // xButton is true when you've x-clicked with a mouse. Otherwise it is false.
                        leftButton = (eventObject.button === _Constants._LEFT_MSPOINTER_BUTTON);
                        rightButton = (eventObject.button === _Constants._RIGHT_MSPOINTER_BUTTON);
                    }

                    this._DragStartBound = this._DragStartBound || this.onDragStart.bind(this);
                    this._PointerEnterBound = this._PointerEnterBound || this.onPointerEnter.bind(this);
                    this._PointerLeaveBound = this._PointerLeaveBound || this.onPointerLeave.bind(this);

                    var isInteractive = this._isInteractive(eventObject.target),
                        currentPressedIndex = site.indexForItemElement(eventObject.target),
                        currentPressedHeaderIndex = site.indexForHeaderElement(eventObject.target),
                        mustSetCapture = !isInteractive && currentPressedIndex !== _Constants._INVALID_INDEX;

                    if ((touchInput || leftButton) && this._site.pressedEntity.index === _Constants._INVALID_INDEX && !isInteractive) {
                        if (currentPressedHeaderIndex === _Constants._INVALID_INDEX) {
                            this._site.pressedEntity = { type: _UI.ObjectType.item, index: currentPressedIndex };
                        } else {
                            this._site.pressedEntity = { type: _UI.ObjectType.groupHeader, index: currentPressedHeaderIndex };
                        }

                        if (this._site.pressedEntity.index !== _Constants._INVALID_INDEX) {
                            this._site.pressedPosition = _ElementUtilities._getCursorPos(eventObject);

                            var allowed = site.verifySelectionAllowed(this._site.pressedEntity);
                            this._canSelect = allowed.canSelect;
                            this._canTapSelect = allowed.canTapSelect;

                            if (this._site.pressedEntity.type === _UI.ObjectType.item) {
                                this._site.pressedItemBox = site.itemBoxAtIndex(this._site.pressedEntity.index);
                                this._site.pressedContainer = site.containerAtIndex(this._site.pressedEntity.index);
                                this._site.animatedElement = this._site.pressedContainer;
                                this._site.pressedHeader = null;
                                this._togglePressed(true, false, eventObject);
                                this._site.pressedContainer.addEventListener('dragstart', this._DragStartBound);
                                if (!touchInput) {
                                    // This only works for non touch input because on touch input we set capture which immediately fires the MSPointerOut.
                                    _ElementUtilities._addEventListener(this._site.pressedContainer, 'pointerenter', this._PointerEnterBound, false);
                                    _ElementUtilities._addEventListener(this._site.pressedContainer, 'pointerleave', this._PointerLeaveBound, false);
                                }
                            } else {
                                this._site.pressedHeader = this._site.headerFromElement(eventObject.target);
                                // Interactions with the headers on phone show an animation
                                if (_BaseUtils.isPhone) {
                                    this._site.animatedElement = this._site.pressedHeader;
                                    this._togglePressed(true, false, eventObject);
                                } else {
                                    this._site.pressedItemBox = null;
                                    this._site.pressedContainer = null;
                                    this._site.animatedElement = null;
                                }
                            }

                            if (!this._resetPointerDownStateBound) {
                                this._resetPointerDownStateBound = this._resetPointerDownStateForPointerId.bind(this);
                            }

                            if (!touchInput) {
                                _ElementUtilities._addEventListener(_Global, "pointerup", this._resetPointerDownStateBound, false);
                                _ElementUtilities._addEventListener(_Global, "pointercancel", this._resetPointerDownStateBound, false);
                            }

                            this._pointerId = eventObject.pointerId;
                            this._pointerRightButton = rightButton;
                        }
                    }

                    if (mustSetCapture) {
                        if (touchInput) {
                            try {
                                // Move pointer capture to avoid hover visual on second finger
                                _ElementUtilities._setPointerCapture(site.canvasProxy, eventObject.pointerId);
                            } catch (e) {
                                _WriteProfilerMark("WinJS.UI._ItemEventsHandler:MSPointerDown,StopTM");
                                return;
                            }
                        }
                    }

                    // Once the shift selection pivot is set, it remains the same until the user
                    // performs a left- or right-click without holding the shift key down.
                    if (this._site.pressedEntity.type === _UI.ObjectType.item &&
                            this._selectionAllowed() && this._multiSelection() &&       // Multi selection enabled
                            this._site.pressedEntity.index !== _Constants._INVALID_INDEX &&    // A valid item was clicked
                            site.selection._getFocused().index !== _Constants._INVALID_INDEX && site.selection._pivot === _Constants._INVALID_INDEX) {
                        site.selection._pivot = site.selection._getFocused().index;
                    }

                    _WriteProfilerMark("WinJS.UI._ItemEventsHandler:MSPointerDown,StopTM");
                },

                onPointerEnter: function ItemEventsHandler_onPointerEnter(eventObject) {
                    if (this._site.pressedContainer && this._pointerId === eventObject.pointerId) {
                        this._togglePressed(true, false, eventObject);
                    }
                },

                onPointerLeave: function ItemEventsHandler_onPointerLeave(eventObject) {
                    if (this._site.pressedContainer && this._pointerId === eventObject.pointerId) {
                        this._togglePressed(false, true /* synchronous */, eventObject);
                    }
                },

                onDragStart: function ItemEventsHandler_onDragStart() {
                    this._resetPressedContainer();
                },

                _resetPressedContainer: function ItemEventsHandler_resetPressedContainer() {
                    if ((this._site.pressedContainer || this._site.pressedHeader) && this._site.animatedElement) {
                        this._togglePressed(false);
                        if (this._site.pressedContainer) {
                            this._site.pressedContainer.style[transformNames.scriptName] = "";
                            this._site.pressedContainer.removeEventListener('dragstart', this._DragStartBound);
                            _ElementUtilities._removeEventListener(this._site.pressedContainer, 'pointerenter', this._PointerEnterBound, false);
                            _ElementUtilities._removeEventListener(this._site.pressedContainer, 'pointerleave', this._PointerLeaveBound, false);
                        }
                    }
                },

                onClick: function ItemEventsHandler_onClick(eventObject) {
                    if (!this._skipClick) {
                        // Handle the UIA invoke action on an item. this._skipClick is false which tells us that we received a click
                        // event without an associated MSPointerUp event. This means that the click event was triggered thru UIA
                        // rather than thru the GUI.
                        var entity = { type: _UI.ObjectType.item, index: this._site.indexForItemElement(eventObject.target) };
                        if (entity.index === _Constants._INVALID_INDEX) {
                            entity.index = this._site.indexForHeaderElement(eventObject.target);
                            if (entity.index !== _Constants._INVALID_INDEX) {
                                entity.type = _UI.ObjectType.groupHeader;
                            }
                        }

                        if (entity.index !== _Constants._INVALID_INDEX &&
                            (_ElementUtilities.hasClass(eventObject.target, this._site.accessibleItemClass) || _ElementUtilities.hasClass(eventObject.target, _Constants._headerClass))) {
                            var allowed = this._site.verifySelectionAllowed(entity);
                            if (allowed.canTapSelect) {
                                this.handleTap(entity);
                            }
                            this._site.fireInvokeEvent(entity, eventObject.target);
                        }
                    }
                },

                onPointerUp: function ItemEventsHandler_onPointerUp(eventObject) {
                    _WriteProfilerMark("WinJS.UI._ItemEventsHandler:MSPointerUp,StartTM");

                    var site = this._site;
                    this._skipClick = true;
                    var that = this;
                    _BaseUtils._yieldForEvents(function () {
                        that._skipClick = false;
                    });

                    try {
                        // Release the pointer capture to allow in air touch pointers to be reused for multiple interactions
                        _ElementUtilities._releasePointerCapture(site.canvasProxy, eventObject.pointerId);
                    } catch (e) {
                        // This can throw if SeZo had capture or if the pointer was not already captured
                    }

                    var touchInput = (eventObject.pointerType === PT_TOUCH),
                        releasedElement = this._releasedElement(eventObject),
                        releasedIndex = site.indexForItemElement(releasedElement),
                        releasedHeaderIndex = releasedElement && _ElementUtilities.hasClass(releasedElement, _Constants._headerContainerClass) ? site.indexForHeaderElement(site.pressedHeader) : site.indexForHeaderElement(releasedElement);

                    if (this._pointerId === eventObject.pointerId) {
                        var releasedEntity;
                        if (releasedHeaderIndex === _Constants._INVALID_INDEX) {
                            releasedEntity = { type: _UI.ObjectType.item, index: releasedIndex };
                        } else {
                            releasedEntity = { type: _UI.ObjectType.groupHeader, index: releasedHeaderIndex };
                        }

                        this._resetPressedContainer();

                        if (this._site.pressedEntity.type === _UI.ObjectType.item && releasedEntity.type === _UI.ObjectType.item &&
                                this._site.pressedContainer && this._site.pressedEntity.index === releasedEntity.index) {

                            if (!eventObject.shiftKey) {
                                // Reset the shift selection pivot when the user clicks w/o pressing shift
                                site.selection._pivot = _Constants._INVALID_INDEX;
                            }

                            if (eventObject.shiftKey) {
                                // Shift selection should work when shift or shift+ctrl are depressed for both left- and right-click
                                if (this._selectionAllowed() && this._multiSelection() && site.selection._pivot !== _Constants._INVALID_INDEX) {
                                    var firstIndex = Math.min(this._site.pressedEntity.index, site.selection._pivot),
                                        lastIndex = Math.max(this._site.pressedEntity.index, site.selection._pivot),
                                        additive = (this._pointerRightButton || eventObject.ctrlKey || site.tapBehavior === _UI.TapBehavior.toggleSelect);
                                    site.selectRange(firstIndex, lastIndex, additive);
                                }
                            } else if (eventObject.ctrlKey) {
                                this.toggleSelectionIfAllowed(this._site.pressedEntity.index);
                            }
                        }

                        if (this._site.pressedHeader || this._site.pressedContainer) {
                            var upPosition = _ElementUtilities._getCursorPos(eventObject);
                            var isTap = Math.abs(upPosition.left - this._site.pressedPosition.left) <= _Constants._TAP_END_THRESHOLD &&
                                Math.abs(upPosition.top - this._site.pressedPosition.top) <= _Constants._TAP_END_THRESHOLD;

                            // We do not care whether or not the pressed and released indices are equivalent when the user is using touch. The only time they won't be is if the user
                            // tapped the edge of an item and the pressed animation shrank the item such that the user's finger was no longer over it. In this case, the item should
                            // be considered tapped.
                            // However, if the user is using touch then we must perform an extra check. Sometimes we receive MSPointerUp events when the user intended to pan.
                            // This extra check ensures that these intended pans aren't treated as taps.
                            if (!this._pointerRightButton && !eventObject.ctrlKey && !eventObject.shiftKey &&
                                    ((touchInput && isTap) ||
                                    (!touchInput && this._site.pressedEntity.index === releasedEntity.index && this._site.pressedEntity.type === releasedEntity.type))) {
                                if (releasedEntity.type === _UI.ObjectType.groupHeader) {
                                    this._site.pressedHeader = site.headerAtIndex(releasedEntity.index);
                                    this._site.pressedItemBox = null;
                                    this._site.pressedContainer = null;
                                } else {
                                    this._site.pressedItemBox = site.itemBoxAtIndex(releasedEntity.index);
                                    this._site.pressedContainer = site.containerAtIndex(releasedEntity.index);
                                    this._site.pressedHeader = null;
                                }

                                if (this._canTapSelect) {
                                    this.handleTap(this._site.pressedEntity);
                                }
                                this._site.fireInvokeEvent(this._site.pressedEntity, this._site.pressedItemBox || this._site.pressedHeader);
                            }
                        }

                        if (this._site.pressedEntity.index !== _Constants._INVALID_INDEX) {
                            site.changeFocus(this._site.pressedEntity, true, false, true);
                        }

                        this.resetPointerDownState();
                    }

                    _WriteProfilerMark("WinJS.UI._ItemEventsHandler:MSPointerUp,StopTM");
                },

                onPointerCancel: function ItemEventsHandler_onPointerCancel(eventObject) {
                    if (this._pointerId === eventObject.pointerId) {
                        _WriteProfilerMark("WinJS.UI._ItemEventsHandler:MSPointerCancel,info");
                        this.resetPointerDownState();
                    }
                },

                onLostPointerCapture: function ItemEventsHandler_onLostPointerCapture(eventObject) {
                    if (this._pointerId === eventObject.pointerId) {
                        _WriteProfilerMark("WinJS.UI._ItemEventsHandler:MSLostPointerCapture,info");
                        this.resetPointerDownState();
                    }
                },

                // In order for the control to play nicely with other UI controls such as the app bar, it calls preventDefault on
                // contextmenu events. It does this only when selection is enabled, the event occurred on or within an item, and
                // the event did not occur on an interactive element.
                onContextMenu: function ItemEventsHandler_onContextMenu(eventObject) {
                    if (this._shouldSuppressContextMenu(eventObject.target)) {
                        eventObject.preventDefault();
                    }
                },

                onMSHoldVisual: function ItemEventsHandler_onMSHoldVisual(eventObject) {
                    if (this._shouldSuppressContextMenu(eventObject.target)) {
                        eventObject.preventDefault();
                    }
                },

                onDataChanged: function ItemEventsHandler_onDataChanged() {
                    this.resetPointerDownState();
                },

                toggleSelectionIfAllowed: function ItemEventsHandler_toggleSelectionIfAllowed(itemIndex) {
                    if (this._selectionAllowed(itemIndex)) {
                        this._toggleItemSelection(itemIndex);
                    }
                },

                handleTap: function ItemEventsHandler_handleTap(entity) {
                    if (entity.type === _UI.ObjectType.groupHeader) {
                        return;
                    }

                    var site = this._site,
                        selection = site.selection;

                    if (this._selectionAllowed(entity.index) && this._selectOnTap()) {
                        if (site.tapBehavior === _UI.TapBehavior.toggleSelect) {
                            this._toggleItemSelection(entity.index);
                        } else {
                            // site.tapBehavior === _UI.TapBehavior.directSelect so ensure only itemIndex is selected
                            if (site.selectionMode === _UI.SelectionMode.multi || !selection._isIncluded(entity.index)) {
                                selection.set(entity.index);
                            }
                        }
                    }
                },

                // In single selection mode, in addition to itemIndex's selection state being toggled,
                // all other items will become deselected
                _toggleItemSelection: function ItemEventsHandler_toggleItemSelection(itemIndex) {
                    var site = this._site,
                        selection = site.selection,
                        selected = selection._isIncluded(itemIndex);

                    if (site.selectionMode === _UI.SelectionMode.single) {
                        if (!selected) {
                            selection.set(itemIndex);
                        } else {
                            selection.clear();
                        }
                    } else {
                        if (!selected) {
                            selection.add(itemIndex);
                        } else {
                            selection.remove(itemIndex);
                        }
                    }
                },

                _getCurrentPoint: function ItemEventsHandler_getCurrentPoint(eventObject) {
                    return _WinRT.Windows.UI.Input.PointerPoint.getCurrentPoint(eventObject.pointerId);
                },

                _containedInElementWithClass: function ItemEventsHandler_containedInElementWithClass(element, className) {
                    if (element.parentNode) {
                        var matches = element.parentNode.querySelectorAll("." + className + ", ." + className + " *");
                        for (var i = 0, len = matches.length; i < len; i++) {
                            if (matches[i] === element) {
                                return true;
                            }
                        }
                    }
                    return false;
                },

                _isSelected: function ItemEventsHandler_isSelected(index) {
                    return this._site.selection._isIncluded(index);
                },

                _isInteractive: function ItemEventsHandler_isInteractive(element) {
                    return this._containedInElementWithClass(element, "win-interactive");
                },

                _shouldSuppressContextMenu: function ItemEventsHandler_shouldSuppressContextMenu(element) {
                    var containerElement = this._site.containerFromElement(element);

                    return (this._selectionAllowed() && containerElement && !this._isInteractive(element));
                },

                _togglePressed: function ItemEventsHandler_togglePressed(add, synchronous, eventObject) {
                    var that = this;
                    var isHeader = this._site.pressedEntity.type === _UI.ObjectType.groupHeader;

                    this._site.animatedDownPromise && this._site.animatedDownPromise.cancel();

                    if (!isHeader && _ElementUtilities.hasClass(this._site.pressedItemBox, _Constants._nonSelectableClass)) {
                        return;
                    }

                    if (!this._staticMode(isHeader)) {
                        if (add) {
                            if (!_ElementUtilities.hasClass(this._site.animatedElement, _Constants._pressedClass)) {
                                _WriteProfilerMark("WinJS.UI._ItemEventsHandler:applyPressedUI,info");
                                _ElementUtilities.addClass(this._site.animatedElement, _Constants._pressedClass);

                                var boundingElement = isHeader ? that._site.pressedHeader : that._site.pressedContainer;
                                var transform = tiltTransform(eventObject.clientX, eventObject.clientY, boundingElement.getBoundingClientRect());
                                // Timeout prevents item from looking like it was pressed down during pans
                                this._site.animatedDownPromise = Promise.timeout(50).then(function () {
                                    applyDownVisual(transform);
                                });
                            }
                        } else {
                            if (_ElementUtilities.hasClass(this._site.animatedElement, _Constants._pressedClass)) {
                                var element = this._site.animatedElement;
                                var expectingStyle = this._site.animatedElementScaleTransform;
                                if (synchronous) {
                                    applyUpVisual(element, expectingStyle);
                                } else {
                                    // Force removal of the _pressedClass to be asynchronous so that users will see at
                                    // least one frame of the shrunken item when doing a quick tap.
                                    //
                                    // setImmediate is used rather than requestAnimationFrame to ensure that the item
                                    // doesn't get stuck down for too long -- apps are told to put long running invoke
                                    // code behind a setImmediate and togglePressed's async code needs to run first.
                                    _BaseUtils._setImmediate(function () {
                                        if (_ElementUtilities.hasClass(element, _Constants._pressedClass)) {
                                            applyUpVisual(element, expectingStyle);
                                        }
                                    });
                                }
                            }
                        }
                    }

                    function applyDownVisual(transform) {
                        if (that._site.isInSelectionMode()) {
                            return;
                        }

                        if (that._site.animatedElement.style[transformNames.scriptName] === "") {
                            that._site.animatedElement.style[transformNames.scriptName] = transform;
                            that._site.animatedElementScaleTransform = that._site.animatedElement.style[transformNames.scriptName];
                        } else {
                            that._site.animatedElementScaleTransform = "";
                        }
                    }

                    function applyUpVisual(element, expectingStyle) {
                        _WriteProfilerMark("WinJS.UI._ItemEventsHandler:removePressedUI,info");
                        _ElementUtilities.removeClass(element, _Constants._pressedClass);
                        if (that._containsTransform(element, expectingStyle)) {
                            _TransitionAnimation.executeTransition(element, {
                                property: transformNames.cssName,
                                delay: 150,
                                duration: 350,
                                timing: "cubic-bezier(0.17,0.17,0.2,1)",
                                to: element.style[transformNames.scriptName].replace(expectingStyle, "")
                            });
                        }
                    }
                },

                _containsTransform: function ItemEventsHandler_containsTransform(element, transform) {
                    return transform && element.style[transformNames.scriptName].indexOf(transform) !== -1;
                },

                _resetPointerDownStateForPointerId: function ItemEventsHandler_resetPointerDownState(eventObject) {
                    if (this._pointerId === eventObject.pointerId) {
                        this.resetPointerDownState();
                    }
                },

                resetPointerDownState: function ItemEventsHandler_resetPointerDownState() {
                    this._site.pressedElement = null;
                    _ElementUtilities._removeEventListener(_Global, "pointerup", this._resetPointerDownStateBound);
                    _ElementUtilities._removeEventListener(_Global, "pointercancel", this._resetPointerDownStateBound);

                    this._resetPressedContainer();

                    this._site.pressedContainer = null;
                    this._site.animatedElement = null;
                    this._site.pressedHeader = null;
                    this._site.pressedItemBox = null;

                    this._site.pressedEntity = { type: _UI.ObjectType.item, index: _Constants._INVALID_INDEX };
                    this._pointerId = null;
                },

                _releasedElement: function ItemEventsHandler_releasedElement(eventObject) {
                    return _Global.document.elementFromPoint(eventObject.clientX, eventObject.clientY);
                },

                _applyUIInBatches: function ItemEventsHandler_applyUIInBatches(work) {
                    var that = this;
                    this._work.push(work);

                    if (!this._paintedThisFrame) {
                        applyUI();
                    }

                    function applyUI() {
                        if (that._work.length > 0) {
                            that._flushUIBatches();
                            that._paintedThisFrame = _BaseUtils._requestAnimationFrame(applyUI.bind(that));
                        } else {
                            that._paintedThisFrame = null;
                        }
                    }
                },

                _flushUIBatches: function ItemEventsHandler_flushUIBatches() {
                    if (this._work.length > 0) {
                        var workItems = this._work;
                        this._work = [];

                        for (var i = 0; i < workItems.length; i++) {
                            workItems[i]();
                        }
                    }
                },

                _selectionAllowed: function ItemEventsHandler_selectionAllowed(itemIndex) {
                    var item = (itemIndex !== undefined ? this._site.itemAtIndex(itemIndex) : null),
                        itemSelectable = !(item && _ElementUtilities.hasClass(item, _Constants._nonSelectableClass));
                    return itemSelectable && this._site.selectionMode !== _UI.SelectionMode.none;
                },

                _multiSelection: function ItemEventsHandler_multiSelection() {
                    return this._site.selectionMode === _UI.SelectionMode.multi;
                },

                _selectOnTap: function ItemEventsHandler_selectOnTap() {
                    return this._site.tapBehavior === _UI.TapBehavior.toggleSelect || this._site.tapBehavior === _UI.TapBehavior.directSelect;
                },

                _staticMode: function ItemEventsHandler_staticMode(isHeader) {
                    if (isHeader) {
                        return this._site.headerTapBehavior === _UI.GroupHeaderTapBehavior.none;
                    } else {
                        return this._site.tapBehavior === _UI.TapBehavior.none && this._site.selectionMode === _UI.SelectionMode.none;
                    }
                },
            }, {
                // Avoids unnecessary UIA selection events by only updating aria-selected if it has changed
                setAriaSelected: function ItemEventsHandler_setAriaSelected(itemElement, isSelected) {
                    var ariaSelected = (itemElement.getAttribute("aria-selected") === "true");

                    if (isSelected !== ariaSelected) {
                        itemElement.setAttribute("aria-selected", isSelected);
                    }
                },

                renderSelection: function ItemEventsHandler_renderSelection(itemBox, element, selected, aria, container) {
                    if (!ItemEventsHandler._selectionTemplate) {
                        ItemEventsHandler._selectionTemplate = [];
                        ItemEventsHandler._selectionTemplate.push(createNodeWithClass(_Constants._selectionBackgroundClass));
                        ItemEventsHandler._selectionTemplate.push(createNodeWithClass(_Constants._selectionBorderClass));
                        ItemEventsHandler._selectionTemplate.push(createNodeWithClass(_Constants._selectionCheckmarkBackgroundClass));
                        var checkmark = createNodeWithClass(_Constants._selectionCheckmarkClass);
                        checkmark.textContent = _Constants._SELECTION_CHECKMARK;
                        ItemEventsHandler._selectionTemplate.push(checkmark);
                    }

                    // Update the selection rendering if necessary
                    if (selected !== _ElementUtilities._isSelectionRendered(itemBox)) {
                        if (selected) {
                            itemBox.insertBefore(ItemEventsHandler._selectionTemplate[0].cloneNode(true), itemBox.firstElementChild);

                            for (var i = 1, len = ItemEventsHandler._selectionTemplate.length; i < len; i++) {
                                itemBox.appendChild(ItemEventsHandler._selectionTemplate[i].cloneNode(true));
                            }
                        } else {
                            var nodes = itemBox.querySelectorAll(_ElementUtilities._selectionPartsSelector);
                            for (var i = 0, len = nodes.length; i < len; i++) {
                                itemBox.removeChild(nodes[i]);
                            }
                        }

                        _ElementUtilities[selected ? "addClass" : "removeClass"](itemBox, _Constants._selectedClass);
                        if (container) {
                            _ElementUtilities[selected ? "addClass" : "removeClass"](container, _Constants._selectedClass);
                        }
                    }

                    // To allow itemPropertyChange to work properly, aria needs to be updated after the selection visuals are added to the itemBox
                    if (aria) {
                        ItemEventsHandler.setAriaSelected(element, selected);
                    }
                },
            });

            return ItemEventsHandler;
        })

    });

});
// Copyright (c) Microsoft Corporation.  All Rights Reserved. Licensed under the MIT License. See License.txt in the project root for license information.
define('WinJS/Controls/ListView/_SelectionManager',[
    'exports',
    '../../Core/_Global',
    '../../Core/_Base',
    '../../Promise',
    '../../_Signal',
    '../../Utilities/_UI',
    '../ItemContainer/_Constants'
    ], function selectionManagerInit(exports, _Global, _Base, Promise, _Signal, _UI, _Constants) {
    "use strict";

    _Base.Namespace._moduleDefine(exports, "WinJS.UI", {
        _ItemSet: _Base.Namespace._lazy(function () {
            var _ItemSet = _Base.Class.define(function _ItemSet_ctor(listView, ranges, count) {
                this._listView = listView;
                this._ranges = ranges;
                this._itemsCount = count;
            });
            _ItemSet.prototype = {
                getRanges: function () {
                    var ranges = [];
                    for (var i = 0, len = this._ranges.length; i < len; i++) {
                        var range = this._ranges[i];
                        ranges.push({
                            firstIndex: range.firstIndex,
                            lastIndex: range.lastIndex,
                            firstKey: range.firstKey,
                            lastKey: range.lastKey
                        });
                    }
                    return ranges;
                },

                getItems: function () {
                    return exports.getItemsFromRanges(this._listView._itemsManager.dataSource, this._ranges);
                },

                isEverything: function () {
                    return this.count() === this._itemsCount;
                },

                count: function () {
                    var count = 0;
                    for (var i = 0, len = this._ranges.length; i < len; i++) {
                        var range = this._ranges[i];
                        count += range.lastIndex - range.firstIndex + 1;
                    }
                    return count;
                },

                getIndices: function () {
                    var indices = [];
                    for (var i = 0, len = this._ranges.length; i < len; i++) {
                        var range = this._ranges[i];
                        for (var n = range.firstIndex; n <= range.lastIndex; n++) {
                            indices.push(n);
                        }
                    }
                    return indices;
                }
            };
            return _ItemSet;
        }),

        getItemsFromRanges: function (dataSource, ranges) {
            var listBinding = dataSource.createListBinding(),
                promises = [];

            function getIndices() {
                var indices = [];
                for (var i = 0, len = ranges.length; i < len; i++) {
                    var range = ranges[i];
                    for (var j = range.firstIndex; j <= range.lastIndex; j++) {
                        indices.push(j);
                    }
                }
                return Promise.wrap(indices);
            }

            return getIndices().then(function (indices) {
                for (var i = 0; i < indices.length; i++) {
                    promises.push(listBinding.fromIndex(indices[i]));
                }

                return Promise.join(promises).then(function (items) {
                    listBinding.release();
                    return items;
                });
            });
        },

        _Selection: _Base.Namespace._lazy(function () {
            function isEverythingRange(ranges) {
                return ranges && ranges.firstIndex === 0 && ranges.lastIndex === Number.MAX_VALUE;
            }

            return _Base.Class.derive(exports._ItemSet, function (listView, indexesAndRanges) {
                this._listView = listView;
                this._itemsCount = -1;
                this._ranges = [];
                if (indexesAndRanges) {
                    this.set(indexesAndRanges);
                }
            }, {
                clear: function () {
                    /// <signature helpKeyword="WinJS.UI._Selection.prototype.clear">
                    /// <summary locid="WinJS.UI._Selection.prototype.clear">
                    /// Clears the selection.
                    /// </summary>
                    /// <returns type="Promise" locid="WinJS.UI._Selection.prototype.clear_returnValue">
                    /// A Promise that is fulfilled when the clear operation completes.
                    /// </returns>
                    /// </signature>

                    this._releaseRanges(this._ranges);
                    this._ranges = [];
                    return Promise.wrap();
                },

                set: function (items) {
                    /// <signature helpKeyword="WinJS.UI._Selection.prototype.set">
                    /// <summary locid="WinJS.UI._Selection.prototype.set">
                    /// Clears the current selection and replaces it with the specified items.
                    /// </summary>
                    /// <param name="items" locid="WinJS.UI._Selection.prototype.set_items">
                    /// The indexes or keys of the items that make up the selection.
                    /// You can provide different types of objects for the items parameter:
                    /// you can specify an index, a key, or a range of indexes.
                    /// It can also be an array that contains one or more of these objects.
                    /// </param>
                    /// <returns type="Promise" locid="WinJS.UI._Selection.prototype.set_returnValue">
                    /// A Promise that is fulfilled when the operation completes.
                    /// </returns>
                    /// </signature>

                    // A range with lastIndex set to Number.MAX_VALUE used to mean selectAll. Passing such range to set was equivalent to selectAll. This code preserves this behavior.
                    if (!isEverythingRange(items)) {
                        this._releaseRanges(this._ranges);
                        this._ranges = [];

                        var that = this;
                        return this._execute("_set", items).then(function () {
                            that._ranges.sort(function (left, right) {
                                return left.firstIndex - right.firstIndex;
                            });
                            return that._ensureKeys();
                        }).then(function () {
                            return that._ensureCount();
                        });
                    } else {
                        return this.selectAll();
                    }
                },

                add: function (items) {
                    /// <signature helpKeyword="WinJS.UI._Selection.prototype.add">
                    /// <summary locid="WinJS.UI._Selection.prototype.add">
                    /// Adds one or more items to the selection.
                    /// </summary>
                    /// <param name="items" locid="WinJS.UI._Selection.prototype.add_items">
                    /// The indexes or keys of the items to add.
                    /// You can provide different types of objects for the items parameter:
                    /// you can specify an index, a key, or a range of indexes.
                    /// It can also be an array that contains one or more of these objects.
                    /// </param>
                    /// <returns type="Promise" locid="WinJS.UI._Selection.prototype.add_returnValue">
                    /// A Promise that is fulfilled when the operation completes.
                    /// </returns>
                    /// </signature>

                    if (!isEverythingRange(items)) {
                        var that = this;
                        return this._execute("_add", items).then(function () {
                            return that._ensureKeys();
                        }).then(function () {
                            return that._ensureCount();
                        });
                    } else {
                        return this.selectAll();
                    }
                },

                remove: function (items) {
                    /// <signature helpKeyword="WinJS.UI._Selection.prototype.remove">
                    /// <summary locid="WinJS.UI._Selection.prototype.remove">
                    /// Removes the specified items from the selection.
                    /// </summary>
                    /// <param name="items" locid="WinJS.UI._Selection.prototype.remove_items">
                    /// The indexes or keys of the items to remove. You can provide different types of objects for the items parameter:
                    /// you can specify an index, a key, or a range of indexes.
                    /// It can also be an array that contains one or more of these objects.
                    /// </param>
                    /// <returns type="Promise" locid="WinJS.UI._Selection.prototype.remove_returnValue">
                    /// A Promise that is fulfilled when the operation completes.
                    /// </returns>
                    /// </signature>

                    var that = this;
                    return this._execute("_remove", items).then(function () {
                        return that._ensureKeys();
                    });
                },

                selectAll: function () {
                    /// <signature helpKeyword="WinJS.UI._Selection.prototype.selectAll">
                    /// <summary locid="WinJS.UI._Selection.prototype.selectAll">
                    /// Adds all the items in the ListView to the selection.
                    /// </summary>
                    /// <returns type="Promise" locid="WinJS.UI._Selection.prototype.selectAll_returnValue">
                    /// A Promise that is fulfilled when the operation completes.
                    /// </returns>
                    /// </signature>

                    var that = this;
                    return that._ensureCount().then(function () {
                        if (that._itemsCount) {
                            var range = {
                                firstIndex: 0,
                                lastIndex: that._itemsCount - 1,
                            };
                            that._retainRange(range);
                            that._releaseRanges(that._ranges);
                            that._ranges = [range];
                            return that._ensureKeys();
                        }
                    });
                },

                _execute: function (operation, items) {
                    var that = this,
                        keysSupported = !!that._getListBinding().fromKey,
                        array = Array.isArray(items) ? items : [items],
                        promises = [Promise.wrap()];

                    function toRange(type, first, last) {
                        var retVal = {};
                        retVal["first" + type] = first;
                        retVal["last" + type] = last;
                        return retVal;
                    }

                    function handleKeys(range) {
                        var binding = that._getListBinding();

                        var promise = Promise.join([binding.fromKey(range.firstKey), binding.fromKey(range.lastKey)]).then(function (items) {
                            if (items[0] && items[1]) {
                                range.firstIndex = items[0].index;
                                range.lastIndex = items[1].index;
                                that[operation](range);
                            }
                            return range;
                        });
                        promises.push(promise);
                    }

                    for (var i = 0, len = array.length; i < len; i++) {
                        var item = array[i];
                        if (typeof item === "number") {
                            this[operation](toRange("Index", item, item));
                        } else if (item) {
                            if (keysSupported && item.key !== undefined) {
                                handleKeys(toRange("Key", item.key, item.key));
                            } else if (keysSupported && item.firstKey !== undefined && item.lastKey !== undefined) {
                                handleKeys(toRange("Key", item.firstKey, item.lastKey));
                            } else if (item.index !== undefined && typeof item.index === "number") {
                                this[operation](toRange("Index", item.index, item.index));
                            } else if (item.firstIndex !== undefined && item.lastIndex !== undefined &&
                                    typeof item.firstIndex === "number" && typeof item.lastIndex === "number") {
                                this[operation](toRange("Index", item.firstIndex, item.lastIndex));
                            }
                        }
                    }

                    return Promise.join(promises);
                },

                _set: function (range) {
                    this._retainRange(range);
                    this._ranges.push(range);
                },

                _add: function (newRange) {
                    var that = this,
                        prev = null,
                        range,
                        inserted;

                    var merge = function (left, right) {
                        if (right.lastIndex > left.lastIndex) {
                            left.lastIndex = right.lastIndex;
                            left.lastKey = right.lastKey;
                            if (left.lastPromise) {
                                left.lastPromise.release();
                            }
                            left.lastPromise = that._getListBinding().fromIndex(left.lastIndex).retain();
                        }
                    };
                    var mergeWithPrev;
                    for (var i = 0, len = this._ranges.length; i < len; i++) {
                        range = this._ranges[i];
                        if (newRange.firstIndex < range.firstIndex) {
                            mergeWithPrev = prev && newRange.firstIndex < (prev.lastIndex + 1);
                            if (mergeWithPrev) {
                                inserted = i - 1;
                                merge(prev, newRange);
                            } else {
                                this._insertRange(i, newRange);
                                inserted = i;
                            }
                            break;
                        } else if (newRange.firstIndex === range.firstIndex) {
                            merge(range, newRange);
                            inserted = i;
                            break;
                        }
                        prev = range;
                    }

                    if (inserted === undefined) {
                        var last = this._ranges.length ? this._ranges[this._ranges.length - 1] : null,
                            mergeWithLast = last && newRange.firstIndex < (last.lastIndex + 1);
                        if (mergeWithLast) {
                            merge(last, newRange);
                        } else {
                            this._retainRange(newRange);
                            this._ranges.push(newRange);
                        }
                    } else {
                        prev = null;
                        for (i = inserted + 1, len = this._ranges.length; i < len; i++) {
                            range = this._ranges[i];
                            if (newRange.lastIndex < range.firstIndex) {
                                mergeWithPrev = prev && prev.lastIndex > newRange.lastIndex;
                                if (mergeWithPrev) {
                                    merge(this._ranges[inserted], prev);
                                }
                                this._removeRanges(inserted + 1, i - inserted - 1);
                                break;
                            } else if (newRange.lastIndex === range.firstIndex) {
                                merge(this._ranges[inserted], range);
                                this._removeRanges(inserted + 1, i - inserted);
                                break;
                            }
                            prev = range;
                        }
                        if (i >= len) {
                            merge(this._ranges[inserted], this._ranges[len - 1]);
                            this._removeRanges(inserted + 1, len - inserted - 1);
                        }
                    }
                },

                _remove: function (toRemove) {
                    var that = this;

                    function retainPromise(index) {
                        return that._getListBinding().fromIndex(index).retain();
                    }

                    // This method is called when a range needs to be unselected.  It is inspecting every range in the current selection comparing
                    // it to the range which is being unselected and it is building an array of new selected ranges
                    var ranges = [];
                    for (var i = 0, len = this._ranges.length; i < len; i++) {
                        var range = this._ranges[i];
                        if (range.lastIndex < toRemove.firstIndex || range.firstIndex > toRemove.lastIndex) {
                            // No overlap with the unselected range
                            ranges.push(range);
                        } else if (range.firstIndex < toRemove.firstIndex && range.lastIndex >= toRemove.firstIndex && range.lastIndex <= toRemove.lastIndex) {
                            // The end of this range is being unselected
                            ranges.push({
                                firstIndex: range.firstIndex,
                                firstKey: range.firstKey,
                                firstPromise: range.firstPromise,
                                lastIndex: toRemove.firstIndex - 1,
                                lastPromise: retainPromise(toRemove.firstIndex - 1)
                            });
                            range.lastPromise.release();
                        } else if (range.lastIndex > toRemove.lastIndex && range.firstIndex >= toRemove.firstIndex && range.firstIndex <= toRemove.lastIndex) {
                            // The beginning of this range is being unselected
                            ranges.push({
                                firstIndex: toRemove.lastIndex + 1,
                                firstPromise: retainPromise(toRemove.lastIndex + 1),
                                lastIndex: range.lastIndex,
                                lastKey: range.lastKey,
                                lastPromise: range.lastPromise
                            });
                            range.firstPromise.release();
                        } else if (range.firstIndex < toRemove.firstIndex && range.lastIndex > toRemove.lastIndex) {
                            // The middle part of this range is being unselected
                            ranges.push({
                                firstIndex: range.firstIndex,
                                firstKey: range.firstKey,
                                firstPromise: range.firstPromise,
                                lastIndex: toRemove.firstIndex - 1,
                                lastPromise: retainPromise(toRemove.firstIndex - 1),
                            });
                            ranges.push({
                                firstIndex: toRemove.lastIndex + 1,
                                firstPromise: retainPromise(toRemove.lastIndex + 1),
                                lastIndex: range.lastIndex,
                                lastKey: range.lastKey,
                                lastPromise: range.lastPromise
                            });
                        } else {
                            // The whole range is being unselected
                            range.firstPromise.release();
                            range.lastPromise.release();
                        }
                    }
                    this._ranges = ranges;
                },

                _ensureKeys: function () {
                    var promises = [Promise.wrap()];
                    var that = this;

                    var ensureKey = function (which, range) {
                        var keyProperty = which + "Key";

                        if (!range[keyProperty]) {
                            var promise = range[which + "Promise"];
                            promise.then(function (item) {
                                if (item) {
                                    range[keyProperty] = item.key;
                                }
                            });
                            return promise;
                        } else {
                            return Promise.wrap();
                        }
                    };

                    for (var i = 0, len = this._ranges.length; i < len; i++) {
                        var range = this._ranges[i];
                        promises.push(ensureKey("first", range));
                        promises.push(ensureKey("last", range));
                    }

                    Promise.join(promises).then(function () {
                        that._ranges = that._ranges.filter(function (range) {
                            return range.firstKey && range.lastKey;
                        });
                    });
                    return Promise.join(promises);
                },

                _mergeRanges: function (target, source) {
                    target.lastIndex = source.lastIndex;
                    target.lastKey = source.lastKey;
                },

                _isIncluded: function (index) {
                    if (this.isEverything()) {
                        return true;
                    } else {
                        for (var i = 0, len = this._ranges.length; i < len; i++) {
                            var range = this._ranges[i];
                            if (range.firstIndex <= index && index <= range.lastIndex) {
                                return true;
                            }
                        }
                        return false;
                    }
                },

                _ensureCount: function () {
                    var that = this;
                    return this._listView._itemsCount().then(function (count) {
                        that._itemsCount = count;
                    });
                },

                _insertRange: function (index, newRange) {
                    this._retainRange(newRange);
                    this._ranges.splice(index, 0, newRange);
                },

                _removeRanges: function (index, howMany) {
                    for (var i = 0; i < howMany; i++) {
                        this._releaseRange(this._ranges[index + i]);
                    }
                    this._ranges.splice(index, howMany);
                },

                _retainRange: function (range) {
                    if (!range.firstPromise) {
                        range.firstPromise = this._getListBinding().fromIndex(range.firstIndex).retain();
                    }
                    if (!range.lastPromise) {
                        range.lastPromise = this._getListBinding().fromIndex(range.lastIndex).retain();
                    }
                },

                _retainRanges: function () {
                    for (var i = 0, len = this._ranges.length; i < len; i++) {
                        this._retainRange(this._ranges[i]);
                    }
                },

                _releaseRange: function (range) {
                    range.firstPromise.release();
                    range.lastPromise.release();
                },

                _releaseRanges: function (ranges) {
                    for (var i = 0, len = ranges.length; i < len; ++i) {
                        this._releaseRange(ranges[i]);
                    }
                },

                _getListBinding: function () {
                    return this._listView._itemsManager._listBinding;
                }
            }, {
                supportedForProcessing: false,
            });
        }),

        // This component is responsible for holding selection state
        _SelectionManager: _Base.Namespace._lazy(function () {
            var _SelectionManager = function (listView) {
                this._listView = listView;
                this._selected = new exports._Selection(this._listView);
                // Don't rename this member. Some apps reference it.
                this._pivot = _Constants._INVALID_INDEX;
                this._focused = { type: _UI.ObjectType.item, index: 0 };
                this._pendingChange = Promise.wrap();
            };
            _SelectionManager.prototype = {
                count: function () {
                    /// <signature helpKeyword="WinJS.UI._SelectionManager.prototype.count">
                    /// <summary locid="WinJS.UI._SelectionManager.prototype.count">
                    /// Returns the number of items in the selection.
                    /// </summary>
                    /// <returns type="Number" locid="WinJS.UI._SelectionManager.prototype.count_returnValue">
                    /// The number of items in the selection.
                    /// </returns>
                    /// </signature>
                    return this._selected.count();
                },

                getIndices: function () {
                    /// <signature helpKeyword="WinJS.UI._SelectionManager.prototype.getIndices">
                    /// <summary locid="WinJS.UI._SelectionManager.prototype.getIndices">
                    /// Returns a list of the indexes for the items in the selection.
                    /// </summary>
                    /// <returns type="Array" locid="WinJS.UI._SelectionManager.prototype.getIndices_returnValue">
                    /// The list of indexes for the items in the selection as an array of Number objects.
                    /// </returns>
                    /// </signature>
                    return this._selected.getIndices();
                },

                getItems: function () {
                    /// <signature helpKeyword="WinJS.UI._SelectionManager.prototype.getItems">
                    /// <summary locid="WinJS.UI._SelectionManager.prototype.getItems">
                    /// Returns an array that contains the items in the selection.
                    /// </summary>
                    /// <returns type="Promise" locid="WinJS.UI._SelectionManager.prototype.getItems_returnValue">
                    /// A Promise that contains an array of the requested IItem objects.
                    /// </returns>
                    /// </signature>

                    return this._selected.getItems();
                },

                getRanges: function () {
                    /// <signature helpKeyword="WinJS.UI._SelectionManager.prototype.getRanges">
                    /// <summary locid="WinJS.UI._SelectionManager.prototype.getRanges">
                    /// Gets an array of the index ranges for the selected items.
                    /// </summary>
                    /// <returns type="Array" locid="WinJS.UI._SelectionManager.prototype.getRanges_returnValue">
                    /// An array that contains an ISelectionRange object for each index range in the selection.
                    /// </returns>
                    /// </signature>
                    return this._selected.getRanges();
                },

                isEverything: function () {
                    /// <signature helpKeyword="WinJS.UI._SelectionManager.prototype.isEverything">
                    /// <summary locid="WinJS.UI._SelectionManager.prototype.isEverything">
                    /// Returns a value that indicates whether the selection contains every item in the data source.
                    /// </summary>
                    /// <returns type="Boolean" locid="WinJS.UI._SelectionManager.prototype.isEverything_returnValue">
                    /// true if the selection contains every item in the data source; otherwise, false.
                    /// </returns>
                    /// </signature>
                    return this._selected.isEverything();
                },

                set: function (items) {
                    /// <signature helpKeyword="WinJS.UI._SelectionManager.prototype.set">
                    /// <summary locid="WinJS.UI._SelectionManager.prototype.set">
                    /// Clears the current selection and replaces it with the specified items.
                    /// </summary>
                    /// <param name="items" locid="WinJS.UI._SelectionManager.prototype.set_items">
                    /// The indexes or keys of the items that make up the selection.
                    /// You can provide different types of objects for the items parameter:
                    /// you can specify an index, a key, or a range of indexes.
                    /// It can also be an array that contains one or more of these objects.
                    /// </param>
                    /// <returns type="Promise" locid="WinJS.UI._SelectionManager.prototype.set_returnValue">
                    /// A Promise that is fulfilled when the operation completes.
                    /// </returns>
                    /// </signature>
                    var that = this,
                        signal = new _Signal();
                    return this._synchronize(signal).then(function () {
                        var newSelection = new exports._Selection(that._listView);
                        return newSelection.set(items).then(
                            function () {
                                that._set(newSelection);
                                signal.complete();
                            },
                            function (error) {
                                newSelection.clear();
                                signal.complete();
                                return Promise.wrapError(error);
                            }
                        );
                    });
                },

                clear: function () {
                    /// <signature helpKeyword="WinJS.UI._SelectionManager.prototype.clear">
                    /// <summary locid="WinJS.UI._SelectionManager.prototype.clear">
                    /// Clears the selection.
                    /// </summary>
                    /// <returns type="Promise" locid="WinJS.UI._SelectionManager.prototype.clear_returnValue">
                    /// A Promise that is fulfilled when the clear operation completes.
                    /// </returns>
                    /// </signature>

                    var that = this,
                        signal = new _Signal();
                    return this._synchronize(signal).then(function () {
                        var newSelection = new exports._Selection(that._listView);
                        return newSelection.clear().then(
                            function () {
                                that._set(newSelection);
                                signal.complete();
                            },
                            function (error) {
                                newSelection.clear();
                                signal.complete();
                                return Promise.wrapError(error);
                            }
                        );
                    });
                },

                add: function (items) {
                    /// <signature helpKeyword="WinJS.UI._SelectionManager.prototype.add">
                    /// <summary locid="WinJS.UI._SelectionManager.prototype.add">
                    /// Adds one or more items to the selection.
                    /// </summary>
                    /// <param name="items" locid="WinJS.UI._SelectionManager.prototype.add_items">
                    /// The indexes or keys of the items to add.
                    /// You can provide different types of objects for the items parameter:
                    /// you can specify an index, a key, or a range of indexes.
                    /// It can also be an array that contains one or more of these objects.
                    /// </param>
                    /// <returns type="Promise" locid="WinJS.UI._SelectionManager.prototype.add_returnValue">
                    /// A Promise that is fulfilled when the operation completes.
                    /// </returns>
                    /// </signature>
                    var that = this,
                        signal = new _Signal();
                    return this._synchronize(signal).then(function () {
                        var newSelection = that._cloneSelection();
                        return newSelection.add(items).then(
                            function () {
                                that._set(newSelection);
                                signal.complete();
                            },
                            function (error) {
                                newSelection.clear();
                                signal.complete();
                                return Promise.wrapError(error);
                            }
                        );
                    });
                },

                remove: function (items) {
                    /// <signature helpKeyword="WinJS.UI._SelectionManager.prototype.remove">
                    /// <summary locid="WinJS.UI._SelectionManager.prototype.remove">
                    /// Removes the specified items from the selection.
                    /// </summary>
                    /// <param name="items" locid="WinJS.UI._SelectionManager.prototype.remove_items">
                    /// The indexes or keys of the items to remove. You can provide different types of objects for the items parameter:
                    /// you can specify an index, a key, or a range of indexes.
                    /// It can also be an array that contains one or more of these objects.
                    /// </param>
                    /// <returns type="Promise" locid="WinJS.UI._SelectionManager.prototype.remove_returnValue">
                    /// A Promise that is fulfilled when the operation completes.
                    /// </returns>
                    /// </signature>
                    var that = this,
                        signal = new _Signal();
                    return this._synchronize(signal).then(function () {
                        var newSelection = that._cloneSelection();
                        return newSelection.remove(items).then(
                            function () {
                                that._set(newSelection);
                                signal.complete();
                            },
                            function (error) {
                                newSelection.clear();
                                signal.complete();
                                return Promise.wrapError(error);
                            }
                        );
                    });
                },

                selectAll: function () {
                    /// <signature helpKeyword="WinJS.UI._SelectionManager.prototype.selectAll">
                    /// <summary locid="WinJS.UI._SelectionManager.prototype.selectAll">
                    /// Adds all the items in the ListView to the selection.
                    /// </summary>
                    /// <returns type="Promise" locid="WinJS.UI._SelectionManager.prototype.selectAll_returnValue">
                    /// A Promise that is fulfilled when the operation completes.
                    /// </returns>
                    /// </signature>
                    var that = this,
                        signal = new _Signal();
                    return this._synchronize(signal).then(function () {
                        var newSelection = new exports._Selection(that._listView);
                        return newSelection.selectAll().then(
                            function () {
                                that._set(newSelection);
                                signal.complete();
                            },
                            function (error) {
                                newSelection.clear();
                                signal.complete();
                                return Promise.wrapError(error);
                            }
                        );
                    });
                },

                _synchronize: function (signal) {
                    var that = this;
                    return this._listView._versionManager.unlocked.then(function () {
                        var currentPendingChange = that._pendingChange;
                        that._pendingChange = Promise.join([currentPendingChange, signal.promise]).then(function () { });
                        return currentPendingChange;
                    });
                },

                _reset: function () {
                    this._pivot = _Constants._INVALID_INDEX;
                    this._setFocused({ type: _UI.ObjectType.item, index: 0 }, this._keyboardFocused());
                    this._pendingChange.cancel();
                    this._pendingChange = Promise.wrap();
                    this._selected.clear();
                    this._selected = new exports._Selection(this._listView);
                },

                _dispose: function () {
                    this._selected.clear();
                    this._selected = null;
                    this._listView = null;
                },

                _set: function (newSelection) {
                    var that = this;
                    return this._fireSelectionChanging(newSelection).then(function (approved) {
                        if (approved) {
                            that._selected.clear();
                            that._selected = newSelection;
                            that._listView._updateSelection();
                            that._fireSelectionChanged();
                        } else {
                            newSelection.clear();
                        }
                        return approved;
                    });
                },

                _fireSelectionChanging: function (newSelection) {
                    var eventObject = _Global.document.createEvent("CustomEvent"),
                        newSelectionUpdated = Promise.wrap();

                    eventObject.initCustomEvent("selectionchanging", true, true, {
                        newSelection: newSelection,
                        preventTapBehavior: function () {
                        },
                        setPromise: function (promise) {
                            /// <signature helpKeyword="WinJS.UI.SelectionManager.selectionchanging.setPromise">
                            /// <summary locid="WinJS.UI.SelectionManager.selectionchanging.setPromise">
                            /// Used to inform the ListView that asynchronous work is being performed, and that this
                            /// event handler should not be considered complete until the promise completes.
                            /// </summary>
                            /// <param name="promise" type="WinJS.Promise" locid="WinJS.UI.SelectionManager.selectionchanging.setPromise_p:promise">
                            /// The promise to wait for.
                            /// </param>
                            /// </signature>

                            newSelectionUpdated = promise;
                        }
                    });

                    var approved = this._listView._element.dispatchEvent(eventObject);
                    return newSelectionUpdated.then(function () {
                        return approved;
                    });
                },

                _fireSelectionChanged: function () {
                    var eventObject = _Global.document.createEvent("CustomEvent");
                    eventObject.initCustomEvent("selectionchanged", true, false, null);
                    this._listView._element.dispatchEvent(eventObject);
                },

                _getFocused: function () {
                    return { type: this._focused.type, index: this._focused.index };
                },

                _setFocused: function (entity, keyboardFocused) {
                    this._focused = { type: entity.type, index: entity.index };
                    this._focusedByKeyboard = keyboardFocused;
                },

                _keyboardFocused: function () {
                    return this._focusedByKeyboard;
                },

                _updateCount: function (count) {
                    this._selected._itemsCount = count;
                },

                _isIncluded: function (index) {
                    return this._selected._isIncluded(index);
                },

                _cloneSelection: function () {
                    var newSelection = new exports._Selection(this._listView);
                    newSelection._ranges = this._selected.getRanges();
                    newSelection._itemsCount = this._selected._itemsCount;
                    newSelection._retainRanges();
                    return newSelection;
                }
            };
            _SelectionManager.supportedForProcessing = false;
            return _SelectionManager;
        })
    });

});

// Copyright (c) Microsoft Corporation.  All Rights Reserved. Licensed under the MIT License. See License.txt in the project root for license information.
define('WinJS/Controls/ListView/_BrowseMode',[
    'exports',
    '../../Core/_Global',
    '../../Core/_Base',
    '../../Core/_BaseUtils',
    '../../Animations',
    '../../Promise',
    '../../Utilities/_ElementUtilities',
    '../../Utilities/_UI',
    '../ItemContainer/_Constants',
    '../ItemContainer/_ItemEventsHandler',
    './_SelectionManager'
    ], function browseModeInit(exports, _Global, _Base, _BaseUtils, Animations, Promise, _ElementUtilities, _UI, _Constants, _ItemEventsHandler, _SelectionManager) {
    "use strict";

    var transformName = _BaseUtils._browserStyleEquivalents["transform"].scriptName;
    // This component is responsible for handling input in Browse Mode.
    // When the user clicks on an item in this mode itemInvoked event is fired.
    _Base.Namespace._moduleDefine(exports, "WinJS.UI", {

        _SelectionMode: _Base.Namespace._lazy(function () {

            function clampToRange(first, last, x) {
                return Math.max(first, Math.min(last, x));
            }

            function dispatchKeyboardNavigating(element, oldEntity, newEntity) {
                var navigationEvent = _Global.document.createEvent("CustomEvent");
                navigationEvent.initCustomEvent("keyboardnavigating", true, true, {
                    oldFocus: oldEntity.index,
                    oldFocusType: oldEntity.type,
                    newFocus: newEntity.index,
                    newFocusType: newEntity.type
                });
                return element.dispatchEvent(navigationEvent);
            }

            var _SelectionMode = _Base.Class.define(function _SelectionMode_ctor(modeSite) {
                this.inboundFocusHandled = false;
                this._pressedContainer = null;
                this._pressedItemBox = null;
                this._pressedHeader = null;
                this._pressedEntity = { type: _UI.ObjectType.item, index: _Constants._INVALID_INDEX };
                this._pressedPosition = null;

                this.initialize(modeSite);
            },{
                _dispose: function () {
                    if (this._itemEventsHandler) {
                        this._itemEventsHandler.dispose();
                    }
                    if (this._setNewFocusItemOffsetPromise) {
                        this._setNewFocusItemOffsetPromise.cancel();
                    }
                },

                initialize: function (modeSite) {
                    this.site = modeSite;

                    this._keyboardNavigationHandlers = {};
                    this._keyboardAcceleratorHandlers = {};

                    var site = this.site,
                        that = this;
                    this._itemEventsHandler = new _ItemEventsHandler._ItemEventsHandler(Object.create({
                        containerFromElement: function (element) {
                            return site._view.items.containerFrom(element);
                        },
                        indexForItemElement: function (element) {
                            return site._view.items.index(element);
                        },
                        indexForHeaderElement: function (element) {
                            return site._groups.index(element);
                        },
                        itemBoxAtIndex: function (index) {
                            return site._view.items.itemBoxAt(index);
                        },
                        itemAtIndex: function (index) {
                            return site._view.items.itemAt(index);
                        },
                        headerAtIndex: function (index) {
                            return site._groups.group(index).header;
                        },
                        headerFromElement: function (element) {
                            return site._groups.headerFrom(element);
                        },
                        containerAtIndex: function (index) {
                            return site._view.items.containerAt(index);
                        },
                        isZombie: function () {
                            return site._isZombie();
                        },
                        getItemPosition: function (entity) {
                            return site._getItemPosition(entity);
                        },
                        rtl: function () {
                            return site._rtl();
                        },
                        fireInvokeEvent: function (entity, itemElement) {
                            return that._fireInvokeEvent(entity, itemElement);
                        },
                        verifySelectionAllowed: function (index) {
                            return that._verifySelectionAllowed(index);
                        },
                        changeFocus: function (newFocus, skipSelection, ctrlKeyDown, skipEnsureVisible, keyboardFocused) {
                            return site._changeFocus(newFocus, skipSelection, ctrlKeyDown, skipEnsureVisible, keyboardFocused);
                        },
                        selectRange: function (firstIndex, lastIndex, additive) {
                            return that._selectRange(firstIndex, lastIndex, additive);
                        },
                        isInSelectionMode: function () {
                            return site._isInSelectionMode();
                        }
                    }, {
                        pressedEntity: {
                            enumerable: true,
                            get: function () {
                                return that._pressedEntity;
                            },
                            set: function (value) {
                                that._pressedEntity = value;
                            }
                        },
                        pressedContainerScaleTransform: {
                            enumerable: true,
                            get: function () {
                                return that._pressedContainerScaleTransform;
                            },
                            set: function (value) {
                                that._pressedContainerScaleTransform = value;
                            }
                        },
                        pressedContainer: {
                            enumerable: true,
                            get: function () {
                                return that._pressedContainer;
                            },
                            set: function (value) {
                                that._pressedContainer = value;
                            }
                        },

                        pressedItemBox: {
                            enumerable: true,
                            get: function () {
                                return that._pressedItemBox;
                            },
                            set: function (value) {
                                that._pressedItemBox = value;
                            }
                        },

                        pressedHeader: {
                            enumerable: true,
                            get: function () {
                                return that._pressedHeader;
                            },
                            set: function (value) {
                                return that._pressedHeader = value;
                            }
                        },

                        pressedPosition: {
                            enumerable: true,
                            get: function () {
                                return that._pressedPosition;
                            },
                            set: function (value) {
                                that._pressedPosition = value;
                            }
                        },

                        pressedElement: {
                            enumerable: true,
                            set: function (value) {
                                that._pressedElement = value;
                            }
                        },
                        eventHandlerRoot: {
                            enumerable: true,
                            get: function () {
                                return site._viewport;
                            }
                        },
                        selectionMode: {
                            enumerable: true,
                            get: function () {
                                return site._selectionMode;
                            }
                        },
                        accessibleItemClass: {
                            enumerable: true,
                            get: function () {
                                // CSS class of the element with the aria role
                                return _Constants._itemClass;
                            }
                        },
                        canvasProxy: {
                            enumerable: true,
                            get: function () {
                                return site._canvasProxy;
                            }
                        },
                        tapBehavior: {
                            enumerable: true,
                            get: function () {
                                return site._tap;
                            }
                        },
                        headerTapBehavior: {
                            enumerable: true,
                            get: function () {
                                return site._groupHeaderTap;
                            }
                        },
                        draggable: {
                            enumerable: true,
                            get: function () {
                                return site.itemsDraggable || site.itemsReorderable;
                            }
                        },
                        selection: {
                            enumerable: true,
                            get: function () {
                                return site._selection;
                            }
                        },
                        customFootprintParent: {
                            enumerable: true,
                            get: function () {
                                return null;
                            }
                        }
                    }));

                    function createArrowHandler(direction, clampToBounds) {
                        var handler = function (oldFocus) {
                            return modeSite._view.getAdjacent(oldFocus, direction);
                        };
                        handler.clampToBounds = clampToBounds;
                        return handler;
                    }

                    var Key = _ElementUtilities.Key;
                    this._keyboardNavigationHandlers[Key.upArrow] = createArrowHandler(Key.upArrow);
                    this._keyboardNavigationHandlers[Key.downArrow] = createArrowHandler(Key.downArrow);
                    this._keyboardNavigationHandlers[Key.leftArrow] = createArrowHandler(Key.leftArrow);
                    this._keyboardNavigationHandlers[Key.rightArrow] = createArrowHandler(Key.rightArrow);
                    this._keyboardNavigationHandlers[Key.pageUp] = createArrowHandler(Key.pageUp, true);
                    this._keyboardNavigationHandlers[Key.pageDown] = createArrowHandler(Key.pageDown, true);
                    this._keyboardNavigationHandlers[Key.home] = function (oldFocus) {
                        if (that.site._header && (oldFocus.type === _UI.ObjectType.groupHeader || oldFocus.type === _UI.ObjectType.footer)) {
                            return Promise.wrap({ type: _UI.ObjectType.header, index: 0 });
                        }


                        return Promise.wrap({ type: (oldFocus.type !== _UI.ObjectType.footer ? oldFocus.type : _UI.ObjectType.groupHeader), index: 0 });
                    };
                    this._keyboardNavigationHandlers[Key.end] = function (oldFocus) {
                        if (that.site._footer && (oldFocus.type === _UI.ObjectType.groupHeader || oldFocus.type === _UI.ObjectType.header)) {
                            return Promise.wrap({ type: _UI.ObjectType.footer, index: 0 });
                        } else if (oldFocus.type === _UI.ObjectType.groupHeader || oldFocus.type === _UI.ObjectType.header) {
                            return Promise.wrap({ type: _UI.ObjectType.groupHeader, index: site._groups.length() - 1 });
                        } else {
                            // Get the index of the last container
                            var lastIndex = that.site._view.lastItemIndex();
                            if (lastIndex >= 0) {
                                return Promise.wrap({ type: oldFocus.type, index: lastIndex });
                            } else {
                                return Promise.cancel;
                            }
                        }
                    };

                    this._keyboardAcceleratorHandlers[Key.a] = function () {
                        if (that.site._multiSelection()) {
                            that._selectAll();
                        }
                    };
                },

                staticMode: function SelectionMode_staticMode() {
                    return this.site._tap === _UI.TapBehavior.none && this.site._selectionMode === _UI.SelectionMode.none;
                },

                itemUnrealized: function SelectionMode_itemUnrealized(index, itemBox) {
                    if (this._pressedEntity.type === _UI.ObjectType.groupHeader) {
                        return;
                    }

                    if (this._pressedEntity.index === index) {
                        this._resetPointerDownState();
                    }

                    if (this._itemBeingDragged(index)) {
                        for (var i = this._draggedItemBoxes.length - 1; i >= 0; i--) {
                            if (this._draggedItemBoxes[i] === itemBox) {
                                _ElementUtilities.removeClass(itemBox, _Constants._dragSourceClass);
                                this._draggedItemBoxes.splice(i, 1);
                            }
                        }
                    }
                },

                _fireInvokeEvent: function SelectionMode_fireInvokeEvent(entity, itemElement) {
                    if (!itemElement) {
                        return;
                    }

                    var that = this;
                    function fireInvokeEventImpl(dataSource, isHeader) {
                        var listBinding = dataSource.createListBinding(),
                             promise = listBinding.fromIndex(entity.index),
                             eventName = isHeader ? "groupheaderinvoked" : "iteminvoked";

                        promise.done(function () {
                            listBinding.release();
                        });

                        var eventObject = _Global.document.createEvent("CustomEvent");
                        eventObject.initCustomEvent(eventName, true, true, isHeader ? {
                            groupHeaderPromise: promise,
                            groupHeaderIndex: entity.index
                        } : {
                            itemPromise: promise,
                            itemIndex: entity.index
                        });

                        // If preventDefault was not called, call the default action on the site
                        if (itemElement.dispatchEvent(eventObject)) {
                            that.site._defaultInvoke(entity);
                        }
                    }

                    if (entity.type === _UI.ObjectType.groupHeader) {
                        if (this.site._groupHeaderTap === _UI.GroupHeaderTapBehavior.invoke &&
                            entity.index !== _Constants._INVALID_INDEX) {
                            fireInvokeEventImpl(this.site.groupDataSource, true);
                        }
                    } else {
                        // We don't want to raise an iteminvoked event when the ListView's tapBehavior is none, or if it's in selection mode.
                        if (this.site._tap !== _UI.TapBehavior.none && entity.index !== _Constants._INVALID_INDEX && !(this.site._isInSelectionMode())) {
                            fireInvokeEventImpl(this.site.itemDataSource, false);
                        }
                    }
                },

                _verifySelectionAllowed: function SelectionMode_verifySelectionAllowed(entity) {
                    if (entity.type === _UI.ObjectType.groupHeader) {
                        return {
                            canSelect: false,
                            canTapSelect: false
                        };
                    }

                    var itemIndex = entity.index;
                    var site = this.site;
                    var item = this.site._view.items.itemAt(itemIndex);
                    if (site._selectionAllowed() && site._selectOnTap() && !(item && _ElementUtilities.hasClass(item, _Constants._nonSelectableClass))) {
                        var selected = site._selection._isIncluded(itemIndex),
                            single = !site._multiSelection(),
                            newSelection = site._selection._cloneSelection();

                        if (selected) {
                            if (single) {
                                newSelection.clear();
                            } else {
                                newSelection.remove(itemIndex);
                            }
                        } else {
                            if (single) {
                                newSelection.set(itemIndex);
                            } else {
                                newSelection.add(itemIndex);
                            }
                        }

                        var eventObject = _Global.document.createEvent("CustomEvent"),
                            newSelectionUpdated = Promise.wrap(),
                            completed = false,
                            preventTap = false,
                            included;

                        eventObject.initCustomEvent("selectionchanging", true, true, {
                            newSelection: newSelection,
                            preventTapBehavior: function () {
                                preventTap = true;
                            },
                            setPromise: function (promise) {
                                /// <signature helpKeyword="WinJS.UI.BrowseMode.selectionchanging.setPromise">
                                /// <summary locid="WinJS.UI.BrowseMode.selectionchanging.setPromise">
                                /// Used to inform the ListView that asynchronous work is being performed, and that this
                                /// event handler should not be considered complete until the promise completes.
                                /// </summary>
                                /// <param name="promise" type="WinJS.Promise" locid="WinJS.UI.BrowseMode.selectionchanging.setPromise_p:promise">
                                /// The promise to wait for.
                                /// </param>
                                /// </signature>

                                newSelectionUpdated = promise;
                            }
                        });

                        var defaultBehavior = site._element.dispatchEvent(eventObject);

                        newSelectionUpdated.then(function () {
                            completed = true;
                            included = newSelection._isIncluded(itemIndex);
                            newSelection.clear();
                        });

                        var canSelect = defaultBehavior && completed && (selected || included);

                        return {
                            canSelect: canSelect,
                            canTapSelect: canSelect && !preventTap
                        };
                    } else {
                        return {
                            canSelect: false,
                            canTapSelect: false
                        };
                    }
                },

                _containedInElementWithClass: function SelectionMode_containedInElementWithClass(element, className) {
                    if (element.parentNode) {
                        var matches = element.parentNode.querySelectorAll("." + className + ", ." + className + " *");
                        for (var i = 0, len = matches.length; i < len; i++) {
                            if (matches[i] === element) {
                                return true;
                            }
                        }
                    }
                    return false;
                },

                _isDraggable: function SelectionMode_isDraggable(element) {
                    return (!this._containedInElementWithClass(element, _Constants._nonDraggableClass));
                },

                _isInteractive: function SelectionMode_isInteractive(element) {
                    return this._containedInElementWithClass(element, "win-interactive");
                },

                _resetPointerDownState: function SelectionMode_resetPointerDownState() {
                    this._itemEventsHandler.resetPointerDownState();
                },

                onPointerDown: function SelectionMode_onPointerDown(eventObject) {
                    this._itemEventsHandler.onPointerDown(eventObject);
                },

                onclick: function SelectionMode_onclick(eventObject) {
                    this._itemEventsHandler.onClick(eventObject);
                },

                onPointerUp: function SelectionMode_onPointerUp(eventObject) {
                    this._itemEventsHandler.onPointerUp(eventObject);
                },

                onPointerCancel: function SelectionMode_onPointerCancel(eventObject) {
                    this._itemEventsHandler.onPointerCancel(eventObject);
                },

                onLostPointerCapture: function SelectionMode_onLostPointerCapture(eventObject) {
                    this._itemEventsHandler.onLostPointerCapture(eventObject);
                },

                onContextMenu: function SelectionMode_onContextMenu(eventObject) {
                    this._itemEventsHandler.onContextMenu(eventObject);
                },

                onMSHoldVisual: function SelectionMode_onMSHoldVisual(eventObject) {
                    this._itemEventsHandler.onMSHoldVisual(eventObject);
                },

                onDataChanged: function SelectionMode_onDataChanged(eventObject) {
                    this._itemEventsHandler.onDataChanged(eventObject);
                },

                _removeTransform: function SelectionMode_removeTransform(element, transform) {
                    if (transform && element.style[transformName].indexOf(transform) !== -1) {
                        element.style[transformName] = element.style[transformName].replace(transform, "");
                    }
                },

                _selectAll: function SelectionMode_selectAll() {
                    var unselectableRealizedItems = [];
                    this.site._view.items.each(function (index, item) {
                        if (item && _ElementUtilities.hasClass(item, _Constants._nonSelectableClass)) {
                            unselectableRealizedItems.push(index);
                        }
                    });

                    this.site._selection.selectAll();
                    if (unselectableRealizedItems.length > 0) {
                        this.site._selection.remove(unselectableRealizedItems);
                    }
                },

                _selectRange: function SelectionMode_selectRange(firstIndex, lastIndex, additive) {
                    var ranges = [];
                    var currentStartRange = -1;
                    for (var i = firstIndex; i <= lastIndex; i++) {
                        var item = this.site._view.items.itemAt(i);
                        if (item && _ElementUtilities.hasClass(item, _Constants._nonSelectableClass)) {
                            if (currentStartRange !== -1) {
                                ranges.push({
                                    firstIndex: currentStartRange,
                                    lastIndex: i - 1
                                });
                                currentStartRange = -1;
                            }
                        } else if (currentStartRange === -1) {
                            currentStartRange = i;
                        }
                    }
                    if (currentStartRange !== -1) {
                        ranges.push({
                            firstIndex: currentStartRange,
                            lastIndex: lastIndex
                        });
                    }
                    if (ranges.length > 0) {
                        this.site._selection[additive ? "add" : "set"](ranges);
                    }
                },

                onDragStart: function SelectionMode_onDragStart(eventObject) {
                    this._pressedEntity = { type: _UI.ObjectType.item, index: this.site._view.items.index(eventObject.target) };
                    this.site._selection._pivot = _Constants._INVALID_INDEX;
                    // Drag shouldn't be initiated when the user holds down the mouse on a win-interactive element and moves.
                    // The problem is that the dragstart event's srcElement+target will both be an itembox (which has draggable=true), so we can't check for win-interactive in the dragstart event handler.
                    // The itemEventsHandler sets our _pressedElement field on MSPointerDown, so we use that instead when checking for interactive.
                    if (this._pressedEntity.index !== _Constants._INVALID_INDEX &&
                            (this.site.itemsDraggable || this.site.itemsReorderable) &&
                            !this.site._view.animating &&
                            this._isDraggable(eventObject.target) &&
                            (!this._pressedElement || !this._isInteractive(this._pressedElement))) {
                        this._dragging = true;
                        this._dragDataTransfer = eventObject.dataTransfer;
                        this._pressedPosition = _ElementUtilities._getCursorPos(eventObject);
                        this._dragInfo = null;
                        this._lastEnteredElement = eventObject.target;

                        if (this.site._selection._isIncluded(this._pressedEntity.index)) {
                            this._dragInfo = this.site.selection;
                        } else {
                            this._draggingUnselectedItem = true;
                            this._dragInfo = new _SelectionManager._Selection(this.site, [{ firstIndex: this._pressedEntity.index, lastIndex: this._pressedEntity.index }]);
                        }

                        var dropTarget = this.site.itemsReorderable;
                        var event = _Global.document.createEvent("CustomEvent");
                        event.initCustomEvent("itemdragstart", true, false, {
                            dataTransfer: eventObject.dataTransfer,
                            dragInfo: this._dragInfo
                        });

                        // Firefox requires setData to be called on the dataTransfer object in order for DnD to continue.
                        // Firefox also has an issue rendering the item's itemBox+element, so we need to use setDragImage, using the item's container, to get it to render.
                        eventObject.dataTransfer.setData("text", "");
                        if (eventObject.dataTransfer.setDragImage) {
                            var pressedItemData = this.site._view.items.itemDataAt(this._pressedEntity.index);
                            if (pressedItemData && pressedItemData.container) {
                                var rect = pressedItemData.container.getBoundingClientRect();
                                eventObject.dataTransfer.setDragImage(pressedItemData.container, eventObject.clientX - rect.left, eventObject.clientY - rect.top);
                            }
                        }
                        this.site.element.dispatchEvent(event);
                        if (this.site.itemsDraggable && !this.site.itemsReorderable) {
                            if (!this._firedDragEnter) {
                                if (this._fireDragEnterEvent(eventObject.dataTransfer)) {
                                    dropTarget = true;
                                    this._dragUnderstood = true;
                                }
                            }
                        }

                        if (dropTarget) {
                            this._addedDragOverClass = true;
                            _ElementUtilities.addClass(this.site._element, _Constants._dragOverClass);
                        }

                        this._draggedItemBoxes = [];

                        var that = this;
                        // A dragged element can be removed from the DOM by a number of actions - datasource removes/changes, being scrolled outside of the realized range, etc.
                        // The dragend event is fired on the original source element of the drag. If that element isn't in the DOM, though, the dragend event will only be fired on the element
                        // itself and not bubble up through the ListView's tree to the _viewport element where all the other drag event handlers are.
                        // The dragend event handler has to be added to the event's srcElement so that we always receive the event, even when the source element is unrealized.
                        var sourceElement = eventObject.target;
                        sourceElement.addEventListener("dragend", function itemDragEnd(eventObject) {
                            sourceElement.removeEventListener("dragend", itemDragEnd);
                            that.onDragEnd(eventObject);
                        });
                        // We delay setting the opacity of the dragged items so that IE has time to create a thumbnail before me make them invisible
                        _BaseUtils._yieldForDomModification(function () {
                            if (that._dragging) {
                                var indicesSelected = that._dragInfo.getIndices();
                                for (var i = 0, len = indicesSelected.length; i < len; i++) {
                                    var itemData = that.site._view.items.itemDataAt(indicesSelected[i]);
                                    if (itemData && itemData.itemBox) {
                                        that._addDragSourceClass(itemData.itemBox);
                                    }
                                }
                            }
                        });
                    } else {
                        eventObject.preventDefault();
                    }
                },

                onDragEnter: function (eventObject) {
                    var eventHandled = this._dragUnderstood;
                    this._lastEnteredElement = eventObject.target;
                    if (this._exitEventTimer) {
                        _Global.clearTimeout(this._exitEventTimer);
                        this._exitEventTimer = 0;
                    }

                    if (!this._firedDragEnter) {
                        if (this._fireDragEnterEvent(eventObject.dataTransfer)) {
                            eventHandled = true;
                        }
                    }

                    if (eventHandled || (this._dragging && this.site.itemsReorderable)) {
                        eventObject.preventDefault();
                        this._dragUnderstood = true;
                        if (!this._addedDragOverClass) {
                            this._addedDragOverClass = true;
                            _ElementUtilities.addClass(this.site._element, _Constants._dragOverClass);
                        }
                    }
                    this._pointerLeftRegion = false;
                },

                onDragLeave: function (eventObject) {
                    if (eventObject.target === this._lastEnteredElement) {
                        this._pointerLeftRegion = true;
                        this._handleExitEvent();
                    }
                },

                fireDragUpdateEvent: function () {
                    var event = _Global.document.createEvent("CustomEvent");
                    event.initCustomEvent("itemdragchanged", true, false, {
                        dataTransfer: this._dragDataTransfer,
                        dragInfo: this._dragInfo
                    });
                    this.site.element.dispatchEvent(event);
                },

                _fireDragEnterEvent: function (dataTransfer) {
                    var event = _Global.document.createEvent("CustomEvent");
                    event.initCustomEvent("itemdragenter", true, true, {
                        dataTransfer: dataTransfer
                    });
                    // The end developer must tell a ListView when a drag can be understood by calling preventDefault() on the event we fire
                    var dropTarget = (!this.site.element.dispatchEvent(event));
                    this._firedDragEnter = true;
                    return dropTarget;
                },

                _fireDragBetweenEvent: function (index, insertAfterIndex, dataTransfer) {
                    var event = _Global.document.createEvent("CustomEvent");
                    event.initCustomEvent("itemdragbetween", true, true, {
                        index: index,
                        insertAfterIndex: insertAfterIndex,
                        dataTransfer: dataTransfer
                    });
                    return this.site.element.dispatchEvent(event);
                },

                _fireDropEvent: function (index, insertAfterIndex, dataTransfer) {
                    var event = _Global.document.createEvent("CustomEvent");
                    event.initCustomEvent("itemdragdrop", true, true, {
                        index: index,
                        insertAfterIndex: insertAfterIndex,
                        dataTransfer: dataTransfer
                    });
                    return this.site.element.dispatchEvent(event);
                },

                _handleExitEvent: function () {
                    if (this._exitEventTimer) {
                        _Global.clearTimeout(this._exitEventTimer);
                        this._exitEventTimer = 0;
                    }
                    var that = this;
                    this._exitEventTimer = _Global.setTimeout(function () {
                        if (that.site._disposed) { return; }

                        if (that._pointerLeftRegion) {
                            that.site._layout.dragLeave && that.site._layout.dragLeave();
                            that._pointerLeftRegion = false;
                            that._dragUnderstood = false;
                            that._lastEnteredElement = null;
                            that._lastInsertPoint = null;
                            that._dragBetweenDisabled = false;
                            if (that._firedDragEnter) {
                                var event = _Global.document.createEvent("CustomEvent");
                                event.initCustomEvent("itemdragleave", true, false, {
                                });
                                that.site.element.dispatchEvent(event);
                                that._firedDragEnter = false;
                            }
                            if (that._addedDragOverClass) {
                                that._addedDragOverClass = false;
                                _ElementUtilities.removeClass(that.site._element, _Constants._dragOverClass);
                            }
                            that._exitEventTimer = 0;
                            that._stopAutoScroll();
                        }
                    }, 40);
                },

                _getEventPositionInElementSpace: function (element, eventObject) {
                    var elementRect = { left: 0, top: 0 };
                    try {
                        elementRect = element.getBoundingClientRect();
                    }
                    catch (err) { }

                    var computedStyle = _Global.getComputedStyle(element, null),
                        paddingLeft = parseInt(computedStyle["paddingLeft"]),
                        paddingTop = parseInt(computedStyle["paddingTop"]),
                        borderLeft = parseInt(computedStyle["borderLeftWidth"]),
                        borderTop = parseInt(computedStyle["borderTopWidth"]),
                        clientX = eventObject.clientX,
                        clientY = eventObject.clientY;

                    var position = {
                        x: +clientX === clientX ? (clientX - elementRect.left - paddingLeft - borderLeft) : 0,
                        y: +clientY === clientY ? (clientY - elementRect.top - paddingTop - borderTop) : 0
                    };

                    if (this.site._rtl()) {
                        position.x = (elementRect.right - elementRect.left) - position.x;
                    }

                    return position;
                },

                _getPositionInCanvasSpace: function (eventObject) {
                    var scrollLeft = this.site._horizontal() ? this.site.scrollPosition : 0,
                        scrollTop = this.site._horizontal() ? 0 : this.site.scrollPosition,
                        position = this._getEventPositionInElementSpace(this.site.element, eventObject);

                    return {
                        x: position.x + scrollLeft,
                        y: position.y + scrollTop
                    };
                },

                _itemBeingDragged: function (itemIndex) {
                    if (!this._dragging) {
                        return false;
                    }

                    return ((this._draggingUnselectedItem && this._dragInfo._isIncluded(itemIndex)) || (!this._draggingUnselectedItem && this.site._isSelected(itemIndex)));
                },

                _addDragSourceClass: function (itemBox) {
                    this._draggedItemBoxes.push(itemBox);
                    _ElementUtilities.addClass(itemBox, _Constants._dragSourceClass);
                    if (itemBox.parentNode) {
                        _ElementUtilities.addClass(itemBox.parentNode, _Constants._footprintClass);
                    }
                },

                renderDragSourceOnRealizedItem: function (itemIndex, itemBox) {
                    if (this._itemBeingDragged(itemIndex)) {
                        this._addDragSourceClass(itemBox);
                    }
                },

                onDragOver: function (eventObject) {
                    if (!this._dragUnderstood) {
                        return;
                    }
                    this._pointerLeftRegion = false;
                    eventObject.preventDefault();

                    var cursorPositionInCanvas = this._getPositionInCanvasSpace(eventObject),
                        cursorPositionInRoot = this._getEventPositionInElementSpace(this.site.element, eventObject);
                    this._checkAutoScroll(cursorPositionInRoot.x, cursorPositionInRoot.y);
                    if (this.site._layout.hitTest) {
                        if (this._autoScrollFrame) {
                            if (this._lastInsertPoint) {
                                this.site._layout.dragLeave();
                                this._lastInsertPoint = null;
                            }
                        } else {
                            var insertPoint = this.site._view.hitTest(cursorPositionInCanvas.x, cursorPositionInCanvas.y);
                            insertPoint.insertAfterIndex = clampToRange(-1, this.site._cachedCount - 1, insertPoint.insertAfterIndex);
                            if (!this._lastInsertPoint || this._lastInsertPoint.insertAfterIndex !== insertPoint.insertAfterIndex || this._lastInsertPoint.index !== insertPoint.index) {
                                this._dragBetweenDisabled = !this._fireDragBetweenEvent(insertPoint.index, insertPoint.insertAfterIndex, eventObject.dataTransfer);
                                if (!this._dragBetweenDisabled) {
                                    this.site._layout.dragOver(cursorPositionInCanvas.x, cursorPositionInCanvas.y, this._dragInfo);
                                } else {
                                    this.site._layout.dragLeave();
                                }
                            }
                            this._lastInsertPoint = insertPoint;
                        }
                    }
                },

                _clearDragProperties: function () {
                    if (this._addedDragOverClass) {
                        this._addedDragOverClass = false;
                        _ElementUtilities.removeClass(this.site._element, _Constants._dragOverClass);
                    }
                    if (this._draggedItemBoxes) {
                        for (var i = 0, len = this._draggedItemBoxes.length; i < len; i++) {
                            _ElementUtilities.removeClass(this._draggedItemBoxes[i], _Constants._dragSourceClass);
                            if (this._draggedItemBoxes[i].parentNode) {
                                _ElementUtilities.removeClass(this._draggedItemBoxes[i].parentNode, _Constants._footprintClass);
                            }
                        }
                        this._draggedItemBoxes = [];
                    }
                    this.site._layout.dragLeave();
                    this._dragging = false;
                    this._dragInfo = null;
                    this._draggingUnselectedItem = false;
                    this._dragDataTransfer = null;
                    this._lastInsertPoint = null;
                    this._resetPointerDownState();
                    this._lastEnteredElement = null;
                    this._dragBetweenDisabled = false;
                    this._firedDragEnter = false;
                    this._dragUnderstood = false;
                    this._stopAutoScroll();
                },

                onDragEnd: function () {
                    var event = _Global.document.createEvent("CustomEvent");
                    event.initCustomEvent("itemdragend", true, false, {});
                    this.site.element.dispatchEvent(event);
                    this._clearDragProperties();
                },

                _findFirstAvailableInsertPoint: function (selectedItems, startIndex, searchForwards) {
                    var indicesSelected = selectedItems.getIndices(),
                        dropIndexInSelection = -1,
                        count = this.site._cachedCount,
                        selectionCount = indicesSelected.length,
                        startIndexInSelection = -1,
                        dropIndex = startIndex;
                    for (var i = 0; i < selectionCount; i++) {
                        if (indicesSelected[i] === dropIndex) {
                            dropIndexInSelection = i;
                            startIndexInSelection = i;
                            break;
                        }
                    }

                    while (dropIndexInSelection >= 0 && dropIndex >= 0) {
                        if (searchForwards) {
                            dropIndex++;
                            if (dropIndexInSelection < selectionCount && indicesSelected[dropIndexInSelection + 1] === dropIndex && dropIndex < count) {
                                dropIndexInSelection++;
                            } else if (dropIndex >= count) {
                                // If we hit the end of the list when looking for a new location ahead of our start index, it means everything from the starting point
                                // to the end is selected, so no valid index can be located to move the items. We need to start searching again, moving backwards
                                // from the starting location, to find the first available insert location to move the selected items.
                                searchForwards = false;
                                dropIndex = startIndex;
                                dropIndexInSelection = startIndexInSelection;
                            } else {
                                dropIndexInSelection = -1;
                            }
                        } else {
                            dropIndex--;
                            if (dropIndexInSelection > 0 && indicesSelected[dropIndexInSelection - 1] === dropIndex) {
                                dropIndexInSelection--;
                            } else {
                                dropIndexInSelection = -1;
                            }
                        }
                    }

                    return dropIndex;
                },

                _reorderItems: function (dropIndex, reorderedItems, reorderingUnselectedItem, useMoveBefore, ensureVisibleAtEnd) {
                    var site = this.site;
                    var updateSelection = function updatedSelectionOnDrop(items) {
                        // Update selection if the items were selected. If there is a range with length > 0 a move operation
                        // on the first or last item removes the range.
                        if (!reorderingUnselectedItem) {
                            site._selection.set({ firstKey: items[0].key, lastKey: items[items.length - 1].key });
                        } else {
                            site._selection.remove({ key: items[0].key });
                        }
                        if (ensureVisibleAtEnd) {
                            site.ensureVisible(site._selection._getFocused());
                        }
                    };
                    reorderedItems.getItems().then(function (items) {
                        var ds = site.itemDataSource;
                        if (dropIndex === -1) {
                            ds.beginEdits();
                            for (var i = items.length - 1; i >= 0; i--) {
                                ds.moveToStart(items[i].key);
                            }
                            ds.endEdits();
                            updateSelection(items);
                        } else {
                            var listBinding = ds.createListBinding();
                            listBinding.fromIndex(dropIndex).then(function (item) {
                                listBinding.release();
                                ds.beginEdits();
                                if (useMoveBefore) {
                                    for (var i = 0, len = items.length; i < len; i++) {
                                        ds.moveBefore(items[i].key, item.key);
                                    }
                                } else {
                                    for (var i = items.length - 1; i >= 0; i--) {
                                        ds.moveAfter(items[i].key, item.key);
                                    }
                                }
                                ds.endEdits();
                                updateSelection(items);
                            });
                        }
                    });
                },

                onDrop: function SelectionMode_onDrop(eventObject) {
                    // If the listview or the handler of the drop event we fire triggers a reorder, the dragged items can end up having different container nodes than what they started with.
                    // Because of that, we need to remove the footprint class from the item boxes' containers before we do any processing of the drop event.
                    if (this._draggedItemBoxes) {
                        for (var i = 0, len = this._draggedItemBoxes.length; i < len; i++) {
                            if (this._draggedItemBoxes[i].parentNode) {
                                _ElementUtilities.removeClass(this._draggedItemBoxes[i].parentNode, _Constants._footprintClass);
                            }
                        }
                    }
                    if (!this._dragBetweenDisabled) {
                        var cursorPosition = this._getPositionInCanvasSpace(eventObject);
                        var dropLocation = this.site._view.hitTest(cursorPosition.x, cursorPosition.y),
                            dropIndex = clampToRange(-1, this.site._cachedCount - 1, dropLocation.insertAfterIndex),
                            allowDrop = true;
                        // We don't fire dragBetween events during autoscroll, so if a user drops during autoscroll, we need to get up to date information
                        // on the drop location, and fire dragBetween before the insert so that the developer can prevent the drop if they choose.
                        if (!this._lastInsertPoint || this._lastInsertPoint.insertAfterIndex !== dropIndex || this._lastInsertPoint.index !== dropLocation.index) {
                            allowDrop = this._fireDragBetweenEvent(dropLocation.index, dropIndex, eventObject.dataTransfer);
                        }
                        if (allowDrop) {
                            this._lastInsertPoint = null;
                            this.site._layout.dragLeave();
                            if (this._fireDropEvent(dropLocation.index, dropIndex, eventObject.dataTransfer) && this._dragging && this.site.itemsReorderable) {
                                if (this._dragInfo.isEverything() || this.site._groupsEnabled()) {
                                    return;
                                }

                                dropIndex = this._findFirstAvailableInsertPoint(this._dragInfo, dropIndex, false);
                                this._reorderItems(dropIndex, this._dragInfo, this._draggingUnselectedItem);
                            }
                        }
                    }
                    this._clearDragProperties();
                    eventObject.preventDefault();
                },

                _checkAutoScroll: function (x, y) {
                    var viewportSize = this.site._getViewportLength(),
                        horizontal = this.site._horizontal(),
                        cursorPositionInViewport = (horizontal ? x : y),
                        canvasSize = this.site._viewport[horizontal ? "scrollWidth" : "scrollHeight"],
                        scrollPosition = Math.floor(this.site.scrollPosition),
                        travelRate = 0;

                    if (cursorPositionInViewport < _Constants._AUTOSCROLL_THRESHOLD) {
                        travelRate = cursorPositionInViewport - _Constants._AUTOSCROLL_THRESHOLD;
                    } else if (cursorPositionInViewport > (viewportSize - _Constants._AUTOSCROLL_THRESHOLD)) {
                        travelRate = (cursorPositionInViewport - (viewportSize - _Constants._AUTOSCROLL_THRESHOLD));
                    }
                    travelRate = Math.round((travelRate / _Constants._AUTOSCROLL_THRESHOLD) * (_Constants._MAX_AUTOSCROLL_RATE - _Constants._MIN_AUTOSCROLL_RATE));

                    // If we're at the edge of the content, we don't need to keep scrolling. We'll set travelRate to 0 to stop the autoscroll timer.
                    if ((scrollPosition === 0 && travelRate < 0) || (scrollPosition >= (canvasSize - viewportSize) && travelRate > 0)) {
                        travelRate = 0;
                    }
                    if (travelRate === 0) {
                        if (this._autoScrollDelay) {
                            _Global.clearTimeout(this._autoScrollDelay);
                            this._autoScrollDelay = 0;
                        }
                    } else {
                        if (!this._autoScrollDelay && !this._autoScrollFrame) {
                            var that = this;
                            this._autoScrollDelay = _Global.setTimeout(function () {
                                if (that._autoScrollRate) {
                                    that._lastDragTimeout = _BaseUtils._now();
                                    var nextFrame = function () {
                                        if ((!that._autoScrollRate && that._autoScrollFrame) || that.site._disposed) {
                                            that._stopAutoScroll();
                                        } else {
                                            // Timeout callbacks aren't reliably timed, so extra math is needed to figure out how far the scroll position should move since the last callback
                                            var currentTime = _BaseUtils._now();
                                            var delta = that._autoScrollRate * ((currentTime - that._lastDragTimeout) / 1000);
                                            delta = (delta < 0 ? Math.min(-1, delta) : Math.max(1, delta));
                                            var newScrollPos = {};
                                            newScrollPos[that.site._scrollProperty] = that.site._viewportScrollPosition + delta;
                                            _ElementUtilities.setScrollPosition(that.site._viewport, newScrollPos);
                                            that._lastDragTimeout = currentTime;
                                            that._autoScrollFrame = _BaseUtils._requestAnimationFrame(nextFrame);
                                        }
                                    };
                                    that._autoScrollFrame = _BaseUtils._requestAnimationFrame(nextFrame);
                                }
                            }, _Constants._AUTOSCROLL_DELAY);
                        }
                    }
                    this._autoScrollRate = travelRate;
                },

                _stopAutoScroll: function () {
                    if (this._autoScrollDelay) {
                        _Global.clearTimeout(this._autoScrollDelay);
                        this._autoScrollDelay = 0;
                    }
                    this._autoScrollRate = 0;
                    this._autoScrollFrame = 0;
                },

                onKeyDown: function SelectionMode_onKeyDown(eventObject) {
                    var that = this,
                        site = this.site,
                        view = site._view,
                        oldEntity = site._selection._getFocused(),
                        handled = true,
                        ctrlKeyDown = eventObject.ctrlKey;

                    function setNewFocus(newEntity, skipSelection, clampToBounds) {
                        function setNewFocusImpl(maxIndex) {
                            var moveView = true,
                                invalidIndex = false;
                            // Since getKeyboardNavigatedItem is purely geometry oriented, it can return us out of bounds numbers, so this check is necessary
                            if (clampToBounds) {
                                newEntity.index = Math.max(0, Math.min(maxIndex, newEntity.index));
                            } else if (newEntity.index < 0 || newEntity.index > maxIndex) {
                                invalidIndex = true;
                            }
                            if (!invalidIndex && (oldEntity.index !== newEntity.index || oldEntity.type !== newEntity.type)) {
                                var changeFocus = dispatchKeyboardNavigating(site._element, oldEntity, newEntity);
                                if (changeFocus) {
                                    moveView = false;

                                    // If the oldEntity is completely off-screen then we mimic the desktop
                                    // behavior. This is consistent with navbar keyboarding.
                                    if (that._setNewFocusItemOffsetPromise) {
                                        that._setNewFocusItemOffsetPromise.cancel();
                                    }
                                    site._batchViewUpdates(_Constants._ViewChange.realize, _Constants._ScrollToPriority.high, function () {
                                        that._setNewFocusItemOffsetPromise = site._getItemOffset(oldEntity, true).then(function (range) {
                                            range = site._convertFromCanvasCoordinates(range);
                                            var oldItemOffscreen = range.end <= site.scrollPosition || range.begin >= site.scrollPosition + site._getViewportLength() - 1;
                                            that._setNewFocusItemOffsetPromise = site._getItemOffset(newEntity).then(function (range) {
                                                that._setNewFocusItemOffsetPromise = null;
                                                var retVal = {
                                                    position: site.scrollPosition,
                                                    direction: "right"
                                                };
                                                if (oldItemOffscreen) {
                                                    // oldEntity is completely off-screen
                                                    site._selection._setFocused(newEntity, true);
                                                    range = site._convertFromCanvasCoordinates(range);
                                                    if (newEntity.index > oldEntity.index) {
                                                        retVal.direction = "right";
                                                        retVal.position = range.end - site._getViewportLength();
                                                    } else {
                                                        retVal.direction = "left";
                                                        retVal.position = range.begin;
                                                    }
                                                }
                                                site._changeFocus(newEntity, skipSelection, ctrlKeyDown, oldItemOffscreen, true);
                                                if (!oldItemOffscreen) {
                                                    return Promise.cancel;
                                                } else {
                                                    return retVal;
                                                }
                                            }, function (error) {
                                                site._changeFocus(newEntity, skipSelection, ctrlKeyDown, true, true);
                                                return Promise.wrapError(error);
                                            });
                                            return that._setNewFocusItemOffsetPromise;
                                        }, function (error) {
                                            site._changeFocus(newEntity, skipSelection, ctrlKeyDown, true, true);
                                            return Promise.wrapError(error);
                                        });
                                        return that._setNewFocusItemOffsetPromise;
                                    }, true);
                                }
                            }
                            // When a key is pressed, we want to make sure the current focus is in view. If the keypress is changing to a new valid index,
                            // _changeFocus will handle moving the viewport for us. If the focus isn't moving, though, we need to put the view back on
                            // the current item ourselves and call setFocused(oldFocus, true) to make sure that the listview knows the focused item was
                            // focused via keyboard and renders the rectangle appropriately.
                            if (moveView) {
                                site._selection._setFocused(oldEntity, true);
                                site.ensureVisible(oldEntity);
                            }
                            if (invalidIndex) {
                                return { type: _UI.ObjectType.item, index: _Constants._INVALID_INDEX };
                            } else {
                                return newEntity;
                            }
                        }

                        // We need to get the final item in the view so that we don't try setting focus out of bounds.
                        if (newEntity.type === _UI.ObjectType.item) {
                            return Promise.wrap(view.lastItemIndex()).then(setNewFocusImpl);
                        } else if (newEntity.type === _UI.ObjectType.groupHeader) {
                            return Promise.wrap(site._groups.length() - 1).then(setNewFocusImpl);
                        } else {
                            return Promise.wrap(0).then(setNewFocusImpl);
                        }
                    }

                    var Key = _ElementUtilities.Key,
                        keyCode = eventObject.keyCode,
                        rtl = site._rtl();

                    if (!this._isInteractive(eventObject.target)) {
                        if (eventObject.ctrlKey && !eventObject.altKey && !eventObject.shiftKey && this._keyboardAcceleratorHandlers[keyCode]) {
                            this._keyboardAcceleratorHandlers[keyCode]();
                        }
                        if (site.itemsReorderable && (!eventObject.ctrlKey && eventObject.altKey && eventObject.shiftKey && oldEntity.type === _UI.ObjectType.item) &&
                            (keyCode === Key.leftArrow || keyCode === Key.rightArrow || keyCode === Key.upArrow || keyCode === Key.downArrow)) {
                            var selection = site._selection,
                                focusedIndex = oldEntity.index,
                                movingUnselectedItem = false,
                                processReorder = true;
                            if (!selection.isEverything()) {
                                if (!selection._isIncluded(focusedIndex)) {
                                    var item = site._view.items.itemAt(focusedIndex);
                                    // Selected items should never be marked as non draggable, so we only need to check for nonDraggableClass when trying to reorder an unselected item.
                                    if (item && _ElementUtilities.hasClass(item, _Constants._nonDraggableClass)) {
                                        processReorder = false;
                                    } else {
                                        movingUnselectedItem = true;
                                        selection = new _SelectionManager._Selection(this.site, [{ firstIndex: focusedIndex, lastIndex: focusedIndex }]);
                                    }
                                }
                                if (processReorder) {
                                    var dropIndex = focusedIndex;
                                    if (keyCode === Key.rightArrow) {
                                        dropIndex += (rtl ? -1 : 1);
                                    } else if (keyCode === Key.leftArrow) {
                                        dropIndex += (rtl ? 1 : -1);
                                    } else if (keyCode === Key.upArrow) {
                                        dropIndex--;
                                    } else {
                                        dropIndex++;
                                    }
                                    // If the dropIndex is larger than the original index, we're trying to move items forward, so the search for the first unselected item to insert after should move forward.
                                    var movingAhead = (dropIndex > focusedIndex),
                                        searchForward = movingAhead;
                                    if (movingAhead && dropIndex >= this.site._cachedCount) {
                                        // If we're at the end of the list and trying to move items forward, dropIndex should be >= cachedCount.
                                        // That doesn't mean we don't have to do any reordering, though. A selection could be broken down into
                                        // a few blocks. We need to make the selection contiguous after this reorder, so we've got to search backwards
                                        // to find the first unselected item, then move everything in the selection after it.
                                        searchForward = false;
                                        dropIndex = this.site._cachedCount - 1;
                                    }
                                    dropIndex = this._findFirstAvailableInsertPoint(selection, dropIndex, searchForward);
                                    dropIndex = Math.min(Math.max(-1, dropIndex), this.site._cachedCount - 1);
                                    var reportedInsertAfterIndex = dropIndex - (movingAhead || dropIndex === -1 ? 0 : 1),
                                        reportedIndex = dropIndex,
                                        groupsEnabled = this.site._groupsEnabled();

                                    if (groupsEnabled) {
                                        // The indices we picked for the index/insertAfterIndex to report in our events is always correct in an ungrouped list,
                                        // and mostly correct in a grouped list. The only problem occurs when you move an item (or items) ahead into a new group,
                                        // or back into a previous group, such that the items should be the first/last in the group. Take this list as an example:
                                        // [Group A] [a] [b] [c] [Group B] [d] [e]
                                        // When [d] is focused, right/down arrow reports index: 4, insertAfterIndex: 4, which is right -- it means move [d] after [e].
                                        // Similarily, when [c] is focused and left/up is pressed, we report index: 1, insertAfterIndex: 0 -- move [c] to after [a].
                                        // Take note that index does not tell us where focus is / what item is being moved.
                                        // Like mouse/touch DnD, index tells us what the dragBetween slots would be were we to animate a dragBetween.
                                        // The problem cases are moving backwards into a previous group, or forward into the next group.
                                        // If [c] were focused and the user pressed right/down, we would report index: 3, insertAfterIndex: 3. In other words, move [c] after [d].
                                        // That's not right at all - [c] needs to become the first element of [Group B]. When we're moving ahead, then, and our dropIndex
                                        // is the first index of a new group, we adjust insertAfterIndex to be dropIndex - 1. Now we'll report index:3, insertAfterIndex: 2, which means
                                        // [c] is now the first element of [Group B], rather than the last element of [Group A]. This is exactly the same as what we would report when
                                        // the user mouse/touch drags [c] right before [d].
                                        // Similarily, when [d] is focused and we press left/up, without the logic below we would report index: 2, insertAfterIndex: 1, so we'd try to move
                                        // [d] ahead of [b]. Again, [d] first needs the opportunity to become the last element in [Group A], so we adjust the insertAfterIndex up by 1.
                                        // We then will report index:2, insertAfterIndex:2, meaning insert [d] in [Group A] after [c], which again mimics the mouse/touch API.
                                        var groups = this.site._groups,
                                            groupIndex = (dropIndex > -1 ? groups.groupFromItem(dropIndex) : 0);
                                        if (movingAhead) {
                                            if (groups.group(groupIndex).startIndex === dropIndex) {
                                                reportedInsertAfterIndex--;
                                            }
                                        } else if (groupIndex < (groups.length() - 1) && dropIndex === (groups.group(groupIndex + 1).startIndex - 1)) {
                                            reportedInsertAfterIndex++;
                                        }
                                    }

                                    if (this._fireDragBetweenEvent(reportedIndex, reportedInsertAfterIndex, null) && this._fireDropEvent(reportedIndex, reportedInsertAfterIndex, null)) {
                                        if (groupsEnabled) {
                                            return;
                                        }

                                        this._reorderItems(dropIndex, selection, movingUnselectedItem, !movingAhead, true);
                                    }
                                }
                            }
                        } else if (!eventObject.altKey) {
                            if (this._keyboardNavigationHandlers[keyCode]) {
                                this._keyboardNavigationHandlers[keyCode](oldEntity).then(function (newEntity) {
                                    if (newEntity.index !== oldEntity.index || newEntity.type !== oldEntity.type) {
                                        var clampToBounds = that._keyboardNavigationHandlers[keyCode].clampToBounds;
                                        if (newEntity.type !== _UI.ObjectType.groupHeader && eventObject.shiftKey && site._selectionAllowed() && site._multiSelection()) {
                                            // Shift selection should work when shift or shift+ctrl are depressed
                                            if (site._selection._pivot === _Constants._INVALID_INDEX) {
                                                site._selection._pivot = oldEntity.index;
                                            }
                                            setNewFocus(newEntity, true, clampToBounds).then(function (newEntity) {
                                                if (newEntity.index !== _Constants._INVALID_INDEX) {
                                                    var firstIndex = Math.min(newEntity.index, site._selection._pivot),
                                                        lastIndex = Math.max(newEntity.index, site._selection._pivot),
                                                        additive = (eventObject.ctrlKey || site._tap === _UI.TapBehavior.toggleSelect);
                                                    that._selectRange(firstIndex, lastIndex, additive);
                                                }
                                            });
                                        } else {
                                            site._selection._pivot = _Constants._INVALID_INDEX;
                                            setNewFocus(newEntity, false, clampToBounds);
                                        }
                                    } else {
                                        handled = false;
                                    }

                                });
                            } else if (!eventObject.ctrlKey && keyCode === Key.enter) {
                                var element = oldEntity.type === _UI.ObjectType.groupHeader ? site._groups.group(oldEntity.index).header : site._view.items.itemBoxAt(oldEntity.index);
                                if (element) {
                                    if (oldEntity.type === _UI.ObjectType.groupHeader) {
                                        this._pressedHeader = element;
                                        this._pressedItemBox = null;
                                        this._pressedContainer = null;
                                    } else {
                                        this._pressedItemBox = element;
                                        this._pressedContainer = site._view.items.containerAt(oldEntity.index);
                                        this._pressedHeader = null;
                                    }

                                    var allowed = this._verifySelectionAllowed(oldEntity);
                                    if (allowed.canTapSelect) {
                                        this._itemEventsHandler.handleTap(oldEntity);
                                    }
                                    this._fireInvokeEvent(oldEntity, element);
                                }
                            } else if (oldEntity.type !== _UI.ObjectType.groupHeader && ((eventObject.ctrlKey && keyCode === Key.enter) || keyCode === Key.space)) {
                                this._itemEventsHandler.toggleSelectionIfAllowed(oldEntity.index);
                                site._changeFocus(oldEntity, true, ctrlKeyDown, false, true);
                            } else if (keyCode === Key.escape && site._selection.count() > 0) {
                                site._selection._pivot = _Constants._INVALID_INDEX;
                                site._selection.clear();
                            } else {
                                handled = false;
                            }
                        } else {
                            handled = false;
                        }

                        this._keyDownHandled = handled;
                        if (handled) {
                            eventObject.stopPropagation();
                            eventObject.preventDefault();
                        }
                    }

                    if (keyCode === Key.tab) {
                        this.site._keyboardFocusInbound = true;
                    }
                },

                onKeyUp: function (eventObject) {
                    if (this._keyDownHandled) {
                        eventObject.stopPropagation();
                        eventObject.preventDefault();
                    }
                },

                onTabEntered: function (eventObject) {
                    if (this.site._groups.length() === 0 && !this.site._hasHeaderOrFooter) {
                        return;
                    }

                    var site = this.site,
                        focused = site._selection._getFocused(),
                        forward = eventObject.detail;

                    // We establish whether focus is incoming on the ListView by checking keyboard focus and the srcElement.
                    // If the ListView did not have keyboard focus, then it is definitely incoming since keyboard focus is cleared
                    // on blur which works for 99% of all scenarios. When the ListView is the only tabbable element on the page,
                    // then tabbing out of the ListView will make focus wrap around and focus the ListView again. The blur event is
                    // handled after TabEnter, so the keyboard focus flag is not yet cleared. Therefore, we examine the srcElement and see
                    // if it is the _viewport since it is the first tabbable element in the ListView DOM tree.
                    var inboundFocus = !site._hasKeyboardFocus || eventObject.target === site._viewport;
                    if (inboundFocus) {
                        this.inboundFocusHandled = true;

                        // We tabbed into the ListView
                        focused.index = (focused.index === _Constants._INVALID_INDEX ? 0 : focused.index);
                        if (forward || !(this.site._supportsGroupHeaderKeyboarding || this.site._hasHeaderOrFooter)) {
                            // We tabbed into the ListView from before the ListView, so focus should go to items
                            var entity = { type: _UI.ObjectType.item };
                            if (focused.type === _UI.ObjectType.groupHeader) {
                                entity.index = site._groupFocusCache.getIndexForGroup(focused.index);
                                if (dispatchKeyboardNavigating(site._element, focused, entity)) {
                                    site._changeFocus(entity, true, false, false, true);
                                } else {
                                    site._changeFocus(focused, true, false, false, true);
                                }
                            } else {
                                entity.index = (focused.type !== _UI.ObjectType.item ? site._groupFocusCache.getLastFocusedItemIndex() : focused.index);
                                site._changeFocus(entity, true, false, false, true);
                            }
                            eventObject.preventDefault();
                        } else {
                            // We tabbed into the ListView from after the ListView, focus should go to headers
                            var entity = { type: _UI.ObjectType.groupHeader };
                            if (this.site._hasHeaderOrFooter) {
                                if (this.site._lastFocusedElementInGroupTrack.type === _UI.ObjectType.groupHeader && this.site._supportsGroupHeaderKeyboarding) {
                                    entity.index = site._groups.groupFromItem(focused.index);
                                    if (dispatchKeyboardNavigating(site._element, focused, entity)) {
                                        site._changeFocus(entity, true, false, false, true);
                                    } else {
                                        site._changeFocus(focused, true, false, false, true);
                                    }
                                } else {
                                    entity.type = this.site._lastFocusedElementInGroupTrack.type;
                                    entity.index = 0;
                                    site._changeFocus(entity, true, false, false, true);
                                }
                            } else if (focused.type !== _UI.ObjectType.groupHeader && this.site._supportsGroupHeaderKeyboarding) {
                                entity.index = site._groups.groupFromItem(focused.index);
                                if (dispatchKeyboardNavigating(site._element, focused, entity)) {
                                    site._changeFocus(entity, true, false, false, true);
                                } else {
                                    site._changeFocus(focused, true, false, false, true);
                                }
                            } else {
                                entity.index = focused.index;
                                site._changeFocus(entity, true, false, false, true);
                            }
                            eventObject.preventDefault();
                        }
                    }
                },

                onTabExiting: function (eventObject) {
                    if (!this.site._hasHeaderOrFooter && (!this.site._supportsGroupHeaderKeyboarding || this.site._groups.length() === 0)) {
                        return;
                    }

                    var site = this.site,
                        focused = site._selection._getFocused(),
                        forward = eventObject.detail;

                    if (forward) {
                        var entity = null;
                        if (focused.type === _UI.ObjectType.item) {
                            // Tabbing and we were focusing an item, go to headers.
                            // If we last saw focus in the header track on the layout header/footer, we'll move focus back to there first. Otherwise, we'll let the group header take it.
                            var lastType = this.site._lastFocusedElementInGroupTrack.type;
                            if (lastType === _UI.ObjectType.header || lastType === _UI.ObjectType.footer || !this.site._supportsGroupHeaderKeyboarding) {
                                var entity = { type: (lastType === _UI.ObjectType.item ? _UI.ObjectType.header : lastType), index: 0 };
                            } else {
                                var entity = { type: _UI.ObjectType.groupHeader, index: site._groups.groupFromItem(focused.index) };
                            }

                        }

                        if (entity && dispatchKeyboardNavigating(site._element, focused, entity)) {
                            site._changeFocus(entity, true, false, false, true);
                            eventObject.preventDefault();
                        }
                    } else if (!forward && focused.type !== _UI.ObjectType.item) {
                        // Shift tabbing and we were focusing a header, go to items
                        var targetIndex = 0;
                        if (focused.type === _UI.ObjectType.groupHeader) {
                            targetIndex = site._groupFocusCache.getIndexForGroup(focused.index);
                        } else {
                            targetIndex = (focused.type === _UI.ObjectType.header ? 0 : site._view.lastItemIndex());
                        }
                        var entity = { type: _UI.ObjectType.item, index: targetIndex };
                        if (dispatchKeyboardNavigating(site._element, focused, entity)) {
                            site._changeFocus(entity, true, false, false, true);
                            eventObject.preventDefault();
                        }
                    }
                }
            });
            return _SelectionMode;
        })
    });

});

// Copyright (c) Microsoft Corporation.  All Rights Reserved. Licensed under the MIT License. See License.txt in the project root for license information.
define('WinJS/Controls/ListView/_ErrorMessages',[
        'exports',
        '../../Core/_Base',
        '../../Core/_Resources'
    ], function errorMessagesInit(exports, _Base, _Resources) {
    "use strict";

    _Base.Namespace._moduleDefine(exports, null, {

        modeIsInvalid: {
            get: function () { return "Invalid argument: mode must be one of following values: 'none', 'single' or 'multi'."; }
        },

        loadingBehaviorIsDeprecated: {
            get: function () { return "Invalid configuration: loadingBehavior is deprecated. The control will default this property to 'randomAccess'. Please refer to the 'ListView loading behaviors' SDK Sample for guidance on how to implement incremental load behavior."; }
        },

        pagesToLoadIsDeprecated: {
            get: function () { return "Invalid configuration: pagesToLoad is deprecated. The control will not use this property. Please refer to the 'ListView loading behaviors' SDK Sample for guidance on how to implement incremental load behavior."; }
        },

        pagesToLoadThresholdIsDeprecated: {
            get: function () { return "Invalid configuration: pagesToLoadThreshold is deprecated.  The control will not use this property. Please refer to the 'ListView loading behaviors' SDK Sample for guidance on how to implement incremental load behavior."; }
        },

        automaticallyLoadPagesIsDeprecated: {
            get: function () { return "Invalid configuration: automaticallyLoadPages is deprecated. The control will default this property to false. Please refer to the 'ListView loading behaviors' SDK Sample for guidance on how to implement incremental load behavior."; }
        },

        invalidTemplate: {
            get: function () { return "Invalid template: Templates must be created before being passed to the ListView, and must contain a valid tree of elements."; }
        },

        loadMorePagesIsDeprecated: {
            get: function () { return "loadMorePages is deprecated. Invoking this function will not have any effect. Please refer to the 'ListView loading behaviors' SDK Sample for guidance on how to implement incremental load behavior."; }
        },

        disableBackdropIsDeprecated: {
            get: function () { return "Invalid configuration: disableBackdrop is deprecated. Style: .win-listview .win-container.win-backdrop { background-color:transparent; } instead."; }
        },

        backdropColorIsDeprecated: {
            get: function () { return "Invalid configuration: backdropColor is deprecated. Style: .win-listview .win-container.win-backdrop { rgba(155,155,155,0.23); } instead."; }
        },

        itemInfoIsDeprecated: {
            get: function () { return "GridLayout.itemInfo may be altered or unavailable in future versions. Instead, use CellSpanningLayout."; }
        },

        groupInfoIsDeprecated: {
            get: function () { return "GridLayout.groupInfo may be altered or unavailable in future versions. Instead, use CellSpanningLayout."; }
        },

        resetItemIsDeprecated: {
            get: function () { return "resetItem may be altered or unavailable in future versions. Instead, mark the element as disposable using WinJS.Utilities.markDisposable."; }
        },

        resetGroupHeaderIsDeprecated: {
            get: function () { return "resetGroupHeader may be altered or unavailable in future versions. Instead, mark the header element as disposable using WinJS.Utilities.markDisposable."; }
        },

        maxRowsIsDeprecated: {
            get: function () { return "GridLayout.maxRows may be altered or unavailable in future versions. Instead, use the maximumRowsOrColumns property."; }
        },
        swipeOrientationDeprecated: {
            get: function () { return "Invalid configuration: swipeOrientation is deprecated. The control will default this property to 'none'"; }
        },
        swipeBehaviorDeprecated: {
            get: function () { return "Invalid configuration: swipeBehavior is deprecated. The control will default this property to 'none'"; }
        }
    });

});

// Copyright (c) Microsoft Corporation.  All Rights Reserved. Licensed under the MIT License. See License.txt in the project root for license information.
define('WinJS/Controls/ListView/_GroupFocusCache',[
    'exports',
    '../../Core/_Base'
    ], function GroupFocusCacheInit(exports, _Base) {
    "use strict";

    _Base.Namespace._moduleDefine(exports, "WinJS.UI", {
        _GroupFocusCache: _Base.Namespace._lazy(function () {
            return _Base.Class.define(function GroupFocusCache_ctor(listView) {
                this._listView = listView;
                this.clear();
            }, {
                // We store indices as strings in the cache so index=0 does not evaluate to false as
                // when we check for the existance of an index in the cache. The index is converted
                // back into a number when calling getIndexForGroup

                updateCache: function (groupKey, itemKey, itemIndex) {
                    this._lastFocusedItemKey = itemKey;
                    this._lastFocusedItemIndex = itemIndex;
                    itemIndex = "" + itemIndex;
                    this._itemToIndex[itemKey] = itemIndex;
                    this._groupToItem[groupKey] = itemKey;
                },

                deleteItem: function (itemKey) {
                    if (itemKey === this._lastFocusedItemKey) {
                        this._lastFocusedItemKey = null;
                        this._lastFocusedItemIndex = 0;
                    }

                    if (!this._itemToIndex[itemKey]) {
                        return;
                    }

                    var that = this;
                    var keys = Object.keys(this._groupToItem);
                    for (var i = 0, len = keys.length; i < len; i++) {
                        var key = keys[i];
                        if (that._groupToItem[key] === itemKey) {
                            that.deleteGroup(key);
                            break;
                        }
                    }
                },

                deleteGroup: function (groupKey) {
                    var itemKey = this._groupToItem[groupKey];
                    if (itemKey) {
                        delete this._itemToIndex[itemKey];
                    }
                    delete this._groupToItem[groupKey];
                },

                updateItemIndex: function (itemKey, itemIndex) {
                    if (itemKey === this._lastFocusedItemKey) {
                        this._lastFocusedItemIndex = itemIndex;
                    }

                    if (!this._itemToIndex[itemKey]) {
                        return;
                    }
                    this._itemToIndex[itemKey] = "" + itemIndex;
                },

                getIndexForGroup: function (groupIndex) {
                    var groupKey = this._listView._groups.group(groupIndex).key;

                    var itemKey = this._groupToItem[groupKey];
                    if (itemKey && this._itemToIndex[itemKey]) {
                        return +this._itemToIndex[itemKey];
                    } else {
                        return this._listView._groups.fromKey(groupKey).group.startIndex;
                    }
                },

                clear: function () {
                    this._groupToItem = {};
                    this._itemToIndex = {};
                    this._lastFocusedItemIndex = 0;
                    this._lastFocusedItemKey = null;
                },

                getLastFocusedItemIndex: function () {
                    return this._lastFocusedItemIndex;
                }
            });
        }),

        _UnsupportedGroupFocusCache: _Base.Namespace._lazy(function () {
            return _Base.Class.define(null, {
                updateCache: function (groupKey, itemKey, itemIndex) {
                    this._lastFocusedItemKey = itemKey;
                    this._lastFocusedItemIndex = itemIndex;
                },

                deleteItem: function (itemKey) {
                    if (itemKey === this._lastFocusedItemKey) {
                        this._lastFocusedItemKey = null;
                        this._lastFocusedItemIndex = 0;
                    }
                },

                deleteGroup: function () {
                },

                updateItemIndex: function (itemKey, itemIndex) {
                    if (itemKey === this._lastFocusedItemKey) {
                        this._lastFocusedItemIndex = itemIndex;
                    }
                },

                getIndexForGroup: function () {
                    return 0;
                },

                clear: function () {
                    this._lastFocusedItemIndex = 0;
                    this._lastFocusedItemKey = null;
                },

                getLastFocusedItemIndex: function () {
                    return this._lastFocusedItemIndex;
                }
            });
        })
    });

});
// Copyright (c) Microsoft Corporation.  All Rights Reserved. Licensed under the MIT License. See License.txt in the project root for license information.
define('WinJS/Controls/ListView/_GroupsContainer',[
    'exports',
    '../../Core/_Base',
    '../../Promise',
    '../../Utilities/_Dispose',
    '../../Utilities/_ElementUtilities',
    '../../Utilities/_ItemsManager',
    '../../Utilities/_UI',
    '../ItemContainer/_Constants'
    ], function groupsContainerInit(exports, _Base, Promise, _Dispose, _ElementUtilities, _ItemsManager, _UI, _Constants) {
    "use strict";

    _Base.Namespace._moduleDefine(exports, "WinJS.UI", {
        _GroupsContainerBase: _Base.Namespace._lazy(function () {
            return _Base.Class.define(function () {
            }, {
                index: function (element) {
                    var header = this.headerFrom(element);
                    if (header) {
                        for (var i = 0, len = this.groups.length; i < len; i++) {
                            if (header === this.groups[i].header) {
                                return i;
                            }
                        }
                    }
                    return _Constants._INVALID_INDEX;
                },

                headerFrom: function (element) {
                    while (element && !_ElementUtilities.hasClass(element, _Constants._headerClass)) {
                        element = element.parentNode;
                    }
                    return element;
                },

                requestHeader: function GroupsContainerBase_requestHeader(index) {
                    this._waitingHeaderRequests = this._waitingHeaderRequests || {};
                    if (!this._waitingHeaderRequests[index]) {
                        this._waitingHeaderRequests[index] = [];
                    }

                    var that = this;
                    return new Promise(function (complete) {
                        var group = that.groups[index];
                        if (group && group.header) {
                            complete(group.header);
                        } else {
                            that._waitingHeaderRequests[index].push(complete);
                        }
                    });
                },

                notify: function GroupsContainerBase_notify(index, header) {
                    if (this._waitingHeaderRequests && this._waitingHeaderRequests[index]) {
                        var requests = this._waitingHeaderRequests[index];
                        for (var i = 0, len = requests.length; i < len; i++) {
                            requests[i](header);
                        }

                        this._waitingHeaderRequests[index] = [];
                    }
                },

                groupFromImpl: function GroupsContainerBase_groupFromImpl(fromGroup, toGroup, comp) {
                    if (toGroup < fromGroup) {
                        return null;
                    }

                    var center = fromGroup + Math.floor((toGroup - fromGroup) / 2),
                        centerGroup = this.groups[center];

                    if (comp(centerGroup, center)) {
                        return this.groupFromImpl(fromGroup, center - 1, comp);
                    } else if (center < toGroup && !comp(this.groups[center + 1], center + 1)) {
                        return this.groupFromImpl(center + 1, toGroup, comp);
                    } else {
                        return center;
                    }
                },

                groupFrom: function GroupsContainerBase_groupFrom(comp) {
                    if (this.groups.length > 0) {
                        var lastGroupIndex = this.groups.length - 1,
                            lastGroup = this.groups[lastGroupIndex];

                        if (!comp(lastGroup, lastGroupIndex)) {
                            return lastGroupIndex;
                        } else {
                            return this.groupFromImpl(0, this.groups.length - 1, comp);
                        }
                    } else {
                        return null;
                    }
                },

                groupFromItem: function GroupsContainerBase_groupFromItem(itemIndex) {
                    return this.groupFrom(function (group) {
                        return itemIndex < group.startIndex;
                    });
                },

                groupFromOffset: function GroupsContainerBase_groupFromOffset(offset) {
                    return this.groupFrom(function (group) {
                        return offset < group.offset;
                    });
                },

                group: function GroupsContainerBase_getGroup(index) {
                    return this.groups[index];
                },

                length: function GroupsContainerBase_length() {
                    return this.groups.length;
                },

                cleanUp: function GroupsContainerBase_cleanUp() {
                    if (this.listBinding) {
                        for (var i = 0, len = this.groups.length; i < len; i++) {
                            var group = this.groups[i];
                            if (group.userData) {
                                this.listBinding.releaseItem(group.userData);
                            }
                        }
                        this.listBinding.release();
                    }
                },

                _dispose: function GroupsContainerBase_dispose() {
                    this.cleanUp();
                },

                synchronizeGroups: function GroupsContainerBase_synchronizeGroups() {
                    var that = this;

                    this.pendingChanges = [];
                    this.ignoreChanges = true;
                    return this.groupDataSource.invalidateAll().then(function () {
                        return Promise.join(that.pendingChanges);
                    }).then(function () {
                        if (that._listView._ifZombieDispose()) {
                            return Promise.cancel;
                        }
                    }).then(
                        function () {
                            that.ignoreChanges = false;
                        },
                        function (error) {
                            that.ignoreChanges = false;
                            return Promise.wrapError(error);
                        }
                    );
                },

                fromKey: function GroupsContainerBase_fromKey(key) {
                    for (var i = 0, len = this.groups.length; i < len; i++) {
                        var group = this.groups[i];
                        if (group.key === key) {
                            return {
                                group: group,
                                index: i
                            };
                        }
                    }
                    return null;
                },

                fromHandle: function GroupsContainerBase_fromHandle(handle) {
                    for (var i = 0, len = this.groups.length; i < len; i++) {
                        var group = this.groups[i];
                        if (group.handle === handle) {
                            return {
                                group: group,
                                index: i
                            };
                        }
                    }
                    return null;
                }
            });
        }),

        _UnvirtualizedGroupsContainer: _Base.Namespace._lazy(function () {
            return _Base.Class.derive(exports._GroupsContainerBase, function (listView, groupDataSource) {
                this._listView = listView;
                this.groupDataSource = groupDataSource;
                this.groups = [];
                this.pendingChanges = [];
                this.dirty = true;

                var that = this,
                notificationHandler = {
                    beginNotifications: function GroupsContainer_beginNotifications() {
                        that._listView._versionManager.beginNotifications();
                    },

                    endNotifications: function GroupsContainer_endNotifications() {
                        that._listView._versionManager.endNotifications();

                        if (that._listView._ifZombieDispose()) { return; }

                        if (!that.ignoreChanges && that._listView._groupsChanged) {
                            that._listView._scheduleUpdate();
                        }
                    },

                    indexChanged: function GroupsContainer_indexChanged() {
                        that._listView._versionManager.receivedNotification();

                        if (that._listView._ifZombieDispose()) { return; }

                        this.scheduleUpdate();
                    },

                    itemAvailable: function GroupsContainer_itemAvailable() {
                    },

                    countChanged: function GroupsContainer_countChanged(newCount) {
                        that._listView._versionManager.receivedNotification();

                        that._listView._writeProfilerMark("groupCountChanged(" + newCount + "),info");

                        if (that._listView._ifZombieDispose()) { return; }

                        this.scheduleUpdate();
                    },

                    changed: function GroupsContainer_changed(newItem) {
                        that._listView._versionManager.receivedNotification();

                        if (that._listView._ifZombieDispose()) { return; }

                        var groupEntry = that.fromKey(newItem.key);
                        if (groupEntry) {
                            that._listView._writeProfilerMark("groupChanged(" + groupEntry.index + "),info");

                            groupEntry.group.userData = newItem;
                            groupEntry.group.startIndex = newItem.firstItemIndexHint;
                            this.markToRemove(groupEntry.group);
                        }

                        this.scheduleUpdate();
                    },

                    removed: function GroupsContainer_removed(itemHandle) {
                        that._listView._versionManager.receivedNotification();
                        that._listView._groupRemoved(itemHandle);

                        if (that._listView._ifZombieDispose()) { return; }

                        var groupEntry = that.fromHandle(itemHandle);
                        if (groupEntry) {
                            that._listView._writeProfilerMark("groupRemoved(" + groupEntry.index + "),info");

                            that.groups.splice(groupEntry.index, 1);
                            var index = that.groups.indexOf(groupEntry.group, groupEntry.index);

                            if (index > -1) {
                                that.groups.splice(index, 1);
                            }

                            this.markToRemove(groupEntry.group);
                        }

                        this.scheduleUpdate();
                    },

                    inserted: function GroupsContainer_inserted(itemPromise, previousHandle, nextHandle) {
                        that._listView._versionManager.receivedNotification();

                        if (that._listView._ifZombieDispose()) { return; }

                        that._listView._writeProfilerMark("groupInserted,info");

                        var notificationHandler = this;
                        itemPromise.retain().then(function (item) {

                            var index;
                            if (!previousHandle && !nextHandle && !that.groups.length) {
                                index = 0;
                            } else {
                                index = notificationHandler.findIndex(previousHandle, nextHandle);
                            }
                            if (index !== -1) {
                                var newGroup = {
                                    key: item.key,
                                    startIndex: item.firstItemIndexHint,
                                    userData: item,
                                    handle: itemPromise.handle
                                };

                                that.groups.splice(index, 0, newGroup);
                            }
                            notificationHandler.scheduleUpdate();
                        });
                        that.pendingChanges.push(itemPromise);
                    },

                    moved: function GroupsContainer_moved(itemPromise, previousHandle, nextHandle) {
                        that._listView._versionManager.receivedNotification();

                        if (that._listView._ifZombieDispose()) { return; }

                        that._listView._writeProfilerMark("groupMoved,info");

                        var notificationHandler = this;
                        itemPromise.then(function (item) {
                            var newIndex = notificationHandler.findIndex(previousHandle, nextHandle),
                                groupEntry = that.fromKey(item.key);

                            if (groupEntry) {
                                that.groups.splice(groupEntry.index, 1);

                                if (newIndex !== -1) {
                                    if (groupEntry.index < newIndex) {
                                        newIndex--;
                                    }

                                    groupEntry.group.key = item.key;
                                    groupEntry.group.userData = item;
                                    groupEntry.group.startIndex = item.firstItemIndexHint;
                                    that.groups.splice(newIndex, 0, groupEntry.group);
                                }

                            } else if (newIndex !== -1) {
                                var newGroup = {
                                    key: item.key,
                                    startIndex: item.firstItemIndexHint,
                                    userData: item,
                                    handle: itemPromise.handle
                                };
                                that.groups.splice(newIndex, 0, newGroup);
                                itemPromise.retain();
                            }

                            notificationHandler.scheduleUpdate();
                        });
                        that.pendingChanges.push(itemPromise);
                    },

                    reload: function GroupsContainer_reload() {
                        that._listView._versionManager.receivedNotification();

                        if (that._listView._ifZombieDispose()) {
                            return;
                        }

                        that._listView._processReload();
                    },

                    markToRemove: function GroupsContainer_markToRemove(group) {
                        if (group.header) {
                            var header = group.header;
                            group.header = null;
                            group.left = -1;
                            group.width = -1;
                            group.decorator = null;
                            group.tabIndex = -1;
                            header.tabIndex = -1;

                            that._listView._groupsToRemove[_ElementUtilities._uniqueID(header)] = { group: group, header: header };
                        }
                    },

                    scheduleUpdate: function GroupsContainer_scheduleUpdate() {
                        that.dirty = true;
                        if (!that.ignoreChanges) {
                            that._listView._groupsChanged = true;
                        }
                    },

                    findIndex: function GroupsContainer_findIndex(previousHandle, nextHandle) {
                        var index = -1,
                            groupEntry;

                        if (previousHandle) {
                            groupEntry = that.fromHandle(previousHandle);
                            if (groupEntry) {
                                index = groupEntry.index + 1;
                            }
                        }

                        if (index === -1 && nextHandle) {
                            groupEntry = that.fromHandle(nextHandle);
                            if (groupEntry) {
                                index = groupEntry.index;
                            }
                        }

                        return index;
                    },

                    removeElements: function GroupsContainer_removeElements(group) {
                        if (group.header) {
                            var parentNode = group.header.parentNode;
                            if (parentNode) {
                                _Dispose.disposeSubTree(group.header);
                                parentNode.removeChild(group.header);
                            }
                            group.header = null;
                            group.left = -1;
                            group.width = -1;
                        }
                    }
                };

                this.listBinding = this.groupDataSource.createListBinding(notificationHandler);
            }, {
                initialize: function UnvirtualizedGroupsContainer_initialize() {
                    if (this.initializePromise) {
                        this.initializePromise.cancel();
                    }

                    this._listView._writeProfilerMark("GroupsContainer_initialize,StartTM");

                    var that = this;
                    this.initializePromise = this.groupDataSource.getCount().then(function (count) {
                        var promises = [];
                        for (var i = 0; i < count; i++) {
                            promises.push(that.listBinding.fromIndex(i).retain());
                        }
                        return Promise.join(promises);
                    }).then(
                        function (groups) {
                            that.groups = [];

                            for (var i = 0, len = groups.length; i < len; i++) {
                                var group = groups[i];

                                that.groups.push({
                                    key: group.key,
                                    startIndex: group.firstItemIndexHint,
                                    handle: group.handle,
                                    userData: group,
                                });
                            }
                            that._listView._writeProfilerMark("GroupsContainer_initialize groups(" + groups.length + "),info");
                            that._listView._writeProfilerMark("GroupsContainer_initialize,StopTM");
                        },
                        function (error) {
                            that._listView._writeProfilerMark("GroupsContainer_initialize,StopTM");
                            return Promise.wrapError(error);
                        });
                    return this.initializePromise;
                },

                renderGroup: function UnvirtualizedGroupsContainer_renderGroup(index) {
                    if (this._listView.groupHeaderTemplate) {
                        var group = this.groups[index];
                        return Promise.wrap(this._listView._groupHeaderRenderer(Promise.wrap(group.userData))).then(_ItemsManager._normalizeRendererReturn);
                    } else {
                        return Promise.wrap(null);
                    }
                },

                setDomElement: function UnvirtualizedGroupsContainer_setDomElement(index, headerElement) {
                    this.groups[index].header = headerElement;
                    this.notify(index, headerElement);
                },

                removeElements: function UnvirtualizedGroupsContainer_removeElements() {
                    var elements = this._listView._groupsToRemove || {},
                        keys = Object.keys(elements),
                        focusedItemPurged = false;

                    var focused = this._listView._selection._getFocused();
                    for (var i = 0, len = keys.length; i < len; i++) {
                        var group = elements[keys[i]],
                            header = group.header,
                            groupData = group.group;

                        if (!focusedItemPurged && focused.type === _UI.ObjectType.groupHeader && groupData.userData.index === focused.index) {
                            this._listView._unsetFocusOnItem();
                            focusedItemPurged = true;
                        }

                        if (header) {
                            var parentNode = header.parentNode;
                            if (parentNode) {
                                _Dispose._disposeElement(header);
                                parentNode.removeChild(header);
                            }
                        }
                    }

                    if (focusedItemPurged) {
                        this._listView._setFocusOnItem(focused);
                    }

                    this._listView._groupsToRemove = {};
                },

                resetGroups: function UnvirtualizedGroupsContainer_resetGroups() {
                    var groups = this.groups.slice(0);

                    for (var i = 0, len = groups.length; i < len; i++) {
                        var group = groups[i];

                        if (this.listBinding && group.userData) {
                            this.listBinding.releaseItem(group.userData);
                        }
                    }

                    // Set the lengths to zero to clear the arrays, rather than setting = [], which re-instantiates
                    this.groups.length = 0;
                    this.dirty = true;
                }
            });
        }),

        _NoGroups: _Base.Namespace._lazy(function () {
            return _Base.Class.derive(exports._GroupsContainerBase, function (listView) {
                this._listView = listView;
                this.groups = [{ startIndex: 0 }];
                this.dirty = true;
            }, {
                synchronizeGroups: function () {
                    return Promise.wrap();
                },

                addItem: function () {
                    return Promise.wrap(this.groups[0]);
                },

                resetGroups: function () {
                    this.groups = [{ startIndex: 0 }];
                    delete this.pinnedItem;
                    delete this.pinnedOffset;
                    this.dirty = true;
                },

                renderGroup: function () {
                    return Promise.wrap(null);
                },

                ensureFirstGroup: function () {
                    return Promise.wrap(this.groups[0]);
                },

                groupOf: function () {
                    return Promise.wrap(this.groups[0]);
                },

                removeElements: function () {
                }
            });
        })
    });

});

// Copyright (c) Microsoft Corporation.  All Rights Reserved. Licensed under the MIT License. See License.txt in the project root for license information.
define('WinJS/Controls/ListView/_Helpers',[
    'exports',
    '../../Core/_Base',
    '../ItemContainer/_Constants'
], function helpersInit(exports, _Base, _Constants) {
    "use strict";

    function nodeListToArray(nodeList) {
        return Array.prototype.slice.call(nodeList);
    }

    function repeat(strings, count) {
        // Continously concatenate a string or set of strings
        // until the specified number of concatenations are made.
        // e.g.
        //  repeat("a", 3) ==> "aaa"
        //  repeat(["a", "b"], 0) ==> ""
        //  repeat(["a", "b", "c"], 2) ==> "ab"
        //  repeat(["a", "b", "c"], 7) ==> "abcabca"
        if (typeof strings === "string") {
            return repeat([strings], count);
        }
        var result = new Array(Math.floor(count / strings.length) + 1).join(strings.join(""));
        result += strings.slice(0, count % strings.length).join("");
        return result;
    }

    function stripedContainers(count, nextItemIndex) {
        var containersMarkup,
            evenStripe = _Constants._containerEvenClass,
            oddStripe = _Constants._containerOddClass,
            stripes = nextItemIndex % 2 === 0 ? [evenStripe, oddStripe] : [oddStripe, evenStripe];

        var pairOfContainers = [
                "<div class='win-container " + stripes[0] + " win-backdrop'></div>",
                "<div class='win-container " + stripes[1] + " win-backdrop'></div>"
        ];

        containersMarkup = repeat(pairOfContainers, count);
        return containersMarkup;
    }

    _Base.Namespace._moduleDefine(exports, "WinJS.UI", {
        _nodeListToArray: nodeListToArray,
        _repeat: repeat,
        _stripedContainers: stripedContainers
    });
});

// Copyright (c) Microsoft Corporation.  All Rights Reserved. Licensed under the MIT License. See License.txt in the project root for license information.
define('WinJS/Controls/ListView/_ItemsContainer',[
    'exports',
    '../../Core/_Base',
    '../../Promise',
    '../../Utilities/_ElementUtilities',
    '../ItemContainer/_Constants'
    ], function itemsContainerInit(exports, _Base, Promise, _ElementUtilities, _Constants) {
    "use strict";

    _Base.Namespace._moduleDefine(exports, "WinJS.UI", {
        _ItemsContainer: _Base.Namespace._lazy(function () {

            var _ItemsContainer = function (site) {
                this.site = site;
                this._itemData = {};
                this.waitingItemRequests = {};
            };
            _ItemsContainer.prototype = {
                requestItem: function ItemsContainer_requestItem(itemIndex) {
                    if (!this.waitingItemRequests[itemIndex]) {
                        this.waitingItemRequests[itemIndex] = [];
                    }

                    var that = this;
                    var promise = new Promise(function (complete) {
                        var itemData = that._itemData[itemIndex];
                        if (itemData && !itemData.detached && itemData.element) {
                            complete(itemData.element);
                        } else {
                            that.waitingItemRequests[itemIndex].push(complete);
                        }
                    });

                    return promise;
                },

                removeItem: function (index) {
                    delete this._itemData[index];
                },

                removeItems: function ItemsContainer_removeItems() {
                    this._itemData = {};
                    this.waitingItemRequests = {};
                },

                setItemAt: function ItemsContainer_setItemAt(itemIndex, itemData) {
                    this._itemData[itemIndex] = itemData;
                    if (!itemData.detached) {
                        this.notify(itemIndex, itemData);
                    }
                },

                notify: function ItemsContainer_notify(itemIndex, itemData) {
                    if (this.waitingItemRequests[itemIndex]) {
                        var requests = this.waitingItemRequests[itemIndex];
                        for (var i = 0; i < requests.length; i++) {
                            requests[i](itemData.element);
                        }

                        this.waitingItemRequests[itemIndex] = [];
                    }
                },

                elementAvailable: function ItemsContainer_elementAvailable(itemIndex) {
                    var itemData = this._itemData[itemIndex];
                    itemData.detached = false;
                    this.notify(itemIndex, itemData);
                },

                itemAt: function ItemsContainer_itemAt(itemIndex) {
                    var itemData = this._itemData[itemIndex];
                    return itemData ? itemData.element : null;
                },

                itemDataAt: function ItemsContainer_itemDataAt(itemIndex) {
                    return this._itemData[itemIndex];
                },

                containerAt: function ItemsContainer_containerAt(itemIndex) {
                    var itemData = this._itemData[itemIndex];
                    return itemData ? itemData.container : null;
                },

                itemBoxAt: function ItemsContainer_itemBoxAt(itemIndex) {
                    var itemData = this._itemData[itemIndex];
                    return itemData ? itemData.itemBox : null;
                },

                itemBoxFrom: function ItemsContainer_containerFrom(element) {
                    while (element && !_ElementUtilities.hasClass(element, _Constants._itemBoxClass)) {
                        element = element.parentNode;
                    }

                    return element;
                },

                containerFrom: function ItemsContainer_containerFrom(element) {
                    while (element && !_ElementUtilities.hasClass(element, _Constants._containerClass)) {
                        element = element.parentNode;
                    }

                    return element;
                },

                index: function ItemsContainer_index(element) {
                    var item = this.containerFrom(element);
                    if (item) {
                        for (var index in this._itemData) {
                            if (this._itemData[index].container === item) {
                                return parseInt(index, 10);
                            }
                        }
                    }

                    return _Constants._INVALID_INDEX;
                },

                each: function ItemsContainer_each(callback) {
                    for (var index in this._itemData) {
                        if (this._itemData.hasOwnProperty(index)) {
                            var itemData = this._itemData[index];
                            callback(parseInt(index, 10), itemData.element, itemData);
                        }
                    }
                },

                eachIndex: function ItemsContainer_each(callback) {
                    for (var index in this._itemData) {
                        if (callback(parseInt(index, 10))) {
                            break;
                        }
                    }
                },

                count: function ItemsContainer_count() {
                    return Object.keys(this._itemData).length;
                }
            };
            return _ItemsContainer;
        })
    });

});

// Copyright (c) Microsoft Corporation.  All Rights Reserved. Licensed under the MIT License. See License.txt in the project root for license information.
define('WinJS/Controls/ListView/_Layouts',[
    'exports',
    '../../Core/_Global',
    '../../Core/_Base',
    '../../Core/_BaseUtils',
    '../../Core/_ErrorFromName',
    '../../Core/_Resources',
    '../../Core/_WriteProfilerMark',
    '../../Animations/_TransitionAnimation',
    '../../Promise',
    '../../Scheduler',
    '../../_Signal',
    '../../Utilities/_Dispose',
    '../../Utilities/_ElementUtilities',
    '../../Utilities/_SafeHtml',
    '../../Utilities/_UI',
    '../ItemContainer/_Constants',
    './_ErrorMessages'
], function layouts2Init(exports, _Global, _Base, _BaseUtils, _ErrorFromName, _Resources, _WriteProfilerMark, _TransitionAnimation, Promise, Scheduler, _Signal, _Dispose, _ElementUtilities, _SafeHtml, _UI, _Constants, _ErrorMessages) {
    "use strict";

    var Key = _ElementUtilities.Key,
        uniqueID = _ElementUtilities._uniqueID;

    var strings = {
        get itemInfoIsInvalid() { return "Invalid argument: An itemInfo function must be provided which returns an object with numeric width and height properties."; },
        get groupInfoResultIsInvalid() { return "Invalid result: groupInfo result for cell spanning groups must include the following numeric properties: cellWidth and cellHeight."; }
    };

    //
    // Helpers for dynamic CSS rules
    //
    // Rule deletions are delayed until the next rule insertion. This helps the
    // scenario where a ListView changes layouts. By doing the rule manipulations
    // in a single synchronous block, IE will do 1 layout pass instead of 2.
    //

    // Dynamic CSS rules will be added to this style element
    var layoutStyleElem = _Global.document.createElement("style");
    _Global.document.head.appendChild(layoutStyleElem);

    var nextCssClassId = 0,
        staleClassNames = [];

    // The prefix for the class name should not contain dashes
    function uniqueCssClassName(prefix) {
        return "_win-dynamic-" + prefix + "-" + (nextCssClassId++);
    }

    var browserStyleEquivalents = _BaseUtils._browserStyleEquivalents;
    var transformNames = browserStyleEquivalents["transform"];
    var transitionScriptName = _BaseUtils._browserStyleEquivalents["transition"].scriptName;
    var dragBetweenTransition = transformNames.cssName + " cubic-bezier(0.1, 0.9, 0.2, 1) 167ms";
    var dragBetweenDistance = 12;

    // Removes the dynamic CSS rules corresponding to the classes in staleClassNames
    // from the DOM.
    function flushDynamicCssRules() {
        var rules = layoutStyleElem.sheet.cssRules,
            classCount = staleClassNames.length,
            i,
            j,
            ruleSuffix;

        for (i = 0; i < classCount; i++) {
            ruleSuffix = "." + staleClassNames[i] + " ";
            for (j = rules.length - 1; j >= 0; j--) {
                if (rules[j].selectorText.indexOf(ruleSuffix) !== -1) {
                    layoutStyleElem.sheet.deleteRule(j);
                }
            }
        }
        staleClassNames = [];
    }

    // Creates a dynamic CSS rule and adds it to the DOM. uniqueToken is a class name
    // which uniquely identifies a set of related rules. These rules may be removed
    // using deleteDynamicCssRule. uniqueToken should be created using uniqueCssClassName.
    function addDynamicCssRule(uniqueToken, site, selector, body) {
        flushDynamicCssRules();
        var rule = "." + _Constants._listViewClass + " ." + uniqueToken + " " + selector + " { " +
             body +
        "}";
        var perfId = "_addDynamicCssRule:" + uniqueToken + ",info";
        if (site) {
            site._writeProfilerMark(perfId);
        } else {
            _WriteProfilerMark("WinJS.UI.ListView:Layout" + perfId);
        }
        layoutStyleElem.sheet.insertRule(rule, 0);
    }

    // Marks the CSS rules corresponding to uniqueToken for deletion. The rules
    // should have been added by addDynamicCssRule.
    function deleteDynamicCssRule(uniqueToken) {
        staleClassNames.push(uniqueToken);
    }

    //
    // Helpers shared by all layouts
    //

    // Clamps x to the range first <= x <= last
    function clampToRange(first, last, x) {
        return Math.max(first, Math.min(last, x));
    }

    function getDimension(element, property) {
        return _ElementUtilities.convertToPixels(element, _Global.getComputedStyle(element, null)[property]);
    }

    // Returns the sum of the margin, border, and padding for the side of the
    // element specified by side. side can be "Left", "Right", "Top", or "Bottom".
    function getOuter(side, element) {
        return getDimension(element, "margin" + side) +
            getDimension(element, "border" + side + "Width") +
            getDimension(element, "padding" + side);
    }

    // Returns the total height of element excluding its content height
    function getOuterHeight(element) {
        return getOuter("Top", element) + getOuter("Bottom", element);
    }

    // Returns the total width of element excluding its content width
    function getOuterWidth(element) {
        return getOuter("Left", element) + getOuter("Right", element);
    }

    function forEachContainer(itemsContainer, callback) {
        if (itemsContainer.items) {
            for (var i = 0, len = itemsContainer.items.length; i < len; i++) {
                callback(itemsContainer.items[i], i);
            }
        } else {
            for (var b = 0, index = 0; b < itemsContainer.itemsBlocks.length; b++) {
                var block = itemsContainer.itemsBlocks[b];
                for (var i = 0, len = block.items.length; i < len; i++) {
                    callback(block.items[i], index++);
                }
            }
        }
    }

    function containerFromIndex(itemsContainer, index) {
        if (index < 0) {
            return null;
        }
        if (itemsContainer.items) {
            return (index < itemsContainer.items.length ? itemsContainer.items[index] : null);
        } else {
            var blockSize = itemsContainer.itemsBlocks[0].items.length,
                blockIndex = Math.floor(index / blockSize),
                offset = index % blockSize;
            return (blockIndex < itemsContainer.itemsBlocks.length && offset < itemsContainer.itemsBlocks[blockIndex].items.length ? itemsContainer.itemsBlocks[blockIndex].items[offset] : null);
        }
    }

    function getItemsContainerTree(itemsContainer, tree) {
        var itemsContainerTree;
        for (var i = 0, treeLength = tree.length; i < treeLength; i++) {
            if (tree[i].itemsContainer.element === itemsContainer) {
                itemsContainerTree = tree[i].itemsContainer;
                break;
            }
        }
        return itemsContainerTree;
    }

    function getItemsContainerLength(itemsContainer) {
        var blocksCount,
            itemsCount;
        if (itemsContainer.itemsBlocks) {
            blocksCount = itemsContainer.itemsBlocks.length;
            if (blocksCount > 0) {
                itemsCount = (itemsContainer.itemsBlocks[0].items.length * (blocksCount - 1)) + itemsContainer.itemsBlocks[blocksCount - 1].items.length;
            } else {
                itemsCount = 0;
            }
        } else {
            itemsCount = itemsContainer.items.length;
        }
        return itemsCount;
    }

    var environmentDetails = null;
    // getEnvironmentSupportInformation does one-time checks on several browser-specific environment details (both to check the existence of styles,
    // and also to see if some environments have layout bugs the ListView needs to work around).
    function getEnvironmentSupportInformation(site) {
        if (!environmentDetails) {
            var surface = _Global.document.createElement("div");
            surface.style.width = "500px";
            surface.style.visibility = "hidden";

            // Set up the DOM
            var flexRoot = _Global.document.createElement("div");
            flexRoot.style.cssText += "width: 500px; height: 200px; display: -webkit-flex; display: flex";
            _SafeHtml.setInnerHTMLUnsafe(flexRoot,
                "<div style='height: 100%; display: -webkit-flex; display: flex; flex-flow: column wrap; align-content: flex-start; -webkit-flex-flow: column wrap; -webkit-align-content: flex-start'>" +
                    "<div style='width: 100px; height: 100px'></div>" +
                    "<div style='width: 100px; height: 100px'></div>" +
                    "<div style='width: 100px; height: 100px'></div>" +
                "</div>");
            surface.appendChild(flexRoot);

            // Read from the DOM and detect the bugs
            site.viewport.insertBefore(surface, site.viewport.firstChild);
            var canMeasure = surface.offsetWidth > 0,
                expectedWidth = 200;
            if (canMeasure) {
                // If we can't measure now (e.g. ListView is display:none), leave environmentDetails as null
                // so that we do the detection later when the app calls recalculateItemPosition/forceLayout.

                environmentDetails = {
                    supportsCSSGrid: !!("-ms-grid-row" in _Global.document.documentElement.style),
                    // Detects Chrome flex issue 345433: Incorrect sizing for nested flexboxes
                    // https://code.google.com/p/chromium/issues/detail?id=345433
                    // With nested flexboxes, the inner flexbox's width is proportional to the number of elements intead
                    // of the number of columns.
                    nestedFlexTooLarge: flexRoot.firstElementChild.offsetWidth > expectedWidth,

                    // Detects Firefox issue 995020
                    // https://bugzilla.mozilla.org/show_bug.cgi?id=995020
                    // The three squares we're adding to the nested flexbox should increase the size of the nestedFlex to be 200 pixels wide. This is the case in IE but
                    // currently not in Firefox. In Firefox, the third square will move to the next column, but the container's width won't update for it.
                    nestedFlexTooSmall: flexRoot.firstElementChild.offsetWidth < expectedWidth
                };
            }

            // Signal ListView's own measurement operation.
            // ListView always needs an opportunity to measure, even if layout cannot.
            site.readyToMeasure();

            // Clean up the DOM
            site.viewport.removeChild(surface);
        }

        return environmentDetails;
    }

    _Base.Namespace._moduleDefine(exports, "WinJS.UI", {
        Layout: _Base.Class.define(function Layout_ctor() {
            /// <signature helpKeyword="WinJS.UI.Layout">
            /// <summary locid="WinJS.UI.Layout.constructor">
            /// Creates a new Layout object.
            /// </summary>
            /// <param name="options" type="Object" locid="WinJS.UI.Layout.constructor_p:options">
            /// The set of options to be applied initially to the new Layout object.
            /// </param>
            /// <returns type="WinJS.UI.Layout" locid="WinJS.UI.Layout.constructor_returnValue">
            /// The new Layout object.
            /// </returns>
            /// </signature>
        }),

        _LayoutCommon: _Base.Namespace._lazy(function () {
            return _Base.Class.derive(exports.Layout, null, {
                /// <field type="String" oamOptionsDatatype="WinJS.UI.HeaderPosition" locid="WinJS.UI._LayoutCommon.groupHeaderPosition" helpKeyword="WinJS.UI._LayoutCommon.groupHeaderPosition">
                /// Gets or sets the position of group headers relative to their items.
                /// The default value is "top".
                /// </field>
                groupHeaderPosition: {
                    enumerable: true,
                    get: function () {
                        return this._groupHeaderPosition;
                    },
                    set: function (position) {
                        this._groupHeaderPosition = position;
                        this._invalidateLayout();
                    }
                },

                // Implementation of part of ILayout interface

                initialize: function _LayoutCommon_initialize(site, groupsEnabled) {
                    site._writeProfilerMark("Layout:initialize,info");
                    if (!this._inListMode) {
                        _ElementUtilities.addClass(site.surface, _Constants._gridLayoutClass);
                    }

                    if (this._backdropColorClassName) {
                        _ElementUtilities.addClass(site.surface, this._backdropColorClassName);
                    }
                    if (this._disableBackdropClassName) {
                        _ElementUtilities.addClass(site.surface, this._disableBackdropClassName);
                    }
                    this._groups = [];
                    this._groupMap = {};
                    this._oldGroupHeaderPosition = null;
                    this._usingStructuralNodes = false;

                    this._site = site;
                    this._groupsEnabled = groupsEnabled;
                    this._resetAnimationCaches(true);
                },

                /// <field type="String" oamOptionsDatatype="WinJS.UI.Orientation" locid="WinJS.UI._LayoutCommon.orientation" helpKeyword="WinJS.UI._LayoutCommon.orientation">
                /// Gets or sets the orientation for the layout.
                /// The default value is "horizontal".
                /// </field>
                orientation: {
                    enumerable: true,
                    get: function () {
                        return this._orientation;
                    },
                    set: function (orientation) {
                        this._orientation = orientation;
                        this._horizontal = (orientation === "horizontal");
                        this._invalidateLayout();
                    }
                },

                uninitialize: function _LayoutCommon_uninitialize() {
                    var perfId = "Layout:uninitialize,info";
                    function cleanGroups(groups) {
                        var len = groups.length,
                            i;
                        for (i = 0; i < len; i++) {
                            groups[i].cleanUp(true);
                        }
                    }

                    this._elementsToMeasure = {};

                    if (this._site) {
                        this._site._writeProfilerMark(perfId);
                        _ElementUtilities.removeClass(this._site.surface, _Constants._gridLayoutClass);
                        _ElementUtilities.removeClass(this._site.surface, _Constants._headerPositionTopClass);
                        _ElementUtilities.removeClass(this._site.surface, _Constants._headerPositionLeftClass);
                        _ElementUtilities.removeClass(this._site.surface, _Constants._structuralNodesClass);
                        _ElementUtilities.removeClass(this._site.surface, _Constants._singleItemsBlockClass);
                        _ElementUtilities.removeClass(this._site.surface, _Constants._noCSSGrid);
                        this._site.surface.style.cssText = "";
                        if (this._groups) {
                            cleanGroups(this._groups);
                            this._groups = null;
                            this._groupMap = null;
                        }
                        if (this._layoutPromise) {
                            this._layoutPromise.cancel();
                            this._layoutPromise = null;
                        }
                        this._resetMeasurements();
                        this._oldGroupHeaderPosition = null;
                        this._usingStructuralNodes = false;
                        this._envInfo = null;
                        // The properties given to us by the app (_groupInfo, _itemInfo,
                        // _groupHeaderPosition) are not cleaned up so that the values are
                        // remembered if the layout is reused.

                        if (this._backdropColorClassName) {
                            _ElementUtilities.removeClass(this._site.surface, this._backdropColorClassName);
                            deleteDynamicCssRule(this._backdropColorClassName);
                            this._backdropColorClassName = null;
                        }
                        if (this._disableBackdropClassName) {
                            _ElementUtilities.removeClass(this._site.surface, this._disableBackdropClassName);
                            deleteDynamicCssRule(this._disableBackdropClassName);
                            this._disableBackdropClassName = null;
                        }

                        this._site = null;
                        this._groupsEnabled = null;
                        if (this._animationsRunning) {
                            this._animationsRunning.cancel();
                        }
                        this._animatingItemsBlocks = {};
                    } else {
                        _WriteProfilerMark("WinJS.UI.ListView:" + perfId);
                    }
                },

                numberOfItemsPerItemsBlock: {
                    get: function _LayoutCommon_getNumberOfItemsPerItemsBlock() {
                        function allGroupsAreUniform() {
                            var groupCount = that._site.groupCount,
                                i;

                            for (i = 0; i < groupCount; i++) {
                                if (that._isCellSpanning(i)) {
                                    return false;
                                }
                            }

                            return true;
                        }

                        var that = this;
                        return that._measureItem(0).then(function () {
                            if (that._sizes.viewportContentSize !== that._getViewportCrossSize()) {
                                that._viewportSizeChanged(that._getViewportCrossSize());
                            }

                            if (!allGroupsAreUniform()) {
                                that._usingStructuralNodes = false;
                                return null;
                            } else if (that._envInfo.nestedFlexTooLarge || that._envInfo.nestedFlexTooSmall) {
                                // Store all items in a single itemsblock
                                that._usingStructuralNodes = true;
                                return Number.MAX_VALUE;
                            } else {
                                that._usingStructuralNodes = exports._LayoutCommon._barsPerItemsBlock > 0;
                                return exports._LayoutCommon._barsPerItemsBlock * that._itemsPerBar;
                            }
                        });
                    }
                },

                layout: function _LayoutCommon_layout(tree, changedRange, modifiedItems, modifiedGroups) {
                    // changedRange implies that the minimum amount of work the layout needs to do is as follows:
                    // - It needs to lay out group shells (header containers and items containers) from
                    //   firstChangedGroup thru lastGroup.
                    // - It needs to ask firstChangedGroup thru lastChangedGroup to lay out their
                    //   contents (i.e. win-containers).
                    // - For each group included in the changedRange, it needs to lay out its
                    //   contents (i.e. win-containers) from firstChangedItem thru lastItem.

                    var that = this;
                    var site = that._site,
                        layoutPerfId = "Layout.layout",
                        realizedRangePerfId = layoutPerfId + ":realizedRange",
                        realizedRangePromise;

                    that._site._writeProfilerMark(layoutPerfId + ",StartTM");
                    that._site._writeProfilerMark(realizedRangePerfId + ",StartTM");

                    // Receives an items container's tree and returns a normalized copy.
                    // This allows us to hold on to a snapshot of the tree without
                    // worrying that items may have been unexpectedly inserted/
                    // removed/moved. The returned tree always appears as though
                    // structural nodes are disabled.
                    function copyItemsContainerTree(itemsContainer) {
                        function copyItems(itemsContainer) {
                            if (that._usingStructuralNodes) {
                                var items = [];
                                itemsContainer.itemsBlocks.forEach(function (itemsBlock) {
                                    items = items.concat(itemsBlock.items.slice(0));
                                });
                                return items;
                            } else {
                                return itemsContainer.items.slice(0);
                            }
                        }

                        return {
                            element: itemsContainer.element,
                            items: copyItems(itemsContainer)
                        };
                    }

                    // Updates the GridLayout's internal state to reflect the current tree.
                    // Similarly tells each group to update its internal state via prepareLayout.
                    // After this function runs, the ILayout functions will return results that
                    // are appropriate for the current tree.
                    function updateGroups() {
                        function createGroup(groupInfo, itemsContainer) {
                            var GroupType = (groupInfo.enableCellSpanning ?
                                Groups.CellSpanningGroup :
                                Groups.UniformGroup);
                            return new GroupType(that, itemsContainer);
                        }

                        var oldRealizedItemRange = (that._groups.length > 0 ?
                                that._getRealizationRange() :
                                null),
                            newGroups = [],
                            prepared = [],
                            cleanUpDom = {},
                            newGroupMap = {},
                            currentIndex = 0,
                            len = tree.length,
                            i;

                        for (i = 0; i < len; i++) {
                            var oldChangedRealizedRangeInGroup = null,
                                groupInfo = that._getGroupInfo(i),
                                groupKey = that._site.groupFromIndex(i).key,
                                oldGroup = that._groupMap[groupKey],
                                wasCellSpanning = oldGroup instanceof Groups.CellSpanningGroup,
                                isCellSpanning = groupInfo.enableCellSpanning;

                            if (oldGroup) {
                                if (wasCellSpanning !== isCellSpanning) {
                                    // The group has changed types so DOM needs to be cleaned up
                                    cleanUpDom[groupKey] = true;
                                } else {
                                    // Compute the range of changed items that is within the group's realized range
                                    var firstChangedIndexInGroup = Math.max(0, changedRange.firstIndex - oldGroup.startIndex),
                                        oldRealizedItemRangeInGroup = that._rangeForGroup(oldGroup, oldRealizedItemRange);
                                    if (oldRealizedItemRangeInGroup && firstChangedIndexInGroup <= oldRealizedItemRangeInGroup.lastIndex) {
                                        // The old changed realized range is non-empty
                                        oldChangedRealizedRangeInGroup = {
                                            firstIndex: Math.max(firstChangedIndexInGroup, oldRealizedItemRangeInGroup.firstIndex),
                                            lastIndex: oldRealizedItemRangeInGroup.lastIndex
                                        };
                                    }
                                }
                            }
                            var group = createGroup(groupInfo, tree[i].itemsContainer.element);
                            var prepareLayoutPromise;
                            if (group.prepareLayoutWithCopyOfTree) {
                                prepareLayoutPromise = group.prepareLayoutWithCopyOfTree(copyItemsContainerTree(tree[i].itemsContainer), oldChangedRealizedRangeInGroup, oldGroup, {
                                    groupInfo: groupInfo,
                                    startIndex: currentIndex,
                                });
                            } else {
                                prepareLayoutPromise = group.prepareLayout(getItemsContainerLength(tree[i].itemsContainer), oldChangedRealizedRangeInGroup, oldGroup, {
                                    groupInfo: groupInfo,
                                    startIndex: currentIndex,
                                });
                            }
                            prepared.push(prepareLayoutPromise);

                            currentIndex += group.count;

                            newGroups.push(group);
                            newGroupMap[groupKey] = group;
                        }

                        return Promise.join(prepared).then(function () {
                            var currentOffset = 0;
                            for (var i = 0, len = newGroups.length; i < len; i++) {
                                var group = newGroups[i];
                                group.offset = currentOffset;
                                currentOffset += that._getGroupSize(group);
                            }

                            // Clean up deleted groups
                            Object.keys(that._groupMap).forEach(function (deletedKey) {
                                var skipDomCleanUp = !cleanUpDom[deletedKey];
                                that._groupMap[deletedKey].cleanUp(skipDomCleanUp);
                            });

                            that._groups = newGroups;
                            that._groupMap = newGroupMap;
                        });
                    }

                    // When doRealizedRange is true, this function is synchronous and has no return value.
                    // When doRealizedRange is false, this function is asynchronous and returns a promise.
                    function layoutGroupContent(groupIndex, realizedItemRange, doRealizedRange) {
                        var group = that._groups[groupIndex],
                            firstChangedIndexInGroup = Math.max(0, changedRange.firstIndex - group.startIndex),
                            realizedItemRangeInGroup = that._rangeForGroup(group, realizedItemRange),
                            beforeRealizedRange;

                        if (doRealizedRange) {
                            group.layoutRealizedRange(firstChangedIndexInGroup, realizedItemRangeInGroup);
                        } else {
                            if (!realizedItemRangeInGroup) {
                                beforeRealizedRange = (group.startIndex + group.count - 1 < realizedItemRange.firstIndex);
                            }

                            return group.layoutUnrealizedRange(firstChangedIndexInGroup, realizedItemRangeInGroup, beforeRealizedRange);
                        }
                    }

                    // Synchronously lays out:
                    // - Realized and unrealized group shells (header containers and items containers).
                    //   This is needed so that each realized group will be positioned at the correct offset.
                    // - Realized items.
                    function layoutRealizedRange() {
                        if (that._groups.length === 0) {
                            return;
                        }

                        var realizedItemRange = that._getRealizationRange(),
                            len = tree.length,
                            i,
                            firstChangedGroup = site.groupIndexFromItemIndex(changedRange.firstIndex);

                        for (i = firstChangedGroup; i < len; i++) {
                            layoutGroupContent(i, realizedItemRange, true);
                            that._layoutGroup(i);
                        }
                    }

                    // Asynchronously lays out the unrealized items
                    function layoutUnrealizedRange() {
                        if (that._groups.length === 0) {
                            return Promise.wrap();
                        }

                        var realizedItemRange = that._getRealizationRange(),
                            // Last group before the realized range which contains 1 or more unrealized items
                            lastGroupBefore = site.groupIndexFromItemIndex(realizedItemRange.firstIndex - 1),
                            // First group after the realized range which contains 1 or more unrealized items
                            firstGroupAfter = site.groupIndexFromItemIndex(realizedItemRange.lastIndex + 1),
                            firstChangedGroup = site.groupIndexFromItemIndex(changedRange.firstIndex),
                            layoutPromises = [],
                            groupCount = that._groups.length;

                        var stop = false;
                        var before = lastGroupBefore;
                        var after = Math.max(firstChangedGroup, firstGroupAfter);
                        after = Math.max(before + 1, after);
                        while (!stop) {
                            stop = true;
                            if (before >= firstChangedGroup) {
                                layoutPromises.push(layoutGroupContent(before, realizedItemRange, false));
                                stop = false;
                                before--;
                            }
                            if (after < groupCount) {
                                layoutPromises.push(layoutGroupContent(after, realizedItemRange, false));
                                stop = false;
                                after++;
                            }
                        }

                        return Promise.join(layoutPromises);
                    }

                    realizedRangePromise = that._measureItem(0).then(function () {
                        _ElementUtilities[(that._usingStructuralNodes) ? "addClass" : "removeClass"]
                            (that._site.surface, _Constants._structuralNodesClass);
                        _ElementUtilities[(that._envInfo.nestedFlexTooLarge || that._envInfo.nestedFlexTooSmall) ? "addClass" : "removeClass"]
                            (that._site.surface, _Constants._singleItemsBlockClass);

                        if (that._sizes.viewportContentSize !== that._getViewportCrossSize()) {
                            that._viewportSizeChanged(that._getViewportCrossSize());
                        }

                        // Move deleted elements to their original positions before calling updateGroups can be slow.
                        that._cacheRemovedElements(modifiedItems, that._cachedItemRecords, that._cachedInsertedItemRecords, that._cachedRemovedItems, false);
                        that._cacheRemovedElements(modifiedGroups, that._cachedHeaderRecords, that._cachedInsertedHeaderRecords, that._cachedRemovedHeaders, true);

                        return updateGroups();
                    }).then(function () {
                        that._syncDomWithGroupHeaderPosition(tree);
                        var surfaceLength = 0;
                        if (that._groups.length > 0) {
                            var lastGroup = that._groups[that._groups.length - 1];
                            surfaceLength = lastGroup.offset + that._getGroupSize(lastGroup);
                        }

                        // Explicitly set the surface width/height. This maintains backwards
                        // compatibility with the original layouts by allowing the items
                        // to be shifted through surface margins.
                        if (that._horizontal) {
                            if (that._groupsEnabled && that._groupHeaderPosition === HeaderPosition.left) {
                                site.surface.style.cssText +=
                                    ";height:" + that._sizes.surfaceContentSize +
                                    "px;-ms-grid-columns: (" + that._sizes.headerContainerWidth + "px auto)[" + tree.length + "]";
                            } else {
                                site.surface.style.height = that._sizes.surfaceContentSize + "px";
                            }
                            if (that._envInfo.nestedFlexTooLarge || that._envInfo.nestedFlexTooSmall) {
                                site.surface.style.width = surfaceLength + "px";
                            }
                        } else {
                            if (that._groupsEnabled && that._groupHeaderPosition === HeaderPosition.top) {
                                site.surface.style.cssText +=
                                    ";width:" + that._sizes.surfaceContentSize +
                                    "px;-ms-grid-rows: (" + that._sizes.headerContainerHeight + "px auto)[" + tree.length + "]";
                            } else {
                                site.surface.style.width = that._sizes.surfaceContentSize + "px";
                            }
                            if (that._envInfo.nestedFlexTooLarge || that._envInfo.nestedFlexTooSmall) {
                                site.surface.style.height = surfaceLength + "px";
                            }
                        }

                        layoutRealizedRange();

                        that._layoutAnimations(modifiedItems, modifiedGroups);

                        that._site._writeProfilerMark(realizedRangePerfId + ":complete,info");
                        that._site._writeProfilerMark(realizedRangePerfId + ",StopTM");
                    }, function (error) {
                        that._site._writeProfilerMark(realizedRangePerfId + ":canceled,info");
                        that._site._writeProfilerMark(realizedRangePerfId + ",StopTM");
                        return Promise.wrapError(error);
                    });

                    that._layoutPromise = realizedRangePromise.then(function () {
                        return layoutUnrealizedRange().then(function () {
                            that._site._writeProfilerMark(layoutPerfId + ":complete,info");
                            that._site._writeProfilerMark(layoutPerfId + ",StopTM");
                        }, function (error) {
                            that._site._writeProfilerMark(layoutPerfId + ":canceled,info");
                            that._site._writeProfilerMark(layoutPerfId + ",StopTM");
                            return Promise.wrapError(error);
                        });
                    });

                    return {
                        realizedRangeComplete: realizedRangePromise,
                        layoutComplete: that._layoutPromise
                    };
                },

                itemsFromRange: function _LayoutCommon_itemsFromRange(firstPixel, lastPixel) {
                    if (this._rangeContainsItems(firstPixel, lastPixel)) {
                        return {
                            firstIndex: this._firstItemFromRange(firstPixel),
                            lastIndex: this._lastItemFromRange(lastPixel)
                        };
                    } else {
                        return {
                            firstIndex: 0,
                            lastIndex: -1
                        };
                    }

                },

                getAdjacent: function _LayoutCommon_getAdjacent(currentItem, pressedKey) {
                    var that = this,
                        groupIndex = that._site.groupIndexFromItemIndex(currentItem.index),
                        group = that._groups[groupIndex],
                        adjustedKey = that._adjustedKeyForOrientationAndBars(that._adjustedKeyForRTL(pressedKey), group instanceof Groups.CellSpanningGroup);

                    if (!currentItem.type) {
                        currentItem.type = _UI.ObjectType.item;
                    }
                    if (currentItem.type !== _UI.ObjectType.item && (pressedKey === Key.pageUp || pressedKey === Key.pageDown)) {
                        // We treat page up and page down keys as if an item had focus
                        var itemIndex = 0;
                        if (currentItem.type === _UI.ObjectType.groupHeader) {
                            itemIndex = that._groups[currentItem.index].startIndex;
                        } else {
                            itemIndex = (currentItem.type === _UI.ObjectType.header ? 0 : that._groups[that._groups.length - 1].count - 1);
                        }
                        currentItem = { type: _UI.ObjectType.item, index: itemIndex };
                    }else if (currentItem.type === _UI.ObjectType.header && adjustedKey === Key.rightArrow) {
                        return { type: (that._groupsEnabled ? _UI.ObjectType.groupHeader : _UI.ObjectType.footer), index: 0 };
                    } else if (currentItem.type === _UI.ObjectType.footer && adjustedKey === Key.leftArrow) {
                        return { type: (that._groupsEnabled ? _UI.ObjectType.groupHeader : _UI.ObjectType.header), index: 0 };
                    } else if (currentItem.type === _UI.ObjectType.groupHeader) {
                        if (adjustedKey === Key.leftArrow) {
                            var desiredIndex = currentItem.index - 1;
                            desiredIndex = (that._site.header ? desiredIndex : Math.max(0, desiredIndex));
                            return {
                                type: (desiredIndex > -1 ? _UI.ObjectType.groupHeader : _UI.ObjectType.header),
                                index: (desiredIndex > -1 ? desiredIndex : 0)
                            };
                        } else if (adjustedKey === Key.rightArrow) {
                            var desiredIndex = currentItem.index + 1;
                            desiredIndex = (that._site.header ? desiredIndex : Math.min(that._groups.length - 1, currentItem.index + 1));
                            return {
                                type: (desiredIndex >= that._groups.length ? _UI.ObjectType.header : _UI.ObjectType.groupHeader),
                                index: (desiredIndex >= that._groups.length ? 0 : desiredIndex)
                            };
                        }
                        return currentItem;
                    }

                    function handleArrowKeys() {
                        var currentItemInGroup = {
                            type: currentItem.type,
                            index: currentItem.index - group.startIndex
                        },
                            newItem = group.getAdjacent(currentItemInGroup, adjustedKey);

                        if (newItem === "boundary") {
                            var prevGroup = that._groups[groupIndex - 1],
                                nextGroup = that._groups[groupIndex + 1],
                                lastGroupIndex = that._groups.length - 1;

                            if (adjustedKey === Key.leftArrow) {
                                if (groupIndex === 0) {
                                    // We're at the beginning of the first group so stay put
                                    return currentItem;
                                } else if (prevGroup instanceof Groups.UniformGroup && group instanceof Groups.UniformGroup) {
                                    // Moving between uniform groups so maintain the row/column if possible
                                    var coordinates = that._indexToCoordinate(currentItemInGroup.index);
                                    var currentSlot = (that._horizontal ? coordinates.row : coordinates.column),
                                        indexOfLastBar = Math.floor((prevGroup.count - 1) / that._itemsPerBar),
                                        startOfLastBar = indexOfLastBar * that._itemsPerBar; // first cell of last bar
                                    return {
                                        type: _UI.ObjectType.item,
                                        index: prevGroup.startIndex + Math.min(prevGroup.count - 1, startOfLastBar + currentSlot)
                                    };
                                } else {
                                    // Moving to or from a cell spanning group so go to the last item
                                    return { type: _UI.ObjectType.item, index: group.startIndex - 1 };
                                }
                            } else if (adjustedKey === Key.rightArrow) {
                                if (groupIndex === lastGroupIndex) {
                                    // We're at the end of the last group so stay put
                                    return currentItem;
                                } else if (group instanceof Groups.UniformGroup && nextGroup instanceof Groups.UniformGroup) {
                                    // Moving between uniform groups so maintain the row/column if possible
                                    var coordinates = that._indexToCoordinate(currentItemInGroup.index),
                                        currentSlot = (that._horizontal ? coordinates.row : coordinates.column);
                                    return {
                                        type: _UI.ObjectType.item,
                                        index: nextGroup.startIndex + Math.min(nextGroup.count - 1, currentSlot)
                                    };
                                } else {
                                    // Moving to or from a cell spanning group so go to the first item
                                    return { type: _UI.ObjectType.item, index: nextGroup.startIndex };
                                }
                            } else {
                                return currentItem;
                            }
                        } else {
                            newItem.index += group.startIndex;
                            return newItem;
                        }
                    }

                    switch (that._adjustedKeyForRTL(pressedKey)) {
                        case Key.upArrow:
                        case Key.leftArrow:
                        case Key.downArrow:
                        case Key.rightArrow:
                            return handleArrowKeys();
                        default:
                            return exports._LayoutCommon.prototype._getAdjacentForPageKeys.call(that, currentItem, pressedKey);
                    }
                },

                hitTest: function _LayoutCommon_hitTest(x, y) {
                    var sizes = this._sizes,
                        result;

                    // Make the coordinates relative to grid layout's content box
                    x -= sizes.layoutOriginX;
                    y -= sizes.layoutOriginY;

                    var groupIndex = this._groupFromOffset(this._horizontal ? x : y),
                        group = this._groups[groupIndex];

                    // Make the coordinates relative to the margin box of the group's items container
                    if (this._horizontal) {
                        x -= group.offset;
                    } else {
                        y -= group.offset;
                    }
                    if (this._groupsEnabled) {
                        if (this._groupHeaderPosition === HeaderPosition.left) {
                            x -= sizes.headerContainerWidth;
                        } else {
                            // Headers above
                            y -= sizes.headerContainerHeight;
                        }
                    }

                    result = group.hitTest(x, y);
                    result.index += group.startIndex;
                    result.insertAfterIndex += group.startIndex;
                    return result;
                },

                // Animation cycle:
                //
                // Edits
                //  ---     UpdateTree        Realize
                // |   |      ---               /\/\
                // |   |     |   |             |    |
                // ------------------------------------------------------- Time
                //      |   |     |   |   |   |      |   |
                //       ---      |   |    ---        ---/\/\/\/\/\/\/\/\/
                //     setupAni   |   | layoutAni    endAni  (animations)
                //                 ---/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\
                //                layout    (outside realized range)
                //
                //
                // When there is a modification to the DataSource, the first thing that happens is setupAnimations is
                // called with the current tree. This allows us to cache the locations of the existing items.
                //
                // The next 3 steps have to be completely synchronous otherwise users will see intermediate states and
                // items will blink or jump between locations.
                //
                // ListView modifies the DOM tree. A container is added/removed from the group's itemsContainer for each
                // item added/removed to the group. The existing itemBoxes are shuffled between the different containers.
                // The itemBoxes for the removed items will be removed from the containers completely. Since the DOM tree
                // has been modified we have to apply a transform to position the itemBoxes at their original location. We
                // compare the new locations with the cached locations to figure out how far to translate the itemBoxes.
                // Also the removed items need to be placed back in the DOM without affecting layout (by using position
                // absolute) so that they also do not jump or blink.
                //
                // We only tranform and add back removed items for items which were on screen or are now on screen.
                //
                // Now the ListView can realize other items asynchronously. The items to realize are items which have been
                // inserted into the DataSource or items which are in the realization range because a removal has occurred
                // or the user has scroll slightly.
                //
                // During the realization pass the user may scroll. If they scroll to a range outside of the realization
                // range the items will just appear in the correct location without any animations. If they scroll to a
                // location within the old realization range we still have the items and they will animate correctly.
                //
                // During the realization pass another data source edit can occur. A realization pass is unable to run when
                // the tree and layout are out of sync. Otherwise it may try to request item at index X and get item at
                // index X + 1. This means that if another data source edit occurs before endAnimations is called we
                // restart the whole animation cycle. To group the animations between the two edits we do not reset the
                // caches of item box locations. We could add to it if there were items outside of the range however they
                // will only play half of the animation and will probably look just as ugly as not playing the animation at
                // all. This means setupAnimations will just be a no op in this scenario.
                //
                // This also shows that batching data source edits and only changing the data source when in loadingstate
                // "complete" is still a large performance win.
                //
                // Once the realization pass has finished ListView calls executeAnimations. This is where the layout
                // effectively fades out the removed items (and then removes them from the dom), moves the itemBoxes back
                // to translate(0,0), and fades in the inserted itemBoxes. ListView waits for the executeAnimations promise
                // to complete before allowing more data source edits to trigger another animation cycle.
                //
                // If a resize occurs during the animation cycle the animations will be canceled and items will jump to
                // their final positions.

                setupAnimations: function _LayoutCommon_setupAnimations() {
                    // This function is called after a data source change so that we can cache the locations
                    // of the realized items.

                    if (this._groups.length === 0) {
                        // No animations if we haven't measured before
                        this._resetAnimationCaches();
                        return;
                    }

                    if (Object.keys(this._cachedItemRecords).length) {
                        // Ignore the second call.
                        return;
                    }

                    this._site._writeProfilerMark("Animation:setupAnimations,StartTM");

                    var realizationRange = this._getRealizationRange();

                    var tree = this._site.tree;
                    var itemIndex = 0;
                    var horizontal = (this.orientation === "horizontal");
                    for (var i = 0, treeLength = tree.length; i < treeLength; i++) {
                        var groupBundle = tree[i];
                        var groupHasAtleastOneItemRealized = false;
                        var group = this._groups[i];
                        var groupIsCellSpanning = group instanceof Groups.CellSpanningGroup;
                        var groupOffset = (group ? group.offset : 0);

                        forEachContainer(groupBundle.itemsContainer, function (container, j) {
                            // Don't try to cache something outside of the realization range.
                            if (realizationRange.firstIndex <= itemIndex && realizationRange.lastIndex >= itemIndex) {
                                groupHasAtleastOneItemRealized = true;

                                if (!this._cachedItemRecords[itemIndex]) {
                                    var itemPosition = this._getItemPositionForAnimations(itemIndex, i, j);
                                    var row = itemPosition.row;
                                    var column = itemPosition.column;
                                    var left = itemPosition.left;
                                    var top = itemPosition.top;

                                    // Setting both old and new variables now in case layoutAnimations is called multiple times.
                                    this._cachedItemRecords[itemIndex] = {
                                        oldRow: row,
                                        oldColumn: column,
                                        oldLeft: left,
                                        oldTop: top,
                                        width: itemPosition.width,
                                        height: itemPosition.height,
                                        element: container,
                                        inCellSpanningGroup: groupIsCellSpanning
                                    };
                                }
                            }
                            itemIndex++;
                        }.bind(this));

                        if (groupHasAtleastOneItemRealized) {
                            var groupIndex = i;
                            if (!this._cachedHeaderRecords[groupIndex]) {
                                var headerPosition = this._getHeaderPositionForAnimations(groupIndex);
                                this._cachedHeaderRecords[groupIndex] = {
                                    oldLeft: headerPosition.left,
                                    oldTop: headerPosition.top,
                                    width: headerPosition.width,
                                    height: headerPosition.height,
                                    element: groupBundle.header,
                                };
                            }
                            if (!this._cachedGroupRecords[uniqueID(groupBundle.itemsContainer.element)]) {
                                this._cachedGroupRecords[uniqueID(groupBundle.itemsContainer.element)] = {
                                    oldLeft: horizontal ? groupOffset : 0,
                                    left: horizontal ? groupOffset : 0,
                                    oldTop: horizontal ? 0 : groupOffset,
                                    top: horizontal ? 0 : groupOffset,
                                    element: groupBundle.itemsContainer.element,
                                };
                            }
                        }
                    }

                    this._site._writeProfilerMark("Animation:setupAnimations,StopTM");
                },

                _layoutAnimations: function _LayoutCommon_layoutAnimations(modifiedItems, modifiedGroups) {
                    // This function is called after the DOM tree has been modified to match the data source.
                    // In this function we update the cached records and apply transforms to hide the modifications
                    // from the user. We will remove the transforms via animations in execute animation.

                    if (!Object.keys(this._cachedItemRecords).length &&
                        !Object.keys(this._cachedGroupRecords).length &&
                        !Object.keys(this._cachedHeaderRecords).length) {
                        return;
                    }

                    this._site._writeProfilerMark("Animation:layoutAnimation,StartTM");

                    this._updateAnimationCache(modifiedItems, modifiedGroups);

                    var realizationRange = this._getRealizationRange();

                    var tree = this._site.tree;
                    var itemIndex = 0;
                    var horizontal = (this.orientation === "horizontal");
                    for (var i = 0, treeLength = tree.length; i < treeLength; i++) {
                        var groupBundle = tree[i];
                        var group = this._groups[i];
                        var groupIsCellSpanning = group instanceof Groups.CellSpanningGroup;
                        var groupOffset = (group ? group.offset : 0);
                        var groupMovementX = 0;
                        var groupMovementY = 0;

                        var cachedGroupRecord = this._cachedGroupRecords[uniqueID(groupBundle.itemsContainer.element)];
                        if (cachedGroupRecord) {
                            if (horizontal) {
                                groupMovementX = cachedGroupRecord.oldLeft - groupOffset;
                            } else {
                                groupMovementY = cachedGroupRecord.oldTop - groupOffset;
                            }
                        }


                        forEachContainer(groupBundle.itemsContainer, function (container, j) {
                            // Don't try to cache something outside of the realization range.
                            if (realizationRange.firstIndex <= itemIndex && realizationRange.lastIndex >= itemIndex) {
                                var cachedItemRecord = this._cachedItemRecords[itemIndex];
                                if (cachedItemRecord) {
                                    var itemPosition = this._getItemPositionForAnimations(itemIndex, i, j);
                                    var row = itemPosition.row;
                                    var column = itemPosition.column;
                                    var left = itemPosition.left;
                                    var top = itemPosition.top;

                                    cachedItemRecord.inCellSpanningGroup = cachedItemRecord.inCellSpanningGroup || groupIsCellSpanning;

                                    // If the item has moved we need to update the cache and apply a transform to make it
                                    // appear like it has not moved yet.
                                    if (cachedItemRecord.oldRow !== row ||
                                        cachedItemRecord.oldColumn !== column ||
                                        cachedItemRecord.oldTop !== top ||
                                        cachedItemRecord.oldLeft !== left) {

                                        cachedItemRecord.row = row;
                                        cachedItemRecord.column = column;
                                        cachedItemRecord.left = left;
                                        cachedItemRecord.top = top;

                                        var xOffset = cachedItemRecord.oldLeft - cachedItemRecord.left - groupMovementX;
                                        var yOffset = cachedItemRecord.oldTop - cachedItemRecord.top - groupMovementY;
                                        xOffset = (this._site.rtl ? -1 : 1) * xOffset;

                                        cachedItemRecord.xOffset = xOffset;
                                        cachedItemRecord.yOffset = yOffset;
                                        if (xOffset !== 0 || yOffset !== 0) {
                                            var element = cachedItemRecord.element;
                                            cachedItemRecord.needsToResetTransform = true;
                                            element.style[transitionScriptName] = "";
                                            element.style[transformNames.scriptName] = "translate(" + xOffset + "px," + yOffset + "px)";
                                        }

                                        var itemsBlock = container.parentNode;
                                        if (_ElementUtilities.hasClass(itemsBlock, _Constants._itemsBlockClass)) {
                                            this._animatingItemsBlocks[uniqueID(itemsBlock)] = itemsBlock;
                                        }
                                    }

                                } else {
                                    // Treat items that came from outside of the realization range into the realization range
                                    // as a "Move" which means fade it in.
                                    this._cachedInsertedItemRecords[itemIndex] = container;
                                    container.style[transitionScriptName] = "";
                                    container.style.opacity = 0;
                                }
                            }

                            itemIndex++;
                        }.bind(this));

                        var groupIndex = i;
                        var cachedHeader = this._cachedHeaderRecords[groupIndex];
                        if (cachedHeader) {
                            var headerPosition = this._getHeaderPositionForAnimations(groupIndex);
                            // Note: If a group changes width we allow the header to immediately grow/shrink instead of
                            // animating it. However if the header is removed we stick the header to the last known size.
                            cachedHeader.height = headerPosition.height;
                            cachedHeader.width = headerPosition.width;
                            if (cachedHeader.oldLeft !== headerPosition.left ||
                                cachedHeader.oldTop !== headerPosition.top) {

                                cachedHeader.left = headerPosition.left;
                                cachedHeader.top = headerPosition.top;

                                var xOffset = cachedHeader.oldLeft - cachedHeader.left;
                                var yOffset = cachedHeader.oldTop - cachedHeader.top;
                                xOffset = (this._site.rtl ? -1 : 1) * xOffset;
                                if (xOffset !== 0 || yOffset !== 0) {
                                    cachedHeader.needsToResetTransform = true;
                                    var headerContainer = cachedHeader.element;
                                    headerContainer.style[transitionScriptName] = "";
                                    headerContainer.style[transformNames.scriptName] = "translate(" + xOffset + "px," + yOffset + "px)";
                                }
                            }
                        }

                        if (cachedGroupRecord) {
                            if ((horizontal && cachedGroupRecord.left !== groupOffset) ||
                                (!horizontal && cachedGroupRecord.top !== groupOffset)) {
                                var element = cachedGroupRecord.element;
                                if (groupMovementX === 0 && groupMovementY === 0) {
                                    if (cachedGroupRecord.needsToResetTransform) {
                                        cachedGroupRecord.needsToResetTransform = false;
                                        element.style[transformNames.scriptName] = "";
                                    }
                                } else {
                                    var groupOffsetX = (this._site.rtl ? -1 : 1) * groupMovementX,
                                        groupOffsetY = groupMovementY;
                                    cachedGroupRecord.needsToResetTransform = true;
                                    element.style[transitionScriptName] = "";
                                    element.style[transformNames.scriptName] = "translate(" + groupOffsetX + "px, " + groupOffsetY + "px)";
                                }
                            }
                        }
                    }

                    if (this._inListMode || this._itemsPerBar === 1) {
                        var itemsBlockKeys = Object.keys(this._animatingItemsBlocks);
                        for (var b = 0, blockKeys = itemsBlockKeys.length; b < blockKeys; b++) {
                            this._animatingItemsBlocks[itemsBlockKeys[b]].style.overflow = 'visible';
                        }
                    }

                    this._site._writeProfilerMark("Animation:layoutAnimation,StopTM");
                },

                executeAnimations: function _LayoutCommon_executeAnimations() {
                    // This function is called when we should perform an animation to reveal the true location of the items.
                    // We fade out removed items, fade in added items, and move items which need to be shifted. If they moved
                    // across columns we do a reflow animation.

                    var animationSignal = new _Signal();

                    // Only animate the items on screen.
                    this._filterInsertedElements();
                    this._filterMovedElements();
                    this._filterRemovedElements();

                    if (this._insertedElements.length === 0 && this._removedElements.length === 0 && this._itemMoveRecords.length === 0 && this._moveRecords.length === 0) {
                        // Nothing to animate.
                        this._resetAnimationCaches(true);
                        animationSignal.complete();
                        return animationSignal.promise;
                    }
                    this._animationsRunning = animationSignal.promise;

                    var slowAnimations = exports.Layout._debugAnimations || exports.Layout._slowAnimations;
                    var site = this._site;
                    var insertedElements = this._insertedElements;
                    var removedElements = this._removedElements;
                    var itemMoveRecords = this._itemMoveRecords;
                    var moveRecords = this._moveRecords;

                    var removeDelay = 0;
                    var moveDelay = 0;
                    var addDelay = 0;

                    var currentAnimationPromise = null;
                    var pendingTransitionPromises = [];

                    var hasMultisizeMove = false;
                    var hasReflow = false;
                    var minOffset = 0;
                    var maxOffset = 0;
                    var itemContainersToExpand = {};
                    var upOutDist = 0;
                    var downOutDist = 0;
                    var upInDist = 0;
                    var downInDist = 0;
                    var reflowItemRecords = [];
                    var horizontal = (this.orientation === "horizontal");
                    var oldReflowLayoutProperty = horizontal ? "oldColumn" : "oldRow",
                        reflowLayoutProperty = horizontal ? "column" : "row",
                        oldReflowLayoutPosition = horizontal ? "oldTop" : "oldLeft",
                        reflowLayoutPosition = horizontal ? "top" : "left";

                    var animatingItemsBlocks = this._animatingItemsBlocks;

                    for (var i = 0, len = itemMoveRecords.length; i < len; i++) {
                        var cachedItemRecord = itemMoveRecords[i];
                        if (cachedItemRecord.inCellSpanningGroup) {
                            hasMultisizeMove = true;
                            break;
                        }
                    }

                    var that = this;

                    function startAnimations() {
                        removePhase();
                        if (hasMultisizeMove) {
                            cellSpanningFadeOutMove();
                        } else {
                            if (that._itemsPerBar > 1) {
                                var maxDistance = that._itemsPerBar * that._sizes.containerCrossSize + that._getHeaderSizeContentAdjustment() +
                                    that._sizes.containerMargins[horizontal ? "top" : (site.rtl ? "right" : "left")] +
                                    (horizontal ? that._sizes.layoutOriginY : that._sizes.layoutOriginX);
                                for (var i = 0, len = itemMoveRecords.length; i < len; i++) {
                                    var cachedItemRecord = itemMoveRecords[i];
                                    if (cachedItemRecord[oldReflowLayoutProperty] > cachedItemRecord[reflowLayoutProperty]) {
                                        upOutDist = Math.max(upOutDist, cachedItemRecord[oldReflowLayoutPosition] + cachedItemRecord[horizontal ? "height" : "width"]);
                                        upInDist = Math.max(upInDist, maxDistance - cachedItemRecord[reflowLayoutPosition]);
                                        hasReflow = true;
                                        reflowItemRecords.push(cachedItemRecord);
                                    } else if (cachedItemRecord[oldReflowLayoutProperty] < cachedItemRecord[reflowLayoutProperty]) {
                                        downOutDist = Math.max(downOutDist, maxDistance - cachedItemRecord[oldReflowLayoutPosition]);
                                        downInDist = Math.max(downInDist, cachedItemRecord[reflowLayoutPosition] + cachedItemRecord[horizontal ? "height" : "width"]);
                                        reflowItemRecords.push(cachedItemRecord);
                                        hasReflow = true;
                                    }
                                }
                            }

                            if (site.rtl && !horizontal) {
                                upOutDist *= -1;
                                upInDist *= -1;
                                downOutDist *= -1;
                                downInDist *= -1;
                            }

                            if (hasReflow) {
                                reflowPhase(that._itemsPerBar);
                            } else {
                                directMovePhase();
                            }
                        }
                    }

                    if (exports.Layout._debugAnimations) {
                        _BaseUtils._requestAnimationFrame(function () {
                            startAnimations();
                        });
                    } else {
                        startAnimations();
                    }

                    function waitForNextPhase(nextPhaseCallback) {
                        currentAnimationPromise = Promise.join(pendingTransitionPromises);
                        currentAnimationPromise.done(function () {
                            pendingTransitionPromises = [];
                            // The success is called even if the animations are canceled due to the WinJS.UI.executeTransition
                            // API. To deal with that we check the animationSignal variable. If it is null the animations were
                            // canceled so we shouldn't continue.
                            if (animationSignal) {
                                if (exports.Layout._debugAnimations) {
                                    _BaseUtils._requestAnimationFrame(function () {
                                        nextPhaseCallback();
                                    });
                                } else {
                                    nextPhaseCallback();
                                }
                            }
                        });
                    }

                    function removePhase() {
                        if (removedElements.length) {
                            site._writeProfilerMark("Animation:setupRemoveAnimation,StartTM");

                            moveDelay += 60;
                            addDelay += 60;

                            var removeDuration = 120;
                            if (slowAnimations) {
                                removeDuration *= 10;
                            }

                            pendingTransitionPromises.push(_TransitionAnimation.executeTransition(removedElements,
                            [{
                                property: "opacity",
                                delay: removeDelay,
                                duration: removeDuration,
                                timing: "linear",
                                to: 0,
                                skipStylesReset: true
                            }]));

                            site._writeProfilerMark("Animation:setupRemoveAnimation,StopTM");
                        }
                    }

                    function cellSpanningFadeOutMove() {
                        site._writeProfilerMark("Animation:cellSpanningFadeOutMove,StartTM");

                        // For multisize items which move we fade out and then fade in (opacity 1->0->1)
                        var moveElements = [];
                        for (var i = 0, len = itemMoveRecords.length; i < len; i++) {
                            var cachedItemRecord = itemMoveRecords[i];
                            var container = cachedItemRecord.element;
                            moveElements.push(container);
                        }
                        // Including groups and headers.
                        for (var i = 0, len = moveRecords.length; i < len; i++) {
                            var cachedItemRecord = moveRecords[i];
                            var container = cachedItemRecord.element;
                            moveElements.push(container);
                        }

                        var fadeOutDuration = 120;
                        if (slowAnimations) {
                            fadeOutDuration *= 10;
                        }

                        pendingTransitionPromises.push(_TransitionAnimation.executeTransition(moveElements,
                        {
                            property: "opacity",
                            delay: removeDelay,
                            duration: fadeOutDuration,
                            timing: "linear",
                            to: 0
                        }));

                        waitForNextPhase(cellSpanningFadeInMove);
                        site._writeProfilerMark("Animation:cellSpanningFadeOutMove,StopTM");
                    }

                    function cellSpanningFadeInMove() {
                        site._writeProfilerMark("Animation:cellSpanningFadeInMove,StartTM");

                        addDelay = 0;

                        var moveElements = [];
                        // Move them to their final location.
                        for (var i = 0, len = itemMoveRecords.length; i < len; i++) {
                            var cachedItemRecord = itemMoveRecords[i];
                            var container = cachedItemRecord.element;
                            container.style[transformNames.scriptName] = "";
                            moveElements.push(container);
                        }
                        // Including groups and headers.
                        for (var i = 0, len = moveRecords.length; i < len; i++) {
                            var cachedItemRecord = moveRecords[i];
                            var container = cachedItemRecord.element;
                            container.style[transformNames.scriptName] = "";
                            moveElements.push(container);
                        }

                        var fadeInDuration = 120;
                        if (slowAnimations) {
                            fadeInDuration *= 10;
                        }

                        // For multisize items which move we fade out and then fade in (opacity 1->0->1)
                        pendingTransitionPromises.push(_TransitionAnimation.executeTransition(moveElements,
                        {
                            property: "opacity",
                            delay: addDelay,
                            duration: fadeInDuration,
                            timing: "linear",
                            to: 1
                        }));

                        site._writeProfilerMark("Animation:cellSpanningFadeInMove,StopTM");

                        addPhase();
                    }

                    function reflowPhase(itemsPerBar) {
                        site._writeProfilerMark("Animation:setupReflowAnimation,StartTM");

                        var itemContainersLastBarIndices = {};
                        for (var i = 0, len = reflowItemRecords.length; i < len; i++) {
                            var reflowItemRecord = reflowItemRecords[i];
                            var xOffset = reflowItemRecord.xOffset;
                            var yOffset = reflowItemRecord.yOffset;
                            if (reflowItemRecord[oldReflowLayoutProperty] > reflowItemRecord[reflowLayoutProperty]) {
                                if (horizontal) {
                                    yOffset -= upOutDist;
                                } else {
                                    xOffset -= upOutDist;
                                }
                            } else if (reflowItemRecord[oldReflowLayoutProperty] < reflowItemRecord[reflowLayoutProperty]) {
                                if (horizontal) {
                                    yOffset += downOutDist;
                                } else {
                                    xOffset += downOutDist;
                                }
                            }

                            var container = reflowItemRecord.element;

                            minOffset = Math.min(minOffset, horizontal ? xOffset : yOffset);
                            maxOffset = Math.max(maxOffset, horizontal ? xOffset : yOffset);
                            var itemsContainer = container.parentNode;
                            if (!_ElementUtilities.hasClass(itemsContainer, "win-itemscontainer")) {
                                itemsContainer = itemsContainer.parentNode;
                            }

                            // The itemscontainer element is always overflow:hidden for two reasons:
                            // 1) Better panning performance
                            // 2) When there is margin betweeen the itemscontainer and the surface elements, items that
                            //    reflow should not be visible while they travel long distances or overlap with headers.
                            // This introduces an issue when updateTree makes the itemscontainer smaller, but we need its size
                            // to remain the same size during the execution of the animation to avoid having some of the animated
                            // items being clipped. This is only an issue when items from the last column (in horizontal mode) or row
                            // (in vertical mode) of the group will reflow. Therefore, we change the padding so that the contents are larger,
                            // and then use margin to reverse the size change. We don't do this expansion when it is unnecessary because the
                            // layout/formatting caused by these style changes has significant cost when the group has thousands of items.
                            var lastBarIndex = itemContainersLastBarIndices[uniqueID(itemsContainer)];
                            if (!lastBarIndex) {
                                var count = getItemsContainerLength(getItemsContainerTree(itemsContainer, site.tree));
                                itemContainersLastBarIndices[uniqueID(itemsContainer)] = lastBarIndex = Math.ceil(count / itemsPerBar) - 1;
                            }
                            if (reflowItemRecords[i][horizontal ? "column" : "row"] === lastBarIndex) {
                                itemContainersToExpand[uniqueID(itemsContainer)] = itemsContainer;
                            }

                            var reflowDuration = 80;
                            if (slowAnimations) {
                                reflowDuration *= 10;
                            }

                            pendingTransitionPromises.push(_TransitionAnimation.executeTransition(container,
                            {
                                property: transformNames.cssName,
                                delay: moveDelay,
                                duration: reflowDuration,
                                timing: "cubic-bezier(0.1, 0.9, 0.2, 1)",
                                to: "translate(" + xOffset + "px," + yOffset + "px)"
                            }));
                        }

                        var itemContainerKeys = Object.keys(itemContainersToExpand);
                        for (var i = 0, len = itemContainerKeys.length; i < len; i++) {
                            var itemContainer = itemContainersToExpand[itemContainerKeys[i]];
                            if (site.rtl && horizontal) {
                                itemContainer.style.paddingLeft = (-1 * minOffset) + 'px';
                                itemContainer.style.marginLeft = minOffset + 'px';
                            } else {
                                itemContainer.style[horizontal ? "paddingRight" : "paddingBottom"] = maxOffset + 'px';
                                itemContainer.style[horizontal ? "marginRight" : "marginBottom"] = '-' + maxOffset + 'px';
                            }
                        }
                        var itemsBlockKeys = Object.keys(animatingItemsBlocks);
                        for (var i = 0, len = itemsBlockKeys.length; i < len; i++) {
                            animatingItemsBlocks[itemsBlockKeys[i]].classList.add(_Constants._clipClass);
                        }

                        waitForNextPhase(afterReflowPhase);

                        site._writeProfilerMark("Animation:setupReflowAnimation,StopTM");
                    }

                    function cleanupItemsContainers() {
                        // Reset the styles used to obtain overflow-y: hidden overflow-x: visible.
                        var itemContainerKeys = Object.keys(itemContainersToExpand);
                        for (var i = 0, len = itemContainerKeys.length; i < len; i++) {
                            var itemContainer = itemContainersToExpand[itemContainerKeys[i]];
                            if (site.rtl && horizontal) {
                                itemContainer.style.paddingLeft = '';
                                itemContainer.style.marginLeft = '';
                            } else {
                                itemContainer.style[horizontal ? "paddingRight" : "paddingBottom"] = '';
                                itemContainer.style[horizontal ? "marginRight" : "marginBottom"] = '';
                            }
                        }
                        itemContainersToExpand = {};

                        var itemsBlockKeys = Object.keys(animatingItemsBlocks);
                        for (var i = 0, len = itemsBlockKeys.length; i < len; i++) {
                            var itemsBlock = animatingItemsBlocks[itemsBlockKeys[i]];
                            itemsBlock.style.overflow = '';
                            itemsBlock.classList.remove(_Constants._clipClass);
                        }
                    }

                    function afterReflowPhase() {
                        site._writeProfilerMark("Animation:prepareReflowedItems,StartTM");

                        // Position the items at the edge ready to slide in.
                        for (var i = 0, len = reflowItemRecords.length; i < len; i++) {
                            var reflowItemRecord = reflowItemRecords[i];
                            var xOffset = 0,
                                yOffset = 0;
                            if (reflowItemRecord[oldReflowLayoutProperty] > reflowItemRecord[reflowLayoutProperty]) {
                                if (horizontal) {
                                    yOffset = upInDist;
                                } else {
                                    xOffset = upInDist;
                                }
                            } else if (reflowItemRecord[oldReflowLayoutProperty] < reflowItemRecord[reflowLayoutProperty]) {
                                if (horizontal) {
                                    yOffset = -1 * downInDist;
                                } else {
                                    xOffset = -1 * downInDist;
                                }
                            }
                            reflowItemRecord.element.style[transitionScriptName] = "";
                            reflowItemRecord.element.style[transformNames.scriptName] = "translate(" + xOffset + "px," + yOffset + "px)";
                        }

                        site._writeProfilerMark("Animation:prepareReflowedItems,StopTM");

                        if (exports.Layout._debugAnimations) {
                            _BaseUtils._requestAnimationFrame(function () {
                                directMovePhase(true);
                            });
                        } else {
                            directMovePhase(true);
                        }
                    }

                    function directMovePhase(fastMode) {
                        // For groups and items which move we transition them from transform: translate(Xpx,Ypx) to translate(0px,0px).
                        var duration = 200;
                        if (fastMode) {
                            duration = 150;
                            moveDelay = 0;
                            addDelay = 0;
                        }

                        if (slowAnimations) {
                            duration *= 10;
                        }

                        if (itemMoveRecords.length > 0 || moveRecords.length > 0) {
                            site._writeProfilerMark("Animation:setupMoveAnimation,StartTM");

                            var moveElements = [];
                            for (var i = 0, len = moveRecords.length; i < len; i++) {
                                var container = moveRecords[i].element;
                                moveElements.push(container);
                            }
                            for (var i = 0, len = itemMoveRecords.length; i < len; i++) {
                                var container = itemMoveRecords[i].element;
                                moveElements.push(container);
                            }
                            pendingTransitionPromises.push(_TransitionAnimation.executeTransition(moveElements,
                            {
                                property: transformNames.cssName,
                                delay: moveDelay,
                                duration: duration,
                                timing: "cubic-bezier(0.1, 0.9, 0.2, 1)",
                                to: ""
                            }));

                            addDelay += 80;

                            site._writeProfilerMark("Animation:setupMoveAnimation,StopTM");
                        }

                        addPhase();
                    }

                    function addPhase() {
                        if (insertedElements.length > 0) {
                            site._writeProfilerMark("Animation:setupInsertAnimation,StartTM");

                            var addDuration = 120;
                            if (slowAnimations) {
                                addDuration *= 10;
                            }

                            pendingTransitionPromises.push(_TransitionAnimation.executeTransition(insertedElements,
                            [{
                                property: "opacity",
                                delay: addDelay,
                                duration: addDuration,
                                timing: "linear",
                                to: 1
                            }]));

                            site._writeProfilerMark("Animation:setupInsertAnimation,StopTM");
                        }

                        waitForNextPhase(completePhase);
                    }
                    function completePhase() {
                        site._writeProfilerMark("Animation:cleanupAnimations,StartTM");

                        cleanupItemsContainers();

                        for (var i = 0, len = removedElements.length; i < len; i++) {
                            var container = removedElements[i];
                            if (container.parentNode) {
                                _Dispose._disposeElement(container);
                                container.parentNode.removeChild(container);
                            }
                        }

                        site._writeProfilerMark("Animation:cleanupAnimations,StopTM");

                        that._animationsRunning = null;
                        animationSignal.complete();
                    }
                    this._resetAnimationCaches(true);

                    // The PVL animation library completes sucessfully even if you cancel an animation.
                    // If the animation promise passed to layout is canceled we should cancel the PVL animations and
                    // set a marker for them to be ignored.
                    animationSignal.promise.then(null, function () {
                        // Since it was canceled make sure we still clean up the styles.
                        cleanupItemsContainers();
                        for (var i = 0, len = moveRecords.length; i < len; i++) {
                            var container = moveRecords[i].element;
                            container.style[transformNames.scriptName] = '';
                            container.style.opacity = 1;
                        }
                        for (var i = 0, len = itemMoveRecords.length; i < len; i++) {
                            var container = itemMoveRecords[i].element;
                            container.style[transformNames.scriptName] = '';
                            container.style.opacity = 1;
                        }
                        for (var i = 0, len = insertedElements.length; i < len; i++) {
                            insertedElements[i].style.opacity = 1;
                        }
                        for (var i = 0, len = removedElements.length; i < len; i++) {
                            var container = removedElements[i];
                            if (container.parentNode) {
                                _Dispose._disposeElement(container);
                                container.parentNode.removeChild(container);
                            }
                        }

                        this._animationsRunning = null;
                        animationSignal = null;
                        currentAnimationPromise && currentAnimationPromise.cancel();

                    }.bind(this));

                    return animationSignal.promise;
                },

                dragOver: function _LayoutCommon_dragOver(x, y, dragInfo) {
                    // The coordinates passed to dragOver should be in ListView's viewport space. 0,0 should be the top left corner of the viewport's padding.
                    var indicesAffected = this.hitTest(x, y),
                        groupAffected = (this._groups ? this._site.groupIndexFromItemIndex(indicesAffected.index) : 0),
                        itemsContainer = this._site.tree[groupAffected].itemsContainer,
                        itemsCount = getItemsContainerLength(itemsContainer),
                        indexOffset = (this._groups ? this._groups[groupAffected].startIndex : 0),
                        visibleRange = this._getVisibleRange();

                    indicesAffected.index -= indexOffset;
                    indicesAffected.insertAfterIndex -= indexOffset;
                    visibleRange.firstIndex = Math.max(visibleRange.firstIndex - indexOffset - 1, 0);
                    visibleRange.lastIndex = Math.min(visibleRange.lastIndex - indexOffset + 1, itemsCount);
                    var indexAfter = Math.max(Math.min(itemsCount - 1, indicesAffected.insertAfterIndex), -1),
                        indexBefore = Math.min(indexAfter + 1, itemsCount);

                    if (dragInfo) {
                        for (var i = indexAfter; i >= visibleRange.firstIndex; i--) {
                            if (!dragInfo._isIncluded(i + indexOffset)) {
                                indexAfter = i;
                                break;
                            } else if (i === visibleRange.firstIndex) {
                                indexAfter = -1;
                            }
                        }

                        for (var i = indexBefore; i < visibleRange.lastIndex; i++) {
                            if (!dragInfo._isIncluded(i + indexOffset)) {
                                indexBefore = i;
                                break;
                            } else if (i === (visibleRange.lastIndex - 1)) {
                                indexBefore = itemsCount;
                            }
                        }
                    }

                    var elementBefore = containerFromIndex(itemsContainer, indexBefore),
                        elementAfter = containerFromIndex(itemsContainer, indexAfter);

                    if (this._animatedDragItems) {
                        for (var i = 0, len = this._animatedDragItems.length; i < len; i++) {
                            var item = this._animatedDragItems[i];
                            if (item) {
                                item.style[transitionScriptName] = this._site.animationsDisabled ? "" : dragBetweenTransition;
                                item.style[transformNames.scriptName] = "";
                            }
                        }
                    }
                    this._animatedDragItems = [];
                    var horizontal = this.orientation === "horizontal",
                        inListMode = this._inListMode || this._itemsPerBar === 1;
                    if (this._groups && this._groups[groupAffected] instanceof Groups.CellSpanningGroup) {
                        inListMode = this._groups[groupAffected]._slotsPerColumn === 1;
                    }
                    var horizontalTransform = 0,
                        verticalTransform = 0;
                    // In general, items should slide in the direction perpendicular to the layout's orientation.
                    // In a horizontal layout, items are laid out top to bottom, left to right. For any two neighboring items in this layout, we want to move the first item up and the second down
                    // to denote that any inserted item would go between those two.
                    // Similarily, vertical layout should have the first item move left and the second move right.
                    // List layout is a special case. A horizontal list layout can only lay things out left to right, so it should slide the two items left and right like a vertical grid.
                    // A vertical list can only lay things out top to bottom, so it should slide items up and down like a horizontal grid.
                    // In other words: Apply horizontal transformations if we're a vertical grid or horizontal list, otherwise use vertical transformations.
                    if ((!horizontal && !inListMode) || (horizontal && inListMode)) {
                        horizontalTransform = this._site.rtl ? -dragBetweenDistance : dragBetweenDistance;
                    } else {
                        verticalTransform = dragBetweenDistance;
                    }
                    if (elementBefore) {
                        elementBefore.style[transitionScriptName] = this._site.animationsDisabled ? "" : dragBetweenTransition;
                        elementBefore.style[transformNames.scriptName] = "translate(" + horizontalTransform + "px, " + verticalTransform + "px)";
                        this._animatedDragItems.push(elementBefore);
                    }
                    if (elementAfter) {
                        elementAfter.style[transitionScriptName] = this._site.animationsDisabled ? "" : dragBetweenTransition;
                        elementAfter.style[transformNames.scriptName] = "translate(" + (-horizontalTransform) + "px, -" + verticalTransform + "px)";
                        this._animatedDragItems.push(elementAfter);
                    }
                },

                dragLeave: function _LayoutCommon_dragLeave() {
                    if (this._animatedDragItems) {
                        for (var i = 0, len = this._animatedDragItems.length; i < len; i++) {
                            this._animatedDragItems[i].style[transitionScriptName] = this._site.animationsDisabled ? "" : dragBetweenTransition;
                            this._animatedDragItems[i].style[transformNames.scriptName] = "";
                        }
                    }
                    this._animatedDragItems = [];
                },

                // Private methods

                _setMaxRowsOrColumns: function _LayoutCommon_setMaxRowsOrColumns(value) {
                    if (value === this._maxRowsOrColumns || this._inListMode) {
                        return;
                    }

                    // If container size is unavailable then we do not need to compute itemsPerBar
                    // as it will be computed along with the container size.
                    if (this._sizes && this._sizes.containerSizeLoaded) {
                        this._itemsPerBar = Math.floor(this._sizes.maxItemsContainerContentSize / this._sizes.containerCrossSize);
                        if (value) {
                            this._itemsPerBar = Math.min(this._itemsPerBar, value);
                        }
                        this._itemsPerBar = Math.max(1, this._itemsPerBar);
                    }
                    this._maxRowsOrColumns = value;

                    this._invalidateLayout();
                },

                _getItemPosition: function _LayoutCommon_getItemPosition(itemIndex) {
                    if (this._groupsEnabled) {
                        var groupIndex = Math.min(this._groups.length - 1, this._site.groupIndexFromItemIndex(itemIndex)),
                            group = this._groups[groupIndex],
                            itemOfGroupIndex = itemIndex - group.startIndex;
                        return this._getItemPositionForAnimations(itemIndex, groupIndex, itemOfGroupIndex);
                    } else {
                        return this._getItemPositionForAnimations(itemIndex, 0, itemIndex);
                    }
                },

                _getRealizationRange: function _LayoutCommon_getRealizationRange() {
                    var realizedRange = this._site.realizedRange;
                    return {
                        firstIndex: this._firstItemFromRange(realizedRange.firstPixel),
                        lastIndex: this._lastItemFromRange(realizedRange.lastPixel)
                    };
                },

                _getVisibleRange: function _LayoutCommon_getVisibleRange() {
                    var visibleRange = this._site.visibleRange;
                    return {
                        firstIndex: this._firstItemFromRange(visibleRange.firstPixel),
                        lastIndex: this._lastItemFromRange(visibleRange.lastPixel)
                    };
                },

                _resetAnimationCaches: function _LayoutCommon_resetAnimationCaches(skipReset) {
                    if (!skipReset) {
                        // Caches with move transforms:
                        this._resetStylesForRecords(this._cachedGroupRecords);
                        this._resetStylesForRecords(this._cachedItemRecords);
                        this._resetStylesForRecords(this._cachedHeaderRecords);

                        // Caches with insert transforms:
                        this._resetStylesForInsertedRecords(this._cachedInsertedItemRecords);
                        this._resetStylesForInsertedRecords(this._cachedInsertedHeaderRecords);

                        // Caches with insert transforms:
                        this._resetStylesForRemovedRecords(this._cachedRemovedItems);
                        this._resetStylesForRemovedRecords(this._cachedRemovedHeaders);

                        var itemsBlockKeys = Object.keys(this._animatingItemsBlocks);
                        for (var i = 0, len = itemsBlockKeys.length; i < len; i++) {
                            var itemsBlock = this._animatingItemsBlocks[itemsBlockKeys[i]];
                            itemsBlock.style.overflow = '';
                            itemsBlock.classList.remove(_Constants._clipClass);
                        }
                    }

                    this._cachedGroupRecords = {};
                    this._cachedItemRecords = {};
                    this._cachedHeaderRecords = {};

                    this._cachedInsertedItemRecords = {};
                    this._cachedInsertedHeaderRecords = {};

                    this._cachedRemovedItems = [];
                    this._cachedRemovedHeaders = [];

                    this._animatingItemsBlocks = {};
                },

                _cacheRemovedElements: function _LayoutCommon_cacheRemovedElements(modifiedElements, cachedRecords, cachedInsertedRecords, removedElements, areHeaders) {
                    var leftStr = "left";
                    if (this._site.rtl) {
                        leftStr = "right";
                    }
                    // Offset between the container's content box and its margin box
                    var outerX, outerY;
                    if (areHeaders) {
                        outerX = this._sizes.headerContainerOuterX;
                        outerY = this._sizes.headerContainerOuterY;
                    } else {
                        outerX = this._sizes.containerMargins[leftStr];
                        outerY = this._sizes.containerMargins.top;
                    }

                    // Cache the removed boxes and place them back in the DOM with position absolute
                    // so that they do not appear like they have moved.
                    for (var i = 0, len = modifiedElements.length; i < len; i++) {
                        var modifiedElementLookup = modifiedElements[i];
                        if (modifiedElementLookup.newIndex === -1) {
                            var container = modifiedElementLookup.element;
                            var cachedItemRecord = cachedRecords[modifiedElementLookup.oldIndex];
                            if (cachedItemRecord) {
                                cachedItemRecord.element = container;
                                // This item can no longer be a moved item.
                                delete cachedRecords[modifiedElementLookup.oldIndex];
                                container.style.position = "absolute";
                                container.style[transitionScriptName] = "";
                                container.style.top = cachedItemRecord.oldTop - outerY + "px";
                                container.style[leftStr] = cachedItemRecord.oldLeft - outerX + "px";
                                container.style.width = cachedItemRecord.width + "px";
                                container.style.height = cachedItemRecord.height + "px";
                                container.style[transformNames.scriptName] = "";
                                this._site.surface.appendChild(container);
                                removedElements.push(cachedItemRecord);
                            }
                            if (cachedInsertedRecords[modifiedElementLookup.oldIndex]) {
                                delete cachedInsertedRecords[modifiedElementLookup.oldIndex];
                            }
                        }
                    }
                },
                _cacheInsertedElements: function _LayoutCommon_cacheInsertedItems(modifiedElements, cachedInsertedRecords, cachedRecords) {
                    var newCachedInsertedRecords = {};

                    for (var i = 0, len = modifiedElements.length; i < len; i++) {
                        var modifiedElementLookup = modifiedElements[i];
                        var wasInserted = cachedInsertedRecords[modifiedElementLookup.oldIndex];
                        if (wasInserted) {
                            delete cachedInsertedRecords[modifiedElementLookup.oldIndex];
                        }

                        if (wasInserted || modifiedElementLookup.oldIndex === -1 || modifiedElementLookup.moved) {
                            var cachedRecord = cachedRecords[modifiedElementLookup.newIndex];
                            if (cachedRecord) {
                                delete cachedRecords[modifiedElementLookup.newIndex];
                            }

                            var modifiedElement = modifiedElementLookup.element;
                            newCachedInsertedRecords[modifiedElementLookup.newIndex] = modifiedElement;
                            modifiedElement.style[transitionScriptName] = "";
                            modifiedElement.style[transformNames.scriptName] = "";
                            modifiedElement.style.opacity = 0;
                        }
                    }

                    var keys = Object.keys(cachedInsertedRecords);
                    for (var i = 0, len = keys.length; i < len; i++) {
                        newCachedInsertedRecords[keys[i]] = cachedInsertedRecords[keys[i]];
                    }

                    return newCachedInsertedRecords;
                },
                _resetStylesForRecords: function _LayoutCommon_resetStylesForRecords(recordsHash) {
                    var recordKeys = Object.keys(recordsHash);
                    for (var i = 0, len = recordKeys.length; i < len; i++) {
                        var record = recordsHash[recordKeys[i]];
                        if (record.needsToResetTransform) {
                            record.element.style[transformNames.scriptName] = "";
                            record.needsToResetTransform = false;
                        }
                    }
                },
                _resetStylesForInsertedRecords: function _LayoutCommon_resetStylesForInsertedRecords(insertedRecords) {
                    var insertedRecordKeys = Object.keys(insertedRecords);
                    for (var i = 0, len = insertedRecordKeys.length; i < len; i++) {
                        var insertedElement = insertedRecords[insertedRecordKeys[i]];
                        insertedElement.style.opacity = 1;
                    }
                },
                _resetStylesForRemovedRecords: function _LayoutCommon_resetStylesForRemovedRecords(removedElements) {
                    for (var i = 0, len = removedElements.length; i < len; i++) {
                        var container = removedElements[i].element;
                        if (container.parentNode) {
                            _Dispose._disposeElement(container);
                            container.parentNode.removeChild(container);
                        }
                    }
                },
                _updateAnimationCache: function _LayoutCommon_updateAnimationCache(modifiedItems, modifiedGroups) {
                    // ItemBoxes can change containers so we have to start them back without transforms
                    // and then update them again. ItemsContainers don't need to do this.
                    this._resetStylesForRecords(this._cachedItemRecords);
                    this._resetStylesForRecords(this._cachedHeaderRecords);
                    // Go through all the inserted records and reset their insert transforms.
                    this._resetStylesForInsertedRecords(this._cachedInsertedItemRecords);
                    this._resetStylesForInsertedRecords(this._cachedInsertedHeaderRecords);

                    var existingContainers = {};
                    var realizationRange = this._getRealizationRange();
                    var tree = this._site.tree;
                    for (var i = 0, itemIndex = 0, treeLength = tree.length; i < treeLength; i++) {
                        forEachContainer(tree[i].itemsContainer, function (container) {
                            if (realizationRange.firstIndex <= itemIndex && realizationRange.lastIndex >= itemIndex) {
                                existingContainers[uniqueID(container)] = true;
                            }
                            itemIndex++;
                        });
                    }

                    // Update the indicies before the insert since insert needs the new containers.
                    function updateIndicies(modifiedElements, cachedRecords) {
                        var updatedCachedRecords = {};

                        for (var i = 0, len = modifiedElements.length; i < len; i++) {
                            var modifiedElementLookup = modifiedElements[i];
                            var cachedRecord = cachedRecords[modifiedElementLookup.oldIndex];
                            if (cachedRecord) {
                                updatedCachedRecords[modifiedElementLookup.newIndex] = cachedRecord;
                                cachedRecord.element = modifiedElementLookup.element;
                                delete cachedRecords[modifiedElementLookup.oldIndex];
                            }
                        }
                        var cachedRecordKeys = Object.keys(cachedRecords);
                        for (var i = 0, len = cachedRecordKeys.length; i < len; i++) {
                            var key = cachedRecordKeys[i],
                                record = cachedRecords[key];
                            // We need to filter out containers which were removed from the DOM. If container's item
                            // wasn't realized container can be removed without adding record to modifiedItems.
                            if (!record.element || existingContainers[uniqueID(record.element)]) {
                                updatedCachedRecords[key] = record;
                            }
                        }
                        return updatedCachedRecords;
                    }

                    this._cachedItemRecords = updateIndicies(modifiedItems, this._cachedItemRecords);
                    this._cachedHeaderRecords = updateIndicies(modifiedGroups, this._cachedHeaderRecords);

                    this._cachedInsertedItemRecords = this._cacheInsertedElements(modifiedItems, this._cachedInsertedItemRecords, this._cachedItemRecords);
                    this._cachedInsertedHeaderRecords = this._cacheInsertedElements(modifiedGroups, this._cachedInsertedHeaderRecords, this._cachedHeaderRecords);
                },
                _filterRemovedElements: function _LayoutCommon_filterRemovedElements() {
                    this._removedElements = [];

                    if (this._site.animationsDisabled) {
                        this._resetStylesForRemovedRecords(this._cachedRemovedItems);
                        this._resetStylesForRemovedRecords(this._cachedRemovedHeaders);
                        return;
                    }

                    var that = this;
                    var oldLeftStr = this.orientation === "horizontal" ? "oldLeft" : "oldTop";
                    var widthStr = this.orientation === "horizontal" ? "width" : "height";

                    var visibleFirstPixel = this._site.scrollbarPos;
                    var visibleLastPixel = visibleFirstPixel + this._site.viewportSize[widthStr] - 1;

                    function filterRemovedElements(removedRecordArray, removedElementsArray) {
                        for (var i = 0, len = removedRecordArray.length; i < len; i++) {
                            var removedItem = removedRecordArray[i];
                            var container = removedItem.element;
                            if (removedItem[oldLeftStr] + removedItem[widthStr] - 1 < visibleFirstPixel || removedItem[oldLeftStr] > visibleLastPixel || !that._site.viewport.contains(container)) {
                                if (container.parentNode) {
                                    _Dispose._disposeElement(container);
                                    container.parentNode.removeChild(container);
                                }
                            } else {
                                removedElementsArray.push(container);
                            }
                        }
                    }

                    filterRemovedElements(this._cachedRemovedItems, this._removedElements);
                    filterRemovedElements(this._cachedRemovedHeaders, this._removedElements);
                },

                _filterInsertedElements: function _LayoutCommon_filterInsertedElements() {
                    this._insertedElements = [];
                    if (this._site.animationsDisabled) {
                        this._resetStylesForInsertedRecords(this._cachedInsertedItemRecords);
                        this._resetStylesForInsertedRecords(this._cachedInsertedHeaderRecords);
                        return;
                    }

                    var that = this;
                    var visibleRange = this._getVisibleRange();

                    function filterInsertedElements(cachedInsertedRecords, insertedElementsArray) {
                        var recordKeys = Object.keys(cachedInsertedRecords);
                        for (var i = 0, len = recordKeys.length; i < len; i++) {
                            var itemIndex = recordKeys[i];
                            var insertedRecord = cachedInsertedRecords[itemIndex];
                            if (itemIndex < visibleRange.firstIndex || itemIndex > visibleRange.lastIndex || that._site.viewport.contains(insertedRecord.element)) {
                                insertedRecord.style.opacity = 1;
                            } else {
                                insertedElementsArray.push(insertedRecord);
                            }
                        }
                    }

                    filterInsertedElements(this._cachedInsertedItemRecords, this._insertedElements);
                    filterInsertedElements(this._cachedInsertedHeaderRecords, this._insertedElements);
                },

                _filterMovedElements: function _LayoutCommon_filterMovedElements() {
                    var that = this;

                    // This filters all the items and groups down which could have moved to just the items on screen.
                    // The items which are not going to animate are immediately shown in their correct final location.
                    var oldLeftStr = this.orientation === "horizontal" ? "oldLeft" : "oldTop";
                    var leftStr = this.orientation === "horizontal" ? "left" : "top";
                    var widthStr = this.orientation === "horizontal" ? "width" : "height";

                    var realizationRange = this._getRealizationRange();
                    var visibleFirstPixel = this._site.scrollbarPos;
                    var visibleLastPixel = visibleFirstPixel + this._site.viewportSize[widthStr] - 1;

                    // ItemMove can reflow across column or fade in/out due to multisize.
                    this._itemMoveRecords = [];
                    this._moveRecords = [];

                    if (!this._site.animationsDisabled) {
                        var tree = this._site.tree;
                        var itemIndex = 0;
                        for (var i = 0, treeLength = tree.length; i < treeLength; i++) {
                            var groupBundle = tree[i];
                            var groupHasItemToAnimate = false;

                            forEachContainer(groupBundle.itemsContainer, function () {
                                if (realizationRange.firstIndex <= itemIndex && realizationRange.lastIndex >= itemIndex) {
                                    var cachedItemRecord = this._cachedItemRecords[itemIndex];
                                    if (cachedItemRecord) {
                                        var shouldAnimate = ((cachedItemRecord[oldLeftStr] + cachedItemRecord[widthStr] - 1 >= visibleFirstPixel && cachedItemRecord[oldLeftStr] <= visibleLastPixel) ||
                                                            (cachedItemRecord[leftStr] + cachedItemRecord[widthStr] - 1 >= visibleFirstPixel && cachedItemRecord[leftStr] <= visibleLastPixel)) &&
                                                            that._site.viewport.contains(cachedItemRecord.element);
                                        if (shouldAnimate) {
                                            groupHasItemToAnimate = true;
                                            if (cachedItemRecord.needsToResetTransform) {
                                                this._itemMoveRecords.push(cachedItemRecord);
                                                delete this._cachedItemRecords[itemIndex];
                                            }
                                        }
                                    }
                                }
                                itemIndex++;
                            }.bind(this));

                            var groupIndex = i;
                            var cachedHeaderRecord = this._cachedHeaderRecords[groupIndex];
                            if (cachedHeaderRecord) {
                                if (groupHasItemToAnimate && cachedHeaderRecord.needsToResetTransform) {
                                    this._moveRecords.push(cachedHeaderRecord);
                                    delete this._cachedHeaderRecords[groupIndex];
                                }
                            }

                            var cachedGroupRecord = this._cachedGroupRecords[uniqueID(groupBundle.itemsContainer.element)];
                            if (cachedGroupRecord) {
                                if (groupHasItemToAnimate && cachedGroupRecord.needsToResetTransform) {
                                    this._moveRecords.push(cachedGroupRecord);
                                    delete this._cachedGroupRecords[uniqueID(groupBundle.itemsContainer.element)];
                                }
                            }
                        }
                    }

                    // Reset transform for groups and items that were never on screen.
                    this._resetStylesForRecords(this._cachedGroupRecords);
                    this._resetStylesForRecords(this._cachedItemRecords);
                    this._resetStylesForRecords(this._cachedHeaderRecords);
                },

                _getItemPositionForAnimations: function _LayoutCommon_getItemPositionForAnimations(itemIndex, groupIndex, itemOfGroupIndex) {
                    // Top/Left are used to know if the item has moved and also used to position the item if removed.
                    // Row/Column are used to know if a reflow animation should occur
                    // Height/Width are used when positioning a removed item without impacting layout.
                    // The returned rectangle refers to the win-container's border/padding/content box. Coordinates
                    // are relative to the viewport.
                    var group = this._groups[groupIndex];
                    var itemPosition = group.getItemPositionForAnimations(itemOfGroupIndex);
                    var groupOffset = (this._groups[groupIndex] ? this._groups[groupIndex].offset : 0);
                    var headerWidth = (this._groupsEnabled && this._groupHeaderPosition === HeaderPosition.left ? this._sizes.headerContainerWidth : 0);
                    var headerHeight = (this._groupsEnabled && this._groupHeaderPosition === HeaderPosition.top ? this._sizes.headerContainerHeight : 0);

                    itemPosition.left += this._sizes.layoutOriginX + headerWidth + this._sizes.itemsContainerOuterX;
                    itemPosition.top += this._sizes.layoutOriginY + headerHeight + this._sizes.itemsContainerOuterY;
                    itemPosition[this._horizontal ? "left" : "top"] += groupOffset;
                    return itemPosition;
                },

                _getHeaderPositionForAnimations: function (groupIndex) {
                    // Top/Left are used to know if the item has moved.
                    // Height/Width are used when positioning a removed item without impacting layout.
                    // The returned rectangle refers to the header container's content box. Coordinates
                    // are relative to the viewport.

                    var headerPosition;

                    if (this._groupsEnabled) {
                        var width = this._sizes.headerContainerWidth - this._sizes.headerContainerOuterWidth,
                            height = this._sizes.headerContainerHeight - this._sizes.headerContainerOuterHeight;
                        if (this._groupHeaderPosition === HeaderPosition.left && !this._horizontal) {
                            height = this._groups[groupIndex].getItemsContainerSize() - this._sizes.headerContainerOuterHeight;
                        } else if (this._groupHeaderPosition === HeaderPosition.top && this._horizontal) {
                            width = this._groups[groupIndex].getItemsContainerSize() - this._sizes.headerContainerOuterWidth;
                        }

                        var offsetX = this._horizontal ? this._groups[groupIndex].offset : 0,
                            offsetY = this._horizontal ? 0 : this._groups[groupIndex].offset;
                        headerPosition = {
                            top: this._sizes.layoutOriginY + offsetY + this._sizes.headerContainerOuterY,
                            left: this._sizes.layoutOriginX + offsetX + this._sizes.headerContainerOuterX,
                            height: height,
                            width: width
                        };
                    } else {
                        headerPosition = {
                            top: 0,
                            left: 0,
                            height: 0,
                            width: 0
                        };
                    }
                    return headerPosition;
                },

                _rangeContainsItems: function _LayoutCommon_rangeContainsItems(firstPixel, lastPixel) {
                    if (this._groups.length === 0) {
                        return false;
                    } else {
                        var lastGroup = this._groups[this._groups.length - 1],
                            lastPixelOfLayout = this._sizes.layoutOrigin + lastGroup.offset + this._getGroupSize(lastGroup) - 1;

                        return lastPixel >= 0 && firstPixel <= lastPixelOfLayout;
                    }
                },

                _itemFromOffset: function _LayoutCommon_itemFromOffset(offset, options) {
                    // supported options are:
                    // - wholeItem: when set to true the fully visible item is returned
                    // - last: if 1 the last item is returned. if 0 the first
                    var that = this;
                    if (this._groups.length === 0) {
                        return 0;
                    }

                    function assignItemMargins(offset) {
                        if (!options.wholeItem) {
                            // This makes it such that if a container's margin is on screen but all of its
                            // content is off screen then we'll treat the container as being off screen.
                            var marginPropLast = (that._horizontal ? (that._site.rtl ? "right" : "left") : "top"),
                                marginPropFirst = (that._horizontal ? (that._site.rtl ? "left" : "right") : "bottom");
                            if (options.last) {
                                // When looking for the *last* item, treat all container margins
                                // as belonging to the container *before* the margin.
                                return offset - that._sizes.containerMargins[marginPropLast];
                            } else {
                                // When looking for the *first* item, treat all container margins
                                // as belonging to the container *after* the margin.
                                return offset + that._sizes.containerMargins[marginPropFirst];
                            }
                        }
                        return offset;
                    }

                    // Assign the headers and margins to the appropriate groups.
                    function assignGroupMarginsAndHeaders(offset) {
                        if (options.last) {
                            // When looking for the *last* group, the *trailing* header and margin belong to the group.
                            return offset - that._getHeaderSizeGroupAdjustment() - that._sizes.itemsContainerOuterStart;
                        } else {
                            // When looking for the *first* group, the *leading* header and margin belong to the group.
                            // No need to make any adjustments to offset because the correct header and margin
                            // already belong to the group.
                            return offset;
                        }
                    }

                    options = options || {};

                    // Make offset relative to layout's content box
                    offset -= this._sizes.layoutOrigin;

                    offset = assignItemMargins(offset);

                    var groupIndex = this._groupFromOffset(assignGroupMarginsAndHeaders(offset)),
                        group = this._groups[groupIndex];

                    // Make offset relative to the margin box of the group's items container
                    offset -= group.offset;
                    offset -= this._getHeaderSizeGroupAdjustment();

                    return group.startIndex + group.itemFromOffset(offset, options);
                },

                _firstItemFromRange: function _LayoutCommon_firstItemFromRange(firstPixel, options) {
                    // supported options are:
                    // - wholeItem: when set to true the first fully visible item is returned
                    options = options || {};
                    options.last = 0;
                    return this._itemFromOffset(firstPixel, options);
                },

                _lastItemFromRange: function _LayoutCommon_lastItemFromRange(lastPixel, options) {
                    // supported options are:
                    // - wholeItem: when set to true the last fully visible item is returned
                    options = options || {};
                    options.last = 1;
                    return this._itemFromOffset(lastPixel, options);
                },

                _adjustedKeyForRTL: function _LayoutCommon_adjustedKeyForRTL(key) {
                    if (this._site.rtl) {
                        if (key === Key.leftArrow) {
                            key = Key.rightArrow;
                        } else if (key === Key.rightArrow) {
                            key = Key.leftArrow;
                        }
                    }
                    return key;
                },

                _adjustedKeyForOrientationAndBars: function _LayoutCommon_adjustedKeyForOrientationAndBars(key, cellSpanningGroup) {
                    var newKey = key;

                    // Don't support cell spanning
                    if (cellSpanningGroup) {
                        return key;
                    }
                    // First, convert the key into a virtual form based off of horizontal layouts.
                    // In a horizontal layout, left/right keys switch between columns (AKA "bars"), and
                    // up/down keys switch between rows (AKA "slots").
                    // In vertical mode, we want up/down to switch between rows (AKA "bars" when vertical),
                    // and left/right to switch between columns (AKA "slots" when vertical).
                    // The first step is to convert keypresses in vertical so that up/down always correspond to moving between slots,
                    // and left/right moving between bars.
                    if (!this._horizontal) {
                        switch (newKey) {
                            case Key.leftArrow:
                                newKey = Key.upArrow;
                                break;
                            case Key.rightArrow:
                                newKey = Key.downArrow;
                                break;
                            case Key.upArrow:
                                newKey = Key.leftArrow;
                                break;
                            case Key.downArrow:
                                newKey = Key.rightArrow;
                                break;
                        }
                    }

                    // Next, if we only have one item per bar, we'll make the change-slots-key the same as the change-bars-key
                    if (this._itemsPerBar === 1) {
                        if (newKey === Key.upArrow) {
                            newKey = Key.leftArrow;
                        } else if (newKey === Key.downArrow) {
                            newKey = Key.rightArrow;
                        }
                    }

                    return newKey;
                },

                _getAdjacentForPageKeys: function _LayoutCommon_getAdjacentForPageKeys(currentItem, pressedKey) {
                    var containerMargins = this._sizes.containerMargins,
                        marginSum = (this.orientation === "horizontal" ?
                            containerMargins.left + containerMargins.right :
                            containerMargins.top + containerMargins.bottom);

                    var viewportLength = this._site.viewportSize[this.orientation === "horizontal" ? "width" : "height"],
                        firstPixel = this._site.scrollbarPos,
                        lastPixel = firstPixel + viewportLength - 1 - containerMargins[(this.orientation === "horizontal" ? "right" : "bottom")],
                        newFocus;

                    // Handles page up by attempting to choose the first fully visible item
                    // on the current page. If that item already has focus, chooses the
                    // first item on the previous page. Page down is handled similarly.

                    var firstIndex = this._firstItemFromRange(firstPixel, { wholeItem: true }),
                        lastIndex = this._lastItemFromRange(lastPixel, { wholeItem: false }),
                        currentItemPosition = this._getItemPosition(currentItem.index);


                    var offscreen = false;
                    if (currentItem.index < firstIndex || currentItem.index > lastIndex) {
                        offscreen = true;
                        if (this.orientation === "horizontal") {
                            firstPixel = currentItemPosition.left - marginSum;
                        } else {
                            firstPixel = currentItemPosition.top - marginSum;
                        }
                        lastPixel = firstPixel + viewportLength - 1;
                        firstIndex = this._firstItemFromRange(firstPixel, { wholeItem: true });
                        lastIndex = this._lastItemFromRange(lastPixel, { wholeItem: false });
                    }

                    if (pressedKey === Key.pageUp) {
                        if (!offscreen && firstIndex !== currentItem.index) {
                            return { type: _UI.ObjectType.item, index: firstIndex };
                        }
                        var end;
                        if (this.orientation === "horizontal") {
                            end = currentItemPosition.left + currentItemPosition.width + marginSum + containerMargins.left;
                        } else {
                            end = currentItemPosition.top + currentItemPosition.height + marginSum + containerMargins.bottom;
                        }
                        var firstIndexOnPrevPage = this._firstItemFromRange(end - viewportLength, { wholeItem: true });
                        if (currentItem.index === firstIndexOnPrevPage) {
                            // The current item is so big that it spanned from the previous page, so we want to at least
                            // move to the previous item.
                            newFocus = Math.max(0, currentItem.index - this._itemsPerBar);
                        } else {
                            newFocus = firstIndexOnPrevPage;
                        }
                    } else {
                        if (!offscreen && lastIndex !== currentItem.index) {
                            return { type: _UI.ObjectType.item, index: lastIndex };
                        }
                        // We need to subtract twice the marginSum from the item's starting position because we need to
                        // consider that ensureVisible will scroll the viewport to include the new items margin as well
                        // which may push the current item just off screen.
                        var start;
                        if (this.orientation === "horizontal") {
                            start = currentItemPosition.left - marginSum - containerMargins.right;
                        } else {
                            start = currentItemPosition.top - marginSum - containerMargins.bottom;
                        }
                        var lastIndexOnNextPage = Math.max(0, this._lastItemFromRange(start + viewportLength - 1, { wholeItem: true }));
                        if (currentItem.index === lastIndexOnNextPage) {
                            // The current item is so big that it spans across the next page, so we want to at least
                            // move to the next item. It is also ok to blindly increment this index w/o bound checking
                            // since the browse mode clamps the bounds for page keys. This way we do not have to
                            // asynchronoulsy request the count here.
                            newFocus = currentItem.index + this._itemsPerBar;
                        } else {
                            newFocus = lastIndexOnNextPage;
                        }
                    }

                    return { type: _UI.ObjectType.item, index: newFocus };
                },

                _isCellSpanning: function _LayoutCommon_isCellSpanning(groupIndex) {
                    var group = this._site.groupFromIndex(groupIndex),
                        groupInfo = this._groupInfo;

                    if (groupInfo) {
                        return !!(typeof groupInfo === "function" ? groupInfo(group) : groupInfo).enableCellSpanning;
                    } else {
                        return false;
                    }
                },

                // Can only be called after measuring has been completed
                _getGroupInfo: function _LayoutCommon_getGroupInfo(groupIndex) {
                    var group = this._site.groupFromIndex(groupIndex),
                        groupInfo = this._groupInfo,
                        margins = this._sizes.containerMargins,
                        adjustedInfo = { enableCellSpanning: false };

                    groupInfo = (typeof groupInfo === "function" ? groupInfo(group) : groupInfo);
                    if (groupInfo) {
                        if (groupInfo.enableCellSpanning && (+groupInfo.cellWidth !== groupInfo.cellWidth || +groupInfo.cellHeight !== groupInfo.cellHeight)) {
                            throw new _ErrorFromName("WinJS.UI.GridLayout.GroupInfoResultIsInvalid", strings.groupInfoResultIsInvalid);
                        }
                        adjustedInfo = {
                            enableCellSpanning: !!groupInfo.enableCellSpanning,
                            cellWidth: groupInfo.cellWidth + margins.left + margins.right,
                            cellHeight: groupInfo.cellHeight + margins.top + margins.bottom
                        };
                    }

                    return adjustedInfo;
                },

                // itemIndex is optional
                _getItemInfo: function _LayoutCommon_getItemInfo(itemIndex) {
                    var result;
                    if (!this._itemInfo || typeof this._itemInfo !== "function") {
                        if (this._useDefaultItemInfo) {
                            result = this._defaultItemInfo(itemIndex);
                        } else {
                            throw new _ErrorFromName("WinJS.UI.GridLayout.ItemInfoIsInvalid", strings.itemInfoIsInvalid);
                        }
                    } else {
                        result = this._itemInfo(itemIndex);
                    }
                    return Promise.as(result).then(function (size) {
                        if (!size || +size.width !== size.width || +size.height !== size.height) {
                            throw new _ErrorFromName("WinJS.UI.GridLayout.ItemInfoIsInvalid", strings.itemInfoIsInvalid);
                        }
                        return size;
                    });
                },

                _defaultItemInfo: function _LayoutCommon_defaultItemInfo(itemIndex) {
                    var that = this;
                    return this._site.renderItem(this._site.itemFromIndex(itemIndex)).then(function (element) {
                        that._elementsToMeasure[itemIndex] = {
                            element: element
                        };
                        return that._measureElements();
                    }).then(
                        function () {
                            var entry = that._elementsToMeasure[itemIndex],
                                size = {
                                    width: entry.width,
                                    height: entry.height
                                };

                            delete that._elementsToMeasure[itemIndex];
                            return size;
                        },
                        function (error) {
                            delete that._elementsToMeasure[itemIndex];
                            return Promise.wrapError(error);
                        }
                    );
                },

                _getGroupSize: function _LayoutCommon_getGroupSize(group) {
                    var headerContainerMinSize = 0;

                    if (this._groupsEnabled) {
                        if (this._horizontal && this._groupHeaderPosition === HeaderPosition.top) {
                            headerContainerMinSize = this._sizes.headerContainerMinWidth;
                        } else if (!this._horizontal && this._groupHeaderPosition === HeaderPosition.left) {
                            headerContainerMinSize = this._sizes.headerContainerMinHeight;
                        }
                    }
                    return Math.max(headerContainerMinSize, group.getItemsContainerSize() + this._getHeaderSizeGroupAdjustment());
                },

                // offset should be relative to the grid layout's content box
                _groupFromOffset: function _LayoutCommon_groupFromOffset(offset) {
                    return offset < this._groups[0].offset ?
                        0 :
                        this._groupFrom(function (group) {
                            return offset < group.offset;
                        });
                },

                _groupFromImpl: function _LayoutCommon_groupFromImpl(fromGroup, toGroup, comp) {
                    if (toGroup < fromGroup) {
                        return null;
                    }

                    var center = fromGroup + Math.floor((toGroup - fromGroup) / 2),
                        centerGroup = this._groups[center];

                    if (comp(centerGroup, center)) {
                        return this._groupFromImpl(fromGroup, center - 1, comp);
                    } else if (center < toGroup && !comp(this._groups[center + 1], center + 1)) {
                        return this._groupFromImpl(center + 1, toGroup, comp);
                    } else {
                        return center;
                    }
                },

                _groupFrom: function _LayoutCommon_groupFrom(comp) {
                    if (this._groups.length > 0) {
                        var lastGroupIndex = this._groups.length - 1,
                            lastGroup = this._groups[lastGroupIndex];

                        if (!comp(lastGroup, lastGroupIndex)) {
                            return lastGroupIndex;
                        } else {
                            return this._groupFromImpl(0, this._groups.length - 1, comp);
                        }
                    } else {
                        return null;
                    }
                },

                _invalidateLayout: function _LayoutCommon_invalidateLayout() {
                    if (this._site) {
                        this._site.invalidateLayout();
                    }
                },

                _resetMeasurements: function _LayoutCommon_resetMeasurements() {
                    if (this._measuringPromise) {
                        this._measuringPromise.cancel();
                        this._measuringPromise = null;
                    }
                    if (this._containerSizeClassName) {
                        _ElementUtilities.removeClass(this._site.surface, this._containerSizeClassName);
                        deleteDynamicCssRule(this._containerSizeClassName);
                        this._containerSizeClassName = null;
                    }
                    this._sizes = null;
                    this._resetAnimationCaches();
                },

                _measureElements: function _LayoutCommon_measureElements() {
                    // batching measurements to minimalize number of layout passes
                    if (!this._measuringElements) {
                        var that = this;
                        // Schedule a job so that:
                        //  1. Calls to _measureElements are batched.
                        //  2. that._measuringElements is set before the promise handler is executed
                        //     (that._measuringElements is used within the handler).
                        that._measuringElements = Scheduler.schedulePromiseHigh(null, "WinJS.UI.GridLayout._measuringElements").then(
                            function measure() {
                                that._site._writeProfilerMark("_measureElements,StartTM");

                                var surface = that._createMeasuringSurface(),
                                    itemsContainer = _Global.document.createElement("div"),
                                    site = that._site,
                                    measuringElements = that._measuringElements,
                                    elementsToMeasure = that._elementsToMeasure,
                                    stopMeasuring = false;

                                itemsContainer.className = _Constants._itemsContainerClass + " " + _Constants._laidOutClass;
                                // This code is executed by CellSpanningGroups where styling is configured for –ms-grid. Let's satisfy these assumptions
                                itemsContainer.style.cssText +=
                                        ";display: -ms-grid" +
                                        ";-ms-grid-column: 1" +
                                        ";-ms-grid-row: 1";

                                var keys = Object.keys(elementsToMeasure),
                                    len,
                                    i;

                                for (i = 0, len = keys.length; i < len; i++) {
                                    var element = elementsToMeasure[keys[i]].element;
                                    element.style["-ms-grid-column"] = i + 1;
                                    element.style["-ms-grid-row"] = i + 1;
                                    itemsContainer.appendChild(element);
                                }

                                surface.appendChild(itemsContainer);
                                site.viewport.insertBefore(surface, site.viewport.firstChild);

                                // Reading from the DOM may cause the app's resize handler to
                                // be run synchronously which may invalidate this measuring
                                // operation. When this happens, stop measuring.
                                measuringElements.then(null, function () {
                                    stopMeasuring = true;
                                });

                                for (i = 0, len = keys.length; i < len && !stopMeasuring; i++) {
                                    var entry = elementsToMeasure[keys[i]],
                                        item = entry.element.querySelector("." + _Constants._itemClass);

                                    entry.width = _ElementUtilities.getTotalWidth(item);
                                    entry.height = _ElementUtilities.getTotalHeight(item);

                                }

                                if (surface.parentNode) {
                                    surface.parentNode.removeChild(surface);
                                }
                                if (measuringElements === that._measuringElements) {
                                    that._measuringElements = null;
                                }

                                site._writeProfilerMark("_measureElements,StopTM");
                            },
                            function (error) {
                                that._measuringElements = null;
                                return Promise.wrapError(error);
                            }
                        );
                    }
                    return this._measuringElements;
                },

                _ensureEnvInfo: function _LayoutCommon_ensureEnvInfo() {
                    if (!this._envInfo) {
                        this._envInfo = getEnvironmentSupportInformation(this._site);
                        if (this._envInfo && !this._envInfo.supportsCSSGrid) {
                            _ElementUtilities.addClass(this._site.surface, _Constants._noCSSGrid);
                        }
                    }
                    return !!this._envInfo;
                },

                _createMeasuringSurface: function _LayoutCommon_createMeasuringSurface() {
                    var surface = _Global.document.createElement("div");

                    surface.style.cssText =
                        "visibility: hidden" +
                        ";-ms-grid-columns: auto" +
                        ";-ms-grid-rows: auto" +
                        ";-ms-flex-align: start" +
                        ";-webkit-align-items: flex-start" +
                        ";align-items: flex-start";
                    surface.className = _Constants._scrollableClass + " " + (this._inListMode ? _Constants._listLayoutClass : _Constants._gridLayoutClass);
                    if (!this._envInfo.supportsCSSGrid) {
                        _ElementUtilities.addClass(surface, _Constants._noCSSGrid);
                    }
                    if (this._groupsEnabled) {
                        if (this._groupHeaderPosition === HeaderPosition.top) {
                            _ElementUtilities.addClass(surface, _Constants._headerPositionTopClass);
                        } else {
                            _ElementUtilities.addClass(surface, _Constants._headerPositionLeftClass);
                        }
                    }

                    return surface;
                },

                // Assumes that the size of the item at the specified index is representative
                // of the size of all items, measures it, and stores the measurements in
                // this._sizes. If necessary, also:
                // - Creates a CSS rule to give the containers a height and width
                // - Stores the name associated with the rule in this._containerSizeClassName
                // - Adds the class name associated with the rule to the surface
                _measureItem: function _LayoutCommon_measureItem(index) {
                    var that = this;
                    var perfId = "Layout:measureItem";
                    var site = that._site;
                    var measuringPromise = that._measuringPromise;

                    // itemPromise is optional. It is provided when taking a second attempt at measuring.
                    function measureItemImpl(index, itemPromise) {
                        var secondTry = !!itemPromise,
                            elementPromises = {},
                            itemPromise,
                            left = site.rtl ? "right" : "left";

                        return site.itemCount.then(function (count) {
                            if (!count || (that._groupsEnabled && !site.groupCount)) {
                                return Promise.cancel;
                            }

                            itemPromise = itemPromise || site.itemFromIndex(index);
                            elementPromises.container = site.renderItem(itemPromise);
                            if (that._groupsEnabled) {
                                elementPromises.headerContainer = site.renderHeader(that._site.groupFromIndex(site.groupIndexFromItemIndex(index)));
                            }

                            return Promise.join(elementPromises);
                        }).then(function (elements) {

                            // Reading from the DOM is tricky because each read may trigger a resize handler which
                            // may invalidate this layout object. To make it easier to minimize bugs in this edge case:
                            //  1. All DOM reads for _LayoutCommon_measureItem should be contained within this function.
                            //  2. This function should remain as simple as possible. Stick to DOM reads, avoid putting
                            //     logic in here, and cache all needed instance variables at the top of the function.
                            //
                            // Returns null if the measuring operation was invalidated while reading from the DOM.
                            // Otherwise, returns an object containing the measurements.
                            function readMeasurementsFromDOM() {
                                var horizontal = that._horizontal;
                                var groupsEnabled = that._groupsEnabled;
                                var stopMeasuring = false;

                                // Reading from the DOM may cause the app's resize handler to
                                // be run synchronously which may invalidate this measuring
                                // operation. When this happens, stop measuring.
                                measuringPromise.then(null, function () {
                                    stopMeasuring = true;
                                });

                                var firstElementOnSurfaceMargins = getMargins(firstElementOnSurface);
                                var firstElementOnSurfaceOffsetX = site.rtl ?
                                    (site.viewport.offsetWidth - (firstElementOnSurface.offsetLeft + firstElementOnSurface.offsetWidth)) :
                                    firstElementOnSurface.offsetLeft;
                                var firstElementOnSurfaceOffsetY = firstElementOnSurface.offsetTop;
                                var sizes = {
                                    // These will be set by _viewportSizeChanged
                                    viewportContentSize: 0,
                                    surfaceContentSize: 0,
                                    maxItemsContainerContentSize: 0,

                                    surfaceOuterHeight: getOuterHeight(surface),
                                    surfaceOuterWidth: getOuterWidth(surface),

                                    // Origin of the grid layout's content in viewport coordinates
                                    layoutOriginX: firstElementOnSurfaceOffsetX - firstElementOnSurfaceMargins[left],
                                    layoutOriginY: firstElementOnSurfaceOffsetY - firstElementOnSurfaceMargins.top,
                                    itemsContainerOuterHeight: getOuterHeight(itemsContainer),
                                    itemsContainerOuterWidth: getOuterWidth(itemsContainer),
                                    // Amount of space between the items container's margin and its content
                                    itemsContainerOuterX: getOuter(site.rtl ? "Right" : "Left", itemsContainer),
                                    itemsContainerOuterY: getOuter("Top", itemsContainer),
                                    itemsContainerMargins: getMargins(itemsContainer),

                                    itemBoxOuterHeight: getOuterHeight(itemBox),
                                    itemBoxOuterWidth: getOuterWidth(itemBox),
                                    containerOuterHeight: getOuterHeight(elements.container),
                                    containerOuterWidth: getOuterWidth(elements.container),
                                    emptyContainerContentHeight: _ElementUtilities.getContentHeight(emptyContainer),
                                    emptyContainerContentWidth: _ElementUtilities.getContentWidth(emptyContainer),

                                    containerMargins: getMargins(elements.container),
                                    // containerWidth/Height are computed when a uniform group is detected
                                    containerWidth: 0,
                                    containerHeight: 0,
                                    // true when both containerWidth and containerHeight have been measured
                                    containerSizeLoaded: false
                                };

                                if (site.header) {
                                    sizes[(horizontal ? "layoutOriginX" : "layoutOriginY")] += _ElementUtilities[(horizontal ? "getTotalWidth" : "getTotalHeight")](site.header);
                                }

                                if (groupsEnabled) {
                                    // Amount of space between the header container's margin and its content
                                    sizes.headerContainerOuterX = getOuter(site.rtl ? "Right" : "Left", elements.headerContainer),
                                    sizes.headerContainerOuterY = getOuter("Top", elements.headerContainer),

                                    sizes.headerContainerOuterWidth = getOuterWidth(elements.headerContainer);
                                    sizes.headerContainerOuterHeight = getOuterHeight(elements.headerContainer);
                                    sizes.headerContainerWidth = _ElementUtilities.getTotalWidth(elements.headerContainer);
                                    sizes.headerContainerHeight = _ElementUtilities.getTotalHeight(elements.headerContainer);
                                    sizes.headerContainerMinWidth = getDimension(elements.headerContainer, "minWidth") + sizes.headerContainerOuterWidth;
                                    sizes.headerContainerMinHeight = getDimension(elements.headerContainer, "minHeight") + sizes.headerContainerOuterHeight;
                                }

                                var measurements = {
                                    // Measurements which are needed after measureItem has returned.
                                    sizes: sizes,

                                    // Measurements which are only needed within measureItem.
                                    viewportContentWidth: _ElementUtilities.getContentWidth(site.viewport),
                                    viewportContentHeight: _ElementUtilities.getContentHeight(site.viewport),
                                    containerContentWidth: _ElementUtilities.getContentWidth(elements.container),
                                    containerContentHeight: _ElementUtilities.getContentHeight(elements.container),
                                    containerWidth: _ElementUtilities.getTotalWidth(elements.container),
                                    containerHeight: _ElementUtilities.getTotalHeight(elements.container)
                                };
                                measurements.viewportCrossSize = measurements[horizontal ? "viewportContentHeight" : "viewportContentWidth"];

                                site.readyToMeasure();

                                return stopMeasuring ? null : measurements;
                            }

                            function cleanUp() {
                                if (surface.parentNode) {
                                    surface.parentNode.removeChild(surface);
                                }
                            }

                            var surface = that._createMeasuringSurface(),
                                itemsContainer = _Global.document.createElement("div"),
                                emptyContainer = _Global.document.createElement("div"),
                                itemBox = elements.container.querySelector("." + _Constants._itemBoxClass),
                                groupIndex = site.groupIndexFromItemIndex(index);

                            emptyContainer.className = _Constants._containerClass;
                            itemsContainer.className = _Constants._itemsContainerClass + " " + _Constants._laidOutClass;
                            // Use display=inline-block so that the width sizes to content when not in list mode.
                            // When in grid mode, put items container and header container in different rows and columns so that the size of the items container does not affect the size of the header container and vice versa.
                            // Use the same for list mode when headers are inline with item containers.
                            // When headers are to the left of a vertical list, or above a horizontal list, put the rows/columns they would be in when laid out normally
                            // into the CSS text for measuring. We have to do this because list item containers should take up 100% of the space left over in the surface
                            // once the group's header is laid out.
                            var itemsContainerRow = 1,
                                itemsContainerColumn = 1,
                                headerContainerRow = 2,
                                headerContainerColumn = 2,
                                firstElementOnSurface = itemsContainer,
                                addHeaderFirst = false;
                            if (that._inListMode && that._groupsEnabled) {
                                if (that._horizontal && that._groupHeaderPosition === HeaderPosition.top) {
                                    itemsContainerRow = 2;
                                    headerContainerColumn = 1;
                                    headerContainerRow = 1;
                                    firstElementOnSurface = elements.headerContainer;
                                    addHeaderFirst = true;
                                } else if (!that._horizontal && that._groupHeaderPosition === HeaderPosition.left) {
                                    itemsContainerColumn = 2;
                                    headerContainerColumn = 1;
                                    headerContainerRow = 1;
                                    firstElementOnSurface = elements.headerContainer;
                                    addHeaderFirst = true;
                                }
                            }
                            // ListMode needs to use display block to proprerly measure items in vertical mode, and display flex to properly measure items in horizontal mode
                            itemsContainer.style.cssText +=
                                    ";display: " + (that._inListMode ? ((that._horizontal ? "flex" : "block") + "; overflow: hidden") : "inline-block") +
                                     ";vertical-align:top" +
                                    ";-ms-grid-column: " + itemsContainerColumn +
                                    ";-ms-grid-row: " + itemsContainerRow;
                            if (!that._inListMode) {
                                elements.container.style.display = "inline-block";
                            }
                            if (that._groupsEnabled) {
                                elements.headerContainer.style.cssText +=
                                    ";display: inline-block" +
                                    ";-ms-grid-column: " + headerContainerColumn +
                                    ";-ms-grid-row: " + headerContainerRow;
                                _ElementUtilities.addClass(elements.headerContainer, _Constants._laidOutClass + " " + _Constants._groupLeaderClass);
                                if ((that._groupHeaderPosition === HeaderPosition.top && that._horizontal) ||
                                    (that._groupHeaderPosition === HeaderPosition.left && !that._horizontal)) {
                                    _ElementUtilities.addClass(itemsContainer, _Constants._groupLeaderClass);
                                }
                            }
                            if (addHeaderFirst) {
                                surface.appendChild(elements.headerContainer);
                            }

                            itemsContainer.appendChild(elements.container);
                            itemsContainer.appendChild(emptyContainer);

                            surface.appendChild(itemsContainer);
                            if (!addHeaderFirst && that._groupsEnabled) {
                                surface.appendChild(elements.headerContainer);
                            }
                            site.viewport.insertBefore(surface, site.viewport.firstChild);

                            var measurements = readMeasurementsFromDOM();

                            if (!measurements) {
                                // While reading from the DOM, the measuring operation was invalidated. Bail out.
                                cleanUp();
                                return Promise.cancel;
                            } else if ((that._horizontal && measurements.viewportContentHeight === 0) || (!that._horizontal && measurements.viewportContentWidth === 0)) {
                                // ListView is invisible so we can't measure. Return a canceled promise.
                                cleanUp();
                                return Promise.cancel;
                            } else if (!secondTry && !that._isCellSpanning(groupIndex) &&
                                    (measurements.containerContentWidth === 0 || measurements.containerContentHeight === 0)) {
                                // win-container has no size. For backwards compatibility, wait for the item promise and then try measuring again.
                                cleanUp();
                                return itemPromise.then(function () {
                                    return measureItemImpl(index, itemPromise);
                                });
                            } else {
                                var sizes = that._sizes = measurements.sizes;

                                // Wrappers for orientation-specific properties.
                                // Sizes prefaced with "cross" refer to the sizes orthogonal to the current layout orientation. Sizes without a preface are in the orientation's direction.
                                Object.defineProperties(sizes, {
                                    surfaceOuterCrossSize: {
                                        get: function () {
                                            return (that._horizontal ? sizes.surfaceOuterHeight : sizes.surfaceOuterWidth);
                                        },
                                        enumerable: true
                                    },
                                    layoutOrigin: {
                                        get: function () {
                                            return (that._horizontal ? sizes.layoutOriginX : sizes.layoutOriginY);
                                        },
                                        enumerable: true
                                    },
                                    itemsContainerOuterSize: {
                                        get: function () {
                                            return (that._horizontal ? sizes.itemsContainerOuterWidth : sizes.itemsContainerOuterHeight);
                                        },
                                        enumerable: true
                                    },
                                    itemsContainerOuterCrossSize: {
                                        get: function () {
                                            return (that._horizontal ? sizes.itemsContainerOuterHeight : sizes.itemsContainerOuterWidth);
                                        },
                                        enumerable: true
                                    },
                                    itemsContainerOuterStart: {
                                        get: function () {
                                            return (that._horizontal ? sizes.itemsContainerOuterX : sizes.itemsContainerOuterY);
                                        },
                                        enumerable: true
                                    },
                                    itemsContainerOuterCrossStart: {
                                        get: function () {
                                            return (that._horizontal ? sizes.itemsContainerOuterY : sizes.itemsContainerOuterX);
                                        },
                                        enumerable: true
                                    },
                                    containerCrossSize: {
                                        get: function () {
                                            return (that._horizontal ? sizes.containerHeight : sizes.containerWidth);
                                        },
                                        enumerable: true
                                    },
                                    containerSize: {
                                        get: function () {
                                            return (that._horizontal ? sizes.containerWidth : sizes.containerHeight);
                                        },
                                        enumerable: true
                                    },
                                });

                                // If the measured group is uniform, measure the container height
                                // and width now. Otherwise, compute them thru itemInfo on demand (via _ensureContainerSize).
                                if (!that._isCellSpanning(groupIndex)) {
                                    if (that._inListMode) {
                                        var itemsContainerContentSize = measurements.viewportCrossSize - sizes.surfaceOuterCrossSize - that._getHeaderSizeContentAdjustment() - sizes.itemsContainerOuterCrossSize;
                                        if (that._horizontal) {
                                            sizes.containerHeight = itemsContainerContentSize;
                                            sizes.containerWidth = measurements.containerWidth;
                                        } else {
                                            sizes.containerHeight = measurements.containerHeight;
                                            sizes.containerWidth = itemsContainerContentSize;
                                        }
                                    } else {
                                        sizes.containerWidth = measurements.containerWidth;
                                        sizes.containerHeight = measurements.containerHeight;
                                    }
                                    sizes.containerSizeLoaded = true;
                                }

                                that._createContainerStyleRule();
                                that._viewportSizeChanged(measurements.viewportCrossSize);

                                cleanUp();
                            }
                        });
                    }

                    if (!measuringPromise) {
                        site._writeProfilerMark(perfId + ",StartTM");
                        // Use a signal to guarantee that measuringPromise is set before the promise
                        // handler is executed (measuringPromise is referenced within measureItemImpl).
                        var promiseStoredSignal = new _Signal();
                        that._measuringPromise = measuringPromise = promiseStoredSignal.promise.then(function () {
                            if (that._ensureEnvInfo()) {
                                return measureItemImpl(index);
                            } else {
                                // Couldn't get envInfo. ListView is invisible. Bail out.
                                return Promise.cancel;
                            }
                        }).then(function () {
                            site._writeProfilerMark(perfId + ":complete,info");
                            site._writeProfilerMark(perfId + ",StopTM");
                        }, function (error) {
                            // The purpose of the measuring promise is so that we only
                            // measure once. If measuring fails, clear the promise because
                            // we still need to measure.
                            that._measuringPromise = null;

                            site._writeProfilerMark(perfId + ":canceled,info");
                            site._writeProfilerMark(perfId + ",StopTM");

                            return Promise.wrapError(error);
                        });
                        promiseStoredSignal.complete();
                    }
                    return measuringPromise;
                },

                _getHeaderSizeGroupAdjustment: function () {
                    if (this._groupsEnabled) {
                        if (this._horizontal && this._groupHeaderPosition === HeaderPosition.left) {
                            return this._sizes.headerContainerWidth;
                        } else if (!this._horizontal && this._groupHeaderPosition === HeaderPosition.top) {
                            return this._sizes.headerContainerHeight;
                        }
                    }

                    return 0;
                },
                _getHeaderSizeContentAdjustment: function () {
                    if (this._groupsEnabled) {
                        if (this._horizontal && this._groupHeaderPosition === HeaderPosition.top) {
                            return this._sizes.headerContainerHeight;
                        } else if (!this._horizontal && this._groupHeaderPosition === HeaderPosition.left) {
                            return this._sizes.headerContainerWidth;
                        }
                    }

                    return 0;
                },

                // Horizontal layouts lay items out top to bottom, left to right, whereas vertical layouts lay items out left to right, top to bottom.
                // The viewport size is the size layouts use to determine how many items can be placed in one bar, so it should be cross to the
                // orientation.
                _getViewportCrossSize: function () {
                    return this._site.viewportSize[this._horizontal ? "height" : "width"];
                },

                // viewportContentSize is the new viewport size
                _viewportSizeChanged: function _LayoutCommon_viewportSizeChanged(viewportContentSize) {
                    var sizes = this._sizes;

                    sizes.viewportContentSize = viewportContentSize;
                    sizes.surfaceContentSize = viewportContentSize - sizes.surfaceOuterCrossSize;
                    sizes.maxItemsContainerContentSize = sizes.surfaceContentSize - sizes.itemsContainerOuterCrossSize - this._getHeaderSizeContentAdjustment();

                    // This calculation is for uniform layouts
                    if (sizes.containerSizeLoaded && !this._inListMode) {
                        this._itemsPerBar = Math.floor(sizes.maxItemsContainerContentSize / sizes.containerCrossSize);
                        if (this.maximumRowsOrColumns) {
                            this._itemsPerBar = Math.min(this._itemsPerBar, this.maximumRowsOrColumns);
                        }
                        this._itemsPerBar = Math.max(1, this._itemsPerBar);
                    } else {
                        if (this._inListMode) {
                            sizes[this._horizontal ? "containerHeight" : "containerWidth"] = sizes.maxItemsContainerContentSize;
                        }
                        this._itemsPerBar = 1;
                    }

                    // Ignore animations if height changed
                    this._resetAnimationCaches();
                },

                _createContainerStyleRule: function _LayoutCommon_createContainerStyleRule() {
                    // Adding CSS rules is expensive. Add a rule to provide a
                    // height and width for containers if the app hasn't provided one.
                    var sizes = this._sizes;
                    if (!this._containerSizeClassName && sizes.containerSizeLoaded && (sizes.emptyContainerContentHeight === 0 || sizes.emptyContainerContentWidth === 0)) {
                        var width = sizes.containerWidth - sizes.containerOuterWidth + "px",
                            height = sizes.containerHeight - sizes.containerOuterHeight + "px";
                        if (this._inListMode) {
                            if (this._horizontal) {
                                height = "calc(100% - " + (sizes.containerMargins.top + sizes.containerMargins.bottom) + "px)";
                            } else {
                                width = "auto";
                            }
                        }

                        if (!this._containerSizeClassName) {
                            this._containerSizeClassName = uniqueCssClassName("containersize");
                            _ElementUtilities.addClass(this._site.surface, this._containerSizeClassName);
                        }
                        var ruleSelector = "." + _Constants._containerClass,
                            ruleBody = "width:" + width + ";height:" + height + ";";
                        addDynamicCssRule(this._containerSizeClassName, this._site, ruleSelector, ruleBody);
                    }
                },

                // Computes container width and height if they haven't been computed yet. This
                // should happen when the first uniform group is created.
                _ensureContainerSize: function _LayoutCommon_ensureContainerSize(group) {
                    var sizes = this._sizes;
                    if (!sizes.containerSizeLoaded && !this._ensuringContainerSize) {
                        var promise;
                        if ((!this._itemInfo || typeof this._itemInfo !== "function") && this._useDefaultItemInfo) {
                            var margins = sizes.containerMargins;
                            promise = Promise.wrap({
                                width: group.groupInfo.cellWidth - margins.left - margins.right,
                                height: group.groupInfo.cellHeight - margins.top - margins.bottom
                            });

                        } else {
                            promise = this._getItemInfo();
                        }

                        var that = this;
                        this._ensuringContainerSize = promise.then(function (itemSize) {
                            sizes.containerSizeLoaded = true;
                            sizes.containerWidth = itemSize.width + sizes.itemBoxOuterWidth + sizes.containerOuterWidth;
                            sizes.containerHeight = itemSize.height + sizes.itemBoxOuterHeight + sizes.containerOuterHeight;
                            if (!that._inListMode) {
                                that._itemsPerBar = Math.floor(sizes.maxItemsContainerContentSize / sizes.containerCrossSize);
                                if (that.maximumRowsOrColumns) {
                                    that._itemsPerBar = Math.min(that._itemsPerBar, that.maximumRowsOrColumns);
                                }
                                that._itemsPerBar = Math.max(1, that._itemsPerBar);
                            } else {
                                that._itemsPerBar = 1;
                            }
                            that._createContainerStyleRule();
                        });

                        promise.done(
                            function () {
                                that._ensuringContainerSize = null;
                            },
                            function () {
                                that._ensuringContainerSize = null;
                            }
                        );

                        return promise;
                    } else {
                        return this._ensuringContainerSize ? this._ensuringContainerSize : Promise.wrap();
                    }
                },

                _indexToCoordinate: function _LayoutCommon_indexToCoordinate(index, itemsPerBar) {
                    itemsPerBar = itemsPerBar || this._itemsPerBar;
                    var bar = Math.floor(index / itemsPerBar);
                    if (this._horizontal) {
                        return {
                            column: bar,
                            row: index - bar * itemsPerBar
                        };
                    } else {
                        return {
                            row: bar,
                            column: index - bar * itemsPerBar
                        };
                    }
                },

                // Empty ranges are represented by null. Non-empty ranges are represented by
                // an object with 2 properties: firstIndex and lastIndex.
                _rangeForGroup: function _LayoutCommon_rangeForGroup(group, range) {
                    var first = group.startIndex,
                        last = first + group.count - 1;

                    if (!range || range.firstIndex > last || range.lastIndex < first) {
                        // There isn't any overlap between range and the group's indices
                        return null;
                    } else {
                        return {
                            firstIndex: Math.max(0, range.firstIndex - first),
                            lastIndex: Math.min(group.count - 1, range.lastIndex - first)
                        };
                    }
                },

                _syncDomWithGroupHeaderPosition: function _LayoutCommon_syncDomWithGroupHeaderPosition(tree) {
                    if (this._groupsEnabled && this._oldGroupHeaderPosition !== this._groupHeaderPosition) {
                        // this._oldGroupHeaderPosition may refer to top, left, or null. It will be null
                        // the first time this function is called which means that no styles have to be
                        // removed.

                        var len = tree.length,
                            i;
                        // Remove styles associated with old group header position
                        if (this._oldGroupHeaderPosition === HeaderPosition.top) {
                            _ElementUtilities.removeClass(this._site.surface, _Constants._headerPositionTopClass);
                            // maxWidth must be cleared because it is used with headers in the top position but not the left position.
                            // The _groupLeaderClass must be removed from the itemsContainer element because the associated styles
                            // should only be applied to it when headers are in the top position.
                            if (this._horizontal) {
                                for (i = 0; i < len; i++) {
                                    tree[i].header.style.maxWidth = "";
                                    _ElementUtilities.removeClass(tree[i].itemsContainer.element, _Constants._groupLeaderClass);
                                }
                            } else {
                                this._site.surface.style.msGridRows = "";
                            }
                        } else if (this._oldGroupHeaderPosition === HeaderPosition.left) {
                            _ElementUtilities.removeClass(this._site.surface, _Constants._headerPositionLeftClass);
                            // msGridColumns is cleared for a similar reason as maxWidth
                            if (!this._horizontal) {
                                for (i = 0; i < len; i++) {
                                    tree[i].header.style.maxHeight = "";
                                    _ElementUtilities.removeClass(tree[i].itemsContainer.element, _Constants._groupLeaderClass);
                                }
                            }
                            this._site.surface.style.msGridColumns = "";
                        }

                        // Add styles associated with new group header position
                        if (this._groupHeaderPosition === HeaderPosition.top) {
                            _ElementUtilities.addClass(this._site.surface, _Constants._headerPositionTopClass);
                            if (this._horizontal) {
                                for (i = 0; i < len; i++) {
                                    _ElementUtilities.addClass(tree[i].itemsContainer.element, _Constants._groupLeaderClass);
                                }
                            }
                        } else {
                            _ElementUtilities.addClass(this._site.surface, _Constants._headerPositionLeftClass);
                            if (!this._horizontal) {
                                for (i = 0; i < len; i++) {
                                    _ElementUtilities.addClass(tree[i].itemsContainer.element, _Constants._groupLeaderClass);
                                }
                            }
                        }

                        this._oldGroupHeaderPosition = this._groupHeaderPosition;
                    }
                },

                _layoutGroup: function _LayoutCommon_layoutGroup(index) {
                    var group = this._groups[index],
                        groupBundle = this._site.tree[index],
                        headerContainer = groupBundle.header,
                        itemsContainer = groupBundle.itemsContainer.element,
                        sizes = this._sizes,
                        groupCrossSize = group.getItemsContainerCrossSize();

                    if (this._groupsEnabled) {
                        if (this._horizontal) {
                            if (this._groupHeaderPosition === HeaderPosition.top) {
                                // Horizontal with headers above
                                //
                                var headerContainerMinContentWidth = sizes.headerContainerMinWidth - sizes.headerContainerOuterWidth,
                                    itemsContainerContentWidth = group.getItemsContainerSize() - sizes.headerContainerOuterWidth;
                                headerContainer.style.maxWidth = Math.max(headerContainerMinContentWidth, itemsContainerContentWidth) + "px";
                                if (this._envInfo.supportsCSSGrid) {
                                    headerContainer.style.msGridColumn = index + 1;
                                    itemsContainer.style.msGridColumn = index + 1;
                                } else {
                                    headerContainer.style.height = (sizes.headerContainerHeight - sizes.headerContainerOuterHeight) + "px";
                                    itemsContainer.style.height = (groupCrossSize - sizes.itemsContainerOuterHeight) + "px";
                                    // If the itemsContainer is too small, the next group's header runs the risk of appearing below the current group's items.
                                    // We need to add a margin to the bottom of the itemsContainer to prevent that from happening.
                                    itemsContainer.style.marginBottom = sizes.itemsContainerMargins.bottom + (sizes.maxItemsContainerContentSize - groupCrossSize + sizes.itemsContainerOuterHeight) + "px";
                                }
                                // itemsContainers only get the _groupLeaderClass when header position is top.
                                _ElementUtilities.addClass(itemsContainer, _Constants._groupLeaderClass);
                            } else {
                                // Horizontal with headers on the left
                                //
                                if (this._envInfo.supportsCSSGrid) {
                                    headerContainer.style.msGridColumn = index * 2 + 1;
                                    itemsContainer.style.msGridColumn = index * 2 + 2;
                                } else {
                                    headerContainer.style.width = sizes.headerContainerWidth - sizes.headerContainerOuterWidth + "px";
                                    headerContainer.style.height = (groupCrossSize - sizes.headerContainerOuterHeight) + "px";
                                    itemsContainer.style.height = (groupCrossSize - sizes.itemsContainerOuterHeight) + "px";
                                }
                            }
                        } else {
                            if (this._groupHeaderPosition === HeaderPosition.left) {
                                // Vertical with headers on the left
                                //
                                var headerContainerMinContentHeight = sizes.headerContainerMinHeight - sizes.headerContainerOuterHeight,
                                    itemsContainerContentHeight = group.getItemsContainerSize() - sizes.headerContainerOuterHeight;
                                headerContainer.style.maxHeight = Math.max(headerContainerMinContentHeight, itemsContainerContentHeight) + "px";
                                if (this._envInfo.supportsCSSGrid) {
                                    headerContainer.style.msGridRow = index + 1;
                                    itemsContainer.style.msGridRow = index + 1;
                                } else {
                                    headerContainer.style.width = (sizes.headerContainerWidth - sizes.headerContainerOuterWidth) + "px";
                                    itemsContainer.style.width = (groupCrossSize - sizes.itemsContainerOuterWidth) + "px";
                                    // If the itemsContainer is too small, the next group's header runs the risk of appearing to the side of the current group's items.
                                    // We need to add a margin to the right of the itemsContainer to prevent that from happening (or the left margin, in RTL).
                                    itemsContainer.style["margin" + (this._site.rtl ? "Left" : "Right")] = (sizes.itemsContainerMargins[(this._site.rtl ? "left" : "right")] +
                                        (sizes.maxItemsContainerContentSize - groupCrossSize + sizes.itemsContainerOuterWidth)) + "px";
                                }
                                // itemsContainers only get the _groupLeaderClass when header position is left.
                                _ElementUtilities.addClass(itemsContainer, _Constants._groupLeaderClass);
                            } else {
                                // Vertical with headers above
                                //
                                headerContainer.style.msGridRow = index * 2 + 1;
                                // It's important to explicitly set the container height in vertical list mode with headers above, since we use flow layout.
                                // When the header's content is taken from the DOM, the headerContainer will shrink unless it has a height set.
                                if (this._inListMode) {
                                    headerContainer.style.height = (sizes.headerContainerHeight - sizes.headerContainerOuterHeight) + "px";
                                } else {
                                    if (this._envInfo.supportsCSSGrid) {
                                        itemsContainer.style.msGridRow = index * 2 + 2;
                                    } else {
                                        headerContainer.style.height = sizes.headerContainerHeight - sizes.headerContainerOuterHeight + "px";
                                        headerContainer.style.width = (groupCrossSize - sizes.headerContainerOuterWidth) + "px";
                                        itemsContainer.style.width = (groupCrossSize - sizes.itemsContainerOuterWidth) + "px";
                                    }
                                }
                            }

                        }
                        // Header containers always get the _groupLeaderClass.
                        _ElementUtilities.addClass(headerContainer, _Constants._laidOutClass + " " + _Constants._groupLeaderClass);
                    }
                    _ElementUtilities.addClass(itemsContainer, _Constants._laidOutClass);
                }
            }, {
                // The maximum number of rows or columns of win-containers to put into each items block.
                // A row/column cannot be split across multiple items blocks. win-containers
                // are grouped into items blocks in order to mitigate the costs of the platform doing
                // a layout in response to insertions and removals of win-containers.
                _barsPerItemsBlock: 4
            });
        }),

        //
        // Layouts
        //

        _LegacyLayout: _Base.Namespace._lazy(function () {
            return _Base.Class.derive(exports._LayoutCommon, null, {
                /// <field type="Boolean" locid="WinJS.UI._LegacyLayout.disableBackdrop" helpKeyword="WinJS.UI._LegacyLayout.disableBackdrop">
                /// Gets or sets a value that indicates whether the layout should disable the backdrop feature
                /// which avoids blank areas while panning in a virtualized list.
                /// <deprecated type="deprecate">
                /// disableBackdrop is deprecated. Style: .win-listview .win-container.win-backdrop { background-color:transparent; } instead.
                /// </deprecated>
                /// </field>
                disableBackdrop: {
                    get: function _LegacyLayout_disableBackdrop_get() {
                        return this._backdropDisabled || false;
                    },
                    set: function _LegacyLayout_disableBackdrop_set(value) {
                        _ElementUtilities._deprecated(_ErrorMessages.disableBackdropIsDeprecated);
                        value = !!value;
                        if (this._backdropDisabled !== value) {
                            this._backdropDisabled = value;
                            if (this._disableBackdropClassName) {
                                deleteDynamicCssRule(this._disableBackdropClassName);
                                this._site && _ElementUtilities.removeClass(this._site.surface, this._disableBackdropClassName);
                                this._disableBackdropClassName = null;
                            }
                            this._disableBackdropClassName = uniqueCssClassName("disablebackdrop");
                            this._site && _ElementUtilities.addClass(this._site.surface, this._disableBackdropClassName);
                            if (value) {
                                var ruleSelector = ".win-container.win-backdrop",
                                    ruleBody = "background-color:transparent;";
                                addDynamicCssRule(this._disableBackdropClassName, this._site, ruleSelector, ruleBody);
                            }
                        }
                    }
                },

                /// <field type="String" locid="WinJS.UI._LegacyLayout.backdropColor" helpKeyword="WinJS.UI._LegacyLayout.backdropColor">
                /// Gets or sets the fill color for the default pattern used for the backdrops.
                /// The default value is "rgba(155,155,155,0.23)".
                /// <deprecated type="deprecate">
                /// backdropColor is deprecated. Style: .win-listview .win-container.win-backdrop { rgba(155,155,155,0.23); } instead.
                /// </deprecated>
                /// </field>
                backdropColor: {
                    get: function _LegacyLayout_backdropColor_get() {
                        return this._backdropColor || "rgba(155,155,155,0.23)";
                    },
                    set: function _LegacyLayout_backdropColor_set(value) {
                        _ElementUtilities._deprecated(_ErrorMessages.backdropColorIsDeprecated);
                        if (value && this._backdropColor !== value) {
                            this._backdropColor = value;
                            if (this._backdropColorClassName) {
                                deleteDynamicCssRule(this._backdropColorClassName);
                                this._site && _ElementUtilities.removeClass(this._site.surface, this._backdropColorClassName);
                                this._backdropColorClassName = null;
                            }
                            this._backdropColorClassName = uniqueCssClassName("backdropcolor");
                            this._site && _ElementUtilities.addClass(this._site.surface, this._backdropColorClassName);
                            var ruleSelector = ".win-container.win-backdrop",
                                ruleBody = "background-color:" + value + ";";
                            addDynamicCssRule(this._backdropColorClassName, this._site, ruleSelector, ruleBody);
                        }
                    }
                }
            });
        }),

        GridLayout: _Base.Namespace._lazy(function () {
            return _Base.Class.derive(exports._LegacyLayout, function (options) {
                /// <signature helpKeyword="WinJS.UI.GridLayout">
                /// <summary locid="WinJS.UI.GridLayout">
                /// Creates a new GridLayout.
                /// </summary>
                /// <param name="options" type="Object" locid="WinJS.UI.GridLayout_p:options">
                /// An object that contains one or more property/value pairs to apply to the new control. Each property of the options
                /// object corresponds to one of the control's properties or events.
                /// </param>
                /// <returns type="WinJS.UI.GridLayout" locid="WinJS.UI.GridLayout_returnValue">
                /// The new GridLayout.
                /// </returns>
                /// </signature>
                options = options || {};
                // Executing setters to display compatibility warning
                this.itemInfo = options.itemInfo;
                this.groupInfo = options.groupInfo;
                this._maxRowsOrColumns = 0;
                this._useDefaultItemInfo = true;
                this._elementsToMeasure = {};
                this._groupHeaderPosition = options.groupHeaderPosition || HeaderPosition.top;
                this.orientation = options.orientation || "horizontal";

                if (options.maxRows) {
                    this.maxRows = +options.maxRows;
                }
                if (options.maximumRowsOrColumns) {
                    this.maximumRowsOrColumns = +options.maximumRowsOrColumns;
                }
            }, {

                // Public

                /// <field type="Number" integer="true" locid="WinJS.UI.GridLayout.maximumRowsOrColumns" helpKeyword="WinJS.UI.GridLayout.maximumRowsOrColumns">
                /// Gets the maximum number of rows or columns, depending on the orientation, that should present before it introduces wrapping to the layout.
                /// A value of 0 indicates that there is no maximum. The default value is 0.
                /// </field>
                maximumRowsOrColumns: {
                    get: function () {
                        return this._maxRowsOrColumns;
                    },
                    set: function (value) {
                        this._setMaxRowsOrColumns(value);
                    }
                },

                /// <field type="Number" integer="true" locid="WinJS.UI.GridLayout.maxRows" helpKeyword="WinJS.UI.GridLayout.maxRows">
                /// Gets or sets the maximum number of rows displayed by the ListView.
                /// <deprecated type="deprecate">
                /// WinJS.UI.GridLayout.maxRows may be altered or unavailable after the Windows Library for JavaScript 2.0. Instead, use the maximumRowsOrColumns property.
                /// </deprecated>
                /// </field>
                maxRows: {
                    get: function () {
                        return this.maximumRowsOrColumns;
                    },
                    set: function (maxRows) {
                        _ElementUtilities._deprecated(_ErrorMessages.maxRowsIsDeprecated);
                        this.maximumRowsOrColumns = maxRows;
                    }
                },

                /// <field type="Function" locid="WinJS.UI.GridLayout.itemInfo" helpKeyword="WinJS.UI.GridLayout.itemInfo">
                /// Determines the size of the item and whether
                /// the item should be placed in a new column.
                /// <deprecated type="deprecate">
                /// GridLayout.itemInfo may be altered or unavailable in future versions. Instead, use CellSpanningLayout.
                /// </deprecated>
                /// </field>
                itemInfo: {
                    enumerable: true,
                    get: function () {
                        return this._itemInfo;
                    },
                    set: function (itemInfo) {
                        itemInfo && _ElementUtilities._deprecated(_ErrorMessages.itemInfoIsDeprecated);
                        this._itemInfo = itemInfo;
                        this._invalidateLayout();
                    }
                },

                /// <field type="Function" locid="WinJS.UI.GridLayout.groupInfo" helpKeyword="WinJS.UI.GridLayout.groupInfo">
                /// Indicates whether a group has cell spanning items and specifies the dimensions of the cell.
                /// <deprecated type="deprecate">
                /// GridLayout.groupInfo may be altered or unavailable in future versions. Instead, use CellSpanningLayout.
                /// </deprecated>
                /// </field>
                groupInfo: {
                    enumerable: true,
                    get: function () {
                        return this._groupInfo;
                    },
                    set: function (groupInfo) {
                        groupInfo && _ElementUtilities._deprecated(_ErrorMessages.groupInfoIsDeprecated);
                        this._groupInfo = groupInfo;
                        this._invalidateLayout();
                    }
                }
            });
        })
    });

    var Groups = _Base.Namespace.defineWithParent(null, null, {

        UniformGroupBase: _Base.Namespace._lazy(function () {
            return _Base.Class.define(null, {
                cleanUp: function UniformGroupBase_cleanUp() {
                },

                itemFromOffset: function UniformGroupBase_itemFromOffset(offset, options) {
                    // supported options are:
                    // - wholeItem: when set to true the fully visible item is returned
                    // - last: if 1 the last item is returned. if 0 the first
                    options = options || {};

                    var sizes = this._layout._sizes;

                    // Make offset relative to the items container's content box
                    offset -= sizes.itemsContainerOuterStart;

                    if (options.wholeItem) {
                        offset += (options.last ? -1 : 1) * (sizes.containerSize - 1);
                    }
                    var lastIndexOfGroup = this.count - 1,
                        lastBar = Math.floor(lastIndexOfGroup / this._layout._itemsPerBar),
                        bar = clampToRange(0, lastBar, Math.floor(offset / sizes.containerSize)),
                        index = (bar + options.last) * this._layout._itemsPerBar - options.last;
                    return clampToRange(0, this.count - 1, index);
                },

                hitTest: function UniformGroupBase_hitTest(x, y) {
                    var horizontal = this._layout._horizontal,
                        itemsPerBar = this._layout._itemsPerBar,
                        useListSemantics = this._layout._inListMode || itemsPerBar === 1,
                        directionalLocation = horizontal ? x : y,
                        crossLocation = horizontal ? y : x,
                        sizes = this._layout._sizes;

                    directionalLocation -= sizes.itemsContainerOuterStart;
                    crossLocation -= sizes.itemsContainerOuterCrossStart;

                    var bar = Math.floor(directionalLocation / sizes.containerSize);
                    var slotInBar = clampToRange(0, itemsPerBar - 1, Math.floor(crossLocation / sizes.containerCrossSize));
                    var index = Math.max(-1, bar * itemsPerBar + slotInBar);

                    // insertAfterIndex is determined by which half of the target element the mouse cursor is currently in.
                    // The trouble is that we can cut the element in half horizontally or cut it in half vertically.
                    // Which one we choose depends on the order that elements are laid out in the grid.
                    // A horizontal grid with multiple rows per column will lay items out starting from top to bottom, and move left to right.
                    // A vertical list is just a horizontal grid with an infinite number of rows per column, so it follows the same order.
                    // In both of these cases, each item is cut in half horizontally, since for any item n, n-1 should be above it and n+1 below (ignoring column changes).
                    // A vertical grid lays items out left to right, top to bottom, and a horizontal list left to right (with infinite items per row).
                    // In this case for item n, n-1 is on the left and n+1 on the right, so we cut the item in half vertically.
                    var insertAfterSlot;
                    if ((!horizontal && useListSemantics) ||
                        (horizontal && !useListSemantics)) {
                        insertAfterSlot = (y - sizes.containerHeight / 2) / sizes.containerHeight;
                    } else {
                        insertAfterSlot = (x - sizes.containerWidth / 2) / sizes.containerWidth;
                    }
                    if (useListSemantics) {
                        insertAfterSlot = Math.floor(insertAfterSlot);
                        return {
                            index: index,
                            insertAfterIndex: (insertAfterSlot >= 0 && index >= 0 ? insertAfterSlot : -1)
                        };
                    }
                    insertAfterSlot = clampToRange(-1, itemsPerBar - 1, insertAfterSlot);
                    var insertAfterIndex;
                    if (insertAfterSlot < 0) {
                        insertAfterIndex = bar * itemsPerBar - 1;
                    } else {
                        insertAfterIndex = bar * itemsPerBar + Math.floor(insertAfterSlot);
                    }

                    return {
                        index: clampToRange(-1, this.count - 1, index),
                        insertAfterIndex: clampToRange(-1, this.count - 1, insertAfterIndex)
                    };
                },

                getAdjacent: function UniformGroupBase_getAdjacent(currentItem, pressedKey) {
                    var index = currentItem.index,
                        currentBar = Math.floor(index / this._layout._itemsPerBar),
                        currentSlot = index % this._layout._itemsPerBar,
                        newFocus;

                    switch (pressedKey) {
                        case Key.upArrow:
                            newFocus = (currentSlot === 0 ? "boundary" : index - 1);
                            break;
                        case Key.downArrow:
                            var isLastIndexOfGroup = (index === this.count - 1),
                                inLastSlot = (this._layout._itemsPerBar > 1 && currentSlot === this._layout._itemsPerBar - 1);
                            newFocus = (isLastIndexOfGroup || inLastSlot ? "boundary" : index + 1);
                            break;
                        case Key.leftArrow:
                            newFocus = (currentBar === 0 && this._layout._itemsPerBar > 1 ? "boundary" : index - this._layout._itemsPerBar);
                            break;
                        case Key.rightArrow:
                            var lastIndexOfGroup = this.count - 1,
                                lastBar = Math.floor(lastIndexOfGroup / this._layout._itemsPerBar);
                            newFocus = (currentBar === lastBar ? "boundary" : Math.min(index + this._layout._itemsPerBar, this.count - 1));
                            break;
                    }
                    return (newFocus === "boundary" ? newFocus : { type: _UI.ObjectType.item, index: newFocus });
                },

                getItemsContainerSize: function UniformGroupBase_getItemsContainerSize() {
                    var sizes = this._layout._sizes,
                        barCount = Math.ceil(this.count / this._layout._itemsPerBar);
                    return barCount * sizes.containerSize + sizes.itemsContainerOuterSize;
                },

                getItemsContainerCrossSize: function UniformGroupBase_getItemsContainerCrossSize() {
                    var sizes = this._layout._sizes;
                    return this._layout._itemsPerBar * sizes.containerCrossSize + sizes.itemsContainerOuterCrossSize;
                },

                getItemPositionForAnimations: function UniformGroupBase_getItemPositionForAnimations(itemIndex) {
                    // Top/Left are used to know if the item has moved and also used to position the item if removed.
                    // Row/Column are used to know if a reflow animation should occur
                    // Height/Width are used when positioning a removed item without impacting layout.
                    // The returned rectangle refers to the win-container's border/padding/content box. Coordinates
                    // are relative to group's items container.

                    var sizes = this._layout._sizes;
                    var leftStr = this._layout._site.rtl ? "right" : "left";
                    var containerMargins = this._layout._sizes.containerMargins;
                    var coordinates = this._layout._indexToCoordinate(itemIndex);
                    var itemPosition = {
                        row: coordinates.row,
                        column: coordinates.column,
                        top: containerMargins.top + coordinates.row * sizes.containerHeight,
                        left: containerMargins[leftStr] + coordinates.column * sizes.containerWidth,
                        height: sizes.containerHeight - sizes.containerMargins.top - sizes.containerMargins.bottom,
                        width: sizes.containerWidth - sizes.containerMargins.left - sizes.containerMargins.right
                    };
                    return itemPosition;
                }
            });
        }),

        //
        // Groups for GridLayout
        //
        // Each group implements a 3 function layout interface. The interface is used
        // whenever GridLayout has to do a layout. The interface consists of:
        // - prepareLayout/prepareLayoutWithCopyOfTree: Called 1st. Group should update all of its internal
        //   layout state. It should not modify the DOM. Group should implement either prepareLayout or
        //   prepareLayoutWithCopyOfTree. The former is preferable because the latter is expensive as calling
        //   it requires copying the group's tree. Implementing prepareLayoutWithCopyOfTree is necessary when
        //   the group is manually laying out items and is laying out unrealized items asynchronously
        //   (e.g. CellSpanningGroup). This requires a copy of the tree from the previous layout pass.
        // - layoutRealizedRange: Called 2nd. Group should update the DOM so that
        //   the realized range reflects the internal layout state computed during
        //   prepareLayout.
        // - layoutUnrealizedRange: Called 3rd. Group should update the DOM for the items
        //   outside of the realized range. This function returns a promise so
        //   it can do its work asynchronously. When the promise completes, layout will
        //   be done.
        //
        // The motivation for this interface is perceived performance. If groups had just 1
        // layout function, all items would have to be laid out before any animations could
        // begin. With this interface, animations can begin playing after
        // layoutRealizedRange is called.
        //
        // Each group also implements a cleanUp function which is called when the group is
        // no longer needed so that it can clean up the DOM and its resources. After cleanUp
        // is called, the group object cannnot be reused.
        //

        UniformGroup: _Base.Namespace._lazy(function () {
            return _Base.Class.derive(Groups.UniformGroupBase, function UniformGroup_ctor(layout, itemsContainer) {
                this._layout = layout;
                this._itemsContainer = itemsContainer;
                _ElementUtilities.addClass(this._itemsContainer, layout._inListMode ? _Constants._uniformListLayoutClass : _Constants._uniformGridLayoutClass);
            }, {
                cleanUp: function UniformGroup_cleanUp(skipDomCleanUp) {
                    if (!skipDomCleanUp) {
                        _ElementUtilities.removeClass(this._itemsContainer, _Constants._uniformGridLayoutClass);
                        _ElementUtilities.removeClass(this._itemsContainer, _Constants._uniformListLayoutClass);
                        this._itemsContainer.style.height = this._itemsContainer.style.width = "";
                    }
                    this._itemsContainer = null;
                    this._layout = null;
                    this.groupInfo = null;
                    this.startIndex = null;
                    this.offset = null;
                    this.count = null;
                },

                prepareLayout: function UniformGroup_prepareLayout(itemsCount, oldChangedRealizedRange, oldState, updatedProperties) {
                    this.groupInfo = updatedProperties.groupInfo;
                    this.startIndex = updatedProperties.startIndex;
                    this.count = itemsCount;
                    return this._layout._ensureContainerSize(this);
                },

                layoutRealizedRange: function UniformGroup_layoutRealizedRange() {
                    // Explicitly set the items container size. This is required so that the
                    // surface, which is a grid, will have its width sized to content.
                    var sizes = this._layout._sizes;
                    this._itemsContainer.style[this._layout._horizontal ? "width" : "height"] = this.getItemsContainerSize() - sizes.itemsContainerOuterSize + "px";
                    this._itemsContainer.style[this._layout._horizontal ? "height" : "width"] = (this._layout._inListMode ? sizes.maxItemsContainerContentSize + "px" :
                                                                                                 this._layout._itemsPerBar * sizes.containerCrossSize + "px");
                },

                layoutUnrealizedRange: function UniformGroup_layoutUnrealizedRange() {
                    return Promise.wrap();
                }
            });
        }),

        UniformFlowGroup: _Base.Namespace._lazy(function () {
            return _Base.Class.derive(Groups.UniformGroupBase, function UniformFlowGroup_ctor(layout, tree) {
                this._layout = layout;
                this._itemsContainer = tree.element;
                _ElementUtilities.addClass(this._itemsContainer, layout._inListMode ? _Constants._uniformListLayoutClass : _Constants._uniformGridLayoutClass);
            }, {
                cleanUp: function UniformFlowGroup_cleanUp(skipDomCleanUp) {
                    if (!skipDomCleanUp) {
                        _ElementUtilities.removeClass(this._itemsContainer, _Constants._uniformListLayoutClass);
                        _ElementUtilities.removeClass(this._itemsContainer, _Constants._uniformGridLayoutClass);
                        this._itemsContainer.style.height = "";
                    }
                },
                layout: function UniformFlowGroup_layout() {
                    this._layout._site._writeProfilerMark("Layout:_UniformFlowGroup:setItemsContainerHeight,info");
                    this._itemsContainer.style.height = this.count * this._layout._sizes.containerHeight + "px";
                }
            });
        }),

        CellSpanningGroup: _Base.Namespace._lazy(function () {
            return _Base.Class.define(function CellSpanningGroup_ctor(layout, itemsContainer) {
                this._layout = layout;
                this._itemsContainer = itemsContainer;
                _ElementUtilities.addClass(this._itemsContainer, _Constants._cellSpanningGridLayoutClass);

                this.resetMap();
            }, {
                cleanUp: function CellSpanningGroup_cleanUp(skipDomCleanUp) {
                    if (!skipDomCleanUp) {
                        this._cleanContainers();
                        _ElementUtilities.removeClass(this._itemsContainer, _Constants._cellSpanningGridLayoutClass);
                        this._itemsContainer.style.cssText = "";
                    }
                    this._itemsContainer = null;

                    if (this._layoutPromise) {
                        this._layoutPromise.cancel();
                        this._layoutPromise = null;
                    }
                    this.resetMap();
                    this._slotsPerColumn = null;
                    this._offScreenSlotsPerColumn = null;
                    this._items = null;
                    this._layout = null;
                    this._containersToHide = null;
                    this.groupInfo = null;
                    this.startIndex = null;
                    this.offset = null;
                    this.count = null;
                },

                prepareLayoutWithCopyOfTree: function CellSpanningGroup_prepareLayoutWithCopyOfTree(tree, oldChangedRealizedRange, oldState, updatedProperties) {
                    var that = this;
                    var i;

                    // Remember the items in the old realized item range that changed.
                    // During layoutRealizedRange, they either need to be relaid out or hidden.
                    this._containersToHide = {};
                    if (oldChangedRealizedRange) {
                        for (i = oldChangedRealizedRange.firstIndex; i <= oldChangedRealizedRange.lastIndex; i++) {
                            this._containersToHide[uniqueID(oldState._items[i])] = oldState._items[i];
                        }
                    }

                    // Update public properties
                    this.groupInfo = updatedProperties.groupInfo;
                    this.startIndex = updatedProperties.startIndex;
                    this.count = tree.items.length;

                    this._items = tree.items;
                    this._slotsPerColumn = Math.floor(this._layout._sizes.maxItemsContainerContentSize / this.groupInfo.cellHeight);
                    if (this._layout.maximumRowsOrColumns) {
                        this._slotsPerColumn = Math.min(this._slotsPerColumn, this._layout.maximumRowsOrColumns);
                    }
                    this._slotsPerColumn = Math.max(this._slotsPerColumn, 1);

                    this.resetMap();
                    var itemInfoPromises = new Array(this.count);
                    for (i = 0; i < this.count; i++) {
                        itemInfoPromises[i] = this._layout._getItemInfo(this.startIndex + i);
                    }
                    return Promise.join(itemInfoPromises).then(function (itemInfos) {
                        itemInfos.forEach(function (itemInfo, index) {
                            that.addItemToMap(index, itemInfo);
                        });
                    });
                },

                layoutRealizedRange: function CellSpanningGroup_layoutRealizedRange(firstChangedIndex, realizedRange) {
                    // Lay out the changed items within the realized range
                    if (realizedRange) {
                        var firstChangedRealizedIndex = Math.max(firstChangedIndex, realizedRange.firstIndex),
                            i;
                        for (i = firstChangedRealizedIndex; i <= realizedRange.lastIndex; i++) {
                            this._layoutItem(i);
                            delete this._containersToHide[uniqueID(this._items[i])];
                        }
                    }

                    // Hide the old containers that are in the realized range but weren't relaid out
                    Object.keys(this._containersToHide).forEach(function (id) {
                        _ElementUtilities.removeClass(this._containersToHide[id], _Constants._laidOutClass);
                    }.bind(this));
                    this._containersToHide = {};

                    // Explicitly set the items container height. This is required so that the
                    // surface, which is a grid, will have its width sized to content.
                    this._itemsContainer.style.cssText +=
                        ";width:" + (this.getItemsContainerSize() - this._layout._sizes.itemsContainerOuterSize) +
                        "px;height:" + this._layout._sizes.maxItemsContainerContentSize +
                        "px;-ms-grid-columns: (" + this.groupInfo.cellWidth + "px)[" + this.getColumnCount() +
                        "];-ms-grid-rows: (" + this.groupInfo.cellHeight + "px)[" + (this._slotsPerColumn + this._offScreenSlotsPerColumn) + "]";
                },

                layoutUnrealizedRange: function CellSpanningGroup_layoutUnrealizedRange(firstChangedIndex, realizedRange, beforeRealizedRange) {
                    var that = this;
                    var layoutJob;

                    that._layoutPromise = new Promise(function (complete) {
                        function completeLayout() {
                            layoutJob = null;
                            complete();
                        }

                        function schedule(fn) {
                            return Scheduler.schedule(fn, Scheduler.Priority.normal, null,
                                "WinJS.UI.GridLayout.CellSpanningGroup.LayoutUnrealizedRange");
                        }

                        // These loops are built to lay out the items that are closest to
                        // the realized range first.

                        if (realizedRange) {
                            var stop = false;
                            // For laying out the changed items that are before the realized range
                            var before = realizedRange.firstIndex - 1;
                            // For laying out the changed items that are after the realized range
                            var after = Math.max(firstChangedIndex, realizedRange.lastIndex + 1);
                            after = Math.max(before + 1, after);

                            // Alternate between laying out items before and after the realized range
                            layoutJob = schedule(function unrealizedRangeWork(info) {
                                while (!stop) {
                                    if (info.shouldYield) {
                                        info.setWork(unrealizedRangeWork);
                                        return;
                                    }

                                    stop = true;

                                    if (before >= firstChangedIndex) {
                                        that._layoutItem(before);
                                        before--;
                                        stop = false;
                                    }
                                    if (after < that.count) {
                                        that._layoutItem(after);
                                        after++;
                                        stop = false;
                                    }
                                }
                                completeLayout();
                            });
                        } else if (beforeRealizedRange) {
                            // The items we are laying out come before the realized range.
                            // so lay them out in descending order.
                            var i = that.count - 1;
                            layoutJob = schedule(function beforeRealizedRangeWork(info) {
                                for (; i >= firstChangedIndex; i--) {
                                    if (info.shouldYield) {
                                        info.setWork(beforeRealizedRangeWork);
                                        return;
                                    }
                                    that._layoutItem(i);
                                }
                                completeLayout();
                            });
                        } else {
                            // The items we are laying out come after the realized range
                            // so lay them out in ascending order.
                            var i = firstChangedIndex;
                            layoutJob = schedule(function afterRealizedRangeWork(info) {
                                for (; i < that.count; i++) {
                                    if (info.shouldYield) {
                                        info.setWork(afterRealizedRangeWork);
                                        return;
                                    }
                                    that._layoutItem(i);
                                }
                                completeLayout();
                            });
                        }
                    }, function () {
                        // Cancellation handler for that._layoutPromise
                        layoutJob && layoutJob.cancel();
                        layoutJob = null;
                    });

                    return that._layoutPromise;
                },

                itemFromOffset: function CellSpanningGroup_itemFromOffset(offset, options) {
                    // supported options are:
                    // - wholeItem: when set to true the fully visible item is returned
                    // - last: if 1 the last item is returned. if 0 the first
                    options = options || {};

                    var sizes = this._layout._sizes,
                        margins = sizes.containerMargins;

                    // Make offset relative to the items container's content box
                    offset -= sizes.itemsContainerOuterX;

                    offset -= ((options.last ? 1 : -1) * margins[options.last ? "left" : "right"]);

                    var value = this.indexFromOffset(offset, options.wholeItem, options.last).item;
                    return clampToRange(0, this.count - 1, value);
                },

                getAdjacent: function CellSpanningGroup_getAdjacent(currentItem, pressedKey) {
                    var index,
                        originalIndex;

                    index = originalIndex = currentItem.index;

                    var newIndex, inMap, inMapIndex;
                    if (this.lastAdjacent === index) {
                        inMapIndex = this.lastInMapIndex;
                    } else {
                        inMapIndex = this.findItem(index);
                    }

                    do {
                        var column = Math.floor(inMapIndex / this._slotsPerColumn),
                            row = inMapIndex - column * this._slotsPerColumn,
                            lastColumn = Math.floor((this.occupancyMap.length - 1) / this._slotsPerColumn);

                        switch (pressedKey) {
                            case Key.upArrow:
                                if (row > 0) {
                                    inMapIndex--;
                                } else {
                                    return { type: _UI.ObjectType.item, index: originalIndex };
                                }
                                break;
                            case Key.downArrow:
                                if (row + 1 < this._slotsPerColumn) {
                                    inMapIndex++;
                                } else {
                                    return { type: _UI.ObjectType.item, index: originalIndex };
                                }
                                break;
                            case Key.leftArrow:
                                inMapIndex = (column > 0 ? inMapIndex - this._slotsPerColumn : -1);
                                break;
                            case Key.rightArrow:
                                inMapIndex = (column < lastColumn ? inMapIndex + this._slotsPerColumn : this.occupancyMap.length);
                                break;
                        }

                        inMap = inMapIndex >= 0 && inMapIndex < this.occupancyMap.length;
                        if (inMap) {
                            newIndex = this.occupancyMap[inMapIndex] ? this.occupancyMap[inMapIndex].index : undefined;
                        }

                    } while (inMap && (index === newIndex || newIndex === undefined));

                    this.lastAdjacent = newIndex;
                    this.lastInMapIndex = inMapIndex;

                    return (inMap ? { type: _UI.ObjectType.item, index: newIndex } : "boundary");
                },

                hitTest: function CellSpanningGroup_hitTest(x, y) {
                    var sizes = this._layout._sizes,
                        itemIndex = 0;

                    // Make the coordinates relative to the items container's content box
                    x -= sizes.itemsContainerOuterX;
                    y -= sizes.itemsContainerOuterY;

                    if (this.occupancyMap.length > 0) {
                        var result = this.indexFromOffset(x, false, 0);

                        var counter = Math.min(this._slotsPerColumn - 1, Math.floor(y / this.groupInfo.cellHeight)),
                            curr = result.index,
                            lastValidIndex = curr;
                        while (counter-- > 0) {
                            curr++;
                            if (this.occupancyMap[curr]) {
                                lastValidIndex = curr;
                            }
                        }
                        if (!this.occupancyMap[lastValidIndex]) {
                            lastValidIndex--;
                        }
                        itemIndex = this.occupancyMap[lastValidIndex].index;
                    }

                    var itemSize = this.getItemSize(itemIndex),
                        itemLeft = itemSize.column * this.groupInfo.cellWidth,
                        itemTop = itemSize.row * this.groupInfo.cellHeight,
                        useListSemantics = this._slotsPerColumn === 1,
                        insertAfterIndex = itemIndex;

                    if ((useListSemantics && (x < (itemLeft + itemSize.contentWidth / 2))) ||
                        (!useListSemantics && (y < (itemTop + itemSize.contentHeight / 2)))) {
                        insertAfterIndex--;
                    }

                    return {
                        type: _UI.ObjectType.item,
                        index: clampToRange(0, this.count - 1, itemIndex),
                        insertAfterIndex: clampToRange(-1, this.count - 1, insertAfterIndex)
                    };
                },

                getItemsContainerSize: function CellSpanningGroup_getItemsContainerSize() {
                    var sizes = this._layout._sizes;
                    return this.getColumnCount() * this.groupInfo.cellWidth + sizes.itemsContainerOuterSize;
                },

                getItemsContainerCrossSize: function CellSpanningGroup_getItemsContainerCrossSize() {
                    var sizes = this._layout._sizes;
                    return sizes.maxItemsContainerContentSize + sizes.itemsContainerOuterCrossSize;
                },

                getItemPositionForAnimations: function CellSpanningGroup_getItemPositionForAnimations(itemIndex) {
                    // Top/Left are used to know if the item has moved and also used to position the item if removed.
                    // Row/Column are used to know if a reflow animation should occur
                    // Height/Width are used when positioning a removed item without impacting layout.
                    // The returned rectangle refers to the win-container's border/padding/content box. Coordinates
                    // are relative to group's items container.

                    var leftStr = this._layout._site.rtl ? "right" : "left";
                    var containerMargins = this._layout._sizes.containerMargins;
                    var coordinates = this.getItemSize(itemIndex);
                    var groupInfo = this.groupInfo;
                    var itemPosition = {
                        row: coordinates.row,
                        column: coordinates.column,
                        top: containerMargins.top + coordinates.row * groupInfo.cellHeight,
                        left: containerMargins[leftStr] + coordinates.column * groupInfo.cellWidth,
                        height: coordinates.contentHeight,
                        width: coordinates.contentWidth
                    };

                    return itemPosition;
                },

                _layoutItem: function CellSpanningGroup_layoutItem(index) {
                    var entry = this.getItemSize(index);
                    this._items[index].style.cssText +=
                        ";-ms-grid-row:" + (entry.row + 1) +
                        ";-ms-grid-column:" + (entry.column + 1) +
                        ";-ms-grid-row-span:" + entry.rows +
                        ";-ms-grid-column-span:" + entry.columns +
                        ";height:" + entry.contentHeight +
                        "px;width:" + entry.contentWidth + "px";
                    _ElementUtilities.addClass(this._items[index], _Constants._laidOutClass);

                    return this._items[index];
                },

                _cleanContainers: function CellSpanningGroup_cleanContainers() {
                    var items = this._items,
                        len = items.length,
                        i;
                    for (i = 0; i < len; i++) {
                        items[i].style.cssText = "";
                        _ElementUtilities.removeClass(items[i], _Constants._laidOutClass);
                    }
                },

                // Occupancy map

                getColumnCount: function CellSpanningGroup_getColumnCount() {
                    return Math.ceil(this.occupancyMap.length / this._slotsPerColumn);
                },

                getOccupancyMapItemCount: function CellSpanningGroup_getOccupancyMapItemCount() {
                    var index = -1;

                    // Use forEach as the map may be sparse
                    this.occupancyMap.forEach(function (item) {
                        if (item.index > index) {
                            index = item.index;
                        }
                    });

                    return index + 1;
                },

                coordinateToIndex: function CellSpanningGroup_coordinateToIndex(c, r) {
                    return c * this._slotsPerColumn + r;
                },

                markSlotAsFull: function CellSpanningGroup_markSlotAsFull(index, itemEntry) {
                    var coordinates = this._layout._indexToCoordinate(index, this._slotsPerColumn),
                        toRow = coordinates.row + itemEntry.rows;
                    for (var r = coordinates.row; r < toRow && r < this._slotsPerColumn; r++) {
                        for (var c = coordinates.column, toColumn = coordinates.column + itemEntry.columns; c < toColumn; c++) {
                            this.occupancyMap[this.coordinateToIndex(c, r)] = itemEntry;
                        }
                    }
                    this._offScreenSlotsPerColumn = Math.max(this._offScreenSlotsPerColumn, toRow - this._slotsPerColumn);
                },

                isSlotEmpty: function CellSpanningGroup_isSlotEmpty(itemSize, row, column) {
                    for (var r = row, toRow = row + itemSize.rows; r < toRow; r++) {
                        for (var c = column, toColumn = column + itemSize.columns; c < toColumn; c++) {
                            if ((r >= this._slotsPerColumn) || (this.occupancyMap[this.coordinateToIndex(c, r)] !== undefined)) {
                                return false;
                            }
                        }
                    }
                    return true;
                },

                findEmptySlot: function CellSpanningGroup_findEmptySlot(startIndex, itemSize, newColumn) {
                    var coordinates = this._layout._indexToCoordinate(startIndex, this._slotsPerColumn),
                        startRow = coordinates.row,
                        lastColumn = Math.floor((this.occupancyMap.length - 1) / this._slotsPerColumn);

                    if (newColumn) {
                        for (var c = coordinates.column + 1; c <= lastColumn; c++) {
                            if (this.isSlotEmpty(itemSize, 0, c)) {
                                return this.coordinateToIndex(c, 0);
                            }
                        }
                    } else {
                        for (var c = coordinates.column; c <= lastColumn; c++) {
                            for (var r = startRow; r < this._slotsPerColumn; r++) {
                                if (this.isSlotEmpty(itemSize, r, c)) {
                                    return this.coordinateToIndex(c, r);
                                }
                            }
                            startRow = 0;
                        }
                    }

                    return (lastColumn + 1) * this._slotsPerColumn;
                },

                findItem: function CellSpanningGroup_findItem(index) {
                    for (var inMapIndex = index, len = this.occupancyMap.length; inMapIndex < len; inMapIndex++) {
                        var entry = this.occupancyMap[inMapIndex];
                        if (entry && entry.index === index) {
                            return inMapIndex;
                        }
                    }
                    return inMapIndex;
                },

                getItemSize: function CellSpanningGroup_getItemSize(index) {
                    var inMapIndex = this.findItem(index),
                        entry = this.occupancyMap[inMapIndex],
                        coords = this._layout._indexToCoordinate(inMapIndex, this._slotsPerColumn);

                    if (index === entry.index) {
                        return {
                            row: coords.row,
                            column: coords.column,
                            contentWidth: entry.contentWidth,
                            contentHeight: entry.contentHeight,
                            columns: entry.columns,
                            rows: entry.rows
                        };
                    } else {
                        return null;
                    }
                },

                resetMap: function CellSpanningGroup_resetMap() {
                    this.occupancyMap = [];
                    this.lastAdded = 0;
                    this._offScreenSlotsPerColumn = 0;
                },

                addItemToMap: function CellSpanningGroup_addItemToMap(index, itemInfo) {
                    var that = this;

                    function add(mapEntry, newColumn) {
                        var inMapIndex = that.findEmptySlot(that.lastAdded, mapEntry, newColumn);
                        that.lastAdded = inMapIndex;
                        that.markSlotAsFull(inMapIndex, mapEntry);
                    }

                    var groupInfo = that.groupInfo,
                        margins = that._layout._sizes.containerMargins,
                        mapEntry = {
                            index: index,
                            contentWidth: itemInfo.width,
                            contentHeight: itemInfo.height,
                            columns: Math.max(1, Math.ceil((itemInfo.width + margins.left + margins.right) / groupInfo.cellWidth)),
                            rows: Math.max(1, Math.ceil((itemInfo.height + margins.top + margins.bottom) / groupInfo.cellHeight))
                        };

                    add(mapEntry, itemInfo.newColumn);
                },

                indexFromOffset: function CellSpanningGroup_indexFromOffset(adjustedOffset, wholeItem, last) {
                    var measuredWidth = 0,
                        lastItem = 0,
                        groupInfo = this.groupInfo,
                        index = 0;

                    if (this.occupancyMap.length > 0) {
                        lastItem = this.getOccupancyMapItemCount() - 1;
                        measuredWidth = Math.ceil((this.occupancyMap.length - 1) / this._slotsPerColumn) * groupInfo.cellWidth;

                        if (adjustedOffset < measuredWidth) {
                            var counter = this._slotsPerColumn,
                                index = (Math.max(0, Math.floor(adjustedOffset / groupInfo.cellWidth)) + last) * this._slotsPerColumn - last;
                            while (!this.occupancyMap[index] && counter-- > 0) {
                                index += (last > 0 ? -1 : 1);
                            }
                            return {
                                index: index,
                                item: this.occupancyMap[index].index
                            };
                        } else {
                            index = this.occupancyMap.length - 1;
                        }
                    }

                    return {
                        index: index,
                        item: lastItem + (Math.max(0, Math.floor((adjustedOffset - measuredWidth) / groupInfo.cellWidth)) + last) * this._slotsPerColumn - last
                    };
                }
            });
        })

    });

    _Base.Namespace._moduleDefine(exports, "WinJS.UI", {

        ListLayout: _Base.Namespace._lazy(function () {
            return _Base.Class.derive(exports._LegacyLayout, function ListLayout_ctor(options) {
                /// <signature helpKeyword="WinJS.UI.ListLayout">
                /// <summary locid="WinJS.UI.ListLayout">
                /// Creates a new ListLayout object.
                /// </summary>
                /// <param name="options" type="Object" locid="WinJS.UI.ListLayout_p:options">
                /// An object that contains one or more property/value pairs to apply to the new control. Each property of the options
                /// object corresponds to one of the object's properties or events. Event names must begin with "on".
                /// </param>
                /// <returns type="WinJS.UI.ListLayout" locid="WinJS.UI.ListLayout_returnValue">
                /// The new ListLayout object.
                /// </returns>
                /// </signature>
                options = options || {};
                this._itemInfo = {};
                this._groupInfo = {};
                this._groupHeaderPosition = options.groupHeaderPosition || HeaderPosition.top;
                this._inListMode = true;
                this.orientation = options.orientation || "vertical";
            }, {
                initialize: function ListLayout_initialize(site, groupsEnabled) {
                    _ElementUtilities.addClass(site.surface, _Constants._listLayoutClass);
                    exports._LegacyLayout.prototype.initialize.call(this, site, groupsEnabled);
                },

                uninitialize: function ListLayout_uninitialize() {
                    if (this._site) {
                        _ElementUtilities.removeClass(this._site.surface, _Constants._listLayoutClass);
                    }
                    exports._LegacyLayout.prototype.uninitialize.call(this);
                },

                layout: function ListLayout_layout(tree, changedRange, modifiedItems, modifiedGroups) {
                    if (!this._groupsEnabled && !this._horizontal) {
                        return this._layoutNonGroupedVerticalList(tree, changedRange, modifiedItems, modifiedGroups);
                    } else {
                        return exports._LegacyLayout.prototype.layout.call(this, tree, changedRange, modifiedItems, modifiedGroups);
                    }
                },

                _layoutNonGroupedVerticalList: function ListLayout_layoutNonGroupedVerticalList(tree, changedRange, modifiedItems, modifiedGroups) {
                    var that = this;
                    var perfId = "Layout:_layoutNonGroupedVerticalList";
                    that._site._writeProfilerMark(perfId + ",StartTM");
                    this._layoutPromise = that._measureItem(0).then(function () {
                        _ElementUtilities[(that._usingStructuralNodes) ? "addClass" : "removeClass"]
                            (that._site.surface, _Constants._structuralNodesClass);
                        _ElementUtilities[(that._envInfo.nestedFlexTooLarge || that._envInfo.nestedFlexTooSmall) ? "addClass" : "removeClass"]
                            (that._site.surface, _Constants._singleItemsBlockClass);


                        if (that._sizes.viewportContentSize !== that._getViewportCrossSize()) {
                            that._viewportSizeChanged(that._getViewportCrossSize());
                        }

                        that._cacheRemovedElements(modifiedItems, that._cachedItemRecords, that._cachedInsertedItemRecords, that._cachedRemovedItems, false);
                        that._cacheRemovedElements(modifiedGroups, that._cachedHeaderRecords, that._cachedInsertedHeaderRecords, that._cachedRemovedHeaders, true);

                        var itemsContainer = tree[0].itemsContainer,
                            group = new Groups.UniformFlowGroup(that, itemsContainer);
                        that._groups = [group];
                        group.groupInfo = { enableCellSpanning: false };
                        group.startIndex = 0;
                        group.count = getItemsContainerLength(itemsContainer);
                        group.offset = 0;
                        group.layout();

                        that._site._writeProfilerMark(perfId + ":setSurfaceWidth,info");
                        that._site.surface.style.width = that._sizes.surfaceContentSize + "px";

                        that._layoutAnimations(modifiedItems, modifiedGroups);
                        that._site._writeProfilerMark(perfId + ":complete,info");
                        that._site._writeProfilerMark(perfId + ",StopTM");
                    }, function (error) {
                        that._site._writeProfilerMark(perfId + ":canceled,info");
                        that._site._writeProfilerMark(perfId + ",StopTM");
                        return Promise.wrapError(error);
                    });
                    return {
                        realizedRangeComplete: this._layoutPromise,
                        layoutComplete: this._layoutPromise
                    };
                },

                numberOfItemsPerItemsBlock: {
                    get: function ListLayout_getNumberOfItemsPerItemsBlock() {
                        var that = this;
                        // Measure when numberOfItemsPerItemsBlock is called so that we measure before ListView has created the full tree structure
                        // which reduces the trident layout required by measure.
                        return this._measureItem(0).then(function () {
                            if (that._envInfo.nestedFlexTooLarge || that._envInfo.nestedFlexTooSmall) {
                                // Store all items in a single itemsblock
                                that._usingStructuralNodes = true;
                                return Number.MAX_VALUE;
                            } else {
                                that._usingStructuralNodes = exports.ListLayout._numberOfItemsPerItemsBlock > 0;
                                return exports.ListLayout._numberOfItemsPerItemsBlock;
                            }
                        });
                    }
                },
            }, {
                // The maximum number of win-containers to put into each items block. win-containers
                // are grouped into items blocks in order to mitigate the costs of the platform doing
                // a layout in response to insertions and removals of win-containers.
                _numberOfItemsPerItemsBlock: 10
            });
        }),

        CellSpanningLayout: _Base.Namespace._lazy(function () {
            return _Base.Class.derive(exports._LayoutCommon, function CellSpanningLayout_ctor(options) {
                /// <signature helpKeyword="WinJS.UI.CellSpanningLayout">
                /// <summary locid="WinJS.UI.CellSpanningLayout">
                /// Creates a new CellSpanningLayout object.
                /// </summary>
                /// <param name="options" type="Object" locid="WinJS.UI.CellSpanningLayout_p:options">
                /// An object that contains one or more property/value pairs to apply to the new object. Each property of the options
                /// object corresponds to one of the object's properties or events. Event names must begin with "on".
                /// </param>
                /// <returns type="WinJS.UI.CellSpanningLayout" locid="WinJS.UI.CellSpanningLayout_returnValue">
                /// The new CellSpanningLayout object.
                /// </returns>
                /// </signature>
                options = options || {};
                this._itemInfo = options.itemInfo;
                this._groupInfo = options.groupInfo;
                this._groupHeaderPosition = options.groupHeaderPosition || HeaderPosition.top;
                this._horizontal = true;
                this._cellSpanning = true;
            }, {

                /// <field type="Number" integer="true" locid="WinJS.UI.CellSpanningLayout.maximumRowsOrColumns" helpKeyword="WinJS.UI.CellSpanningLayout.maximumRowsOrColumns">
                /// Gets or set the maximum number of rows or columns, depending on the orientation, to display before content begins to wrap.
                /// A value of 0 indicates that there is no maximum.
                /// </field>
                maximumRowsOrColumns: {
                    get: function () {
                        return this._maxRowsOrColumns;
                    },
                    set: function (value) {
                        this._setMaxRowsOrColumns(value);
                    }
                },

                /// <field type="Function" locid="WinJS.UI.CellSpanningLayout.itemInfo" helpKeyword="WinJS.UI.CellSpanningLayout.itemInfo">
                /// Gets or sets a function that returns the width and height of an item, as well as whether
                /// it should  appear in a new column. Setting this function improves performance because
                /// the ListView can allocate space for an item without having to measure it first.
                /// The function takes a single parameter: the index of the item to render.
                /// The function returns an object that has three properties:
                /// width: The  total width of the item.
                /// height: The total height of the item.
                /// newColumn: Set to true to create a column break; otherwise, false.
                /// </field>
                itemInfo: {
                    enumerable: true,
                    get: function () {
                        return this._itemInfo;
                    },
                    set: function (itemInfo) {
                        this._itemInfo = itemInfo;
                        this._invalidateLayout();
                    }
                },

                /// <field type="Function" locid="WinJS.UI.CellSpanningLayout.groupInfo" helpKeyword="WinJS.UI.CellSpanningLayout.groupInfo">
                /// Gets or sets a function that enables cell-spanning and establishes base cell dimensions.
                /// The function returns an object that has these properties:
                /// enableCellSpanning: Set to true to allow the ListView to contain items of multiple sizes.
                /// cellWidth: The width of the base cell.
                /// cellHeight: The height of the base cell.
                /// </field>
                groupInfo: {
                    enumerable: true,
                    get: function () {
                        return this._groupInfo;
                    },
                    set: function (groupInfo) {
                        this._groupInfo = groupInfo;
                        this._invalidateLayout();
                    }
                },

                /// <field type="String" oamOptionsDatatype="WinJS.UI.Orientation" locid="WinJS.UI.CellSpanningLayout.orientation" helpKeyword="WinJS.UI.CellSpanningLayout.orientation">
                /// Gets the orientation of the layout. CellSpanning layout only supports horizontal orientation.
                /// </field>
                orientation: {
                    enumerable: true,
                    get: function () {
                        return "horizontal";
                    }
                }
            });
        }),

        _LayoutWrapper: _Base.Namespace._lazy(function () {
            return _Base.Class.define(function LayoutWrapper_ctor(layout) {
                this.defaultAnimations = true;

                // Initialize and hitTest are required
                this.initialize = function LayoutWrapper_initialize(site, groupsEnabled) {
                    layout.initialize(site, groupsEnabled);
                };
                this.hitTest = function LayoutWrapper_hitTest(x, y) {
                    return layout.hitTest(x, y);
                };

                // These methods are optional
                layout.uninitialize && (this.uninitialize = function LayoutWrapper_uninitialize() {
                    layout.uninitialize();
                });

                if ("numberOfItemsPerItemsBlock" in layout) {
                    Object.defineProperty(this, "numberOfItemsPerItemsBlock", {
                        get: function LayoutWrapper_getNumberOfItemsPerItemsBlock() {
                            return layout.numberOfItemsPerItemsBlock;
                        }
                    });
                }

                layout._getItemPosition && (this._getItemPosition = function LayoutWrapper_getItemPosition(index) {
                    return layout._getItemPosition(index);
                });

                layout.itemsFromRange && (this.itemsFromRange = function LayoutWrapper_itemsFromRange(start, end) {
                    return layout.itemsFromRange(start, end);
                });

                layout.getAdjacent && (this.getAdjacent = function LayoutWrapper_getAdjacent(currentItem, pressedKey) {
                    return layout.getAdjacent(currentItem, pressedKey);
                });

                layout.dragOver && (this.dragOver = function LayoutWrapper_dragOver(x, y, dragInfo) {
                    return layout.dragOver(x, y, dragInfo);
                });

                layout.dragLeave && (this.dragLeave = function LayoutWrapper_dragLeave() {
                    return layout.dragLeave();
                });
                var propertyDefinition = {
                    enumerable: true,
                    get: function () {
                        return "vertical";
                    }
                };
                if (layout.orientation !== undefined) {
                    propertyDefinition.get = function () {
                        return layout.orientation;
                    };
                    propertyDefinition.set = function (value) {
                        layout.orientation = value;
                    };
                }
                Object.defineProperty(this, "orientation", propertyDefinition);

                if (layout.setupAnimations || layout.executeAnimations) {
                    this.defaultAnimations = false;
                    this.setupAnimations = function LayoutWrapper_setupAnimations() {
                        return layout.setupAnimations();
                    };
                    this.executeAnimations = function LayoutWrapper_executeAnimations() {
                        return layout.executeAnimations();
                    };
                }

                if (layout.layout) {
                    if (this.defaultAnimations) {
                        var that = this;
                        this.layout = function LayoutWrapper_layout(tree, changedRange, modifiedItems, modifiedGroups) {
                            var promises = normalizeLayoutPromises(layout.layout(tree, changedRange, [], [])),
                                synchronous;
                            promises.realizedRangeComplete.then(function () {
                                synchronous = true;
                            });
                            synchronous && that._layoutAnimations(modifiedItems, modifiedGroups);
                            return promises;
                        };
                    } else {
                        this.layout = function LayoutWrapper_layout(tree, changedRange, modifiedItems, modifiedGroups) {
                            return normalizeLayoutPromises(layout.layout(tree, changedRange, modifiedItems, modifiedGroups));
                        };
                    }
                }
            }, {
                uninitialize: function LayoutWrapper_uninitialize() {
                },
                numberOfItemsPerItemsBlock: {
                    get: function LayoutWrapper_getNumberOfItemsPerItemsBlock() {
                    }
                },
                layout: function LayoutWrapper_layout(tree, changedRange, modifiedItems, modifiedGroups) {
                    if (this.defaultAnimations) {
                        this._layoutAnimations(modifiedItems, modifiedGroups);
                    }
                    return normalizeLayoutPromises();
                },
                itemsFromRange: function LayoutWrapper_itemsFromRange() {
                    return { firstIndex: 0, lastIndex: Number.MAX_VALUE };
                },
                getAdjacent: function LayoutWrapper_getAdjacent(currentItem, pressedKey) {

                    switch (pressedKey) {
                        case Key.pageUp:
                        case Key.upArrow:
                        case Key.leftArrow:
                            return { type: currentItem.type, index: currentItem.index - 1 };
                        case Key.downArrow:
                        case Key.rightArrow:
                        case Key.pageDown:
                            return { type: currentItem.type, index: currentItem.index + 1 };
                    }
                },
                dragOver: function LayoutWrapper_dragOver() {
                },
                dragLeave: function LayoutWrapper_dragLeaver() {
                },
                setupAnimations: function LayoutWrapper_setupAnimations() {
                },
                executeAnimations: function LayoutWrapper_executeAnimations() {
                },
                _getItemPosition: function LayoutWrapper_getItemPosition() {
                },
                _layoutAnimations: function LayoutWrapper_layoutAnimations() {
                },
            });
        }),
    });

    function normalizeLayoutPromises(retVal) {
        if (Promise.is(retVal)) {
            return {
                realizedRangeComplete: retVal,
                layoutComplete: retVal
            };
        } else if (typeof retVal === "object" && retVal && retVal.layoutComplete) {
            return retVal;
        } else {
            return {
                realizedRangeComplete: Promise.wrap(),
                layoutComplete: Promise.wrap()
            };
        }
    }

    var HeaderPosition = {
        left: "left",
        top: "top"
    };

    function getMargins(element) {
        return {
            left: getDimension(element, "marginLeft"),
            right: getDimension(element, "marginRight"),
            top: getDimension(element, "marginTop"),
            bottom: getDimension(element, "marginBottom")
        };
    }

    // Layout, _LayoutCommon, and _LegacyLayout are defined ealier so that their fully
    // qualified names can be used in _Base.Class.derive. This is required by Blend.
    _Base.Namespace._moduleDefine(exports, "WinJS.UI", {
        HeaderPosition: HeaderPosition,
        _getMargins: getMargins
    });
});

// Copyright (c) Microsoft Corporation.  All Rights Reserved. Licensed under the MIT License. See License.txt in the project root for license information.
define('WinJS/Controls/ListView/_VirtualizeContentsView',[
    'exports',
    '../../Core/_Global',
    '../../Core/_Base',
    '../../Core/_BaseUtils',
    '../../Promise',
    '../../_Signal',
    '../../Scheduler',
    '../../Utilities/_Dispose',
    '../../Utilities/_ElementUtilities',
    '../../Utilities/_SafeHtml',
    '../../Utilities/_UI',
    '../ItemContainer/_Constants',
    '../ItemContainer/_ItemEventsHandler',
    './_Helpers',
    './_ItemsContainer'
], function virtualizeContentsViewInit(exports, _Global, _Base, _BaseUtils, Promise, _Signal, Scheduler, _Dispose, _ElementUtilities, _SafeHtml, _UI, _Constants, _ItemEventsHandler, _Helpers, _ItemsContainer) {
    "use strict";

    function setFlow(from, to) {
        _ElementUtilities._setAttribute(from, "aria-flowto", to.id);
        _ElementUtilities._setAttribute(to, "x-ms-aria-flowfrom", from.id);
    }

    _Base.Namespace._moduleDefine(exports, "WinJS.UI", {
        _VirtualizeContentsView: _Base.Namespace._lazy(function () {

            function cooperativeQueueWorker(info) {
                var workItems = info.job._workItems;
                var work;
                while (workItems.length && !info.shouldYield) {
                    work = workItems.shift();
                    work();
                }

                info.setWork(cooperativeQueueWorker);

                if (!workItems.length) {
                    info.job.pause();
                }
            }

            function scheduleQueueJob(priority, name) {

                var job = Scheduler.schedule(cooperativeQueueWorker, priority, null, name);

                job._workItems = [];

                job.addWork = function (work, head) {
                    if (head) {
                        this._workItems.unshift(work);
                    } else {
                        this._workItems.push(work);
                    }
                    this.resume();
                };

                job.clearWork = function () {
                    this._workItems.length = 0;
                };

                job.dispose = function () {
                    this.cancel();
                    this._workItems.length = 0;
                };

                return job;
            }

            function shouldWaitForSeZo(listView) {
                return listView._zooming || listView._pinching;
            }

            function waitForSeZo(listView, timeout) {
                // waitForSeZo will block until sezo calls endZoom on listview, or a timeout duration has elapsed to
                // unblock a potential deadlock between the sezo waiting on container creation, and container creation
                // waiting on endZoom.

                if (listView._isZombie()) { return Promise.wrap(); }
                if (shouldWaitForSeZo(listView)) {
                    if (+timeout !== timeout) {
                        timeout = _VirtualizeContentsView._waitForSeZoTimeoutDuration;
                    }
                    //To improve SeZo's zoom animation and pinch detection perf, we want to ensure unimportant task
                    //is only run while zooming or pinching is not in progress.
                    return Promise.timeout(_VirtualizeContentsView._waitForSeZoIntervalDuration).then(function () {
                        timeout -= _VirtualizeContentsView._waitForSeZoIntervalDuration;
                        if (timeout <= 0) {
                            return true;
                        }
                        return waitForSeZo(listView, timeout);
                    });
                } else {
                    return Promise.wrap();
                }
            }

            function makeFunctor(scrollToFunctor) {
                if (typeof scrollToFunctor === "number") {
                    var pos = scrollToFunctor;

                    scrollToFunctor = function () {
                        return {
                            position: pos,
                            direction: "right"
                        };
                    };
                }
                return scrollToFunctor;
            }

            var _VirtualizeContentsView = _Base.Class.define(function VirtualizeContentsView_ctor(listView) {

                this._listView = listView;
                this._forceRelayout = false;
                this.maxLeadingPages = _BaseUtils._isiOS ? _VirtualizeContentsView._iOSMaxLeadingPages : _VirtualizeContentsView._defaultPagesToPrefetch;
                this.maxTrailingPages = _BaseUtils._isiOS ? _VirtualizeContentsView._iOSMaxTrailingPages : _VirtualizeContentsView._defaultPagesToPrefetch;
                this.items = new _ItemsContainer._ItemsContainer(listView);
                this.firstIndexDisplayed = -1;
                this.lastIndexDisplayed = -1;
                this.begin = 0;
                this.end = 0;
                this._realizePass = 1;
                this._firstLayoutPass = true;
                this._runningAnimations = null;
                this._renderCompletePromise = Promise.wrap();
                this._state = new CreatedState(this);
                this._createLayoutSignal();
                this._createTreeBuildingSignal();
                this._layoutWork = null;
                this._onscreenJob = scheduleQueueJob(Scheduler.Priority.aboveNormal, "on-screen items");
                this._frontOffscreenJob = scheduleQueueJob(Scheduler.Priority.normal, "front off-screen items");
                this._backOffscreenJob = scheduleQueueJob(Scheduler.Priority.belowNormal, "back off-screen items");
                this._scrollbarPos = 0;
                this._direction = "right";
                this._scrollToFunctor = makeFunctor(0);
            },
            {

                _dispose: function VirtualizeContentsView_dispose() {
                    this.cleanUp();
                    this.items = null;
                    this._renderCompletePromise && this._renderCompletePromise.cancel();
                    this._renderCompletePromise = null;
                    this._onscreenJob.dispose();
                    this._frontOffscreenJob.dispose();
                    this._backOffscreenJob.dispose();
                },

                _createItem: function VirtualizeContentsView_createItem(itemIndex, itemPromise, available, unavailable) {
                    this._listView._writeProfilerMark("createItem(" + itemIndex + ") " + this._getBoundingRectString(itemIndex) + ",info");

                    var that = this;
                    that._listView._itemsManager._itemFromItemPromiseThrottled(itemPromise).done(
                        function (element) {
                            if (element) {
                                available(itemIndex, element, that._listView._itemsManager._recordFromElement(element));
                            } else {
                                unavailable(itemIndex);
                            }
                        },
                        function (err) {
                            unavailable(itemIndex);
                            return Promise.wrapError(err);
                        }
                    );
                },

                _addItem: function VirtualizeContentsView_addItem(fragment, itemIndex, element, currentPass) {
                    if (this._realizePass === currentPass) {
                        var record = this._listView._itemsManager._recordFromElement(element);

                        delete this._pendingItemPromises[record.itemPromise.handle];

                        this.items.setItemAt(itemIndex, {
                            itemBox: null,
                            container: null,
                            element: element,
                            detached: true,
                            itemsManagerRecord: record
                        });
                    }
                },

                lastItemIndex: function VirtualizeContentsView_lastItemIndex() {
                    return (this.containers ? (this.containers.length - 1) : -1);
                },

                _setSkipRealizationForChange: function (skip) {
                    if (skip) {
                        if (this._realizationLevel !== _VirtualizeContentsView._realizationLevel.realize) {
                            this._realizationLevel = _VirtualizeContentsView._realizationLevel.skip;
                        }
                    } else {
                        this._realizationLevel = _VirtualizeContentsView._realizationLevel.realize;
                    }
                },

                _realizeItems: function VirtualizeContentsView_realizeItems(fragment, begin, end, count, currentPass, scrollbarPos, direction, firstInView, lastInView, ignoreGaps) {
                    var perfId = "_realizeItems(" + begin + "-" + (end - 1) + ") visible(" + firstInView + "-" + lastInView + ")";

                    this._listView._writeProfilerMark(perfId + ",StartTM");

                    direction = direction || "right";

                    var counter = end - begin;
                    var inView = lastInView - firstInView + 1,
                        inViewCounter = inView,
                        rightOffscreenCount = end - lastInView - 1,
                        leftOffscreenCount = firstInView - begin;
                    var renderCompletePromises = [];
                    var entranceAnimationSignal = new _Signal();
                    var viewportItemsRealized = new _Signal();
                    var frontItemsRealized = new _Signal();

                    var that = this;

                    function itemIsReady(itemIndex, itemsManagerRecord) {
                        renderCompletePromises.push(Promise._cancelBlocker(itemsManagerRecord.renderComplete));

                        delivered(itemIndex);
                    }

                    function appendItemsToDom(startIndex, endIndex) {
                        that._listView._writeProfilerMark("_realizeItems_appendedItemsToDom,StartTM");
                        if (that._listView._isZombie()) { return; }

                        function updateDraggable(itemData, element) {
                            if (!itemData.updatedDraggableAttribute && (that._listView.itemsDraggable || that._listView.itemsReorderable)) {
                                itemData.itemsManagerRecord.renderComplete.done(function () {
                                    if (that._realizePass === currentPass) {
                                        if (!_ElementUtilities.hasClass(element, _Constants._nonDraggableClass)) {
                                            itemData.itemBox.draggable = true;
                                        }
                                        itemData.updatedDraggableAttribute = true;
                                    }
                                });
                            }
                        }

                        var itemIndex;
                        var appendItemsCount = 0;
                        var firstIndex = -1;
                        var lastIndex = -1;
                        for (itemIndex = startIndex; itemIndex <= endIndex; itemIndex++) {
                            var itemData = that.items.itemDataAt(itemIndex);
                            if (itemData) {
                                var element = itemData.element,
                                    itemBox = itemData.itemBox;

                                if (!itemBox) {
                                    itemBox = that._listView._itemBoxTemplate.cloneNode(true);
                                    itemData.itemBox = itemBox;

                                    itemBox.appendChild(element);
                                    _ElementUtilities.addClass(element, _Constants._itemClass);

                                    that._listView._setupAriaSelectionObserver(element);

                                    if (that._listView._isSelected(itemIndex)) {
                                        _ItemEventsHandler._ItemEventsHandler.renderSelection(itemBox, element, true, true);
                                    }

                                    that._listView._currentMode().renderDragSourceOnRealizedItem(itemIndex, itemBox);
                                }

                                updateDraggable(itemData, element);

                                var container = that.getContainer(itemIndex);
                                if (itemBox.parentNode !== container) {
                                    itemData.container = container;
                                    that._appendAndRestoreFocus(container, itemBox);

                                    appendItemsCount++;
                                    if (firstIndex < 0) {
                                        firstIndex = itemIndex;
                                    }
                                    lastIndex = itemIndex;

                                    if (that._listView._isSelected(itemIndex)) {
                                        _ElementUtilities.addClass(container, _Constants._selectedClass);
                                    }

                                    _ElementUtilities.removeClass(container, _Constants._backdropClass);

                                    // elementAvailable needs to be called after fragment.appendChild. elementAvailable fulfills a promise for items requested
                                    // by the keyboard focus handler. That handler will explicitly call .focus() on the element, so in order for
                                    // the focus handler to work, the element needs to be in a tree prior to focusing.

                                    that.items.elementAvailable(itemIndex);
                                }
                            }
                        }

                        that._listView._writeProfilerMark("_realizeItems_appendedItemsToDom,StopTM");
                        if (appendItemsCount > 0) {
                            that._listView._writeProfilerMark("_realizeItems_appendedItemsToDom:" + appendItemsCount + " (" + firstIndex + "-" + lastIndex + "),info");
                            that._reportElementsLevel(direction);
                        }
                    }

                    function removeGaps(first, last, begin, end) {
                        if (ignoreGaps) {
                            return;
                        }
                        // If we realized items 0 through 20 and then scrolled so that items 25 - 30 are on screen when we
                        // append them to the dom we should remove items 0 - 20 from the dom so there are no gaps between the
                        // two realized spots.

                        // Walk backwards from the beginning and if we find an item which is missing remove the rest
                        var foundMissing = false;
                        while (first >= begin) {
                            foundMissing = testGap(first, foundMissing);
                            first--;
                        }

                        // Walk forwards from the end and if we find an item which is missing remove the rest
                        foundMissing = false;
                        while (last <= end) {
                            foundMissing = testGap(last, foundMissing);
                            last++;
                        }

                        function testGap(itemIndex, foundMissing) {
                            // This helper method is called for each index and once an item is missing from the dom
                            // it removes any future one it encounters.
                            var itemData = that.items.itemDataAt(itemIndex);
                            if (itemData) {
                                var itemBox = itemData.itemBox;
                                if (!itemBox || !itemBox.parentNode) {
                                    return true;
                                } else if (foundMissing) {
                                    _ElementUtilities.addClass(itemBox.parentNode, _Constants._backdropClass);
                                    itemBox.parentNode.removeChild(itemBox);
                                    return true;
                                } else {
                                    return false;
                                }
                            } else {
                                return true;
                            }
                        }
                    }

                    function scheduleReadySignal(first, last, job, dir, head) {
                        var promises = [];

                        for (var i = first; i <= last; i++) {
                            var itemData = that.items.itemDataAt(i);
                            if (itemData) {
                                promises.push(itemData.itemsManagerRecord.itemPromise);
                            }
                        }

                        function schedule(itemIndex) {
                            var itemData = that.items.itemDataAt(itemIndex);
                            if (itemData) {
                                var record = itemData.itemsManagerRecord;
                                if (!record.readyComplete && that._realizePass === currentPass) {
                                    job.addWork(function () {
                                        if (that._listView._isZombie()) {
                                            return;
                                        }
                                        if (record.pendingReady && that._realizePass === currentPass) {
                                            that._listView._writeProfilerMark("pendingReady(" + itemIndex + "),info");
                                            record.pendingReady();
                                        }
                                    }, head);
                                }
                            }
                        }

                        Promise.join(promises).then(function () {
                            if (dir === "right") {
                                for (var i = first; i <= last; i++) {
                                    schedule(i);
                                }
                            } else {
                                for (var i = last; i >= first; i--) {
                                    schedule(i);
                                }
                            }
                        });
                    }

                    function delivered(index) {
                        if (that._realizePass !== currentPass) {
                            return;
                        }

                        if (index >= firstInView && index <= lastInView) {
                            if (--inViewCounter === 0) {
                                appendItemsToDom(firstInView, lastInView);
                                removeGaps(firstInView, lastInView, begin, end);

                                if (that._firstLayoutPass) {
                                    scheduleReadySignal(firstInView, lastInView, that._frontOffscreenJob, direction === "right" ? "left" : "right", true);

                                    var entranceAnimation = Scheduler.schedulePromiseHigh(null, "WinJS.UI.ListView.entranceAnimation").then(function () {
                                        if (that._listView._isZombie()) { return; }
                                        that._listView._writeProfilerMark("entranceAnimation,StartTM");
                                        var promise = that._listView._animateListEntrance(!that._firstEntranceAnimated);
                                        that._firstEntranceAnimated = true;
                                        return promise;
                                    });

                                    that._runningAnimations = Promise.join([that._runningAnimations, entranceAnimation]);
                                    that._runningAnimations.done(function () {
                                        that._listView._writeProfilerMark("entranceAnimation,StopTM");
                                        if (that._realizePass === currentPass) {
                                            that._runningAnimations = null;
                                            entranceAnimationSignal.complete();
                                        }
                                    });
                                    that._firstLayoutPass = false;

                                    if (that._listView._isCurrentZoomView) {
                                        Scheduler.requestDrain(that._onscreenJob.priority);
                                    }
                                } else {
                                    // during scrolling ready for onscreen items after front off screen items
                                    scheduleReadySignal(firstInView, lastInView, that._frontOffscreenJob, direction);
                                    entranceAnimationSignal.complete();
                                }

                                that._updateHeaders(that._listView._canvas, firstInView, lastInView + 1).done(function () {
                                    viewportItemsRealized.complete();
                                });
                            }
                        } else if (index < firstInView) {
                            --leftOffscreenCount;
                            if (leftOffscreenCount % inView === 0) {
                                appendItemsToDom(begin, firstInView - 1);
                            }
                            if (!leftOffscreenCount) {
                                that._updateHeaders(that._listView._canvas, begin, firstInView).done(function () {
                                    if (direction !== "right") {
                                        frontItemsRealized.complete();
                                    }
                                });
                                scheduleReadySignal(begin, firstInView - 1, direction !== "right" ? that._frontOffscreenJob : that._backOffscreenJob, "left");
                            }
                        } else if (index > lastInView) {
                            --rightOffscreenCount;
                            if (rightOffscreenCount % inView === 0) {
                                appendItemsToDom(lastInView + 1, end - 1);
                            }
                            if (!rightOffscreenCount) {
                                that._updateHeaders(that._listView._canvas, lastInView + 1, end).then(function () {
                                    if (direction === "right") {
                                        frontItemsRealized.complete();
                                    }
                                });
                                scheduleReadySignal(lastInView + 1, end - 1, direction === "right" ? that._frontOffscreenJob : that._backOffscreenJob, "right");
                            }
                        }
                        counter--;

                        if (counter === 0) {
                            that._renderCompletePromise = Promise.join(renderCompletePromises).then(null, function (e) {
                                var error = Array.isArray(e) && e.some(function (item) { return item && !(item instanceof Error && item.name === "Canceled"); });
                                if (error) {
                                    // rethrow
                                    return Promise.wrapError(e);
                                }
                            });

                            (that._headerRenderPromises || Promise.wrap()).done(function () {
                                Scheduler.schedule(function VirtualizeContentsView_async_delivered() {
                                    if (that._listView._isZombie()) {
                                        workCompleteSignal.cancel();
                                    } else {
                                        workCompleteSignal.complete();
                                    }
                                }, Math.min(that._onscreenJob.priority, that._backOffscreenJob.priority), null, "WinJS.UI.ListView._allItemsRealized");
                            });
                        }
                    }

                    function newItemIsReady(itemIndex, element, itemsManagerRecord) {
                        if (that._realizePass === currentPass) {
                            var element = itemsManagerRecord.element;
                            that._addItem(fragment, itemIndex, element, currentPass);
                            itemIsReady(itemIndex, itemsManagerRecord);
                        }
                    }

                    if (counter > 0) {
                        var createCount = 0;
                        var updateCount = 0;
                        var cleanCount = 0;
                        that.firstIndexDisplayed = firstInView;
                        that.lastIndexDisplayed = lastInView;

                        var isCurrentZoomView = that._listView._isCurrentZoomView;
                        if (that._highPriorityRealize && (that._firstLayoutPass || that._hasAnimationInViewportPending)) {
                            // startup or edits that will animate items in the viewport
                            that._highPriorityRealize = false;
                            that._onscreenJob.priority = Scheduler.Priority.high;
                            that._frontOffscreenJob.priority = Scheduler.Priority.normal;
                            that._backOffscreenJob.priority = Scheduler.Priority.belowNormal;
                        } else if (that._highPriorityRealize) {
                            // edits that won't animate items in the viewport
                            that._highPriorityRealize = false;
                            that._onscreenJob.priority = Scheduler.Priority.high;
                            that._frontOffscreenJob.priority = Scheduler.Priority.high - 1;
                            that._backOffscreenJob.priority = Scheduler.Priority.high - 1;
                        } else if (isCurrentZoomView) {
                            // scrolling
                            that._onscreenJob.priority = Scheduler.Priority.aboveNormal;
                            that._frontOffscreenJob.priority = Scheduler.Priority.normal;
                            that._backOffscreenJob.priority = Scheduler.Priority.belowNormal;
                        } else {
                            // hidden ListView in SeZo
                            that._onscreenJob.priority = Scheduler.Priority.belowNormal;
                            that._frontOffscreenJob.priority = Scheduler.Priority.idle;
                            that._backOffscreenJob.priority = Scheduler.Priority.idle;
                        }

                        // Create a promise to wrap the work in the queue. When the queue gets to the last item we can mark
                        // the work promise complete and if the work promise is canceled we cancel the queue.
                        //
                        var workCompleteSignal = new _Signal();

                        // If the version manager recieves a notification we clear the work in the work queues
                        //
                        var cancelToken = that._listView._versionManager.cancelOnNotification(workCompleteSignal.promise);

                        var queueStage1AfterStage0 = function (job, record) {
                            if (record.startStage1) {
                                record.stage0.then(function () {
                                    if (that._realizePass === currentPass && record.startStage1) {
                                        job.addWork(record.startStage1);
                                    }
                                });
                            }
                        };

                        var queueWork = function (job, itemIndex) {
                            var itemData = that.items.itemDataAt(itemIndex);
                            if (!itemData) {
                                var itemPromise = that._listView._itemsManager._itemPromiseAtIndex(itemIndex);

                                // Remember this pending item promise and avoid canceling it from the previous realization pass.
                                that._pendingItemPromises[itemPromise.handle] = itemPromise;
                                delete that._previousRealizationPendingItemPromises[itemPromise.handle];

                                job.addWork(function VirtualizeContentsView_realizeItemsWork() {
                                    if (that._listView._isZombie()) {
                                        return;
                                    }

                                    createCount++;
                                    that._createItem(itemIndex, itemPromise, newItemIsReady, delivered);

                                    // _createItem runs user code
                                    if (that._listView._isZombie() || that._realizePass !== currentPass) {
                                        return;
                                    }

                                    if (itemPromise.handle) {
                                        var record = that._listView._itemsManager._recordFromHandle(itemPromise.handle);
                                        queueStage1AfterStage0(job, record);
                                    }
                                });
                            }

                        };

                        var queueRight = function (job, first, last) {
                            for (var itemIndex = first; itemIndex <= last; itemIndex++) {
                                queueWork(job, itemIndex);
                            }
                        };

                        var queueLeft = function (job, first, last) {
                            // Always build the left side in the direction away from the center.
                            for (var itemIndex = last; itemIndex >= first; itemIndex--) {
                                queueWork(job, itemIndex);
                            }
                        };

                        var handleExistingRange = function (job, first, last) {
                            for (var itemIndex = first; itemIndex <= last; itemIndex++) {
                                var itemData = that.items.itemDataAt(itemIndex);
                                if (itemData) {
                                    var record = itemData.itemsManagerRecord;
                                    itemIsReady(itemIndex, record);
                                    updateCount++;
                                    queueStage1AfterStage0(job, record);
                                }
                            }
                        };

                        // PendingItemPromises are the item promises which we have requested from the ItemsManager
                        // which have not returned an element (placeholder or real). Since we only clean up items
                        // which have an element in _unrealizeItems we need to remember these item promises. We cancel
                        // the item promises from the previous realization iteration if those item promises are not
                        // used for the current realization.
                        this._previousRealizationPendingItemPromises = this._pendingItemPromises || {};
                        this._pendingItemPromises = {};

                        var emptyFront;
                        if (direction === "left") {
                            queueLeft(that._onscreenJob, firstInView, lastInView);
                            queueLeft(that._frontOffscreenJob, begin, firstInView - 1);
                            emptyFront = begin > (firstInView - 1);
                        } else {
                            queueRight(that._onscreenJob, firstInView, lastInView);
                            queueRight(that._frontOffscreenJob, lastInView + 1, end - 1);
                            emptyFront = lastInView + 1 > (end - 1);
                        }

                        // Anything left in _previousRealizationPendingItemPromises can be canceled here.
                        // Note: we are doing this synchronously. If we didn't do it synchronously we would have had to merge
                        // _previousRealizationPendingItemPromises and _pendingItemPromises together. This also has the great
                        // benefit to cancel item promises in the backOffScreenArea which are much less important.
                        for (var i = 0, handles = Object.keys(this._previousRealizationPendingItemPromises), len = handles.length; i < len; i++) {
                            var handle = handles[i];
                            that._listView._itemsManager.releaseItemPromise(this._previousRealizationPendingItemPromises[handle]);
                        }
                        this._previousRealizationPendingItemPromises = {};


                        // Handle existing items in the second pass to make sure that raising ready signal is added to the queues after creating items
                        handleExistingRange(that._onscreenJob, firstInView, lastInView);
                        if (direction === "left") {
                            handleExistingRange(that._frontOffscreenJob, begin, firstInView - 1);
                        } else {
                            handleExistingRange(that._frontOffscreenJob, lastInView + 1, end - 1);
                        }

                        var showProgress = (inViewCounter === lastInView - firstInView + 1);

                        if (that._firstLayoutPass) {
                            that._listView._canvas.style.opacity = 0;
                        } else {
                            if (showProgress) {
                                that._listView._showProgressBar(that._listView._element, "50%", "50%");
                            } else {
                                that._listView._hideProgressBar();
                            }
                        }

                        that._frontOffscreenJob.pause();
                        that._backOffscreenJob.pause();

                        viewportItemsRealized.promise.done(
                            function () {
                                that._frontOffscreenJob.resume();

                                if (emptyFront) {
                                    frontItemsRealized.complete();
                                }
                            },
                            function () {
                                workCompleteSignal.cancel();
                            }
                        );

                        frontItemsRealized.promise.done(function () {
                            that._listView._writeProfilerMark("frontItemsRealized,info");

                            if (direction === "left") {
                                queueRight(that._backOffscreenJob, lastInView + 1, end - 1);
                                handleExistingRange(that._backOffscreenJob, lastInView + 1, end - 1);
                            } else {
                                queueLeft(that._backOffscreenJob, begin, firstInView - 1);
                                handleExistingRange(that._backOffscreenJob, begin, firstInView - 1);
                            }

                            that._backOffscreenJob.resume();
                        });

                        workCompleteSignal.promise.done(
                            function () {
                                that._listView._versionManager.clearCancelOnNotification(cancelToken);

                                that._listView._writeProfilerMark(perfId + " complete(created:" + createCount + " updated:" + updateCount + "),info");
                            },
                            function (err) {
                                that._listView._versionManager.clearCancelOnNotification(cancelToken);
                                that._onscreenJob.clearWork();
                                that._frontOffscreenJob.clearWork();
                                that._backOffscreenJob.clearWork();

                                entranceAnimationSignal.cancel();
                                viewportItemsRealized.cancel();

                                that._listView._writeProfilerMark(perfId + " canceled(created:" + createCount + " updated:" + updateCount + " clean:" + cleanCount + "),info");
                                return Promise.wrapError(err);
                            }
                        );

                        that._listView._writeProfilerMark(perfId + ",StopTM");
                        return {
                            viewportItemsRealized: viewportItemsRealized.promise,
                            allItemsRealized: workCompleteSignal.promise,
                            loadingCompleted: Promise.join([workCompleteSignal.promise, entranceAnimationSignal.promise]).then(function () {
                                var promises = [];

                                for (var i = begin; i < end; i++) {
                                    var itemData = that.items.itemDataAt(i);
                                    if (itemData) {
                                        promises.push(itemData.itemsManagerRecord.itemReadyPromise);
                                    }
                                }
                                return Promise._cancelBlocker(Promise.join(promises));
                            })
                        };
                    } else {
                        that._listView._writeProfilerMark(perfId + ",StopTM");
                        return {
                            viewportItemsRealized: Promise.wrap(),
                            allItemsRealized: Promise.wrap(),
                            loadingCompleted: Promise.wrap()
                        };
                    }
                },

                _setAnimationInViewportState: function VirtualizeContentsView_setAnimationInViewportState(modifiedElements) {
                    this._hasAnimationInViewportPending = false;
                    if (modifiedElements && modifiedElements.length > 0) {
                        var viewportLength = this._listView._getViewportLength(),
                            range = this._listView._layout.itemsFromRange(this._scrollbarPos, this._scrollbarPos + viewportLength - 1);
                        for (var i = 0, len = modifiedElements.length; i < len; i++) {
                            var modifiedElement = modifiedElements[i];
                            if (modifiedElement.newIndex >= range.firstIndex && modifiedElement.newIndex <= range.lastIndex && modifiedElement.newIndex !== modifiedElement.oldIndex) {
                                this._hasAnimationInViewportPending = true;
                                break;
                            }
                        }
                    }
                },

                _addHeader: function VirtualizeContentsView_addHeader(fragment, groupIndex) {
                    var that = this;
                    return this._listView._groups.renderGroup(groupIndex).then(function (header) {
                        if (header) {
                            header.element.tabIndex = 0;
                            var placeholder = that._getHeaderContainer(groupIndex);
                            if (header.element.parentNode !== placeholder) {
                                placeholder.appendChild(header.element);
                                _ElementUtilities.addClass(header.element, _Constants._headerClass);
                            }

                            that._listView._groups.setDomElement(groupIndex, header.element);
                        }
                    });
                },

                _updateHeaders: function VirtualizeContentsView_updateHeaders(fragment, begin, end) {
                    var that = this;

                    function updateGroup(index) {
                        var group = that._listView._groups.group(index);
                        if (group && !group.header) {
                            var headerPromise = group.headerPromise;
                            if (!headerPromise) {
                                headerPromise = group.headerPromise = that._addHeader(fragment, index);
                                headerPromise.done(function () {
                                    group.headerPromise = null;
                                }, function () {
                                    group.headerPromise = null;
                                });
                            }
                            return headerPromise;
                        }
                        return Promise.wrap();
                    }

                    this._listView._groups.removeElements();

                    var groupStart = this._listView._groups.groupFromItem(begin),
                        groupIndex = groupStart,
                        groupEnd = this._listView._groups.groupFromItem(end - 1),
                        realizationPromises = [];

                    if (groupIndex !== null) {
                        for (; groupIndex <= groupEnd; groupIndex++) {
                            realizationPromises.push(updateGroup(groupIndex));
                        }
                    }

                    function done() {
                        that._headerRenderPromises = null;
                    }
                    this._headerRenderPromises = Promise.join(realizationPromises, this._headerRenderPromises).then(done, done);
                    return this._headerRenderPromises || Promise.wrap();
                },

                _unrealizeItem: function VirtualizeContentsView_unrealizeItem(itemIndex) {
                    var listView = this._listView,
                        focusedItemPurged;

                    this._listView._writeProfilerMark("_unrealizeItem(" + itemIndex + "),info");

                    var focused = listView._selection._getFocused();
                    if (focused.type === _UI.ObjectType.item && focused.index === itemIndex) {
                        listView._unsetFocusOnItem();
                        focusedItemPurged = true;
                    }
                    var itemData = this.items.itemDataAt(itemIndex),
                        item = itemData.element,
                        itemBox = itemData.itemBox;

                    if (itemBox && itemBox.parentNode) {
                        _ElementUtilities.removeClass(itemBox.parentNode, _Constants._selectedClass);
                        _ElementUtilities.removeClass(itemBox.parentNode, _Constants._footprintClass);
                        _ElementUtilities.addClass(itemBox.parentNode, _Constants._backdropClass);
                        itemBox.parentNode.removeChild(itemBox);
                    }
                    itemData.container = null;

                    if (listView._currentMode().itemUnrealized) {
                        listView._currentMode().itemUnrealized(itemIndex, itemBox);
                    }

                    this.items.removeItem(itemIndex);

                    // If this wasn't released by the itemsManager already, then
                    // we remove it. This handles the special case of delete
                    // occuring on an item that is outside of the current view, but
                    // has not been cleaned up yet.
                    //
                    if (!itemData.removed) {
                        listView._itemsManager.releaseItem(item);
                    }


                    _Dispose._disposeElement(item);

                    if (focusedItemPurged) {
                        // If the focused item was purged, we'll still want to focus on it if it comes into view sometime in the future.
                        // calling _setFocusOnItem once the item is removed from this.items will set up a promise that will be fulfilled
                        // if the item ever gets reloaded
                        listView._setFocusOnItem(listView._selection._getFocused());
                    }
                },

                _unrealizeGroup: function VirtualizeContentsView_unrealizeGroup(group) {
                    var headerElement = group.header,
                        focusedItemPurged;

                    var focused = this._listView._selection._getFocused();
                    if (focused.type === _UI.ObjectType.groupHeader && this._listView._groups.group(focused.index) === group) {
                        this._listView._unsetFocusOnItem();
                        focusedItemPurged = true;
                    }

                    if (headerElement.parentNode) {
                        headerElement.parentNode.removeChild(headerElement);
                    }

                    _Dispose._disposeElement(headerElement);

                    group.header = null;
                    group.left = -1;
                    group.top = -1;

                    if (focusedItemPurged) {
                        this._listView._setFocusOnItem(this._listView._selection._getFocused());
                    }
                },

                _unrealizeItems: function VirtualizeContentsView_unrealizeItems(remove) {
                    var that = this,
                        removedCount = 0;

                    this.items.eachIndex(function (index) {
                        if (index < that.begin || index >= that.end) {
                            that._unrealizeItem(index);
                            return remove && ++removedCount >= remove;
                        }
                    });

                    var groups = this._listView._groups,
                        beginGroup = groups.groupFromItem(this.begin);

                    if (beginGroup !== null) {
                        var endGroup = groups.groupFromItem(this.end - 1);
                        for (var i = 0, len = groups.length() ; i < len; i++) {
                            var group = groups.group(i);
                            if ((i < beginGroup || i > endGroup) && group.header) {
                                this._unrealizeGroup(group);
                            }
                        }
                    }
                },

                _unrealizeExcessiveItems: function VirtualizeContentsView_unrealizeExcessiveItems() {
                    var realized = this.items.count(),
                        needed = this.end - this.begin,
                        approved = needed + this._listView._maxDeferredItemCleanup;

                    this._listView._writeProfilerMark("_unrealizeExcessiveItems realized(" + realized + ") approved(" + approved + "),info");
                    if (realized > approved) {
                        this._unrealizeItems(realized - approved);
                    }
                },

                _lazilyUnrealizeItems: function VirtualizeContentsView_lazilyUnrealizeItems() {
                    this._listView._writeProfilerMark("_lazilyUnrealizeItems,StartTM");
                    var that = this;
                    return waitForSeZo(this._listView).then(function () {

                        function done() {
                            that._listView._writeProfilerMark("_lazilyUnrealizeItems,StopTM");
                        }

                        if (that._listView._isZombie()) {
                            done();
                            return;
                        }

                        var itemsToUnrealize = [];
                        that.items.eachIndex(function (index) {
                            if (index < that.begin || index >= that.end) {
                                itemsToUnrealize.push(index);
                            }
                        });

                        that._listView._writeProfilerMark("_lazilyUnrealizeItems itemsToUnrealize(" + itemsToUnrealize.length + "),info");

                        var groupsToUnrealize = [],
                            groups = that._listView._groups,
                            beginGroup = groups.groupFromItem(that.begin);

                        if (beginGroup !== null) {
                            var endGroup = groups.groupFromItem(that.end - 1);
                            for (var i = 0, len = groups.length() ; i < len; i++) {
                                var group = groups.group(i);
                                if ((i < beginGroup || i > endGroup) && group.header) {
                                    groupsToUnrealize.push(group);
                                }
                            }
                        }

                        if (itemsToUnrealize.length || groupsToUnrealize.length) {
                            var job;

                            var promise = new Promise(function (complete) {

                                function unrealizeWorker(info) {
                                    if (that._listView._isZombie()) { return; }

                                    var firstIndex = -1,
                                        lastIndex = -1,
                                        removeCount = 0,
                                        zooming = shouldWaitForSeZo(that._listView);

                                    while (itemsToUnrealize.length && !zooming && !info.shouldYield) {
                                        var itemIndex = itemsToUnrealize.shift();
                                        that._unrealizeItem(itemIndex);

                                        removeCount++;
                                        if (firstIndex < 0) {
                                            firstIndex = itemIndex;
                                        }
                                        lastIndex = itemIndex;
                                    }
                                    that._listView._writeProfilerMark("unrealizeWorker removeItems:" + removeCount + " (" + firstIndex + "-" + lastIndex + "),info");

                                    while (groupsToUnrealize.length && !zooming && !info.shouldYield) {
                                        that._unrealizeGroup(groupsToUnrealize.shift());
                                    }

                                    if (itemsToUnrealize.length || groupsToUnrealize.length) {
                                        if (zooming) {
                                            info.setPromise(waitForSeZo(that._listView).then(function () {
                                                return unrealizeWorker;
                                            }));
                                        } else {
                                            info.setWork(unrealizeWorker);
                                        }
                                    } else {
                                        complete();
                                    }
                                }

                                job = Scheduler.schedule(unrealizeWorker, Scheduler.Priority.belowNormal, null, "WinJS.UI.ListView._lazilyUnrealizeItems");
                            });

                            return promise.then(done, function (error) {
                                job.cancel();
                                that._listView._writeProfilerMark("_lazilyUnrealizeItems canceled,info");
                                that._listView._writeProfilerMark("_lazilyUnrealizeItems,StopTM");
                                return Promise.wrapError(error);
                            });

                        } else {
                            done();
                            return Promise.wrap();
                        }
                    });
                },

                _getBoundingRectString: function VirtualizeContentsView_getBoundingRectString(itemIndex) {
                    var result;
                    if (itemIndex >= 0 && itemIndex < this.containers.length) {
                        var itemPos = this._listView._layout._getItemPosition(itemIndex);
                        if (itemPos) {
                            result = "[" + itemPos.left + "; " + itemPos.top + "; " + itemPos.width + "; " + itemPos.height + " ]";
                        }
                    }
                    return result || "";
                },

                _clearDeferTimeout: function VirtualizeContentsView_clearDeferTimeout() {
                    if (this.deferTimeout) {
                        this.deferTimeout.cancel();
                        this.deferTimeout = null;
                    }
                    if (this.deferredActionCancelToken !== -1) {
                        this._listView._versionManager.clearCancelOnNotification(this.deferredActionCancelToken);
                        this.deferredActionCancelToken = -1;
                    }
                },

                _setupAria: function VirtualizeContentsView_setupAria(timedOut) {
                    if (this._listView._isZombie()) { return; }
                    var that = this;

                    function done() {
                        that._listView._writeProfilerMark("aria work,StopTM");
                    }

                    function calcLastRealizedIndexInGroup(groupIndex) {
                        var groups = that._listView._groups,
                            nextGroup = groups.group(groupIndex + 1);
                        return (nextGroup ? Math.min(nextGroup.startIndex - 1, that.end - 1) : that.end - 1);
                    }

                    this._listView._createAriaMarkers();
                    return this._listView._itemsCount().then(function (count) {
                        if (count > 0 && that.firstIndexDisplayed !== -1 && that.lastIndexDisplayed !== -1) {
                            that._listView._writeProfilerMark("aria work,StartTM");
                            var startMarker = that._listView._ariaStartMarker,
                                endMarker = that._listView._ariaEndMarker,
                                index = that.begin,
                                item = that.items.itemAt(that.begin),
                                job,
                                // These are only used when the ListView is using groups
                                groups,
                                startGroup,
                                currentGroup,
                                group,
                                lastRealizedIndexInGroup;

                            if (item) {
                                _ElementUtilities._ensureId(item);
                                if (that._listView._groupsEnabled()) {
                                    groups = that._listView._groups;
                                    startGroup = currentGroup = groups.groupFromItem(that.begin);
                                    group = groups.group(currentGroup);
                                    lastRealizedIndexInGroup = calcLastRealizedIndexInGroup(currentGroup);
                                    _ElementUtilities._ensureId(group.header);
                                    _ElementUtilities._setAttribute(group.header, "role", that._listView._headerRole);
                                    _ElementUtilities._setAttribute(group.header, "x-ms-aria-flowfrom", startMarker.id);
                                    setFlow(group.header, item);
                                    _ElementUtilities._setAttribute(group.header, "tabindex", that._listView._tabIndex);
                                } else {
                                    _ElementUtilities._setAttribute(item, "x-ms-aria-flowfrom", startMarker.id);
                                }

                                return new Promise(function (completeJobPromise) {
                                    var skipWait = timedOut;
                                    job = Scheduler.schedule(function ariaWorker(jobInfo) {
                                        if (that._listView._isZombie()) {
                                            done();
                                            return;
                                        }

                                        for (; index < that.end; index++) {
                                            if (!skipWait && shouldWaitForSeZo(that._listView)) {
                                                jobInfo.setPromise(waitForSeZo(that._listView).then(function (timedOut) {
                                                    skipWait = timedOut;
                                                    return ariaWorker;
                                                }));
                                                return;
                                            } else if (jobInfo.shouldYield) {
                                                jobInfo.setWork(ariaWorker);
                                                return;
                                            }

                                            item = that.items.itemAt(index);
                                            var nextItem = that.items.itemAt(index + 1);

                                            if (nextItem) {
                                                _ElementUtilities._ensureId(nextItem);
                                            }

                                            _ElementUtilities._setAttribute(item, "role", that._listView._itemRole);
                                            _ElementUtilities._setAttribute(item, "aria-setsize", count);
                                            _ElementUtilities._setAttribute(item, "aria-posinset", index + 1);
                                            _ElementUtilities._setAttribute(item, "tabindex", that._listView._tabIndex);

                                            if (that._listView._groupsEnabled()) {
                                                if (index === lastRealizedIndexInGroup || !nextItem) {
                                                    var nextGroup = groups.group(currentGroup + 1);

                                                    // If group is the last realized group, then nextGroup won't exist in the DOM.
                                                    // When this is the case, nextItem shouldn't exist.
                                                    if (nextGroup && nextGroup.header && nextItem) {
                                                        _ElementUtilities._setAttribute(nextGroup.header, "tabindex", that._listView._tabIndex);
                                                        _ElementUtilities._setAttribute(nextGroup.header, "role", that._listView._headerRole);
                                                        _ElementUtilities._ensureId(nextGroup.header);
                                                        setFlow(item, nextGroup.header);
                                                        setFlow(nextGroup.header, nextItem);
                                                    } else {
                                                        // We're at the last group so flow to the end marker
                                                        _ElementUtilities._setAttribute(item, "aria-flowto", endMarker.id);
                                                    }

                                                    currentGroup++;
                                                    group = nextGroup;
                                                    lastRealizedIndexInGroup = calcLastRealizedIndexInGroup(currentGroup);
                                                } else {
                                                    // This is not the last item in the group so flow to the next item
                                                    setFlow(item, nextItem);
                                                }
                                            } else if (nextItem) {
                                                // Groups are disabled so as long as we aren't at the last item, flow to the next one
                                                setFlow(item, nextItem);
                                            } else {
                                                // Groups are disabled and we're at the last item, so flow to the end marker
                                                _ElementUtilities._setAttribute(item, "aria-flowto", endMarker.id);
                                            }
                                            if (!nextItem) {
                                                break;
                                            }
                                        }

                                        that._listView._fireAccessibilityAnnotationCompleteEvent(that.begin, index, startGroup, currentGroup - 1);

                                        done();
                                        completeJobPromise();
                                    }, Scheduler.Priority.belowNormal, null, "WinJS.UI.ListView._setupAria");
                                }, function () {
                                    // Cancellation handler for promise returned by setupAria
                                    job.cancel();
                                    done();
                                });
                            } else {
                                // the first item is null
                                done();
                            }
                        } else {
                            // The count is 0
                            return Promise.wrap();
                        }
                    });
                },

                _setupDeferredActions: function VirtualizeContentsView_setupDeferredActions() {
                    this._listView._writeProfilerMark("_setupDeferredActions,StartTM");
                    var that = this;

                    this._clearDeferTimeout();

                    function cleanUp() {
                        if (that._listView._isZombie()) { return; }
                        that.deferTimeout = null;
                        that._listView._versionManager.clearCancelOnNotification(that.deferredActionCancelToken);
                        that.deferredActionCancelToken = -1;
                    }

                    this.deferTimeout = this._lazilyRemoveRedundantItemsBlocks().then(function () {
                        return Promise.timeout(_Constants._DEFERRED_ACTION);
                    }).
                        then(function () {
                            return waitForSeZo(that._listView);
                        }).
                        then(function (timedOut) {
                            return that._setupAria(timedOut);
                        }).
                        then(cleanUp, function (error) {
                            cleanUp();
                            return Promise.wrapError(error);
                        });

                    this.deferredActionCancelToken = this._listView._versionManager.cancelOnNotification(this.deferTimeout);
                    this._listView._writeProfilerMark("_setupDeferredActions,StopTM");
                },

                // Sets aria-flowto on _ariaStartMarker and x-ms-aria-flowfrom on _ariaEndMarker. The former
                // points to either the first visible group header or the first visible item. The latter points
                // to the last visible item.
                _updateAriaMarkers: function VirtualizeContentsView_updateAriaMarkers(listViewIsEmpty, firstIndexDisplayed, lastIndexDisplayed) {
                    var that = this;
                    if (this._listView._isZombie()) {
                        return;
                    }

                    function getFirstVisibleItem() {
                        return that.items.itemAt(firstIndexDisplayed);
                    }

                    // At a certain index, the VDS may return null for all items at that index and
                    // higher. When this is the case, the end marker should point to the last
                    // non-null item in the visible range.
                    function getLastVisibleItem() {
                        for (var i = lastIndexDisplayed; i >= firstIndexDisplayed; i--) {
                            if (that.items.itemAt(i)) {
                                return that.items.itemAt(i);
                            }
                        }
                        return null;
                    }

                    this._listView._createAriaMarkers();
                    var startMarker = this._listView._ariaStartMarker,
                        endMarker = this._listView._ariaEndMarker,
                        firstVisibleItem,
                        lastVisibleItem;

                    if (firstIndexDisplayed !== -1 && lastIndexDisplayed !== -1 && firstIndexDisplayed <= lastIndexDisplayed) {
                        firstVisibleItem = getFirstVisibleItem();
                        lastVisibleItem = getLastVisibleItem();
                    }

                    if (listViewIsEmpty || !firstVisibleItem || !lastVisibleItem) {
                        setFlow(startMarker, endMarker);
                        this._listView._fireAccessibilityAnnotationCompleteEvent(-1, -1);
                    } else {
                        _ElementUtilities._ensureId(firstVisibleItem);
                        _ElementUtilities._ensureId(lastVisibleItem);

                        // Set startMarker's flowto
                        if (this._listView._groupsEnabled()) {
                            var groups = this._listView._groups,
                                firstVisibleGroup = groups.group(groups.groupFromItem(firstIndexDisplayed));

                            if (firstVisibleGroup.header) {
                                _ElementUtilities._ensureId(firstVisibleGroup.header);

                                if (firstIndexDisplayed === firstVisibleGroup.startIndex) {
                                    _ElementUtilities._setAttribute(startMarker, "aria-flowto", firstVisibleGroup.header.id);
                                } else {
                                    _ElementUtilities._setAttribute(startMarker, "aria-flowto", firstVisibleItem.id);
                                }
                            }
                        } else {
                            _ElementUtilities._setAttribute(startMarker, "aria-flowto", firstVisibleItem.id);
                        }

                        // Set endMarker's flowfrom
                        _ElementUtilities._setAttribute(endMarker, "x-ms-aria-flowfrom", lastVisibleItem.id);
                    }
                },

                // Update the ARIA attributes on item that are needed so that Narrator can announce it.
                // item must be in the items container.
                updateAriaForAnnouncement: function VirtualizeContentsView_updateAriaForAnnouncement(item, count) {
                    if (item === this._listView.header || item === this._listView.footer) {
                        return;
                    }

                    var index = -1;
                    var type = _UI.ObjectType.item;
                    if (_ElementUtilities.hasClass(item, _Constants._headerClass)) {
                        index = this._listView._groups.index(item);
                        type = _UI.ObjectType.groupHeader;
                        _ElementUtilities._setAttribute(item, "role", this._listView._headerRole);
                        _ElementUtilities._setAttribute(item, "tabindex", this._listView._tabIndex);
                    } else {
                        index = this.items.index(item);
                        _ElementUtilities._setAttribute(item, "aria-setsize", count);
                        _ElementUtilities._setAttribute(item, "aria-posinset", index + 1);
                        _ElementUtilities._setAttribute(item, "role", this._listView._itemRole);
                        _ElementUtilities._setAttribute(item, "tabindex", this._listView._tabIndex);
                    }

                    if (type === _UI.ObjectType.groupHeader) {
                        this._listView._fireAccessibilityAnnotationCompleteEvent(-1, -1, index, index);
                    } else {
                        this._listView._fireAccessibilityAnnotationCompleteEvent(index, index, -1, -1);
                    }
                },

                _reportElementsLevel: function VirtualizeContentsView_reportElementsLevel(direction) {
                    var items = this.items;

                    function elementsCount(first, last) {
                        var count = 0;
                        for (var itemIndex = first; itemIndex <= last; itemIndex++) {
                            var itemData = items.itemDataAt(itemIndex);
                            if (itemData && itemData.container) {
                                count++;
                            }
                        }
                        return count;
                    }

                    var level;
                    if (direction === "right") {
                        level = Math.floor(100 * elementsCount(this.firstIndexDisplayed, this.end - 1) / (this.end - this.firstIndexDisplayed));
                    } else {
                        level = Math.floor(100 * elementsCount(this.begin, this.lastIndexDisplayed) / (this.lastIndexDisplayed - this.begin + 1));
                    }

                    this._listView._writeProfilerMark("elementsLevel level(" + level + "),info");
                },

                _createHeaderContainer: function VirtualizeContentsView_createHeaderContainer(insertAfter) {
                    return this._createSurfaceChild(_Constants._headerContainerClass, insertAfter);
                },

                _createItemsContainer: function VirtualizeContentsView_createItemsContainer(insertAfter) {
                    var itemsContainer = this._createSurfaceChild(_Constants._itemsContainerClass, insertAfter);
                    var padder = _Global.document.createElement("div");
                    padder.className = _Constants._padderClass;
                    itemsContainer.appendChild(padder);
                    return itemsContainer;
                },

                _ensureContainerInDOM: function VirtualizeContentsView_ensureContainerInDOM(index) {
                    var container = this.containers[index];
                    if (container && !this._listView._canvas.contains(container)) {
                        this._forceItemsBlocksInDOM(index, index + 1);
                        return true;
                    }
                    return false;
                },

                _ensureItemsBlocksInDOM: function VirtualizeContentsView_ensureItemsBlocksInDOM(begin, end) {
                    if (this._expandedRange) {
                        var oldBegin = this._expandedRange.first.index,
                            oldEnd = this._expandedRange.last.index + 1;

                        if (begin <= oldBegin && end > oldBegin) {
                            end = Math.max(end, oldEnd);
                        } else if (begin < oldEnd && end >= oldEnd) {
                            begin = Math.min(begin, oldBegin);
                        }
                    }
                    this._forceItemsBlocksInDOM(begin, end);
                },

                _removeRedundantItemsBlocks: function VirtualizeContentsView_removeRedundantItemsBlocks() {
                    if (this.begin !== -1 && this.end !== -1) {
                        this._forceItemsBlocksInDOM(this.begin, this.end);
                    }
                },

                _lazilyRemoveRedundantItemsBlocks: function VirtualizeContentsView_lazilyRemoveRedundantItemsBlocks() {
                    this._listView._writeProfilerMark("_lazilyRemoveRedundantItemsBlocks,StartTM");
                    var that = this;
                    return waitForSeZo(this._listView).then(function () {

                        function done() {
                            that._listView._writeProfilerMark("_lazilyRemoveRedundantItemsBlocks,StopTM");
                        }

                        if (that._listView._isZombie()) {
                            done();
                            return;
                        }

                        if (that._expandedRange && that.begin !== -1 && that.end !== -1 && (that._expandedRange.first.index < that.begin || that._expandedRange.last.index + 1 > that.end)) {
                            var job;

                            var promise = new Promise(function (complete) {

                                function blocksCleanupWorker(info) {
                                    if (that._listView._isZombie()) { return; }

                                    var zooming = shouldWaitForSeZo(that._listView);

                                    while (that._expandedRange.first.index < that.begin && !zooming && !info.shouldYield) {
                                        var begin = Math.min(that.begin, that._expandedRange.first.index + that._blockSize * _VirtualizeContentsView._blocksToRelease);
                                        that._forceItemsBlocksInDOM(begin, that.end);
                                    }

                                    while (that._expandedRange.last.index + 1 > that.end && !zooming && !info.shouldYield) {
                                        var end = Math.max(that.end, that._expandedRange.last.index - that._blockSize * _VirtualizeContentsView._blocksToRelease);
                                        that._forceItemsBlocksInDOM(that.begin, end);
                                    }

                                    if (that._expandedRange.first.index < that.begin || that._expandedRange.last.index + 1 > that.end) {
                                        if (zooming) {
                                            info.setPromise(waitForSeZo(that._listView).then(function () {
                                                return blocksCleanupWorker;
                                            }));
                                        } else {
                                            info.setWork(blocksCleanupWorker);
                                        }
                                    } else {
                                        complete();
                                    }
                                }

                                job = Scheduler.schedule(blocksCleanupWorker, Scheduler.Priority.belowNormal, null, "WinJS.UI.ListView._lazilyRemoveRedundantItemsBlocks");
                            });

                            return promise.then(done, function (error) {
                                job.cancel();
                                that._listView._writeProfilerMark("_lazilyRemoveRedundantItemsBlocks canceled,info");
                                that._listView._writeProfilerMark("_lazilyRemoveRedundantItemsBlocks,StopTM");
                                return Promise.wrapError(error);
                            });

                        } else {
                            done();
                            return Promise.wrap();
                        }
                    });
                },

                _forceItemsBlocksInDOM: function VirtualizeContentsView_forceItemsBlocksInDOM(begin, end) {
                    if (!this._blockSize) {
                        return;
                    }
                    var perfId = "_forceItemsBlocksInDOM begin(" + begin + ") end(" + end + "),";
                    this._listView._writeProfilerMark(perfId + "StartTM");

                    var that = this,
                        added = 0,
                        removed = 0,
                        paddingProperty = "padding" + (this._listView._horizontal() ? "Left" : "Top");

                    function setPadder(itemsContainer, padding) {
                        var padder = itemsContainer.element.firstElementChild;
                        padder.style[paddingProperty] = padding;
                    }

                    function forEachBlock(callback) {
                        for (var g = 0; g < that.tree.length; g++) {
                            var itemsContainer = that.tree[g].itemsContainer;
                            for (var b = 0, len = itemsContainer.itemsBlocks.length; b < len; b++) {
                                if (callback(itemsContainer, itemsContainer.itemsBlocks[b])) {
                                    return;
                                }
                            }
                        }
                    }

                    function measureItemsBlock(itemsBlock) {
                        that._listView._writeProfilerMark("_itemsBlockExtent,StartTM");
                        that._listView._itemsBlockExtent = _ElementUtilities[that._listView._horizontal() ? "getTotalWidth" : "getTotalHeight"](itemsBlock.element);
                        that._listView._writeProfilerMark("_itemsBlockExtent(" + that._listView._itemsBlockExtent + "),info");
                        that._listView._writeProfilerMark("_itemsBlockExtent,StopTM");
                    }

                    function getItemsBlockExtent() {
                        if (that._listView._itemsBlockExtent === -1) {
                            // first try blocks already added to the DOM
                            forEachBlock(function (itemsContainer, itemsBlock) {
                                if (itemsBlock.items.length === that._blockSize && itemsBlock.element.parentNode === itemsContainer.element) {
                                    measureItemsBlock(itemsBlock);
                                    return true;
                                }
                                return false;
                            });
                        }

                        if (that._listView._itemsBlockExtent === -1) {
                            forEachBlock(function (itemsContainer, itemsBlock) {
                                if (itemsBlock.items.length === that._blockSize) {
                                    itemsContainer.element.appendChild(itemsBlock.element);
                                    measureItemsBlock(itemsBlock);
                                    itemsContainer.element.removeChild(itemsBlock.element);
                                    return true;
                                }
                                return false;
                            });
                        }
                        return that._listView._itemsBlockExtent;
                    }

                    function removeBlocks(itemsContainer, begin, end) {

                        function remove(blockIndex) {
                            var block = itemsContainer.itemsBlocks[blockIndex];
                            if (block && block.element.parentNode === itemsContainer.element) {
                                itemsContainer.element.removeChild(block.element);
                                removed++;
                            }
                        }

                        if (Array.isArray(begin)) {
                            begin.forEach(remove);
                        } else {
                            for (var i = begin; i < end; i++) {
                                remove(i);
                            }
                        }
                    }

                    function addBlocks(itemsContainer, begin, end) {
                        var padder = itemsContainer.element.firstElementChild,
                            previous = padder;

                        for (var i = begin; i < end; i++) {
                            var block = itemsContainer.itemsBlocks[i];
                            if (block) {
                                if (block.element.parentNode !== itemsContainer.element) {
                                    itemsContainer.element.insertBefore(block.element, previous.nextElementSibling);
                                    added++;
                                }
                                previous = block.element;
                            }
                        }
                    }

                    function collapseGroup(groupIndex) {
                        if (groupIndex < that.tree.length) {
                            that._listView._writeProfilerMark("collapseGroup(" + groupIndex + "),info");
                            var itemsContainer = that.tree[groupIndex].itemsContainer;
                            removeBlocks(itemsContainer, 0, itemsContainer.itemsBlocks.length);
                            setPadder(itemsContainer, "");
                        }
                    }

                    function expandGroup(groupIndex) {
                        if (groupIndex < that.tree.length) {
                            that._listView._writeProfilerMark("expandGroup(" + groupIndex + "),info");
                            var itemsContainer = that.tree[groupIndex].itemsContainer;
                            addBlocks(itemsContainer, 0, itemsContainer.itemsBlocks.length);
                            setPadder(itemsContainer, "");
                        }
                    }

                    function removedFromRange(oldRange, newRange) {
                        function expand(first, last) {
                            var array = [];
                            for (var i = first; i <= last; i++) {
                                array.push(i);
                            }
                            return array;
                        }

                        var newL = newRange[0];
                        var newR = newRange[1];
                        var oldL = oldRange[0];
                        var oldR = oldRange[1];

                        if (newR < oldL || newL > oldR) {
                            return expand(oldL, oldR);
                        } else if (newL > oldL && newR < oldR) {
                            return expand(oldL, newL - 1).concat(expand(newR + 1, oldR));
                        } else if (oldL < newL) {
                            return expand(oldL, newL - 1);
                        } else if (oldR > newR) {
                            return expand(newR + 1, oldR);
                        } else {
                            return null;
                        }
                    }

                    var firstGroupIndex = this._listView._groups.groupFromItem(begin),
                        lastGroupIndex = this._listView._groups.groupFromItem(end - 1);

                    var firstGroup = this._listView._groups.group(firstGroupIndex),
                        firstItemsContainer = that.tree[firstGroupIndex].itemsContainer;

                    var firstBlock = Math.floor((begin - firstGroup.startIndex) / this._blockSize);

                    var lastGroup = this._listView._groups.group(lastGroupIndex),
                        lastItemsContainer = that.tree[lastGroupIndex].itemsContainer;

                    var lastBlock = Math.floor((end - 1 - lastGroup.startIndex) / this._blockSize);

                    // if size of structure block is needed try to obtain it before modifying the tree to avoid a layout pass
                    if (firstBlock && that._listView._itemsBlockExtent === -1) {
                        forEachBlock(function (itemsContainer, itemsBlock) {
                            if (itemsBlock.items.length === that._blockSize && itemsBlock.element.parentNode === itemsContainer.element) {
                                measureItemsBlock(itemsBlock);
                                return true;
                            }
                            return false;
                        });
                    }

                    var groupsToCollapse = this._expandedRange ? removedFromRange([this._expandedRange.first.groupIndex, this._expandedRange.last.groupIndex], [firstGroupIndex, lastGroupIndex]) : null;
                    if (groupsToCollapse) {
                        groupsToCollapse.forEach(collapseGroup);
                    }

                    if (this._expandedRange && this._expandedRange.first.groupKey === firstGroup.key) {
                        var blocksToRemove = removedFromRange([this._expandedRange.first.block, Number.MAX_VALUE], [firstBlock, Number.MAX_VALUE]);
                        if (blocksToRemove) {
                            removeBlocks(firstItemsContainer, blocksToRemove);
                        }
                    } else if (this._expandedRange && firstGroupIndex >= this._expandedRange.first.groupIndex && firstGroupIndex <= this._expandedRange.last.groupIndex) {
                        removeBlocks(firstItemsContainer, 0, firstBlock);
                    }

                    if (firstGroupIndex !== lastGroupIndex) {
                        addBlocks(firstItemsContainer, firstBlock, firstItemsContainer.itemsBlocks.length);
                        addBlocks(lastItemsContainer, 0, lastBlock + 1);
                    } else {
                        addBlocks(firstItemsContainer, firstBlock, lastBlock + 1);
                    }

                    if (this._expandedRange && this._expandedRange.last.groupKey === lastGroup.key) {
                        var blocksToRemove = removedFromRange([0, this._expandedRange.last.block], [0, lastBlock]);
                        if (blocksToRemove) {
                            removeBlocks(lastItemsContainer, blocksToRemove);
                        }
                    } else if (this._expandedRange && lastGroupIndex >= this._expandedRange.first.groupIndex && lastGroupIndex <= this._expandedRange.last.groupIndex) {
                        removeBlocks(lastItemsContainer, lastBlock + 1, lastItemsContainer.itemsBlocks.length);
                    }

                    setPadder(firstItemsContainer, firstBlock ? firstBlock * getItemsBlockExtent() + "px" : "");

                    if (firstGroupIndex !== lastGroupIndex) {
                        setPadder(lastItemsContainer, "");
                    }

                    // groups between first and last
                    for (var i = firstGroupIndex + 1; i < lastGroupIndex; i++) {
                        expandGroup(i);
                    }

                    this._expandedRange = {
                        first: {
                            index: begin,
                            groupIndex: firstGroupIndex,
                            groupKey: firstGroup.key,
                            block: firstBlock
                        },
                        last: {
                            index: end - 1,
                            groupIndex: lastGroupIndex,
                            groupKey: lastGroup.key,
                            block: lastBlock
                        },
                    };
                    this._listView._writeProfilerMark("_forceItemsBlocksInDOM groups(" + firstGroupIndex + "-" + lastGroupIndex + ") blocks(" + firstBlock + "-" + lastBlock + ") added(" + added + ") removed(" + removed + "),info");
                    this._listView._writeProfilerMark(perfId + "StopTM");
                },

                _realizePageImpl: function VirtualizeContentsView_realizePageImpl() {
                    var that = this;

                    var perfId = "realizePage(scrollPosition:" + this._scrollbarPos + " forceLayout:" + this._forceRelayout + ")";
                    this._listView._writeProfilerMark(perfId + ",StartTM");

                    // It's safe to skip realizePage, so we just queue up the last request to run when the version manager
                    // get unlocked.
                    //
                    if (this._listView._versionManager.locked) {
                        this._listView._versionManager.unlocked.done(function () {
                            if (!that._listView._isZombie()) {
                                that._listView._batchViewUpdates(_Constants._ViewChange.realize, _Constants._ScrollToPriority.low, that._listView.scrollPosition);
                            }
                        });
                        this._listView._writeProfilerMark(perfId + ",StopTM");
                        return Promise.cancel;
                    }

                    return new Promise(function (c) {
                        var renderingCompleteSignal = new _Signal();

                        function complete() {
                            c();
                            renderingCompleteSignal.complete();
                        }

                        function viewPortPageRealized() {
                            that._listView._hideProgressBar();
                            that._state.setLoadingState(that._listView._LoadingState.viewPortLoaded);
                            if (that._executeAnimations) {
                                that._setState(RealizingAnimatingState, renderingCompleteSignal.promise);
                            }
                        }

                        function pageRealized(count) {
                            that._updateAriaMarkers(count === 0, that.firstIndexDisplayed, that.lastIndexDisplayed);
                            that._state.setLoadingState && that._state.setLoadingState(that._listView._LoadingState.itemsLoaded);
                        }

                        function finish(count) {
                            that._listView._clearInsertedItems();
                            that._listView._groups.removeElements();
                            viewPortPageRealized();
                            pageRealized(count);
                            complete();
                        }

                        that._state.setLoadingState(that._listView._LoadingState.itemsLoading);
                        if (that._firstLayoutPass) {
                            that._listView._showProgressBar(that._listView._element, "50%", "50%");
                        }

                        var count = that.containers.length;

                        if (count) {
                            // While the zoom animation is played we want to minimize the # of pages
                            // being fetched to improve TtFF for SeZo scenarios
                            var pagesToPrefetch = that.maxLeadingPages;
                            var pagesToRetain = that.maxTrailingPages;
                            var viewportLength = that._listView._getViewportLength();
                            var pagesBefore, pagesAfter;

                            if (that._listView._zooming) {
                                pagesBefore = pagesAfter = 0;
                            } else if (_VirtualizeContentsView._disableCustomPagesPrefetch) {
                                pagesBefore = pagesAfter = _VirtualizeContentsView._defaultPagesToPrefetch;
                            } else {
                                pagesBefore = (that._direction === "left" ? pagesToPrefetch : pagesToRetain);

                                // Optimize the beginning of the list such that if you scroll, then change direction and start going back towards the beginning of the list,
                                // we maintain a remainder of pages that can be added to pagesAfter. This ensures that at beginning of the list, which is the common case,
                                // we always have pagesToPrefetch ahead, even when the scrolling direction is constantly changing.
                                var pagesShortBehind = Math.max(0, (pagesBefore - (that._scrollbarPos / viewportLength)));
                                pagesAfter = Math.min(pagesToPrefetch, pagesShortBehind + (that._direction === "right" ? pagesToPrefetch : pagesToRetain));
                            }

                            var beginningOffset = Math.max(0, that._scrollbarPos - pagesBefore * viewportLength),
                                  endingOffset = that._scrollbarPos + (1 + pagesAfter) * viewportLength;

                            var range = that._listView._layout.itemsFromRange(beginningOffset, endingOffset - 1);
                            if ((range.firstIndex < 0 || range.firstIndex >= count) && (range.lastIndex < 0 || range.lastIndex >= count)) {
                                that.begin = -1;
                                that.end = -1;
                                that.firstIndexDisplayed = -1;
                                that.lastIndexDisplayed = -1;
                                finish(count);
                            } else {
                                var begin = _ElementUtilities._clamp(range.firstIndex, 0, count - 1),
                                    end = _ElementUtilities._clamp(range.lastIndex + 1, 0, count);

                                var inView = that._listView._layout.itemsFromRange(that._scrollbarPos, that._scrollbarPos + viewportLength - 1),
                                    firstInView = _ElementUtilities._clamp(inView.firstIndex, 0, count - 1),
                                    lastInView = _ElementUtilities._clamp(inView.lastIndex, 0, count - 1);

                                if (that._realizationLevel === _VirtualizeContentsView._realizationLevel.skip && !that.lastRealizePass && firstInView === that.firstIndexDisplayed && lastInView === that.lastIndexDisplayed) {
                                    that.begin = begin;
                                    that.end = begin + Object.keys(that.items._itemData).length;
                                    that._updateHeaders(that._listView._canvas, that.begin, that.end).done(function () {
                                        that.lastRealizePass = null;
                                        finish(count);
                                    });
                                } else if ((that._forceRelayout || begin !== that.begin || end !== that.end || firstInView !== that.firstIndexDisplayed || lastInView !== that.lastIndexDisplayed) && (begin < end) && (beginningOffset < endingOffset)) {
                                    that._listView._writeProfilerMark("realizePage currentInView(" + firstInView + "-" + lastInView + ") previousInView(" + that.firstIndexDisplayed + "-" + that.lastIndexDisplayed + ") change(" + (firstInView - that.firstIndexDisplayed) + "),info");
                                    that._cancelRealize();
                                    // cancelRealize changes the realizePass and resets begin/end
                                    var currentPass = that._realizePass;
                                    that.begin = begin;
                                    that.end = end;
                                    that.firstIndexDisplayed = firstInView;
                                    that.lastIndexDisplayed = lastInView;
                                    that.deletesWithoutRealize = 0;

                                    that._ensureItemsBlocksInDOM(that.begin, that.end);

                                    var realizeWork = that._realizeItems(
                                        that._listView._itemCanvas,
                                        that.begin,
                                        that.end,
                                        count,
                                        currentPass,
                                        that._scrollbarPos,
                                        that._direction,
                                        firstInView,
                                        lastInView,
                                        that._forceRelayout);

                                    that._forceRelayout = false;

                                    var realizePassWork = realizeWork.viewportItemsRealized.then(function () {
                                        viewPortPageRealized();
                                        return realizeWork.allItemsRealized;
                                    }).then(function () {
                                        if (that._realizePass === currentPass) {
                                            return that._updateHeaders(that._listView._canvas, that.begin, that.end).then(function () {
                                                pageRealized(count);
                                            });
                                        }
                                    }).then(function () {
                                        return realizeWork.loadingCompleted;
                                    }).then(
                                        function () {
                                            that._unrealizeExcessiveItems();
                                            that.lastRealizePass = null;
                                            complete();
                                        },
                                        function (e) {
                                            if (that._realizePass === currentPass) {
                                                that.lastRealizePass = null;
                                                that.begin = -1;
                                                that.end = -1;
                                            }
                                            return Promise.wrapError(e);
                                        }
                                    );

                                    that.lastRealizePass = Promise.join([realizeWork.viewportItemsRealized, realizeWork.allItemsRealized, realizeWork.loadingCompleted, realizePassWork]);

                                    that._unrealizeExcessiveItems();

                                } else if (!that.lastRealizePass) {
                                    // We are currently in the "itemsLoading" state and need to get back to "complete". The
                                    // previous realize pass has been completed so proceed to the other states.
                                    finish(count);
                                } else {
                                    that.lastRealizePass.then(complete);
                                }
                            }
                        } else {
                            that.begin = -1;
                            that.end = -1;
                            that.firstIndexDisplayed = -1;
                            that.lastIndexDisplayed = -1;

                            finish(count);
                        }

                        that._reportElementsLevel(that._direction);

                        that._listView._writeProfilerMark(perfId + ",StopTM");
                    });
                },

                realizePage: function VirtualizeContentsView_realizePage(scrollToFunctor, forceRelayout, scrollEndPromise, StateType) {
                    this._scrollToFunctor = makeFunctor(scrollToFunctor);
                    this._forceRelayout = this._forceRelayout || forceRelayout;
                    this._scrollEndPromise = scrollEndPromise;

                    this._listView._writeProfilerMark(this._state.name + "_realizePage,info");
                    this._state.realizePage(StateType || RealizingState);
                },

                onScroll: function VirtualizeContentsView_onScroll(scrollToFunctor, scrollEndPromise) {
                    this.realizePage(scrollToFunctor, false, scrollEndPromise, ScrollingState);
                },

                reload: function VirtualizeContentsView_reload(scrollToFunctor, highPriority) {
                    if (this._listView._isZombie()) { return; }

                    this._scrollToFunctor = makeFunctor(scrollToFunctor);
                    this._forceRelayout = true;
                    this._highPriorityRealize = !!highPriority;

                    this.stopWork(true);

                    this._listView._writeProfilerMark(this._state.name + "_rebuildTree,info");
                    this._state.rebuildTree();
                },

                refresh: function VirtualizeContentsView_refresh(scrollToFunctor) {
                    if (this._listView._isZombie()) { return; }

                    this._scrollToFunctor = makeFunctor(scrollToFunctor);
                    this._forceRelayout = true;
                    this._highPriorityRealize = true;

                    this.stopWork();

                    this._listView._writeProfilerMark(this._state.name + "_relayout,info");
                    this._state.relayout();
                },

                waitForValidScrollPosition: function VirtualizeContentsView_waitForValidScrollPosition(newPosition) {
                    var that = this;
                    var currentMaxScroll = this._listView._viewport[this._listView._scrollLength] - this._listView._getViewportLength();
                    if (newPosition > currentMaxScroll) {
                        return that._listView._itemsCount().then(function (count) {
                            // Wait until we have laid out enough containers to be able to set the scroll position to newPosition
                            if (that.containers.length < count) {
                                return Promise._cancelBlocker(that._creatingContainersWork && that._creatingContainersWork.promise).then(function () {
                                    return that._getLayoutCompleted();
                                }).then(function () {
                                    return newPosition;
                                });
                            } else {
                                return newPosition;
                            }
                        });
                    } else {
                        return Promise.wrap(newPosition);
                    }
                },

                waitForEntityPosition: function VirtualizeContentsView_waitForEntityPosition(entity) {
                    var that = this;
                    if (entity.type === _UI.ObjectType.header || entity.type === _UI.ObjectType.footer) {
                        // Headers and footers are always laid out by the ListView as soon as it gets them, so there's nothing to wait on
                        return Promise.wrap();
                    }
                    this._listView._writeProfilerMark(this._state.name + "_waitForEntityPosition" + "(" + entity.type + ": " + entity.index + ")" + ",info");
                    return Promise._cancelBlocker(this._state.waitForEntityPosition(entity).then(function () {
                        if ((entity.type !== _UI.ObjectType.groupHeader && entity.index >= that.containers.length) ||
                            (entity.type === _UI.ObjectType.groupHeader && that._listView._groups.group(entity.index).startIndex >= that.containers.length)) {
                            return that._creatingContainersWork && that._creatingContainersWork.promise;
                        }
                    }).then(function () {
                        return that._getLayoutCompleted();
                    }));
                },

                stopWork: function VirtualizeContentsView_stopWork(stopTreeCreation) {
                    this._listView._writeProfilerMark(this._state.name + "_stop,info");
                    this._state.stop(stopTreeCreation);

                    if (this._layoutWork) {
                        this._layoutWork.cancel();
                    }

                    if (stopTreeCreation && this._creatingContainersWork) {
                        this._creatingContainersWork.cancel();
                    }

                    if (stopTreeCreation) {
                        this._state = new CreatedState(this);
                    }
                },

                _cancelRealize: function VirtualizeContentsView_cancelRealize() {
                    this._listView._writeProfilerMark("_cancelRealize,StartTM");

                    if (this.lastRealizePass || this.deferTimeout) {
                        this._forceRelayout = true;
                    }

                    this._clearDeferTimeout();
                    this._realizePass++;

                    if (this._headerRenderPromises) {
                        this._headerRenderPromises.cancel();
                        this._headerRenderPromises = null;
                    }

                    var last = this.lastRealizePass;
                    if (last) {
                        this.lastRealizePass = null;
                        this.begin = -1;
                        this.end = -1;
                        last.cancel();
                    }
                    this._listView._writeProfilerMark("_cancelRealize,StopTM");
                },

                resetItems: function VirtualizeContentsView_resetItems(unparent) {
                    if (!this._listView._isZombie()) {
                        this.firstIndexDisplayed = -1;
                        this.lastIndexDisplayed = -1;
                        this._runningAnimations = null;
                        this._executeAnimations = false;

                        var listView = this._listView;
                        this._firstLayoutPass = true;
                        listView._unsetFocusOnItem();
                        if (listView._currentMode().onDataChanged) {
                            listView._currentMode().onDataChanged();
                        }

                        this.items.each(function (index, item) {
                            if (unparent && item.parentNode && item.parentNode.parentNode) {
                                item.parentNode.parentNode.removeChild(item.parentNode);
                            }
                            listView._itemsManager.releaseItem(item);
                            _Dispose._disposeElement(item);
                        });

                        this.items.removeItems();
                        this._deferredReparenting = [];

                        if (unparent) {
                            listView._groups.removeElements();
                        }

                        listView._clearInsertedItems();
                    }
                },

                reset: function VirtualizeContentsView_reset() {
                    this.stopWork(true);
                    this._state = new CreatedState(this);

                    this.resetItems();

                    // when in the zombie state, we let disposal cleanup the ScrollView state
                    //
                    if (!this._listView._isZombie()) {
                        var listView = this._listView;
                        listView._groups.resetGroups();
                        listView._resetCanvas();

                        this.tree = null;
                        this.keyToGroupIndex = null;
                        this.containers = null;
                        this._expandedRange = null;
                    }
                },

                cleanUp: function VirtualizeContentsView_cleanUp() {
                    this.stopWork(true);

                    this._runningAnimations && this._runningAnimations.cancel();
                    var itemsManager = this._listView._itemsManager;
                    this.items.each(function (index, item) {
                        itemsManager.releaseItem(item);
                        _Dispose._disposeElement(item);
                    });
                    this._listView._unsetFocusOnItem();
                    this.items.removeItems();
                    this._deferredReparenting = [];
                    this._listView._groups.resetGroups();
                    this._listView._resetCanvas();

                    this.tree = null;
                    this.keyToGroupIndex = null;
                    this.containers = null;
                    this._expandedRange = null;

                    this.destroyed = true;
                },

                getContainer: function VirtualizeContentsView_getContainer(itemIndex) {
                    return this.containers[itemIndex];
                },

                _getHeaderContainer: function VirtualizeContentsView_getHeaderContainer(groupIndex) {
                    return this.tree[groupIndex].header;
                },

                _getGroups: function VirtualizeContentsView_getGroups(count) {
                    if (this._listView._groupDataSource) {
                        var groupsContainer = this._listView._groups.groups,
                            groups = [];
                        if (count) {
                            for (var i = 0, len = groupsContainer.length; i < len; i++) {
                                var group = groupsContainer[i],
                                    nextStartIndex = i + 1 < len ? groupsContainer[i + 1].startIndex : count;
                                groups.push({
                                    key: group.key,
                                    size: nextStartIndex - group.startIndex
                                });
                            }
                        }
                        return groups;
                    } else {
                        return [{ key: "-1", size: count }];
                    }
                },

                // Overridden by tests.
                // Tests should have _createChunk return true when they want _createContainers to stop creating containers.
                _createChunk: function VirtualizeContentsView_createChunk(groups, count, chunkSize) {
                    var that = this;

                    this._listView._writeProfilerMark("createChunk,StartTM");

                    function addToGroup(itemsContainer, groupSize) {
                        var children = itemsContainer.element.children,
                            oldSize = children.length,
                            toAdd = Math.min(groupSize - itemsContainer.items.length, chunkSize);

                        _SafeHtml.insertAdjacentHTMLUnsafe(itemsContainer.element, "beforeend", _Helpers._repeat("<div class='win-container win-backdrop'></div>", toAdd));

                        for (var i = 0; i < toAdd; i++) {
                            var container = children[oldSize + i];
                            itemsContainer.items.push(container);
                            that.containers.push(container);
                        }
                    }

                    function newGroup(group) {
                        var node = {
                            header: that._listView._groupDataSource ? that._createHeaderContainer() : null,
                            itemsContainer: {
                                element: that._createItemsContainer(),
                                items: []
                            }
                        };


                        that.tree.push(node);
                        that.keyToGroupIndex[group.key] = that.tree.length - 1;
                        addToGroup(node.itemsContainer, group.size);
                    }

                    if (this.tree.length && this.tree.length <= groups.length) {
                        var last = this.tree[this.tree.length - 1],
                            finalSize = groups[this.tree.length - 1].size;

                        // check if the last group in the tree already has all items. If not add items to this group
                        if (last.itemsContainer.items.length < finalSize) {
                            addToGroup(last.itemsContainer, finalSize);
                            this._listView._writeProfilerMark("createChunk,StopTM");
                            return;
                        }
                    }

                    if (this.tree.length < groups.length) {
                        newGroup(groups[this.tree.length]);
                    }

                    this._listView._writeProfilerMark("createChunk,StopTM");
                },

                // Overridden by tests.
                // Tests should have _createChunkWithBlocks return true when they want _createContainers to stop creating containers.
                _createChunkWithBlocks: function VirtualizeContentsView_createChunkWithBlocks(groups, count, blockSize, chunkSize) {
                    var that = this;
                    this._listView._writeProfilerMark("createChunk,StartTM");

                    function addToGroup(itemsContainer, toAdd) {
                        var indexOfNextGroupItem;
                        var lastExistingBlock = itemsContainer.itemsBlocks.length ? itemsContainer.itemsBlocks[itemsContainer.itemsBlocks.length - 1] : null;

                        toAdd = Math.min(toAdd, chunkSize);

                        // 1) Add missing containers to the latest itemsblock if it was only partially filled during the previous pass.
                        if (lastExistingBlock && lastExistingBlock.items.length < blockSize) {
                            var emptySpotsToFill = Math.min(toAdd, blockSize - lastExistingBlock.items.length),
                                sizeOfOldLastBlock = lastExistingBlock.items.length,

                            indexOfNextGroupItem = (itemsContainer.itemsBlocks.length - 1) * blockSize + sizeOfOldLastBlock;
                            var containersMarkup = _Helpers._stripedContainers(emptySpotsToFill, indexOfNextGroupItem);

                            _SafeHtml.insertAdjacentHTMLUnsafe(lastExistingBlock.element, "beforeend", containersMarkup);
                            children = lastExistingBlock.element.children;

                            for (var j = 0; j < emptySpotsToFill; j++) {
                                var child = children[sizeOfOldLastBlock + j];
                                lastExistingBlock.items.push(child);
                                that.containers.push(child);
                            }

                            toAdd -= emptySpotsToFill;
                        }
                        indexOfNextGroupItem = itemsContainer.itemsBlocks.length * blockSize;

                        // 2) Generate as many full itemblocks of containers as we can.
                        var newBlocksCount = Math.floor(toAdd / blockSize),
                            markup = "",
                            firstBlockFirstItemIndex = indexOfNextGroupItem,
                            secondBlockFirstItemIndex = indexOfNextGroupItem + blockSize;

                        if (newBlocksCount > 0) {
                            var pairOfItemBlocks = [
                                // Use pairs to ensure that the container striping pattern is maintained regardless if blockSize is even or odd.
                                "<div class='win-itemsblock'>" + _Helpers._stripedContainers(blockSize, firstBlockFirstItemIndex) + "</div>",
                                "<div class='win-itemsblock'>" + _Helpers._stripedContainers(blockSize, secondBlockFirstItemIndex) + "</div>"
                            ];
                            markup = _Helpers._repeat(pairOfItemBlocks, newBlocksCount);
                            indexOfNextGroupItem += (newBlocksCount * blockSize);
                        }

                        // 3) Generate and partially fill, one last itemblock if there are any remaining containers to add.
                        var sizeOfNewLastBlock = toAdd % blockSize;
                        if (sizeOfNewLastBlock > 0) {
                            markup += "<div class='win-itemsblock'>" + _Helpers._stripedContainers(sizeOfNewLastBlock, indexOfNextGroupItem) + "</div>";
                            indexOfNextGroupItem += sizeOfNewLastBlock;
                            newBlocksCount++;
                        }

                        var blocksTemp = _Global.document.createElement("div");
                        _SafeHtml.setInnerHTMLUnsafe(blocksTemp, markup);
                        var children = blocksTemp.children;

                        for (var i = 0; i < newBlocksCount; i++) {
                            var block = children[i],
                                blockNode = {
                                    element: block,
                                    items: _Helpers._nodeListToArray(block.children)
                                };
                            itemsContainer.itemsBlocks.push(blockNode);
                            for (var n = 0; n < blockNode.items.length; n++) {
                                that.containers.push(blockNode.items[n]);
                            }
                        }
                    }

                    function newGroup(group) {
                        var node = {
                            header: that._listView._groupDataSource ? that._createHeaderContainer() : null,
                            itemsContainer: {
                                element: that._createItemsContainer(),
                                itemsBlocks: []
                            }
                        };

                        that.tree.push(node);
                        that.keyToGroupIndex[group.key] = that.tree.length - 1;

                        addToGroup(node.itemsContainer, group.size);
                    }

                    if (this.tree.length && this.tree.length <= groups.length) {
                        var lastContainer = this.tree[this.tree.length - 1].itemsContainer,
                            finalSize = groups[this.tree.length - 1].size,
                            currentSize = 0;

                        if (lastContainer.itemsBlocks.length) {
                            currentSize = (lastContainer.itemsBlocks.length - 1) * blockSize + lastContainer.itemsBlocks[lastContainer.itemsBlocks.length - 1].items.length;
                        }

                        if (currentSize < finalSize) {
                            addToGroup(lastContainer, finalSize - currentSize);
                            this._listView._writeProfilerMark("createChunk,StopTM");
                            return;
                        }
                    }

                    if (this.tree.length < groups.length) {
                        newGroup(groups[this.tree.length]);
                    }

                    this._listView._writeProfilerMark("createChunk,StopTM");
                },

                _generateCreateContainersWorker: function VirtualizeContentsView_generateCreateContainersWorker() {
                    var that = this,
                        counter = 0,
                        skipWait = false;

                    return function work(info) {
                        if (!that._listView._versionManager.locked) {
                            that._listView._itemsCount().then(function (count) {
                                var zooming = !skipWait && shouldWaitForSeZo(that._listView);

                                if (!zooming) {
                                    if (that._listView._isZombie()) { return; }

                                    skipWait = false;

                                    var end = _BaseUtils._now() + _VirtualizeContentsView._createContainersJobTimeslice,
                                        groups = that._getGroups(count),
                                        startLength = that.containers.length,
                                        realizedToEnd = that.end === that.containers.length,
                                        chunkSize = _VirtualizeContentsView._chunkSize;

                                    do {
                                        that._blockSize ? that._createChunkWithBlocks(groups, count, that._blockSize, chunkSize) : that._createChunk(groups, count, chunkSize);
                                        counter++;
                                    } while (that.containers.length < count && _BaseUtils._now() < end);

                                    that._listView._writeProfilerMark("createContainers yields containers(" + that.containers.length + "),info");

                                    that._listView._affectedRange.add({ start: startLength, end: that.containers.length }, count);

                                    if (realizedToEnd) {
                                        that.stopWork();
                                        that._listView._writeProfilerMark(that._state.name + "_relayout,info");
                                        that._state.relayout();
                                    } else {
                                        that._listView._writeProfilerMark(that._state.name + "_layoutNewContainers,info");
                                        that._state.layoutNewContainers();
                                    }

                                    if (that.containers.length < count) {
                                        info.setWork(work);
                                    } else {
                                        that._listView._writeProfilerMark("createContainers completed steps(" + counter + "),info");
                                        that._creatingContainersWork.complete();
                                    }
                                } else {
                                    // Waiting on zooming
                                    info.setPromise(waitForSeZo(that._listView).then(function (timedOut) {
                                        skipWait = timedOut;
                                        return work;
                                    }));
                                }
                            });
                        } else {
                            // Version manager locked
                            info.setPromise(that._listView._versionManager.unlocked.then(function () {
                                return work;
                            }));
                        }
                    };
                },

                _scheduleLazyTreeCreation: function VirtualizeContentsView_scheduleLazyTreeCreation() {
                    return Scheduler.schedule(this._generateCreateContainersWorker(), Scheduler.Priority.idle, this, "WinJS.UI.ListView.LazyTreeCreation");
                },

                _createContainers: function VirtualizeContentsView_createContainers() {
                    this.tree = null;
                    this.keyToGroupIndex = null;
                    this.containers = null;
                    this._expandedRange = null;

                    var that = this,
                        count;

                    return this._listView._itemsCount().then(function (c) {
                        if (c === 0) {
                            that._listView._hideProgressBar();
                        }
                        count = c;
                        that._listView._writeProfilerMark("createContainers(" + count + "),StartTM");
                        if (that._listView._groupDataSource) {
                            return that._listView._groups.initialize();
                        }
                    }).then(function () {
                        that._listView._writeProfilerMark("numberOfItemsPerItemsBlock,StartTM");
                        return (count && that._listView._groups.length() ? that._listView._layout.numberOfItemsPerItemsBlock : null);
                    }).then(function (blockSize) {
                        that._listView._writeProfilerMark("numberOfItemsPerItemsBlock(" + blockSize + "),info");
                        that._listView._writeProfilerMark("numberOfItemsPerItemsBlock,StopTM");

                        that._listView._resetCanvas();

                        that.tree = [];
                        that.keyToGroupIndex = {};
                        that.containers = [];
                        that._blockSize = blockSize;

                        var groups = that._getGroups(count);

                        var end = _BaseUtils._now() + _VirtualizeContentsView._maxTimePerCreateContainers,
                            chunkSize = Math.min(_VirtualizeContentsView._startupChunkSize, _VirtualizeContentsView._chunkSize);
                        var stop;
                        do {
                            // Tests override _createChunk/_createChunkWithBlocks and take advantage of its boolean return value
                            // to stop initial container creation after a certain number of containers have been created.
                            stop = blockSize ? that._createChunkWithBlocks(groups, count, blockSize, chunkSize) : that._createChunk(groups, count, chunkSize);
                        } while (_BaseUtils._now() < end && that.containers.length < count && !stop);

                        that._listView._writeProfilerMark("createContainers created(" + that.containers.length + "),info");

                        that._listView._affectedRange.add({ start: 0, end: that.containers.length }, count);

                        if (that.containers.length < count) {
                            var jobNode = that._scheduleLazyTreeCreation();

                            that._creatingContainersWork.promise.done(null, function () {
                                jobNode.cancel();
                            });
                        } else {
                            that._listView._writeProfilerMark("createContainers completed synchronously,info");
                            that._creatingContainersWork.complete();
                        }

                        that._listView._writeProfilerMark("createContainers(" + count + "),StopTM");
                    });
                },

                _updateItemsBlocks: function VirtualizeContentsView_updateItemsBlocks(blockSize) {
                    var that = this;
                    var usingStructuralNodes = !!blockSize;

                    function createNewBlock() {
                        var element = _Global.document.createElement("div");
                        element.className = _Constants._itemsBlockClass;
                        return element;
                    }

                    function updateGroup(itemsContainer, startIndex) {
                        var blockElements = [],
                            itemsCount = 0,
                            blocks = itemsContainer.itemsBlocks,
                            b;

                        function rebuildItemsContainer() {
                            itemsContainer.itemsBlocks = null;
                            itemsContainer.items = [];
                            for (var i = 0; i < itemsCount; i++) {
                                var container = that.containers[startIndex + i];
                                itemsContainer.element.appendChild(container);
                                itemsContainer.items.push(container);
                            }
                        }

                        function rebuildItemsContainerWithBlocks() {
                            itemsContainer.itemsBlocks = [{
                                element: blockElements.length ? blockElements.shift() : createNewBlock(),
                                items: []
                            }];
                            var currentBlock = itemsContainer.itemsBlocks[0];
                            for (var i = 0; i < itemsCount; i++) {
                                if (currentBlock.items.length === blockSize) {
                                    var nextBlock = blockElements.length ? blockElements.shift() : createNewBlock();
                                    itemsContainer.itemsBlocks.push({
                                        element: nextBlock,
                                        items: []
                                    });
                                    currentBlock = itemsContainer.itemsBlocks[itemsContainer.itemsBlocks.length - 1];
                                }

                                var container = that.containers[startIndex + i];
                                currentBlock.element.appendChild(container);
                                currentBlock.items.push(container);
                            }
                            itemsContainer.items = null;
                        }

                        if (blocks) {
                            for (b = 0; b < blocks.length; b++) {
                                itemsCount += blocks[b].items.length;
                                blockElements.push(blocks[b].element);
                            }
                        } else {
                            itemsCount = itemsContainer.items.length;
                        }

                        if (usingStructuralNodes) {
                            rebuildItemsContainerWithBlocks();
                        } else {
                            rebuildItemsContainer();
                        }

                        for (b = 0; b < blockElements.length; b++) {
                            var block = blockElements[b];
                            if (block.parentNode === itemsContainer.element) {
                                itemsContainer.element.removeChild(block);
                            }
                        }

                        return itemsCount;
                    }

                    for (var g = 0, startIndex = 0; g < this.tree.length; g++) {
                        startIndex += updateGroup(this.tree[g].itemsContainer, startIndex);
                    }

                    that._blockSize = blockSize;
                },

                _layoutItems: function VirtualizeContentsView_layoutItems() {
                    var that = this;
                    return this._listView._itemsCount().then(function () {
                        return Promise.as(that._listView._layout.numberOfItemsPerItemsBlock).then(function (blockSize) {
                            that._listView._writeProfilerMark("numberOfItemsPerItemsBlock(" + blockSize + "),info");
                            if (blockSize !== that._blockSize) {
                                that._updateItemsBlocks(blockSize);
                                that._listView._itemsBlockExtent = -1;
                            }

                            var affectedRange = that._listView._affectedRange.get();
                            var changedRange;

                            // We accumulate all changes that occur between layouts in _affectedRange. If layout is interrupted due to additional
                            // modifications, _affectedRange will become the union of the previous range of changes and the new range of changes
                            // and will be passed to layout again. _affectedRange is reset whenever layout completes.
                            if (affectedRange) {
                                changedRange = {
                                    // _affectedRange is stored in the format [start, end), layout expects a range in the form of [firstIndex , lastIndex]
                                    // To ensure that layout can successfully use the expected range to find all of the groups which need to be re-laid out
                                    // we will pad an extra index at the front end such that layout receives [start - 1, end] in form of [lastIndex, firstIndex].
                                    firstIndex: Math.max(affectedRange.start - 1, 0),
                                    lastIndex: Math.min(that.containers.length - 1, affectedRange.end) // Account for any constrained upper limits from lazily loaded win-container's.
                                };
                                if (changedRange.firstIndex < that.containers.length || that.containers.length === 0) {
                                    return that._listView._layout.layout(that.tree, changedRange,
                                        that._modifiedElements || [], that._modifiedGroups || []);
                                }
                            }

                            // There is nothing to layout.
                            that._listView._affectedRange.clear();
                            return {
                                realizedRangeComplete: Promise.wrap(),
                                layoutComplete: Promise.wrap()
                            };
                        });
                    });
                },

                updateTree: function VirtualizeContentsView_updateTree(count, delta, modifiedElements) {
                    this._listView._writeProfilerMark(this._state.name + "_updateTree,info");
                    return this._state.updateTree(count, delta, modifiedElements);
                },

                _updateTreeImpl: function VirtualizeContentsView_updateTreeImpl(count, delta, modifiedElements, skipUnrealizeItems) {
                    this._executeAnimations = true;
                    this._modifiedElements = modifiedElements;

                    if (modifiedElements.handled) {
                        return;
                    }
                    modifiedElements.handled = true;

                    this._listView._writeProfilerMark("_updateTreeImpl,StartTM");

                    var that = this,
                        i;

                    if (!skipUnrealizeItems) {
                        // If we skip unrealize items, this work will eventually happen when we reach the UnrealizingState. Sometimes,
                        // it is appropriate to defer the unrealize work in order to optimize scenarios (e.g, edits that happen when we are
                        // in the CompletedState, that way the animation can start sooner).
                        this._unrealizeItems();
                    }

                    function removeElements(array) {
                        for (var i = 0, len = array.length; i < len; i++) {
                            var itemBox = array[i];
                            itemBox.parentNode.removeChild(itemBox);
                        }
                    }

                    for (var i = 0, len = modifiedElements.length; i < len; i++) {
                        if (modifiedElements[i]._itemBox && modifiedElements[i]._itemBox.parentNode) {
                            _ElementUtilities.removeClass(modifiedElements[i]._itemBox.parentNode, _Constants._selectedClass);
                        }
                    }

                    this.items.each(function (index, item, itemData) {
                        itemData.container && _ElementUtilities.removeClass(itemData.container, _Constants._selectedClass);
                        itemData.container && _ElementUtilities.addClass(itemData.container, _Constants._backdropClass);
                    });

                    var removedGroups = this._listView._updateContainers(this._getGroups(count), count, delta, modifiedElements);

                    removeElements(removedGroups.removedHeaders);
                    removeElements(removedGroups.removedItemsContainers);

                    for (var i = 0, len = modifiedElements.length; i < len; i++) {
                        var modifiedElement = modifiedElements[i];
                        if (modifiedElement.newIndex !== -1) {
                            modifiedElement.element = this.getContainer(modifiedElement.newIndex);
                            if (!modifiedElement.element) {
                                throw "Container missing after updateContainers.";
                            }
                        } else {
                            _ElementUtilities.removeClass(modifiedElement.element, _Constants._backdropClass);
                        }
                    }

                    // We only need to restore focus if the current focus is within surface
                    var activeElement = _Global.document.activeElement;
                    if (this._listView._canvas.contains(activeElement)) {
                        this._requireFocusRestore = activeElement;
                    }

                    this._deferredReparenting = [];
                    this.items.each(function (index, item, itemData) {
                        var container = that.getContainer(index),
                            itemBox = itemData.itemBox;

                        if (itemBox && container) {
                            itemData.container = container;
                            if (itemBox.parentNode !== container) {
                                if (index >= that.firstIndexDisplayed && index <= that.lastIndexDisplayed) {
                                    that._appendAndRestoreFocus(container, itemBox);
                                } else {
                                    that._deferredReparenting.push({ itemBox: itemBox, container: container });
                                }
                            }
                            _ElementUtilities.removeClass(container, _Constants._backdropClass);

                            _ElementUtilities[that._listView.selection._isIncluded(index) ? "addClass" : "removeClass"](container, _Constants._selectedClass);
                            if (!that._listView.selection._isIncluded(index) && _ElementUtilities.hasClass(itemBox, _Constants._selectedClass)) {
                                _ItemEventsHandler._ItemEventsHandler.renderSelection(itemBox, itemData.element, false, true);
                            }
                        }
                    });

                    this._listView._writeProfilerMark("_updateTreeImpl,StopTM");
                },

                _completeUpdateTree: function () {
                    if (this._deferredReparenting) {
                        var deferredCount = this._deferredReparenting.length;
                        if (deferredCount > 0) {
                            var perfId = "_completeReparenting(" + deferredCount + ")";
                            this._listView._writeProfilerMark(perfId + ",StartTM");
                            var deferredItem;
                            for (var i = 0; i < deferredCount; i++) {
                                deferredItem = this._deferredReparenting[i];
                                this._appendAndRestoreFocus(deferredItem.container, deferredItem.itemBox);
                            }
                            this._deferredReparenting = [];
                            this._listView._writeProfilerMark(perfId + ",StopTM");
                        }
                    }
                    this._requireFocusRestore = null;
                },

                _appendAndRestoreFocus: function VirtualizeContentsView_appendAndRestoreFocus(container, itemBox) {
                    if (itemBox.parentNode !== container) {
                        var activeElement;
                        if (this._requireFocusRestore) {
                            activeElement = _Global.document.activeElement;
                        }

                        if (this._requireFocusRestore && this._requireFocusRestore === activeElement && (container.contains(activeElement) || itemBox.contains(activeElement))) {
                            this._listView._unsetFocusOnItem();
                            activeElement = _Global.document.activeElement;
                        }

                        _ElementUtilities.empty(container);
                        container.appendChild(itemBox);

                        if (this._requireFocusRestore && activeElement === this._listView._keyboardEventsHelper) {
                            var focused = this._listView._selection._getFocused();
                            if (focused.type === _UI.ObjectType.item && this.items.itemBoxAt(focused.index) === itemBox) {
                                _ElementUtilities._setActive(this._requireFocusRestore);
                                this._requireFocusRestore = null;
                            }
                        }
                    }
                },

                _startAnimations: function VirtualizeContentsView_startAnimations() {
                    this._listView._writeProfilerMark("startAnimations,StartTM");

                    var that = this;
                    this._hasAnimationInViewportPending = false;
                    var animationPromise = Promise.as(this._listView._layout.executeAnimations()).then(function () {
                        that._listView._writeProfilerMark("startAnimations,StopTM");
                    });
                    return animationPromise;
                },

                _setState: function VirtualizeContentsView_setState(NewStateType, arg) {
                    if (!this._listView._isZombie()) {
                        var prevStateName = this._state.name;
                        this._state = new NewStateType(this, arg);
                        this._listView._writeProfilerMark(this._state.name + "_enter from(" + prevStateName + "),info");
                        this._state.enter();
                    }
                },

                getAdjacent: function VirtualizeContentsView_getAdjacent(currentFocus, direction) {
                    var that = this;
                    return this.waitForEntityPosition(currentFocus).then(function () {
                        return that._listView._layout.getAdjacent(currentFocus, direction);
                    });
                },

                hitTest: function VirtualizeContentsView_hitTest(x, y) {
                    if (!this._realizedRangeLaidOut) {
                        var retVal = this._listView._layout.hitTest(x, y);
                        retVal.index = _ElementUtilities._clamp(retVal.index, -1, this._listView._cachedCount - 1, 0);
                        retVal.insertAfterIndex = _ElementUtilities._clamp(retVal.insertAfterIndex, -1, this._listView._cachedCount - 1, 0);
                        return retVal;
                    } else {
                        return {
                            index: -1,
                            insertAfterIndex: -1
                        };
                    }
                },

                _createTreeBuildingSignal: function VirtualizeContentsView__createTreeBuildingSignal() {
                    if (!this._creatingContainersWork) {
                        this._creatingContainersWork = new _Signal();

                        var that = this;
                        this._creatingContainersWork.promise.done(
                            function () {
                                that._creatingContainersWork = null;
                            },
                            function () {
                                that._creatingContainersWork = null;
                            }
                        );
                    }
                },

                _createLayoutSignal: function VirtualizeContentsView_createLayoutSignal() {
                    var that = this;

                    if (!this._layoutCompleted) {
                        this._layoutCompleted = new _Signal();

                        this._layoutCompleted.promise.done(
                            function () {
                                that._layoutCompleted = null;
                            },
                            function () {
                                that._layoutCompleted = null;
                            }
                        );
                    }

                    if (!this._realizedRangeLaidOut) {
                        this._realizedRangeLaidOut = new _Signal();
                        this._realizedRangeLaidOut.promise.done(
                            function () {
                                that._realizedRangeLaidOut = null;
                            },
                            function () {
                                that._realizedRangeLaidOut = null;
                            }
                        );
                    }
                },

                _getLayoutCompleted: function VirtualizeContentsView_getLayoutCompleted() {
                    return this._layoutCompleted ? Promise._cancelBlocker(this._layoutCompleted.promise) : Promise.wrap();
                },

                _createSurfaceChild: function VirtualizeContentsView_createSurfaceChild(className, insertAfter) {
                    var element = _Global.document.createElement("div");
                    element.className = className;
                    this._listView._canvas.insertBefore(element, insertAfter ? insertAfter.nextElementSibling : null);
                    return element;
                },

                _executeScrollToFunctor: function VirtualizeContentsView_executeScrollToFunctor() {
                    var that = this;
                    return Promise.as(this._scrollToFunctor ? this._scrollToFunctor() : null).then(function (scroll) {
                        that._scrollToFunctor = null;

                        scroll = scroll || {};
                        // _scrollbarPos is initialized to 0 in the constructor, and we only set it when a valid integer
                        // value is passed in order to account for cases when there is not a _scrollToFunctor
                        if (+scroll.position === scroll.position) {
                            that._scrollbarPos = scroll.position;
                        }
                        that._direction = scroll.direction || "right";
                    });
                }
            }, {
                _defaultPagesToPrefetch: 2,
                _iOSMaxLeadingPages: 6,
                _iOSMaxTrailingPages: 2,
                _disableCustomPagesPrefetch: false,
                _waitForSeZoIntervalDuration: 100,
                _waitForSeZoTimeoutDuration: 500,
                _chunkSize: 500,
                _startupChunkSize: 100,
                _maxTimePerCreateContainers: 5,
                _createContainersJobTimeslice: 15,
                _blocksToRelease: 10,
                _realizationLevel: {
                    skip: "skip",
                    realize: "realize",
                    normal: "normal"
                }
            });


            function nop() { }

            /*
            View is in this state before reload is called so during startup, after datasource change etc.
            */

            var CreatedState = _Base.Class.define(function CreatedState_ctor(view) {
                this.view = view;
                this.view._createTreeBuildingSignal();
                this.view._createLayoutSignal();
            }, {
                name: 'CreatedState',
                enter: function CreatedState_enter() {
                    this.view._createTreeBuildingSignal();
                    this.view._createLayoutSignal();
                },
                stop: nop,
                realizePage: nop,
                rebuildTree: function CreatedState_rebuildTree() {
                    this.view._setState(BuildingState);
                },
                relayout: function CreatedState_relayout() {
                    this.view._setState(BuildingState);
                },
                layoutNewContainers: nop,
                waitForEntityPosition: function CreatedState_waitForEntityPosition() {
                    this.view._setState(BuildingState);
                    return this.view._getLayoutCompleted();
                },
                updateTree: nop
            });

            /*
            In this state View is building its DOM tree with win-container element for each item in the data set.
            To build the tree the view needs to know items count or for grouped case the count of groups and the
            count of items in each group. The view enters this state when the tree needs to be built during
            startup or rebuild after data source change and etc.

            BuildingState => LayingoutState | CreatedState
            */
            var BuildingState = _Base.Class.define(function BuildingState_ctor(view) {
                this.view = view;
            }, {
                name: 'BuildingState',
                enter: function BuildingState_enter() {
                    this.canceling = false;
                    this.view._createTreeBuildingSignal();
                    this.view._createLayoutSignal();

                    var that = this;

                    // Use a signal to guarantee that this.promise is set before the promise
                    // handler is executed.
                    var promiseStoredSignal = new _Signal();
                    this.promise = promiseStoredSignal.promise.then(function () {
                        return that.view._createContainers();
                    }).then(
                        function () {
                            that.view._setState(LayingoutState);
                        },
                        function (error) {
                            if (!that.canceling) {
                                // this is coming from layout. ListView is hidden. We need to raise complete and wait in initial state for further actions
                                that.view._setState(CreatedState);
                                that.view._listView._raiseViewComplete();
                            }
                            return Promise.wrapError(error);
                        }
                    );
                    promiseStoredSignal.complete();
                },
                stop: function BuildingState_stop() {
                    this.canceling = true;
                    this.promise.cancel();
                    this.view._setState(CreatedState);
                },
                realizePage: nop,
                rebuildTree: function BuildingState_rebuildTree() {
                    this.canceling = true;
                    this.promise.cancel();
                    this.enter();
                },
                relayout: nop,
                layoutNewContainers: nop,
                waitForEntityPosition: function BuildingState_waitForEntityPosition() {
                    return this.view._getLayoutCompleted();
                },
                updateTree: nop
            });

            /*
            In this state View waits for the layout to lay out win-container elements. The view enters this state
            after edits or resize.

            LayingoutState => RealizingState | BuildingState | CanceledState | CompletedState | LayoutCanceledState
            */
            var LayingoutState = _Base.Class.define(function LayingoutState_ctor(view, NextStateType) {
                this.view = view;
                this.nextStateType = NextStateType || RealizingState;
            }, {
                name: 'LayingoutState',
                enter: function LayingoutState_enter() {
                    var that = this;
                    this.canceling = false;
                    this.view._createLayoutSignal();

                    this.view._listView._writeProfilerMark(this.name + "_enter_layoutItems,StartTM");

                    // Use a signal to guarantee that this.promise is set before the promise
                    // handler is executed.
                    var promiseStoredSignal = new _Signal();
                    this.promise = promiseStoredSignal.promise.then(function () {
                        return that.view._layoutItems();
                    }).then(function (layoutPromises) {

                        // View is taking ownership of this promise and it will cancel it in stopWork
                        that.view._layoutWork = layoutPromises.layoutComplete;

                        return layoutPromises.realizedRangeComplete;
                    }).then(
                        function () {
                            that.view._listView._writeProfilerMark(that.name + "_enter_layoutItems,StopTM");

                            that.view._listView._clearInsertedItems();
                            that.view._setAnimationInViewportState(that.view._modifiedElements);
                            that.view._modifiedElements = [];
                            that.view._modifiedGroups = [];

                            that.view._realizedRangeLaidOut.complete();

                            that.view._layoutWork.then(function () {
        